/*
** Copyright 1998 - 2004 Double Precision, Inc.
** See COPYING for distribution information.
*/

#include	"config.h"
#include	"submit.h"
#include	"bofh.h"
#include	"rw.h"
#include	"maxlongsize.h"
#include	"courier.h"
#include	"libexecdir.h"
#include	"comreadtime.h"
#include	"comctlfile.h"
#include	"cdfilters.h"
#include	"localstatedir.h"
#include	"sysconfdir.h"
#include	"numlib/numlib.h"
#include	"rfc822/rfc822hdr.h"
#include	"rfc2045/rfc2045.h"
#include	"rfc2045/rfc2045charset.h"
#include	<stdio.h>
#include	<string.h>
#include	<fcntl.h>
#include	<errno.h>
#include	<iostream>
#include	<iomanip>
#if HAVE_SYS_TYPES_H
#include	<sys/types.h>
#endif
#include        <sys/uio.h>
#if HAVE_SYS_WAIT_H
#include	<sys/wait.h>
#endif
#include	<signal.h>
#include	<stdlib.h>

#define	SPILLLEVEL	100

extern CString security;
extern time_t	submit_time;
extern const char *submitdelay;
extern int checkfreespace(unsigned *);
SubmitFile *SubmitFile::current_submit_file=0;

static time_t	queuetime, faxqueuetime, warntime;
static unsigned batchsize=100;
static int spamtrap_flag=0;

extern int verpflag;

//
// Messages are owned by the submitting user, but MAILGID must still be
// able to read/write them!
//

#define	PERMISSION	0660

SubmitFile::SubmitFile() : rwrfcptr(0)
{
}

SubmitFile::~SubmitFile()
{
	interrupt();
	current_submit_file=0;
	if (rwrfcptr)	rfc2045_free(rwrfcptr);
}

// Start the ball rolling by specifying the envelope sender.

void SubmitFile::Sender(const char *f,
	const char *p, const char *e, char t)
{
char buf[2];

	frommta=f;
	num_control_files_created=0;	// No control files created yet
	rcptcount=0;
	sender=p;
	envid= (e ? e:"");

       	buf[0]=t;
       	buf[1]=0;
       	dsnformat=buf;

	addrlist_map.RemoveAll();
	addrlist_gdbm.Close();
	// Read various configuration parameters

	queuetime=config_time_queuetime();
	faxqueuetime=config_time_faxqueuetime();
	warntime=config_time_warntime();
	batchsize=atoi(config_batchsize());
}

// Close the current to-be control file.

void SubmitFile::closectl()
{
	if (security.GetLength() > 0)
		ctlfile << COMCTLFILE_SECURITY << (const char *)security
			<< endl;
	if (verpflag)
		ctlfile << COMCTLFILE_VERP << endl;

	if (warntime == 0)
		ctlfile << COMCTLFILE_WARNINGSENT << endl;

	ctlfile << COMCTLFILE_EXPIRES << submit_time + queuetime << endl
		<< COMCTLFILE_FAXEXPIRES << submit_time + faxqueuetime << endl
		<< COMCTLFILE_WARNING << submit_time + warntime << endl << flush;

	if (ctlfile.fail())	clog_msg_errno();
#if EXPLICITSYNC
	ctlfile.sync();
	fsync(ctlfile.fd());
#endif
	ctlfile.close();
	if (ctlfile.fail())	clog_msg_errno();
}

int SubmitFile::ChkRecipient(const char *key)
{
	receipient=key;

int	receipientl=receipient.GetLength();

	// Ignore duplicate addresses.

	if (addrlist_gdbm.IsOpen())
	{
		if (addrlist_gdbm.Exists(receipient, receipientl))
			return (-1);	/* Already exists */

		if (addrlist_gdbm.Store(receipient, receipientl, "", 1, "R"))
			clog_msg_errno();
	}
	else
	{
	AFXBOOL	dummy;

		if (addrlist_map.Lookup(receipient, dummy))
			return (-1);	/* Already exists */
		addrlist_map[receipient]=1;
		if (addrlist_map.GetCount() > SPILLLEVEL)
		{
//
// Store the GDBM file in the name that's reserved for the first hard link
// link to the message file (which is never used).
//
		char	*gdbmname=namefile("D",1);
		POSITION	pos;

			if (addrlist_gdbm.Open(gdbmname, "N"))
				clog_msg_errno();

			for (pos=addrlist_map.GetStartPosition(); pos; )
			{
				addrlist_map.GetNextAssoc(pos, receipient,
					dummy);

				if (addrlist_gdbm.Store(receipient,
						receipient.GetLength(),
						"", 1, "R"))
					clog_msg_errno();
			}
			addrlist_map.RemoveAll();
		}
	}
	return (0);
}

void SubmitFile::AddReceipient(const char *r,
	const char *orig, const char *dsn, int delivered)
{
	// If # of receipients in the current control file exceeds the
	// defined batch size, close the current control file.

	if (rcptcount >= batchsize)
	{
		closectl();
//
// The first time we create another control file, rename the first
// control file to Cnnnn.1, which is a flag not to process Cnnnn.2, .3, etc...
//
// Cnnnn.2, .3, ... will be processed only after Cnnnn.1 is renamed to Cnnnn
//
		if (num_control_files_created == 1)
		{
		char	*p=strdup(name1stctlfile());

			if (!p)	clog_msg_errno();

		char	*q=namefile("C", 1);

			if (rename(p, q))	clog_msg_errno();
			free(p);
		}
	}

	// Open a new control file, if necessary.

	if (ctlfile.fd() < 0)
		openctl();
	ctlfile << COMCTLFILE_RECEIPIENT << r << endl
		<< COMCTLFILE_ORECEIPIENT << (orig ? orig:"") << endl
		<< COMCTLFILE_DSN << (dsn ? dsn:"") << endl;

	if (delivered)
	{
		ctlfile << COMCTLFILE_DELINFO << rcptcount << ' '
			<< COMCTLFILE_DELINFO_REPLY
			<< " 250 Ok - delivered to alias." << endl
			<< COMCTLFILE_DELSUCCESS << rcptcount << ' '
				<< submit_time << " r" << endl;
	}
	ctlfile << flush;
	++rcptcount;
	if (bofh_chkspamtrap(r))
	{
		spamtrap_flag=1;
	}
}

//
//  Save original recipient list for recipient-specific filtering.
//

void SubmitFile::ReceipientFilter(struct rw_transport *rw,
		const char *host,
		const char *addr,
		unsigned rcptnum)
{

	if (rcptfilterlist_file.is_open())
	{
		rcptfilterlist_file
			<< num_control_files_created - 1 << endl
			<< rcptcount - 1 << endl
			<< rw->name << endl
			<< host << endl
			<< addr << endl
			<< rcptnum << endl;

		return;
	}

POSITION added_pos= rcptfilterlist.AddTail( RcptFilterInfo() );
RcptFilterInfo &last_pos=rcptfilterlist.GetAt(added_pos);

	last_pos.num_control_file=num_control_files_created - 1;
	last_pos.num_receipient=rcptcount - 1;
	last_pos.driver=rw;
	last_pos.host=host;
	last_pos.address=addr;
	last_pos.rcptnum=rcptnum;

	if (rcptfilterlist.GetCount() > SPILLLEVEL)
	{
		const char *filename=namefile("R", 0);

		rcptfilterlist_file.open(filename, ios::in | ios::out | ios::trunc);
		if (!rcptfilterlist_file.is_open())
			clog_msg_errno();
		unlink(filename);	/* Immediately delete it */

		for (added_pos=rcptfilterlist.GetHeadPosition(); 
			added_pos; )
		{
		RcptFilterInfo &p=rcptfilterlist.GetNext(added_pos);

			rcptfilterlist_file << p.num_control_file << endl
				<< p.num_receipient << endl
				<< p.driver->name << endl
				<< p.host << endl
				<< p.address << endl
				<< p.rcptnum << endl;
		}
		rcptfilterlist.RemoveAll();
	}
}

void SubmitFile::openctl()
{
	char	*filename;

	ctlfile.close();
	if (num_control_files_created == 0)	// First recipient
	{
		for (;;)
		{
		const char *timeptr, *pidptr, *hostnameptr;

			getnewtmpfilenameargs(&timeptr, &pidptr,
						&hostnameptr);

			// Allocate in advance the buffer where the various
			// filenames are created.

		int	l=strlen(timeptr)+strlen(pidptr)+
				strlen(hostnameptr)+MAXLONGSIZE*2+15;

			submit_filename_buf_ptr=
				submit_filename_buf.GetBuffer(l);
			submit_filename_buf.ReleaseBuffer(l);

			filename=name1stctlfile();
			current_submit_file=this;

			int nfd=open(filename,
				     O_WRONLY | O_TRUNC | O_CREAT | O_EXCL,
				     PERMISSION);

			if (nfd >= 0)
			  ctlfile.fd(nfd);

			if (nfd >= 0 || errno != EEXIST)
				break;
			current_submit_file=0;
			sleep(3);
		}
		++num_control_files_created;

		if (ctlfile.fd() < 0)
		{
		//
		// One reason why we may not be able to create it
		// would be if the subdirectory, based on current time,
		// does not exist, so fork a tiny program to create it,
		// with the right permissions
		//

		pid_t p=fork();
		pid_t w;
		int wait_stat;

			if (p == -1)
				clog_msg_errno();
			if (p == 0)
			{
				*strchr(filename, '/')=0;
				execl(LIBEXECDIR "/courier/submitmkdir",
						"submitmkdir", filename,
						(char *)0);
				exit(0);
			}
			while ((w=wait(&wait_stat)) != p)
				if (w == -1 && errno == ECHILD)	break;

			int nfd=open(filename,
				     O_WRONLY | O_TRUNC | O_CREAT,
				     PERMISSION);

			if (nfd >= 0)
			  ctlfile.fd(nfd);
		}
	}
	else
	{
		++num_control_files_created;
		filename=namefile("C", num_control_files_created);

		int nfd=open(filename,
			     O_WRONLY | O_TRUNC | O_CREAT,
			     PERMISSION);

		if (nfd >= 0)
			ctlfile.fd(nfd);
	}
	if (ctlfile.fd() < 0)	clog_msg_errno();

struct	stat	stat_buf;
char	ino_buf[sizeof(ino_t)*2+1];

	if (fstat(ctlfile.fd(), &stat_buf) != 0)
		clog_msg_errno();

	rcptcount=0;
	if (num_control_files_created == 1)
	{
	char	time_buf[sizeof(time_t)*2+1];
	char	pid_buf[sizeof(time_t)*2+1];
	char	msgidbuf[sizeof(time_buf)+sizeof(pid_buf)];

		ctltimestamp=stat_buf.st_mtime;
		ctlpid=getpid();

		strcat(strcat(strcpy(msgidbuf,
			libmail_strh_time_t(ctltimestamp, time_buf)), "."),
			libmail_strh_pid_t(getpid(), pid_buf));

		basemsgid=msgidbuf;
		ctlinodenum=stat_buf.st_ino;
	}

	libmail_strh_ino_t( stat_buf.st_ino, ino_buf );

	ctlfile << COMCTLFILE_SENDER << (const char *)sender << endl
		<< COMCTLFILE_FROMMTA << frommta << endl
		<< COMCTLFILE_ENVID << envid << endl
		<< COMCTLFILE_DSNFORMAT << dsnformat << endl
		<< COMCTLFILE_MSGID << ino_buf << '.' << basemsgid << endl;

	if (submitdelay)
		ctlfile << COMCTLFILE_SUBMITDELAY << submitdelay << endl;

	ctlfile << flush;
}

char *SubmitFile::name1stctlfile()
{
const char *timeptr, *pidptr, *hostnameptr;

	gettmpfilenameargs(&timeptr, &pidptr, &hostnameptr);

	strcpy(submit_filename_buf_ptr, timeptr);
int	l=strlen(submit_filename_buf_ptr);

	submit_filename_buf_ptr[l-4]=0;

	return (
		strcat(
		strcat(
		strcat(
		strcat(
		strcat(
		strcat(submit_filename_buf_ptr, "/")
			, timeptr)
			, ".")
			, pidptr)
			, ".")
			, hostnameptr)
		) ;
}

char *SubmitFile::QueueID()
{
	static char result[NUMBUFSIZE+1];

	libmail_strh_ino_t(ctlinodenum, result);

	strcat(strcat(result, "."), basemsgid) ;

	return (result);
}

char *SubmitFile::namefile(const char *pfix, unsigned n)
{
char	buf[MAXLONGSIZE], *p;
const char *timeptr, *pidptr, *hostnameptr;

	gettmpfilenameargs(&timeptr, &pidptr, &hostnameptr);

	strcpy(submit_filename_buf_ptr, timeptr);

int	l=strlen(submit_filename_buf_ptr);

	if (l > 4)
		submit_filename_buf_ptr[l-4]=0;
	else
		strcpy(submit_filename_buf_ptr, "0");

	strcat(strcat(submit_filename_buf_ptr, "/"), pfix);

	p=buf+MAXLONGSIZE-1;

ino_t	inum=ctlinodenum;

	*p=0;
	do
	{
		*--p= '0' + (inum % 10);
		inum=inum/10;
	} while (inum);
	strcat(submit_filename_buf_ptr, p);
	if (n > 0)
	{
		p=buf+MAXLONGSIZE-1;
		*p=0;
		do
		{
			*--p= '0' + (n % 10);
			n=n/10;
		} while (n);
		strcat(strcat(submit_filename_buf_ptr, "."), p);
	}
	return (submit_filename_buf_ptr);
}


// Process about to terminate.  Remove all possible files we might've
// created.

void SubmitFile::interrupt()
{
	if (!current_submit_file)	return;

	unlink(current_submit_file->name1stctlfile());
	if (current_submit_file->ctlinodenum == 0)	return;

unsigned	n;

	for (n=0; n<= 1 || n <= current_submit_file->num_control_files_created;
		n++)
		unlink(current_submit_file->namefile("D", n));

	n=current_submit_file->num_control_files_created;
	while ( n > 0)
	{
		unlink(current_submit_file->namefile("C", n));
		--n;
	}
}

//
// And now, process the message
//

void SubmitFile::MessageStart()
{
	bytecount=0;
	sizelimit=config_sizelimit();
	addrlist_map.RemoveAll();
	addrlist_gdbm.Close();
	if (ctlfile.fd() < 0)
		openctl();
	unlink(namefile("D", 1));	// Might be the GDBM file

	int nfd=open(namefile("D",0), O_RDWR | O_CREAT | O_TRUNC, PERMISSION);
	if (nfd < 0) clog_msg_errno();

	datfile.fd(nfd);

	rwrfcptr=rfc2045_alloc_ac();

	if (rwrfcptr == NULL)
		clog_msg_errno();
	diskfull=checkfreespace(&diskspacecheck);
}

void SubmitFile::Message(const char *p)
{
size_t	l=strlen(p);

	if (sizelimit && bytecount > sizelimit)	return;
	bytecount += l;
	if (sizelimit && bytecount > sizelimit)	return;

	if (diskfull)	return;
	datfile << p;
	if (l > diskspacecheck)
	{
		if (checkfreespace(&diskspacecheck))
		{
			diskfull=1;
			return;
		}
		diskspacecheck += l;
	}
	diskspacecheck -= l;

	if (datfile.fail())	clog_msg_errno();
	rfc2045_parse(rwrfcptr, p, strlen(p));
}

/* -------- Encapsulate a broken message --------- */

static int copy_orig_headers(FILE *, FILE *, const char *);
static int fmtmessage(FILE *, const char *);
static int copymessage(FILE *, FILE *, int, const char *);

static char *mkboundary(FILE *);

static int encapsulate(int fdin, int fdout, const char *errfile, int is8bit)
{
	int fdin_dup=dup(fdin), fdout_dup;
	FILE *fpin, *fpout;
	int rc=0;
	char *boundary;

	if (fdin_dup < 0)
	{
		clog_msg_errno();
		return (-1);
	}

	if ((fdout_dup=dup(fdout)) < 0)
	{
		clog_msg_errno();
		close(fdin_dup);
		return (-1);
	}

	if ((fpin=fdopen(fdin_dup, "r")) == NULL)
	{
		clog_msg_errno();
		close(fdin_dup);
		close(fdout_dup);
		return (-1);
	}

	if ((fpout=fdopen(fdout_dup, "w")) == NULL)
	{
		clog_msg_errno();
		fclose(fpin);
		close(fdin_dup);
		close(fdout_dup);
		return (-1);
	}

	if ((boundary=mkboundary(fpin)) == NULL
	    || copy_orig_headers(fpin, fpout, boundary)
	    || fmtmessage(fpout, errfile)
	    || copymessage(fpin, fpout, is8bit, boundary)
	    || fflush(fpout) || ferror(fpout) || ferror(fpin))
		rc= -1;

	if (boundary)
		free(boundary);
	fclose(fpin);
	fclose(fpout);
	close(fdin_dup);
	close(fdout_dup);
	return (rc);
}

static char *mkboundary(FILE *fp)
{
	char boundary_buf[NUMBUFSIZE*2+60];
	pid_t p=getpid();
	int n=0;
	int good;

	do
	{
		char buf[BUFSIZ];
		int l;

		good=1;
		sprintf(boundary_buf, "=_boundary-%04d-", ++n);
		libmail_str_pid_t(p, boundary_buf+strlen(boundary_buf));
		l=strlen(boundary_buf);

		rewind(fp);

		while (fgets(buf, sizeof(buf), fp))
		{
			if (buf[0] != '-' ||
			    buf[1] != '-')
				continue;

			if (strncasecmp(buf+2, boundary_buf, l) == 0)
			{
				good=0;
				break;
			}
		}
	} while (!good);

	rewind(fp);
	return (strdup(boundary_buf));
}

static int copy_orig_headers(FILE *fpin, FILE *fpout, const char *b)
{
	struct rfc822hdr h;

	rfc822hdr_init(&h, BUFSIZ);
	while (rfc822hdr_read(&h, fpin, NULL, 0) == 0)
	{
		if (h.header && (strcasecmp(h.header, "subject") == 0 ||
				 strcasecmp(h.header, "from") == 0 ||
				 strcasecmp(h.header, "reply-to") == 0 ||
				 strcasecmp(h.header, "sender") == 0 ||
				 strcasecmp(h.header, "date") == 0 ||
				 strcasecmp(h.header, "to") == 0 ||
				 strcasecmp(h.header, "cc") == 0 ||
				 strcasecmp(h.header, "message-id") == 0))
		{
			const char *p=h.value ? h.value:"";

			fprintf(fpout, "%s: ", h.header);

			while (*p)
			{
				putc( (*p) & 0x7F, fpout);
				++p;
			}
			putc('\n', fpout);
		}
	}
	rfc822hdr_free(&h);

	fprintf(fpout, "Mime-Version: 1.0\n"
		"Content-Type: multipart/mixed; boundary=\"%s\"\n\n"
		RFC2045MIMEMSG
		"\n--%s\n", b, b);
	return (fseek(fpin, 0L, SEEK_SET) < 0 ? -1:0);
}

static int fmtmessage(FILE *fpout, const char *errfile)
{
	static const char hdr[]=SYSCONFDIR "/rfcerrheader.txt";

	FILE *fp=fopen(hdr, "r");
	int c;

	if (!fp)
	{
		clog_msg_start_err();
		clog_msg_str(hdr);
		clog_msg_str(":");
		clog_msg_send();
		return (-1);
	}

	while ((c=getc(fp)) != EOF)
	{
		if (c == '~')
			fprintf(fpout, "%s", config_me());
		else
			putc(c, fpout);
	}
	if (ferror(fp))
	{
		fclose(fp);
		clog_msg_start_err();
		clog_msg_str(hdr);
		clog_msg_str(":");
		clog_msg_send();
		clog_msg_errno();
		return (-1);
	}

	fp=fopen(errfile, "r");
	if (!fp)
	{
		clog_msg_start_err();
		clog_msg_str(errfile);
		clog_msg_str(":");
		clog_msg_send();
		clog_msg_errno();
		return (0);
	}

	while ((c=getc(fp)) != EOF)
	{
		putc(c, fpout);
	}

	if (ferror(fp))
	{
		fclose(fp);
		clog_msg_start_err();
		clog_msg_str(errfile);
		clog_msg_str(":");
		clog_msg_send();
		clog_msg_errno();
		return (-1);
	}
	fclose(fp);
	return (0);
}

static int copymessage(FILE *fpin, FILE *fpout, int is8bit, const char *b)
{
	int c;

	fprintf(fpout, "\n--%s\n"
		"Content-Type: text/plain; charset=%s\n"
		"X-Original-Content-Type: message/rfc822\n"
		"Content-Disposition: attachment; filename=\"message.txt\"\n"
		"Content-Transfer-Encoding: %s\n\n",
		b,
		(is8bit ? "iso-8859-1":"us-ascii"),
		(is8bit ? "8bit":"7bit"));

	while ((c=getc(fpin)) != EOF)
	{
		c=(unsigned char)c;
		putc(c, fpout);
	}
	fprintf(fpout, "\n--%s--\n", b);
	return (0);
}

static int isbase64text(struct rfc2045 *rfcp)
{
	const char *content_type_s;
	const char *content_transfer_encoding_s;
	const char *charset_s;

	if (!rfcp)
		return 0;

	rfc2045_mimeinfo(rfcp, &content_type_s,
			 &content_transfer_encoding_s,
			 &charset_s);

	if ((strcasecmp(content_type_s, "text/plain") == 0 ||
	     strcasecmp(content_type_s, "text/html") == 0) &&
	    strcasecmp(content_transfer_encoding_s, "base64") == 0)
		return 1;

	for (rfcp=rfcp->firstpart; rfcp; rfcp=rfcp->next)
		if (!rfcp->isdummy && isbase64text(rfcp))
			return 1;
	return 0;
}

/* ------------ */

int SubmitFile::MessageEnd(unsigned rcptnum, int iswhitelisted,
			   const char *sending_module)
{
int	is8bit=0, dorewrite=0, rwmode=0;
const	char *mime=getenv("MIME");
unsigned	n;
struct	stat	stat_buf;
const char *rfcerr=NULL;
const char *bofhbadmime=getenv("BOFHBADMIME");
int bofhbadmimebounce= bofhbadmime && strcmp(bofhbadmime, "reject") == 0;
int bofhbadmimeaccept= bofhbadmime && strcmp(bofhbadmime, "accept") == 0;
const char *bofhbase64=getenv("BOFHNOBASE64TEXT");

	if (sizelimit && bytecount > sizelimit)
	{
		cout << "523 Message length exceeds administrative limit."
			<< endl << flush;
		return (1);
	}

	if (diskfull)
	{
		cout << "431 Mail system full." << endl << flush;
		return (1);
	}

	if (spamtrap_flag)
	{
		cout << "550 Spam refused." << endl << flush;
		return (1);
	}

	if (rwrfcptr->rfcviolation & RFC2045_ERR8BITHEADER)
	{
		rfcerr= SYSCONFDIR "/rfcerr2047.txt";
		dorewrite=1;

		if (bofhbadmimebounce)
		{
			cout <<
				"550-The headers in this message contain improperly-formatted binary content." << endl <<
				"550-This is often used by viruses that attempt to infect remote systems." << endl <<
				"550-Your mail software may also have a bug that results in this error." << endl <<
				"550 Invalid content rejected, see <URL:ftp://ftp.isi.edu/in-notes/rfc2047.txt>." << endl << flush;
			return (1);
		}

		if (bofhbadmimeaccept)
		{
			dorewrite=0;
			mime="none";
		}
        }
	else if (rwrfcptr->rfcviolation & RFC2045_ERR8BITCONTENT)
	{
		rfcerr= SYSCONFDIR "/rfcerr2045.txt";
		dorewrite=1;

		if (bofhbadmimebounce)
		{
			cout <<
				"550-This message contains improperly-formatted binary content." << endl <<
				"550-This is often used by viruses that attempt to infect remote systems." << endl <<
				"550-Your mail software may also have a bug that results in this error." << endl <<
				"550 Invalid content rejected, see <URL:ftp://ftp.isi.edu/in-notes/rfc2045.txt>" << endl << flush;
			return (1);
		}

		if (bofhbadmimeaccept)
		{
			dorewrite=0;
			mime="none";
		}
        }
	else if (rwrfcptr->rfcviolation & RFC2045_ERRBADBOUNDARY)
	{
		rfcerr= SYSCONFDIR "/rfcerr2046.txt";
		dorewrite=1;

		if (bofhbadmimebounce)
		{
			cout <<
				"550-This message contains invalid MIME headers. This is sometimes used by" << endl <<
				"550-viruses that attempt to infect remote systems, but the usual reason is" << endl <<
				"550-a bug in your mail software.  See 'IMPLEMENTORS NOTE' in" << endl <<
				"550 section 5.1.1 of <URL:ftp://ftp.isi.edu/in-notes/rfc2046.txt>." << endl << flush;
			return (1);
		}

		if (bofhbadmimeaccept)
		{
			dorewrite=0;
			mime="none";
		}
        }
	else
	if (rwrfcptr->rfcviolation & RFC2045_ERR2COMPLEX)
	{
                cout <<
                   "550 Message MIME complexity exceeds the policy maximum."
		     << endl << flush;
		return (1);
	}

	if (bofhbase64 && atoi(bofhbase64) && isbase64text(rwrfcptr))
	{
		cout << "550-This message contains text that uses unnecessary base64 encoding." << endl
		     << "550-This is often a characteristic of spam.  The message has been rejected" << endl
		     << "550 due to policy restrictions."
		     << endl << flush;

		return (1);
	}

	datfile << flush;
	if (datfile.fail())	clog_msg_errno();

	ctlfile << flush;
	if (ctlfile.fail())	clog_msg_errno();

	/* Run global filters for this message */

CString	dfile=namefile("D", 0);

SubmitFile *voidp=this;

	if (strcmp(sending_module, "dsn") &&
	    run_filter(dfile, num_control_files_created,
		       iswhitelisted,
		       &SubmitFile::get_msgid_for_filtering, &voidp))
		return (1);

	if (!mime || strcmp(mime, "none"))
	{
		if (mime && strcmp(mime, "7bit") == 0)
		{
			rwmode=RFC2045_RW_7BIT;
			is8bit=0;
		}
		if (mime && strcmp(mime, "8bit") == 0)
			rwmode=RFC2045_RW_8BIT;
		if (rfc2045_ac_check(rwrfcptr, rwmode))
			dorewrite=1;
	}
	else
		(void)rfc2045_ac_check(rwrfcptr, 0);

	if (rwrfcptr->has8bitchars)
		is8bit=1;

	unlink(namefile("D", 1));	// Might be the GDBM file
					// if receipients read from headers.
	if (dorewrite)
	{
		int	fd1=dup(datfile.fd());
		int	fd2;

		if (fd1 < 0)	clog_msg_errno();
		datfile.close();
		if (datfile.fail())	clog_msg_errno();

		if ((fd2=open(namefile("D", 1),
			O_RDWR|O_CREAT|O_TRUNC, PERMISSION)) < 0)
			clog_msg_errno();

		if (rfcerr ?
		    encapsulate(fd1, fd2, rfcerr,
				(is8bit=rwrfcptr->has8bitchars ||
				 (rwrfcptr->rfcviolation &
				  RFC2045_ERR8BITHEADER))):
		    rfc2045_rewrite(rwrfcptr, fd1, fd2, PACKAGE " " VERSION))
		{
			clog_msg_errno();
			cout << "431 Mail system full." << endl << flush;
			return (1);
		}
		close(fd1);

#if	EXPLICITSYNC
		fsync(fd2);
#endif
		fstat(fd2, &stat_buf);
		close(fd2);

	char	*p=strdup(namefile("D", 0));
		if (!p)	clog_msg_errno();

		unlink(p);
		if (rename(namefile("D", 1), p) != 0)	clog_msg_errno();
		free(p);
	}
	else
	{
		datfile.sync();
#if EXPLICITSYNC
		fsync(datfile.fd());
#endif
		fstat(datfile.fd(), &stat_buf);
		datfile.close();
		if (datfile.fail())	clog_msg_errno();
	}
	if (is8bit)
	{
		ctlfile << COMCTLFILE_8BIT << "\n" << flush;
		closectl();

		if (num_control_files_created > 1)
		{
			for (n=1; n < num_control_files_created; n++)
			{
				char	*p=namefile("C", n);

				int nfd=open(p, O_WRONLY | O_APPEND);

				if (nfd < 0)	clog_msg_errno();
				ctlfile.fd(nfd);
				ctlfile << COMCTLFILE_8BIT << "\n" << flush;
				if (ctlfile.fail())	clog_msg_errno();
#if EXPLICITSYNC
				ctlfile.sync();
				fsync(ctlfile.fd());
#endif
				ctlfile.close();
				if (ctlfile.fail())	clog_msg_errno();
			}
		}
	}
	else
	{
		closectl();
	}

CString	cfile=namefile("C", 0);

	for (n=2; n <= num_control_files_created; n++)
	{
		if (link(dfile, namefile("D", n)) != 0)	clog_msg_errno();
	}

CString okmsg("250 Ok. ");

	okmsg += basemsgid;

int	hasxerror=datafilter(dfile, rcptnum, okmsg);

	current_submit_file=0;
	if (num_control_files_created == 1)
	{
		if (rename(name1stctlfile(), cfile) != 0) clog_msg_errno();
	}
	else
	{
		if (rename(namefile("C", 1), cfile) != 0) clog_msg_errno();
	}

	if (!hasxerror)
	{
#if EXPLICITDIRSYNC
		int p=cfile.ReverseFind('/');

		if (p >= 0)
		{
			CString dir=cfile.Left(p);

			int fd=open(dir, O_RDONLY);

			if (fd >= 0)
			{
				fsync(fd);
				close(fd);
			}
		}
#endif

		cout << okmsg << endl << flush;
	}

	trigger(TRIGGER_NEWMSG);
	return (0);
}

CString SubmitFile::get_msgid_for_filtering(unsigned n, void *p)
{
SubmitFile *objptr= *(SubmitFile **)p;
CString	ctlname(TMPDIR "/");

	if (objptr->num_control_files_created == 1)
		ctlname += objptr->name1stctlfile();
	else
		ctlname += objptr->namefile("C", n+1);

	return (ctlname);
}

static void print_xerror(const char *address, const char *errbuf, int isfinal)
{
unsigned	c;
const char *p;
int	errcode=0;

	do
	{
		for (c=0; errbuf[c] && errbuf[c] != '\n'; c++)
			;
		p=errbuf;
		errbuf += c;
		if (*errbuf)	++errbuf;

		if (!errcode)
		{
			if (c > 3 && p[0] >= '2' && p[0] <= '5' &&
				p[1] >= '0' && p[1] <= '9' &&
				p[2] >= '0' && p[2] <= '9')
				errcode= (p[0] - '0') * 100 +
					(p[1] - '0') * 10 + p[2] - '0';
			else
				errcode=450;
		}
		if (c > 4 &&  p[0] >= '2' && p[0] <= '5' &&
			p[1] >= '0' && p[1] <= '9' &&
			p[2] >= '0' && p[2] <= '9' &&
			(p[3] == ' ' || p[3] == '-'))
		{
			p += 4;
			c -= 4;
		}
		if (address)
		{
			cout << "558-" << errcode << "-"
				<< address << ":" << endl;
			address=0;
		}
		cout << (( *errbuf == 0 && isfinal) ? "558 ":"558-")
			<< errcode << (*errbuf == 0 ? " ":"-");
		cout.write(p, c);
		cout << endl;
	} while (*errbuf);
}

//
// SubmitFile::datafilter runs recipient-specific filters.  This function
// will generate a data extended error message if any recipient's filters
// reject the message.  Each rejection results in the control file being
// reopened and appended a delivery record for the failed recipients, so
// we don't actually try to deliver the message to this address (the addy
// is officially rejected).
//

int SubmitFile::datafilter(const char *datfilename, unsigned nrcpts,
				const char *okmsg)
{
int	fd=open(datfilename, O_RDONLY);
unsigned last_error=0;
int	flag=0;
POSITION pos;

	if (fd < 0)	clog_msg_errno();

	if (rcptfilterlist_file.is_open())
	{
		// List of receipients was large enough to be dumped into a
		// file.

		rcptfilterlist_file.seekg(0);
		if (rcptfilterlist_file.bad())
			clog_msg_errno();

		// Read recipient list from the dump file, and filter each
		// one.

	RcptFilterInfo r;
	CString	s;
	struct rw_transport *rw;
	CString	buf;

		while ( (buf << rcptfilterlist_file) == 0)
		{
			if (sscanf((const char *)buf, "%u",
					&r.num_control_file) != 1
				|| (buf << rcptfilterlist_file)
				|| sscanf((const char *)buf, "%u",
					&r.num_receipient) != 1
				|| (buf << rcptfilterlist_file))
					clog_msg_errno();

			for (rw=rw_transport_first; rw; rw=rw->next)
				if (rw->name == buf)
					break;
			if (!rw)	clog_msg_errno();
			r.driver=rw;
			if ( (r.host << rcptfilterlist_file)
				|| (r.address << rcptfilterlist_file)
				|| (buf << rcptfilterlist_file)
				|| sscanf((const char *)buf, "%u",
					&r.rcptnum) != 1)
				clog_msg_errno();

			do_datafilter(last_error, flag, fd,
				r.driver,
				r.host,
				r.address,
				r.rcptnum,

				r.num_control_file,
				r.num_receipient,

				okmsg, nrcpts);
		}
		if ( rcptfilterlist_file.bad())
			clog_msg_errno();
		rcptfilterlist_file.close();
	}
	else for (pos=rcptfilterlist.GetHeadPosition(); pos; )
	{
	RcptFilterInfo &r=rcptfilterlist.GetNext(pos);

		do_datafilter(last_error, flag, fd,
			r.driver,
			r.host,
			r.address,
			r.rcptnum,

			r.num_control_file,
			r.num_receipient,

			okmsg, nrcpts);
	}
	close(fd);
	while (last_error && last_error < nrcpts)
	{
		print_xerror(0, okmsg, ++last_error == nrcpts);
	}
	cout << flush;
	return (flag);
}

// This is where the dirty work of running a filter actually happens.
// We get here whether the list of recipients was small enough to fit
// into memory, or not.

void SubmitFile::do_datafilter( unsigned &last_error, int &flag, int fd,
        struct rw_transport *driver,
        CString host,
        CString address,
	unsigned rcptnum,

	unsigned num_control_file,
	unsigned num_receipient,

	const char *okmsg,
	unsigned nrcpts)
{
char	buf[2048];
int	ctf;
int	rc;

	if (driver->rw_ptr->filter_msg == 0)	return;

	buf[0]=0;

	if (lseek(fd, 0L, SEEK_SET) < 0)
		clog_msg_errno();

	// Call the driver's filter function.

	rc=driver->rw_ptr->filter_msg
		? (*driver->rw_ptr->filter_msg)(
		sending_module, fd,
		host,
		address,
		sender,
		buf, sizeof(buf)):0;

	if (rc == 0)	return;

	if (buf[0] == 0)	// Error but no msg, make one up
		strcpy(buf, "Access denied.");

	ctf=open ( num_control_files_created == 1 ?
			name1stctlfile(): namefile( "C", num_control_file+1),
				O_WRONLY|O_APPEND);
	if (ctf < 0)	clog_msg_errno();

	/*
	** Mark the recipient as delivered, so no more processing is
	** done.
	*/
	ctlfile_append_replyfd(ctf, num_receipient,
		buf, COMCTLFILE_DELSUCCESS, 0);
	close(ctf);

	/*
	** We are now required to return an EXDATA extended error.  If there
	** were any good recipients up until now, we need to return an OK
	** message for those recipients.
	*/

	while (last_error < rcptnum)
	{
		print_xerror(0, okmsg, 0);
		++last_error;
	}

	print_xerror( address, buf, rcptnum + 1 == nrcpts);
	++last_error;
	flag=1;
}

static RETSIGTYPE sighandler(int signum)
{
	SubmitFile::interrupt();
	signal(SIGINT, SIG_DFL);
	kill(getpid(), SIGINT);
#if	RETSIGTYPE != void
	return (0);
#endif
}

void SubmitFile::trapsignals()
{
	signal(SIGINT, sighandler);
	signal(SIGTERM, sighandler);
	signal(SIGHUP, sighandler);
	signal(SIGALRM, sighandler);
	if (atexit(SubmitFile::interrupt)) clog_msg_errno();
}
