/* dirtools.c - reads and creates a list of directory entries
   Copyright (C) 1996-2000 Paul Sheer

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <my_string.h>
#include "stringtools.h"
#include <sys/types.h>

#include <my_string.h>
#include <sys/stat.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "coolwidget.h"
#include "pool.h"
#include "mad.h"

char *dname (struct dirent *directentry)
{
    int l;
    static char t[MAX_PATH_LEN];
    l = NAMLEN (directentry);
    if (l >= MAX_PATH_LEN)
	l = MAX_PATH_LEN - 1;
    strncpy (t, directentry->d_name, l);
    t[l] = 0;
    return t;
}

/* 
   Returns a \n separate list of directories or files in the
   directory *directory. The files are sorted alphabetacally.
   The list is malloc'ed and must be free'd.
   If f is FILELIST_DIRECTORIES_ONLY then only directories are returned.
   If f & FILELIST_ALL_FILES all files are returned.
   If f & FILELIST_FILES_ONLY, only files are returned.
 */
char *get_file_list (const char *directory, unsigned long f, char *filter)
{
    struct dirent *directentry;
    struct stat stats;
    DIR *dir;
    char *list;
    int numentries = 0;
    long listsize;
    char path_fname[MAX_PATH_LEN];
    POOL *p;

    p = pool_init ();

    if (filter) {
	if (!*filter)
	    filter = "*";
    } else
	filter = "*";

    if ((dir = opendir (directory)) == NULL)
/* No spaces here */
	return (char *) strdup (_("Error: Cannot open directory.\n"));

    while ((directentry = readdir (dir))) {
	strcpy (path_fname, directory);
	strcat (path_fname, "/");
	strcat (path_fname, dname (directentry));
	if (!stat (path_fname, &stats) && strcmp (dname (directentry), ".")) {
	    if (S_ISDIR (stats.st_mode)) {
		if (f & FILELIST_DIRECTORIES_ONLY) {
		    if (regexp_match (filter, dname (directentry), match_file) == 1) {
			if (!pool_printf (p, "/%s\n", dname (directentry))) {
			    closedir (dir);
			    return 0;
			}
			numentries++;
		    }
		}
	    } else {
		if (f & FILELIST_FILES_ONLY) {
		    if (regexp_match (filter, dname (directentry), match_file) == 1) {
			if (!pool_printf (p, "%s\n", dname (directentry))) {
			    closedir (dir);
			    return 0;
			}
			numentries++;
		    }
		}
	    }
	}
    }
/*
   Now do a bubble sort on the list. (a directory list isn't long enough
   to warrant a quick sort) and the qsort command won't work on unevenly
   sized entries.
 */
    pool_null (p);
    listsize = pool_length (p);
    list = (char *) pool_break (p);

    f = 1;
    if (numentries) {
	char *firststr, *secondstr;
	unsigned long i, r, q;
	while (f) {
	    numentries--;
	    r = 0;
	    f = 0;
	    for (i = 0; i < numentries; i++) {
		char *t;
		t = strchr (list + r, '\n');
		if (t) {
		    q = (unsigned long) t - (unsigned long) list + 1;
		    if (strcmp (firststr = strline (list, r), secondstr = strline (list, q)) > 0) {
			strcpy (list + r, secondstr);
			r += strlen (secondstr);
			*(list + r++) = '\n';
			memcpy (list + r, firststr, strlen (firststr));
			f = 1;
		    } else
			r = q;
		} else
		    break;
	    }
	}
	list[listsize - 1] = 0;	/* remove the last \n */
    }
    closedir (dir);
    return list;
}


int compare_fileentries (struct file_entry *file_entry1, struct file_entry *file_entry2)
{
#if 0
    if (file_entry->options & FILELIST_SORT_...);
#endif
    return (strcmp (file_entry1->name, file_entry2->name));
}

struct file_entry *get_file_entry_list (const char *directory, unsigned long options, char *filter)
{
    struct file_entry entry;
    struct file_entry *list;
    struct dirent *directentry;
    struct stat stats;
    DIR *dir;
    int numentries = 0;
    char path_fname[MAX_PATH_LEN];
    POOL *p;

    p = pool_init ();

    if (filter) {
	if (!*filter)
	    filter = "*";
    } else
	filter = "*";

    if ((dir = opendir (directory)) == NULL) {
	pool_free (p);
	return 0;
    }

    while ((directentry = readdir (dir))) {
	strcpy (path_fname, directory);
	strcat (path_fname, "/");
	strcat (path_fname, dname (directentry));
	if (!stat (path_fname, &stats) && strcmp (dname (directentry), ".")) {
	    if (S_ISDIR (stats.st_mode)) {
		if (options & FILELIST_DIRECTORIES_ONLY) {
		    if (regexp_match (filter, dname (directentry), match_file) == 1) {
			lstat (path_fname, &entry.stat);
			strcpy (entry.name, dname (directentry));
			entry.options = options;
			if (!pool_write (p, (unsigned char *) &entry, sizeof (entry))) {
			    pool_free (p);
			    closedir (dir);
			    return 0;
			}
			numentries++;
		    }
		}
	    } else {
		if (options & FILELIST_FILES_ONLY) {
		    if (regexp_match (filter, dname (directentry), match_file) == 1) {
			lstat (path_fname, &entry.stat);
			strcpy (entry.name, dname (directentry));
			entry.options = options;
			if (!pool_write (p, (unsigned char *) &entry, sizeof (entry))) {
			    pool_free (p);
			    closedir (dir);
			    return 0;
			}
			numentries++;
		    }
		}
	    }
	}
    }

    memset (&entry, 0, sizeof (entry));
    entry.options = FILELIST_LAST_ENTRY;
    if (!pool_write (p, (unsigned char *) &entry, sizeof (entry))) {
	pool_free (p);
	closedir (dir);
	return 0;
    }
    list = (struct file_entry *) pool_break (p);

    qsort((void *) list, numentries, sizeof (struct file_entry), (int (*) (const void *, const void *)) compare_fileentries);

    closedir (dir);
    return list;
}

