/*
 *  Class:           Session
 *
 *  Inherits from:   NSObject
 *
 *  Description:     PPP session controller.
 */

#import <Foundation/NSAutoreleasePool.h>
#import <Foundation/NSThread.h>
#import <Foundation/NSFileHandle.h>
#import <Foundation/NSTask.h>
#import <Foundation/NSNotification.h>
#import <Foundation/NSValue.h>
#import <Foundation/NSConnection.h>

#import "AppController.h"
#import "AppDefaults.h"
#import "PPPSession.h"
#import "SessionWindow.h"

#include <termios.h>

int keepRunningFlag;

@implementation PPPSession

+ (void) connectWithPorts: (NSArray *)portArray
{
  NSAutoreleasePool *pool;
  PPPSession        *serverObject;
  NSConnection      *serverConnection;

  // setup own autorelease pool
  pool = [[NSAutoreleasePool alloc] init];

  // Create connection to SessionWindow client
  serverConnection = [NSConnection
    connectionWithReceivePort: [portArray objectAtIndex: 0]
    sendPort: [portArray objectAtIndex: 1]];

  // Send PPPSession instance to SessionWindow via setServer method
  serverObject = [[PPPSession alloc] 
    initWithClientObject: [serverConnection rootProxy]];
  [(SessionWindow *)[serverConnection rootProxy] setServer: serverObject];
  [serverObject release];

  NSLog (@"PPPSession: waiting for messages");

  // Waiting for messages
  // akind of [[NSRunLoop currentRunLoop] run];
  while ( keepRunningFlag && [[NSRunLoop currentRunLoop] 
      runMode: NSDefaultRunLoopMode beforeDate: [NSDate distantFuture]] );

  NSLog (@"PPPSession: thread exiting...");
  
  [pool release];
}

- (id) initWithClientObject: (id)anObject
{
  [super init];

  portSettings = [[NSMutableString alloc] init];
  task = nil;
  keepRunningFlag = 1;
  clientObject = anObject;

  return self;
}

- (oneway void) start: (NSString *)sessionName 
{
  NSDictionary *sessionPrefs;
  int          terminationStatus = 1;

  NSLog (@"PPPSession: start %@", sessionName);
  // Getting preferences
  sessionPrefs = [[ConnectApp appDefaults] defaultsForSession: sessionName];

  portName = [[sessionPrefs objectForKey: @"Modem"] objectForKey: @"Device"];
  portSpeed = [[sessionPrefs objectForKey: @"Modem"] objectForKey: @"Speed"];
  
  phoneNumbers = [[sessionPrefs objectForKey: @"Connection"] 
    objectForKey: @"PhoneNumbers"];

  chatScriptPath = [NSMutableString stringWithString: 
    [[ConnectApp appDefaults] bundleDirForSessionWithName: sessionName]];
  [chatScriptPath appendString: @"chat.script"];

  pppdOptionsPath = [NSMutableString stringWithString: 
    [[ConnectApp appDefaults] bundleDirForSessionWithName: sessionName]];
  [pppdOptionsPath appendString: @"pppd.options"];

  // Stndard Error for stty, chat and pppd (here for saving stty output)
  errorPipe = [[NSPipe alloc] init];
  errorFileIn = [errorPipe fileHandleForReading];
  errorFileOut = [errorPipe fileHandleForWriting];
  
  // port settings
  [clientObject processStatusText: @"Initializing modem: save settings..."];
  [self savePort];

  [clientObject processStatusText: @"Initializing modem: set settings..."];
  terminationStatus = [self setPort];

  // Standard Input, Output "chat" and "pppd"
  outputFile = [NSFileHandle fileHandleForWritingAtPath: portName];
  inputFile = [NSFileHandle fileHandleForReadingAtPath: portName];

  // dialing
  if (!terminationStatus)
  {
    terminationStatus = [self runChat];

    // pppd 
    if (!terminationStatus)
    {
      [clientObject statsPPPSession];
      [clientObject 
        processStatusText: @"Connected. Logging in to the network..."];
      terminationStatus = [self runPPPD];
    }
  }

  // flush data not sent
  // Is it portable to use termios?
  tcflush ([outputFile fileDescriptor], TCIOFLUSH);
  [inputFile closeFile];
  [outputFile closeFile];

  // restore port settings
  [self restorePort];

  RELEASE (errorPipe);
  RELEASE (portSettings);

  // inform SessionWindow
  [clientObject finishPPPSession];

  keepRunningFlag = 0;
}

- (void) stop
{
  NSLog (@"PPPSession stop");
  if (task != nil && [task isRunning])
  {
    NSLog (@"PPPSession: terminate running task");
    [task interrupt];
  }

  return;
}

/*-------------------------------------------------------------------------*
 *  Session methods
 *-------------------------------------------------------------------------*/
- (int) savePort
{
  int    terminationStatus = 0;
  NSTask *savePortTask = nil;

  NSLog (@"-- Save port task start");

  // Add observer processing output
  [[NSNotificationCenter defaultCenter] 
    addObserver: self
    selector: @selector (processSTTYOutput:)
    name: NSFileHandleReadCompletionNotification
    object: errorFileIn];
    
  // stty Arguments
  args = [[NSMutableArray alloc] initWithCapacity: 1];
  [args addObject: @"-g"];
  [args addObject: @"-F"];
  [args addObject: portName];

  // Set tty settings for "chat" task
  savePortTask = [[NSTask alloc] init];
  [savePortTask setStandardOutput: errorFileOut];
  [savePortTask setStandardError: errorFileOut];
  [savePortTask setLaunchPath: @"/bin/stty"];
  [savePortTask setArguments: args];

  task = savePortTask;
  [savePortTask launch];
  [errorFileIn readInBackgroundAndNotify];
  [savePortTask waitUntilExit];

  terminationStatus = [savePortTask terminationStatus];
  
  RELEASE (args);
  RELEASE (savePortTask);
  task = nil;

  // remove trailing new line character
  if ([portSettings length] > 0)
  {
    [portSettings deleteCharactersInRange: 
      NSMakeRange ([portSettings length]-1, 1)];
  }
  else
  {
    NSLog (@"savePort: not saved");
  }

  [[NSNotificationCenter defaultCenter] 
    removeObserver: self
    name: NSFileHandleReadCompletionNotification
    object: errorFileIn];

  NSLog (@"-- Save port task end");

  return terminationStatus;
}

- (int) setPort
{
  int    terminationStatus = 0;
  NSTask *setPortTask = nil;

  NSLog (@"-- Set port task start");
  
  // stty Arguments
  args = [[NSMutableArray alloc] initWithCapacity: 1];
  [args addObject: @"-F"];
  [args addObject: portName];
  [args addObject: portSpeed];
  [args addObject: @"clocal"];
  [args addObject: @"hupcl"];
  [args addObject: @"crtscts"];
  [args addObject: @"ignbrk"];
  [args addObject: @"ignpar"];
  [args addObject: @"istrip"];
  [args addObject: @"-icrnl"];
  [args addObject: @"-icrnl"];
  [args addObject: @"-ixon"];
  [args addObject: @"-opost"];
  [args addObject: @"-onlcr"];
  [args addObject: @"-isig"];
  [args addObject: @"-icanon"];
  [args addObject: @"-iexten"];
  [args addObject: @"-echo"];
  [args addObject: @"-echoe"];
  [args addObject: @"-echok"];
  [args addObject: @"-echoctl"];
  [args addObject: @"-echoke"];

  // Set tty settings for "chat" task
  setPortTask = [[NSTask alloc] init];
  [setPortTask setLaunchPath: @"/bin/stty"];
  [setPortTask setArguments: args];

  task = setPortTask;
  [setPortTask launch];
  [setPortTask waitUntilExit];

  terminationStatus = [setPortTask terminationStatus];

  RELEASE (args);
  RELEASE (setPortTask);
  task = nil;

  NSLog (@"-- Set port task end");

  return terminationStatus;
}

- (int) restorePort
{
  int    terminationStatus = 0;
  NSTask *restorePortTask;

  if ([portSettings length] <= 0)
    return 1;

  NSLog (@"-- Restore port task start");

  // stty Arguments
  args = [[NSMutableArray alloc] initWithCapacity: 1];
  [args addObject: @"-F"];
  [args addObject: portName];
  [args addObject: portSettings];

  // Set tty settings for "chat" task
  restorePortTask = [[NSTask alloc] init];
  [restorePortTask setLaunchPath: @"/bin/stty"];
  [restorePortTask setArguments: args];

  task = restorePortTask;
  [restorePortTask launch];
  [restorePortTask waitUntilExit];

  terminationStatus = [restorePortTask terminationStatus];

  RELEASE (args);
  RELEASE (restorePortTask);
  task = nil;

  NSLog (@"-- Restore port task end");

  return terminationStatus;
}

- (int) runChat
{
  NSString *phoneNumber = nil;
  NSTask   *runChatTask = nil;
  int      i = 0;
  int      terminationStatus = 0;

  // chat Arguments
  args = [[NSMutableArray alloc] initWithCapacity: 1];

  // Add observer processing chat output
  [[NSNotificationCenter defaultCenter] 
    addObserver: self
    selector: @selector (processChatOutput:)
    name: NSFileHandleReadCompletionNotification
    object: errorFileIn];

  NSLog (@"-- Chat task start");

  // Running through the phone list when busy
  for (i = 0; i < [phoneNumbers count]; i++)
  { 
    [clientObject processStatusText: [NSString stringWithFormat: 
      @"Dialing number %@...", [phoneNumbers objectAtIndex: i]]];

    phoneNumber = [NSString 
      stringWithFormat: @"-T%@", [phoneNumbers objectAtIndex: i]];
//    NSLog (@"Dialing number %@", [phoneNumbers objectAtIndex: i]);
    [args addObject: @"-V"];
    [args addObject: @"-S"];
    [args addObject: phoneNumber];
    [args addObject: @"-f"];
    [args addObject: chatScriptPath];

    // "chat" script: dialing and awating of CONNECT string
    runChatTask = [[NSTask alloc] init];
    [runChatTask setStandardInput: inputFile];
    [runChatTask setStandardOutput: outputFile];
    [runChatTask setStandardError: errorFileOut];
    [runChatTask setLaunchPath: @"/usr/sbin/chat"];
    [runChatTask setArguments: args];

    task = runChatTask;
    [runChatTask launch];
    [errorFileIn readInBackgroundAndNotify];
    [runChatTask waitUntilExit];

    terminationStatus = [runChatTask terminationStatus];
    RELEASE (runChatTask);
    task = nil;

    if (terminationStatus != 4)
    {
      break;
    }

    [args removeAllObjects];
  }

  switch (terminationStatus)
  {
    case 1:
      NSLog (@"chat: Invalid parameters (code 1)");
      break;
    case 2:
      NSLog (@"chat: Error during execution (code 2)");
      break;
    case 3:
      NSLog (@"chat: Timeout waiting expect string (code 3)");
      break;
    default:
      break;
  }

NSLog (@"-- Chat task WILL end");

  RELEASE (args);

  // Remove observer processing chat output
  [[NSNotificationCenter defaultCenter] 
    removeObserver: self
    name: NSFileHandleReadCompletionNotification
    object: errorFileIn];

  [clientObject processServerText: @"\n"];
  NSLog (@"-- Chat task end");

  return terminationStatus;
}

- (int) runPPPD
{
  NSNumber *errorFD = nil;
  int      terminationStatus = 0;
  NSTask   *runPPPDTask;
  
  [[NSNotificationCenter defaultCenter] 
    addObserver: self
    selector: @selector (processPPPOutput:)
    name: NSFileHandleReadCompletionNotification
    object: errorFileIn];

  // pppd Arguments
  errorFD = [NSNumber numberWithInt: [errorFileOut fileDescriptor]];
  args = [[NSMutableArray alloc] initWithCapacity: 1];
  [args addObject: @"logfd"];
  [args addObject: @"2"];
  [args addObject: @"file"];
  [args addObject: pppdOptionsPath];

  // pppd
  runPPPDTask = [[NSTask alloc] init];
  [runPPPDTask setStandardInput: inputFile];
  [runPPPDTask setStandardOutput: outputFile];
  [runPPPDTask setStandardError: errorFileOut];
  [runPPPDTask setLaunchPath: @"/usr/sbin/pppd"];
  [runPPPDTask setArguments: args];

  NSLog (@"--- PPPD task start");

  task = runPPPDTask;
  [runPPPDTask launch];
  [errorFileIn readInBackgroundAndNotify];
  [runPPPDTask waitUntilExit];

  terminationStatus = [runPPPDTask terminationStatus];

  switch (terminationStatus)
  {
    case 1:
      NSLog (@"pppd: Fatal error (1)");
      break;
    case 2:
      NSLog (@"pppd: Error processing options (2)");
      break;
    case 3:
      NSLog (@"pppd: Must be suid root (3)");
      break;
    case 4:
      NSLog (@"pppd: Kernel does not support PPP (4)");
      break;
    case 5:
      NSLog (@"pppd: Terminated by signal (5)");
      break;
    case 6:
      NSLog (@"pppd: Serial port lock error. Wow! (6)");
      break;
    case 7:
      NSLog (@"pppd: Serial port open error. Wow! (7)");
      break;
    case 8:
      NSLog (@"pppd: Connect script failed. Wow! (8)");
      break;
    case 9:
      NSLog (@"pppd: pty option could not be run (9)");
      break;
    case 10:
      NSLog (@"pppd: Negotiation failed (10)");
      break;
    case 11:
      NSLog (@"pppd: Peer refused to authenticate (11)");
      break;
    case 12:
      NSLog (@"pppd: Idle's exceeded (12)");
      break;
    case 13:
      NSLog (@"pppd: Connect time limit was reached (13)");
      break;
    case 14:
      NSLog (@"pppd: Waiting for callback incoming call (14)");
      break;
    case 15:
      NSLog (@"pppd: Peer is not responding to echo requests (15)");
      break;
    case 16:
      NSLog (@"pppd: Terminated by the modem hanging up (16)");
      break;
    case 17:
      NSLog (@"pppd: Serial loopback detected (17)");
      break;
    case 18:
      NSLog (@"pppd: Init script failed. Wow! (18)");
      break;
    case 19:
      NSLog (@"pppd: Failed to authenticate to peer (19)");
      break;
    default:
      break;
  }
  
  RELEASE (args);
  RELEASE (runPPPDTask);
  task = nil;

  [[NSNotificationCenter defaultCenter] 
    removeObserver: self
    name: NSFileHandleReadCompletionNotification
    object: errorFileIn];

  NSLog (@"--- PPPD task end");

  return terminationStatus;
}

/*-------------------------------------------------------------------------*
 *  Utility methods
 *-------------------------------------------------------------------------*/
- (void) processSTTYOutput: (NSNotification *)notif
{
  NSData          *inData = nil;
  NSMutableString *inString = nil;

  inData = [[notif userInfo] objectForKey: NSFileHandleNotificationDataItem];

  NSLog (@"STTY output length: %i", [inData length]);
  if ([inData length])
  {
    inString = [[NSMutableString alloc]
      initWithData: inData 
      encoding: [NSString defaultCStringEncoding]];

    [portSettings appendString: inString];

    RELEASE (inString);
  }

  if (task != nil && [task isRunning])
  {
    [[notif object] readInBackgroundAndNotify];
  }
}

- (void) processChatOutput: (NSNotification *)notif
{
  NSData          *inData = nil;
  NSMutableString *inString = nil;

  inData = [[notif userInfo] objectForKey: NSFileHandleNotificationDataItem];

  if ([inData length])
  {
    inString = [[NSMutableString alloc]
      initWithData: inData 
      encoding: [NSString defaultCStringEncoding]];

    // put into details view
    [clientObject processServerText: inString];

    RELEASE (inString);
  }

  if (task != nil && [task isRunning])
  {
    [[notif object] readInBackgroundAndNotify];
  }
}

- (void) processPPPOutput: (NSNotification *)notif
{
  NSData          *inData = nil;
  NSMutableString *inString = nil;

  inData = [[notif userInfo] objectForKey: NSFileHandleNotificationDataItem];

  if ([inData length])
  {
    inString = [[NSMutableString alloc]
      initWithData: inData 
      encoding: [NSString defaultCStringEncoding]];

    // put into details view
    [clientObject processServerText: inString];
    if ([inString rangeOfString: @"remote IP address"].length)
    {
      [clientObject processStatusText: @"Connected"];
    }

    RELEASE (inString);
  }

  if (task != nil && [task isRunning])
  {
    [[notif object] readInBackgroundAndNotify];
  }
}

@end

