/*
//
//  Copyright 1997-2009 Torsten Rohlfing
//
//  Copyright 2004-2012 SRI International
//
//  This file is part of the Computational Morphometry Toolkit.
//
//  http://www.nitrc.org/projects/cmtk/
//
//  The Computational Morphometry Toolkit is free software: you can
//  redistribute it and/or modify it under the terms of the GNU General Public
//  License as published by the Free Software Foundation, either version 3 of
//  the License, or (at your option) any later version.
//
//  The Computational Morphometry Toolkit is distributed in the hope that it
//  will be useful, but WITHOUT ANY WARRANTY; without even the implied
//  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License along
//  with the Computational Morphometry Toolkit.  If not, see
//  <http://www.gnu.org/licenses/>.
//
//  $Revision: 2266 $
//
//  $LastChangedDate: 2010-08-19 15:30:11 -0700 (Thu, 19 Aug 2010) $
//
//  $LastChangedBy: torstenrohlfing $
//
*/

namespace
cmtk
{

/** \addtogroup Base */
//@{
template <class TInterpolationFunction>
typename ImageInterpolator::ImageType::PixelType
ImageInterpolator<TInterpolationFunction>
::GetDataAt(const Vector3D& v ) const
{
  Types::Coordinate lScaled[3];
  int imageGridPoint[3];
  for ( int n = 0; n < 3; ++n )
    {
    lScaled[n] = (v[n] - this->m_Image->Offset[n]) / this->m_Image->Deltas[n];
    imageGridPoint[n] = (int) floor( lScaled[n] );
    if ( ( imageGridPoint[n] < 0 ) || ( imageGridPoint[n] >= this->m_Image->Dims[n]-1 ) )
      return ImageInterpolator::ImageType::PixelType();

    relative[n] = lScaled[n] - imageGridPoint[n];
    }

  return this->GetDataDirect( imageGridPoint, relative );
}

template <class TInterpolationFunction>
typename ImageInterpolator::ImageType::PixelType
ImageInterpolator<TInterpolationFunction>
::GetDataDirect( const int* imageGridPoint, const Types::Coordinate* insidePixel ) const
{
  const int xx = imageGridPoint[0] + 1 - TInterpolationFunction::RegionSizeLeftRight;
  const int yy = imageGridPoint[1] + 1 - TInterpolationFunction::RegionSizeLeftRight;
  const int zz = imageGridPoint[2] + 1 - TInterpolationFunction::RegionSizeLeftRight;

  Types::Coordinate interpolationWeights[3][2 * TInterpolationFunction::RegionSizeLeftRight];

  for ( int n = 0; n < 3; ++n )
    {
    for ( int m = 1-TInterpolationFunction::RegionSizeLeftRight; m <= TInterpolationFunction::RegionSizeLeftRight; ++m )
      {
      interpolationWeights[n][m+TInterpolationFunction::RegionSizeLeftRight-1] = TInterpolationFunction::GetWeight( m, insidePixel[n] );
      }
    }

  const int iMin = std::max( 0, -xx );
  const int iMax = std::min( 2 * TInterpolationFunction::RegionSizeLeftRight, this->m_Image->Dims[0] - xx );

  const int jMin = std::max( 0, -yy );
  const int jMax = std::min( 2 * TInterpolationFunction::RegionSizeLeftRight, this->m_Image->Dims[1] - yy );

  const int kMin = std::max( 0, -zz );
  const int kMax = std::min( 2 * TInterpolationFunction::RegionSizeLeftRight, this->m_Image->Dims[2] - zz );

  typename Self::ImageType::PixelType value;
  Types::Coordinate totalWeight = 0;

  for ( int k = kMin; k < kMax; ++k )
    {
    for ( int j = jMin; j < jMax; ++j )
      {
      const Types::Coordinate weightJK = interpolationWeights[1][j] * interpolationWeights[2][k];
      size_t offset = this->GetOffsetFromIndex( xx + iMin, yy + j, zz + k );
      for ( int i = iMin; i < iMax; ++i, ++offset )
        {
	const Types::DataItem data = this->m_Image[offset];

	const Types::Coordinate weightIJK = interpolationWeights[0][i] * weightJK;
	value += weightIJK * data;
	totalWeight += weightIJK;
        }
      }
    }
  
  return value;
}

} // namespace cmtk
