/*
 * Time-stamp: <98/11/05 16:04:44 panic>
 * Author:	The C-Mix Project <cmix@diku.dk>
 *              Peter Holst Andersen <txix@diku.dk>
 *              Arne John Glenstrup  <panic@diku.dk>
 * Contents:	The core of the ray tracer.
 *		Intersection routines: intersection_*, intersect_all
 *		The shading function: shade
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#define EXTERNS_DEFINED
#include "ray.h"
#ifdef USE_PLOT
#  include "plot.h"
#endif
#ifdef HAVE_LIBTIFF
#  include "tiff.h"
#endif
#include <math.h>

#include "vector.h"
#include "scenes.h"


#ifndef __CMIX
#  define __CMIXDEBUG(S,X)
#  define __CMIX(X)
#else
/*#  define __CMIXDEBUG(S,X) __CMIX(pure)fprintf(stderr,S,X)*/
#  define __CMIXDEBUG(S,X) __CMIX(pure)fprintf(stderr,".")
/*#  define __CMIXDEBUG(S,X)*/
#endif


screenType  screen;
myfloat     minweight;
colorType   background;
lightType   light[MAX_LIGHTS_SCENE];
objectType  scene[MAX_OBJECTS_SCENE + 1];
surfaceType surface[MAX_SURFACES_SCENE];
viewType    view, defaultview;
int         root_object;

IdxType surfaceIdx;
IdxType objectIdx;
IdxType lightIdx;

char* surfaceIdxname = "surfaces";
char* objectIdxname  = "objects"; 
char* lightIdxname   = "lights";  

char* surfaceIdxmaxname = "MAX_SURFACES_SCENE";
char* objectIdxmaxname  = "MAX_OBJECTS_SCENE"; 
char* lightIdxmaxname   = "MAX_LIGHTS_SCENE";  

int stat_intersections;
int stat_eye;
int stat_shadow;
int stat_reflected;
int stat_refracted;

int maxlevel;

extern char prg_name[256];
extern char image_file[256];

extern int to_screen, to_file;
extern int compress;
extern int eyepointSelected, lookpointSelected, updirSelected;
extern int fieldOfViewSelected;
extern viewType userview;

void init_scene(int, int, int);
int  flat_object(ush);
int  bounding_object(ush);
void compute_primary_ray(myfloat, myfloat, rayType *);
void trace(int, myfloat, rayType, colorType *);
void shade(int, myfloat, rayType, intersectionType, colorType *);
void background_color(colorType *);
int  shadow(rayType, myfloat);
INLINE_EXTERN int intersect_all(rayType, double *);
INLINE_EXTERN myfloat intersection(int, rayType);
INLINE_EXTERN myfloat intersection_sphere(sphereType, rayType);
INLINE_EXTERN myfloat intersection_disc(discType, rayType);
INLINE_EXTERN myfloat intersection_square(squareType, rayType);
INLINE_EXTERN myfloat intersection_plane(vectorType, myfloat, rayType);
void compute_normal(intersectionType, vectorType *);
void compute_normal_sphere(intersectionType, vectorType *);
void specular_direction(myfloat, vectorType *, vectorType *, vectorType *);
int  transmission_direction(myfloat, myfloat, vectorType *, vectorType *, vectorType *);
void calculate_uv(intersectionType, myfloat *, myfloat *);
void calculate_uv_square(intersectionType, myfloat *, myfloat *);

void vector_copy(vectorType A, vectorType *B)
{
  B->x = A.x;
  B->y = A.y;
  B->z = A.z;
}

#define vector_copy_SD(A,B) B.x = A.x; B.y = A.y; B.z = A.z;

void light_falloff(colorType *col, myfloat d)
{
  if (d < 1.0)
    return;
  col->red = col->red / d;
  col->green = col->green / d;
  col->blue = col->blue / d;
}

void init_scene(int image_x, int image_y, int use_scene)
{
  vectorType dir, up;
  myfloat magnitude;

  __CMIXDEBUG("/* Calling init scene... */\n", 0);

  surfaceIdx.last = -1;
  surfaceIdx.max  = MAX_SURFACES_SCENE;
  surfaceIdx.name = surfaceIdxname;
  surfaceIdx.maxname = surfaceIdxmaxname;
  objectIdx.last = -1;
  objectIdx.max  = MAX_OBJECTS_SCENE;
  objectIdx.name = objectIdxname;
  objectIdx.maxname = objectIdxmaxname;
  lightIdx.last = -1;
  lightIdx.max  = MAX_LIGHTS_SCENE;
  lightIdx.name = lightIdxname;
  lightIdx.maxname = lightIdxmaxname;
  
  defaultview.lookp.x = defaultview.lookp.y = defaultview.lookp.z = 0.0;
  defaultview.pos.x = 0.0;
  defaultview.pos.y = -8.0;
  defaultview.pos.z = 0.0;
  defaultview.up.x  = 0.0;
  defaultview.up.y  = 0.0;
  defaultview.up.z  = 1.0;
  defaultview.hfov  = defaultview.vfov = 45.0;

  root_object = 0;
  background.red   = BACKGROUND_RED;
  background.green = BACKGROUND_GREEN;
  background.blue  = BACKGROUND_BLUE;
  maxlevel  = MAXLEVEL;
  minweight = MINWEIGHT;
        
  create_scene(use_scene);

  screen.height = image_x;
  screen.width  = image_y;
  if (eyepointSelected)
    view.pos = userview.pos;
  else {
    vector_copy_SD(defaultview.pos, view.pos);
  }
  if (lookpointSelected)
    view.lookp = userview.lookp;
  else {
    vector_copy_SD(defaultview.lookp, view.lookp);
  }
  if (updirSelected)
    view.up = userview.up;
  else {
    vector_copy_SD(defaultview.up, view.up);
  }
  if (fieldOfViewSelected) {
    view.hfov = userview.hfov;
    view.vfov = userview.vfov;
  } else {
    view.hfov = defaultview.hfov;
    view.vfov = defaultview.vfov;
  }
  {
    vectorType lookp, pos, dir;
    lookp = view.lookp; pos = view.pos;
    vector_sub(&lookp, &pos, &dir);
    view.dir = dir;
  }      
  screen.firstv.x = view.dir.x;
  screen.firstv.y = view.dir.y;
  screen.firstv.z = view.dir.z;
  {
    vectorType dir;
    dir = view.dir;
    view.lookdist = vector_norm(&dir);
    view.dir = dir;
  }
  vector_copy_SD(view.dir, dir);
  vector_copy_SD(view.up, up);
  if (vector_norm_cross(&dir, &up, &screen.scrni) == 0.0)
    fprintf(stderr, "%s: The view and up directions are identical\n",
	    prg_name);
  vector_norm_cross(&screen.scrni, &dir, &screen.scrnj);
  magnitude = 2.0 * view.lookdist * tan((double) 0.5*deg2rad(view.hfov)) /
              screen.width;
  vector_scale(magnitude, &screen.scrni, &screen.scrnx);
  magnitude = 2.0 * view.lookdist * tan((double) 0.5*deg2rad(view.vfov)) /
              screen.height;
  vector_scale(magnitude, &screen.scrnj, &screen.scrny);
  screen.firstv.x -= 0.5*screen.height*screen.scrny.x +
                     0.5*screen.width*screen.scrnx.x;
  screen.firstv.y -= 0.5*screen.height*screen.scrny.y +
                     0.5*screen.width*screen.scrnx.y;
  screen.firstv.z -= 0.5*screen.height*screen.scrny.z +
                     0.5*screen.width*screen.scrnx.z;
}

int bounding_object(ush i)
{
  return (scene[i].tag == BOUNDING_SPHERE);
}

int flat_object(ush i)
{
  return (scene[i].tag == DISC || scene[i].tag == SQUARE);
}

void compute_primary_ray(myfloat x, myfloat y, rayType *ray)
{
  __CMIXDEBUG("/* compute_primary_ray called */\n", 0);
  vector_copy_SD(view.pos, ray->p);
  ray->v.x = screen.firstv.x + x*screen.scrnx.x + y*screen.scrny.x;
  ray->v.y = screen.firstv.y + x*screen.scrnx.y + y*screen.scrny.y;
  ray->v.z = screen.firstv.z + x*screen.scrnx.z + y*screen.scrny.z;
  vector_norm(&ray->v);
}

INLINE_EXTERN myfloat intersection_sphere(sphereType sphere, rayType ray)
{
  static myfloat x, y, z, b, d, t;

  x = sphere.c.x - ray.p.x;
  y = sphere.c.y - ray.p.y;
  z = sphere.c.z - ray.p.z;
  b = x * ray.v.x + y * ray.v.y + z * ray.v.z;
  d = b*b - x*x - y*y - z*z + sphere.r2;
  if (d <= 0.0) return 0.0;
  d = sqrt((double) d);
  t = b - d;
  if (t <= MIN_DISTANCE) {
    t = b + d;
    if (t <= MIN_DISTANCE)
      return 0.0;
  }
  return t;
}

INLINE_EXTERN myfloat intersection_plane(vectorType n, myfloat d, rayType ray)
{
  static myfloat t;

  t = n.x * ray.v.x + n.y * ray.v.y + n.z * ray.v.z;
  if (t == 0.0) return 0.0;
  t = -(n.x * ray.p.x + n.y * ray.p.y + n.z * ray.p.z - d) / t;
  if (t <= MIN_DISTANCE) return 0.0;
  return t;
}

INLINE_EXTERN myfloat intersection_disc(discType disc, rayType ray)
{
  static myfloat a, d, t;

  t = intersection_plane(disc.n, disc.d, ray);
  if (t > MIN_DISTANCE) {
    a = disc.c.x - ray.p.x - t * ray.v.x;
    d = a*a;
    if (d >= disc.r2)
      return 0.0;
    a = disc.c.y - ray.p.y - t * ray.v.y;
    d += a*a;
    if (d >= disc.r2)
      return 0.0;
    a = disc.c.z - ray.p.z - t * ray.v.z;
    if (d + a*a >= disc.r2)
      return 0.0;
    else
      return t;
  }
  return 0.0;
}

INLINE_EXTERN myfloat intersection_square(squareType square, rayType ray)
{
  static myfloat x, y, z, a, b, t;

  t = intersection_plane(square.n, square.d, ray);
  if (t > MIN_DISTANCE) {
    x = square.c.x - (ray.p.x + t * ray.v.x);
    y = square.c.y - (ray.p.y + t * ray.v.y);
    z = square.c.z - (ray.p.z + t * ray.v.z);
    a = x * square.v1.x + y * square.v1.y + z * square.v1.z;
    a = a*a;
    b = x*x + y*y + z*z;
    if ((a < square.r2) && (b - a < square.r2))
      return t;
  }
  return 0.0;
}

INLINE_EXTERN myfloat intersection(int i, rayType ray)
{
  __CMIXDEBUG("/* intersection called with i=%d */", i);
  /*    objectType* debug;*/
#ifdef COLLECT_STATISTICS
  stat_intersections++;
#endif
  /*    debug = scene + i;*/
  switch(scene[i].tag) {
  case BOUNDING_SPHERE:
  case SPHERE:
    return intersection_sphere(scene[i].obj.sphere, ray);
  case DISC:
    return intersection_disc(scene[i].obj.disc, ray);
  case SQUARE:
    return intersection_square(scene[i].obj.square, ray);
  default:
    fprintf(stderr, "%s: intersection: invalid object tag: %d, object = %d\n",
	    prg_name, scene[i].tag, i);
  }
  return 0.0;
}

INLINE_EXTERN int intersect_all(rayType ray, double *isect_t)
{
  int n, i;
  double t1;

  __CMIXDEBUG("/* intersect_all called */\n", 0);
  n = -1;
  i = 0;
  while(scene[i].tag != NONE) {
    __CMIXDEBUG("/* intersection called from intersect_all, i=%d */\n",i);
    t1 = intersection(i, ray);
    __CMIXDEBUG("/* intersection finished */\n", 0);
    if (MIN_DISTANCE < t1 && t1 < *isect_t - MIN_DISTANCE) {
      *isect_t = t1;
      n = i;
    }
    i += 1;
  }
  return n;
}

void trace(int level, myfloat weight, rayType ray, colorType *color)
{
  intersectionType isect;
    
  __CMIXDEBUG("/* trace called */\n", 0);
  isect.t = 1.0e+20;
  isect.object = intersect_all(ray, &isect.t);
  __CMIXDEBUG("/* intersect_all finished */\n", 0);
#ifdef DEBUG_INTERSECT
  fprintf(stderr, "intersect all returns (%d,%f)\n",
	  isect.object, isect.t);
#endif
  if (isect.object >= 0) {
    shade(level, weight, ray, isect, color);
    __CMIXDEBUG("/* shade finished */\n", 0);
  } else
    background_color(color);
}

void calculate_uv_square(intersectionType isect, myfloat *u, myfloat *v)
{
  int i;
  squareType *square = NULL;
  
  __CMIXDEBUG("/* calculate_uv_square called */\n", 0);
  /* The Trick: */
  for (i = 0; i <= objectIdx.last; i++)
    if (i == isect.object)
      break ;
  square = &scene[i].obj.square;
  if (square != NULL) {
    __CMIXDEBUG("/* i = %d */\n", i);
    *u = (isect.p.x-square->c.x)*square->v1.x
       + (isect.p.y-square->c.y)*square->v1.y
       + (isect.p.z-square->c.z)*square->v1.z;
    *v = (isect.p.x-square->c.x)*square->v2.x
       + (isect.p.y-square->c.y)*square->v2.y
       + (isect.p.z-square->c.z)*square->v2.z;
  }
}

void calculate_uv_sphere(intersectionType isect, myfloat *u, myfloat *v)
{
  int i;
  sphereType *sphere = NULL;
  vectorType p, b;
  myfloat alen, blen;
  
  __CMIXDEBUG("/* calculate_uv_sphere called */\n", 0);
  /* The Trick: */
  for (i = 0; i <= objectIdx.last; i++)
    if (i == isect.object)
      break ;
  sphere = &scene[i].obj.sphere;
  if (sphere != NULL) {
    p.x = isect.p.x - sphere->c.x;
    p.y = isect.p.y - sphere->c.y;
    p.z = isect.p.z - sphere->c.z;
    alen = p.x * sphere->n.x + p.y * sphere->n.y + p.z * sphere->n.z;
    *u = sphere->r * asin(alen / sphere->r);
    b.x = p.x - alen * sphere->n.x;
    b.y = p.y - alen * sphere->n.y;
    b.z = p.z - alen * sphere->n.z;
    blen = sqrt(b.x * b.x + b.y * b.y + b.z * b.z);
    if (fzero(blen))
      *v = 0.0;
    else
      *v = copysign(sphere->r,
		    p.x * sphere->v2.x +
		    p.y * sphere->v2.y +
		    p.z * sphere->v2.z)
	* acos((b.x * sphere->v1.x +
		b.y * sphere->v1.y +
		b.z * sphere->v1.z) / blen);
  }
}

void calculate_uv(intersectionType isect, myfloat *u, myfloat *v)
{
  int i;
  __CMIXDEBUG("/* calculate_uv called */\n", 0);

  /* The Trick: */
  for (i = 0; i <= objectIdx.last; i++)
    if (i == isect.object)
      break ;
  switch (scene[i].tag) {
  case SQUARE:
  case DISC:
    calculate_uv_square(isect, u, v);
    __CMIXDEBUG("/* calculate_uv_square finished */\n", 0);
    break;
  case SPHERE:
    calculate_uv_sphere(isect, u, v);
    __CMIXDEBUG("/* calculate_uv_sphere finished */\n", 0);
    break;
  default:
    fprintf(stderr, "%s: unknow object tag %d\n",
            prg_name, scene[i].tag);
    exit(-1);
  }
}

void compute_normal_sphere(intersectionType isect, vectorType *N)
{
  int i;
    sphereType *sphere = NULL;

    __CMIXDEBUG("/* compute_normal_sphere called */\n", 0);
    /* The Trick: */
    for (i = 0; i <= objectIdx.last; i++)
      if (i == isect.object)
        break ;
    __CMIXDEBUG("/* i = %d */\n", i);
    sphere = &scene[i].obj.sphere;
    if (sphere != NULL) {
      N->x = (isect.p.x - sphere->c.x) / sphere->r;
      N->y = (isect.p.y - sphere->c.y) / sphere->r;
      N->z = (isect.p.z - sphere->c.z) / sphere->r;
    }
}

void compute_normal(intersectionType isect, vectorType *N)
{
  int i;
  __CMIXDEBUG("/* compute_normal called */\n", 0);

  /* The Trick: */
  for (i = 0; i <= objectIdx.last; i++)
    if (i == isect.object)
      break ;
  switch (scene[i].tag) {
  case SPHERE:
    compute_normal_sphere(isect, N);
    __CMIXDEBUG("/* compute_normal_sphere finished, i = %d */\n", i);
    return;
  case DISC:
    vector_copy(scene[i].obj.disc.n, N);
    return;
  case SQUARE:
    vector_copy(scene[i].obj.square.n, N);
    return;
  default:
    fprintf(stderr,
            "%s: compute_normal: invalid object tag: %d, object=%d\n",
            prg_name, scene[i].tag, i);
  }
}

/* 
 * transmission_direction: compute transmission vector
 * n_dot_v: dot product of N and V
 * refrindex: refraction index
 * V: view vector
 * N: normal vector
 * T: transmission vector (result) 
 * returns: FALSE if no transmission occurs, TRUE otherwise
 *
 * If the ray is entering an object (n_dot_v > 0.0), it's supposed that the
 * ray is leaving air. Otherwise it is supposed that the ray is entering air.
 * All vectors are normalized.
 */
int transmission_direction(myfloat n_dot_v,
			   myfloat r,
			   vectorType *V,
			   vectorType *N,
			   vectorType *T)
{
  myfloat c2, c3;

#ifdef COLLECT_STATISTICS
  stat_refracted++;
#endif
  if (n_dot_v > 0.0)
    r = 1.0/r;

  c2 = r*r*(1.0 - n_dot_v * n_dot_v);
#ifdef DEBUG_TRANS
  if (c2 > 1.0)
    fprintf(stderr, "transmission direction: c2 = %f > 1.0\n", c2);
#endif
  if (c2 > 1.0)
    return FALSE;
  c2 = sqrt(1.0 - c2);
  if (n_dot_v < 0.0)
    c2 = -c2;
  c3 = r * n_dot_v - c2;
  T->x = c3 * N->x + r * V->x;
  T->y = c3 * N->y + r * V->y;
  T->z = c3 * N->z + r * V->z;
#ifdef DEBUG_TRANS
  fprintf(stderr, "TRANS: r = %f, c1 = %f, c2 = %f, r*c1 - c2 = %f\n",
	  r, n_dot_v, c2, c3);
  fprintf(stderr, "trn = (%f, %f, %f)\n", T->x, T->y, T->z);
#endif
  return TRUE;
}

void shade(int level,
	   myfloat weight,
	   rayType ray,
	   intersectionType isect,
	   colorType *color)
{
  int n = -1;
  int saved_isect_object = -1 ;
  surfaceInfoType sinfo;
  vectorType N, L;
  colorType diffuse, specular;
  rayType tray;
  myfloat n_dot_l, n_dot_v, distance, a, u, v;
  colorType tcol;

  __CMIXDEBUG("/* shade called with level = %d */\n", level);

#ifdef DEBUG1
  fprintf(stderr, "shade: level = %d, weight = %f, object = %d, dir = (%f, %f, %f)\n",
	  level, weight, isect.object, ray.v.x, ray.v.y, ray.v.z);
#endif

  color->red = 0.0;
  color->green = 0.0;
  color->blue = 0.0;

  isect.p.x = ray.p.x + isect.t * ray.v.x;
  isect.p.y = ray.p.y + isect.t * ray.v.y;
  isect.p.z = ray.p.z + isect.t * ray.v.z;

  /* The Trick: */
  for (saved_isect_object = 0;
       saved_isect_object <= objectIdx.last;
       saved_isect_object++)
    if (saved_isect_object == isect.object)
      break ;
  
  n = scene[saved_isect_object].prop.surface;
  if (n >= 0) {
    __CMIXDEBUG("/* while loop start, n = %d */\n", n);
    while (surface[n].tag != SIMPLE) {
      myfloat dbl_checksize;
      __CMIXDEBUG("/* n = %d   */\n", n);
      __CMIXDEBUG("/* tag = %d */\n", surface[n].tag);
      switch (surface[n].tag) {
      case CHECKED:
	calculate_uv(isect, &u, &v);
	__CMIXDEBUG("/* calculate_uv finished */\n",0);
	while (u < 0)
	  u += 100.0*surface[n].u.checked.checksize;
	while (v < 0)
	  v += 100.0*surface[n].u.checked.checksize;
	dbl_checksize = surface[n].u.checked.dbl_checksize;
	if ((fmod(u, dbl_checksize) < surface[n].u.checked.checksize) ^
	    (fmod(v, dbl_checksize) < surface[n].u.checked.checksize)) 
	  n = surface[n].u.checked.s1;
	else
	  n = surface[n].u.checked.s2;
	break;
      default:
	fprintf(stderr, "%s: no such surface %d\n", prg_name, n);
	exit(-1);
      }
    }

    sinfo = surface[n].u.sinfo;

    /* isect.p is needed to compute normal for spheres */
  
    compute_normal(isect, &N);
    __CMIXDEBUG("/* compute_normal finished */\n", 0);

    n_dot_v = - vector_dot(&N, &ray.v);

    /* Flip normal if the object has no inside (if it's flat) */
    if (flat_object(saved_isect_object)) {
      if (n_dot_v < 0.0) {
        N.x = -N.x;
        N.y = -N.y;
        N.z = -N.z;
        n_dot_v = - n_dot_v;
      }
      isect.enter = TRUE;
    } else {
      if (n_dot_v < 0.0)
        isect.enter = FALSE;
      else
        isect.enter = TRUE;
    }

#ifdef DEBUG1
    fprintf(stderr, "ray = (%f, %f, %f)\nnml = (%f, %f, %f)\n",
	    ray.v.x, ray.v.y, ray.v.z, N.x, N.y, N.z);
#endif

    /* if we are leaving an object, only transmitted light contributes */
  
    if (!isect.enter) {
      if (level < maxlevel && sinfo.transparency * weight > minweight) {
	tray.p = isect.p;
	if (transmission_direction(n_dot_v, sinfo.refrindex, &ray.v,
				   &N, &tray.v)) {
	  trace(level + 1, sinfo.transparency * weight, tray, &tcol);
	  __CMIXDEBUG("/* trace finished */\n", 0);
	  color->red   = sinfo.transparency * tcol.red;
	  color->green = sinfo.transparency * tcol.green;
	  color->blue  = sinfo.transparency * tcol.blue;
#ifdef DEBUG_TRANS
	  fprintf(stderr, "shade: Leaving object: transmitted light = (%f, %f, %f)\n", color->red, color->green, color->blue);
#endif
	}
      }
#ifdef LIGHT_FALLOFF
      light_falloff(color, isect.t);
#endif
#ifdef DEBUG
      fprintf(stderr, "shade returns (%f, %f, %f)\n",
	      color->red, color->green, color->blue);
#endif
      return;
    }

    /* Ambient Light */
  
    color->red   = sinfo.ambient.red;
    color->green = sinfo.ambient.green;
    color->blue  = sinfo.ambient.blue;
    
    /* Diffuse light and specular reflectance */
  
    diffuse.red = diffuse.green = diffuse.blue = 0.0;
    specular.red = specular.green = specular.blue = 0.0;
    {
      int n;
      for(n = 0; n <= lightIdx.last; n++) {
	pointType p;
	p.x = light[n].p.x;
	p.y = light[n].p.y;
	p.z = light[n].p.z;
	vector_sub(&isect.p, &p, &L);
	distance = vector_norm(&L);
	n_dot_l = vector_dot(&N, &L);
	if (n_dot_l < 0.0) {
	  tray.p.x = light[n].p.x;
	  tray.p.y = light[n].p.y;
	  tray.p.z = light[n].p.z;
	  tray.v = L;
#ifdef COLLECT_STATISTICS
	  stat_shadow++;
#endif
	  if (intersect_all(tray, &distance) == -1) {
	    int cmixhacktmp;
	    diffuse.red   += light[n].c.red * sinfo.diffuse.red;
	    diffuse.green += light[n].c.green * sinfo.diffuse.green;
	    diffuse.blue  += light[n].c.blue * sinfo.diffuse.blue;
	    if (sinfo.specpow != 0.0) {
	      a = - n_dot_l + n_dot_v;
	      a = a*a / (2.0 * (1.0 + ray.v.x * L.x + ray.v.y * L.y +
				ray.v.z * L.z));
#ifdef DEBUG_HIGHLIGHT
	      fprintf(stderr, "shade: a = %f, specpow = %d\n",
		      a, sinfo.specpow);
#endif
	      cmixhacktmp = sinfo.specpow;
	      a = mypower(a, cmixhacktmp);
#ifdef DEBUG_HIGHLIGHT
	      fprintf(stderr, "shade: a = %f\n", a);
#endif
	      specular.red   += a * light[n].c.red;
	      specular.green += a * light[n].c.green;
	      specular.blue  += a * light[n].c.blue;
	    }
	  }
	}
      }
    }
    
#ifdef DEBUG
    fprintf(stderr, "shade: diffuse: (%f, %f, %f)\n",
	    diffuse.red, diffuse.green, diffuse.blue);
    fprintf(stderr, "shade: shadow rays: (%f, %f, %f)\n",
	    specular.red, specular.green, specular.blue);
#endif

    color->red   += diffuse.red + sinfo.reflectivity * specular.red;
    color->green += diffuse.green + sinfo.reflectivity * specular.green;
    color->blue  += diffuse.blue + sinfo.reflectivity * specular.blue;
    
    if (level < maxlevel) {
      tray.p = isect.p;
	
      /* Specular reflection */
      if (sinfo.reflectivity * weight > minweight) {
	specular_direction(n_dot_v, &ray.v, &N, &tray.v);
	trace(level + 1, sinfo.reflectivity * weight, tray, &tcol);
	color->red   += sinfo.reflectivity * tcol.red;
	color->green += sinfo.reflectivity * tcol.green;
	color->blue  += sinfo.reflectivity * tcol.blue;
#ifdef DEBUG_SPECULAR
	fprintf(stderr, "shade: reflected light: (%f, %f, %f)\n",
		tcol.red, tcol.green, tcol.blue);
#endif
      }
	
      /* Transmission ray */
      if (sinfo.transparency * weight > minweight) {
	ray.p = isect.p;
        if (flat_object(saved_isect_object)) {
          trace(level + 1, sinfo.transparency * weight, ray, &tcol);
          color->red   += sinfo.transparency * tcol.red;
          color->green += sinfo.transparency * tcol.green;
          color->blue  += sinfo.transparency * tcol.blue;
#ifdef DEBUG_TRANS
          fprintf(stderr, "shade: transmitted light: (%f, %f, %f)\n",
                  tcol.red, tcol.green, tcol.blue);
#endif
        }
        else {
          if (transmission_direction(n_dot_v, sinfo.refrindex,
                                     &ray.v, &N, &tray.v)) {
            trace(level + 1, sinfo.transparency * weight, tray, &tcol);
            color->red   += sinfo.transparency * tcol.red;
            color->green += sinfo.transparency * tcol.green;
            color->blue  += sinfo.transparency * tcol.blue;
#ifdef DEBUG_TRANS
            fprintf(stderr, "shade: transmitted light: (%f, %f, %f)\n",
                    tcol.red, tcol.green, tcol.blue);
#endif
          }
        }
      }
    }
#ifdef LIGHT_FALLOFF
    light_falloff(color, isect.t);
#endif
#ifdef DEBUG
    fprintf(stderr, "shade returns (%f, %f, %f)\n",
	    color->red, color->green, color->blue);
#endif
  } 
}

void background_color(colorType *color)
{
  color->red = background.red;
  color->green = background.green;
  color->blue = background.blue;
}

/* 
 * specular_direction: compute specular vector in specular direction
 * n_dot_v: dot product of N and V
 * V: view vector
 * N: normal vector
 * S: specular vector (result)
 * All vectors are normalized
 */
void specular_direction(myfloat n_dot_v,
			vectorType *V,
			vectorType *N,
			vectorType *S)
{
#ifdef COLLECT_STATISTICS
  stat_reflected++;
#endif
  S->x = 2.0 * n_dot_v * N->x + V->x;
  S->y = 2.0 * n_dot_v * N->y + V->y;
  S->z = 2.0 * n_dot_v * N->z + V->z;
#ifdef DEBUG_SPECULAR
  fprintf(stderr, "specular direction = (%f,%f,%f)\n", S->x, S->y, S->z);
#endif
}

/* 
 * shadow: returns true if the ray doesn't hit any objects for t < tmax
 *         false othwerwise
 */
/*
int shadow(rayType ray, myfloat tmax)
{
  intersectionType isect;
  
  isect.t = tmax;
  intersect_all(root_object, ray, &isect);
  return (isect.object == -1);
}
*/


void drawscene(int image_x, int image_y, int use_scene, int use_color)
{ 
  char title[256];
  colorType col;
  rayType   ray;

  int x, y;

#ifdef COLLECT_STATISTICS
  stat_intersections = 0;
  stat_eye = 0;
  stat_shadow = 0;
  stat_reflected = 0;
  stat_refracted = 0;
#endif

  if (use_scene == -1) use_scene = DEFAULT_SCENE;
  init_scene(image_x, image_y, use_scene);
  __CMIXDEBUG("/* Init scene done */\n", 0);
fflush(stderr);    
#ifdef USE_PLOT
  if (to_screen) {
    sprintf(title, "%s: scene %d", prg_name, use_scene);
    initWindow(title, screen.height, screen.width, use_color);
  }
#endif
  
#ifdef HAVE_LIBTIFF
  if (to_file) {
    sprintf
      (title,
       "Scene %d traced by the C-Mix Project Ray Tracer\n<cmix@diku.dk>\n",
       use_scene);
    tiffOpenFile(prg_name, image_file, title,
		 screen.width, screen.height, use_color, compress);
  }
#endif

  fprintf(stderr, "%s: tracing scene %d...\n", prg_name, use_scene);

  for(y = 0; y < screen.height; y++) {
    __CMIXDEBUG("/* y loop entered */\n", 0);
    for(x = 0; x < screen.width; x++) {
      __CMIXDEBUG("/* x loop entered */\n", 0);
      compute_primary_ray((double) x + 0.5, (double) y + 0.5, &ray);
      __CMIXDEBUG("/* compute_primary_ray finished */\n", 0);
#ifdef COLLECT_STATISTICS
      stat_eye++;
#endif	   
      trace(0, 1.0, ray, &col);
      __CMIXDEBUG("/* trace finished */\n", 0);
      if (col.red   > 0.9999) col.red   = 0.9999;
      if (col.green > 0.9999) col.green = 0.9999;
      if (col.blue  > 0.9999) col.blue  = 0.9999;
	
#ifdef USE_PLOT
      if (to_screen) plot(x, y, col.red, col.green, col.blue);
#endif
#ifdef HAVE_LIBTIFF
      if (to_file)
	tiffPlot(x, col.red, col.green, col.blue);
#else
#  ifndef USE_PLOT
      /* prevent compiler from optimizing everything away when
	 we are not displaying the traced ray */
      col.red = col.red + col.green + col.blue;
#  endif
#endif
    }
#ifdef USE_PLOT
    if (to_screen) refreshLine(y);
#endif
#ifdef HAVE_LIBTIFF
    if (to_file) tiffFlushLine(y);
#endif
  }
  fprintf(stderr, "%s: tracing done.\n", prg_name, use_scene);
    
#ifdef HAVE_LIBTIFF
  if (to_file) {
    fprintf(stderr, "%s: writing TIFF file `%s'...", prg_name, image_file);
    tiffCloseFile();
    fprintf(stderr, "done.\n");
  }
#endif
  
#ifdef COLLECT_STATISTICS
  fprintf(stderr, "Intersections test = %d\n", stat_intersections);
  fprintf(stderr, "Eye rays           = %d\n", stat_eye);
  fprintf(stderr, "Shadow rays        = %d\n", stat_shadow);
  fprintf(stderr, "Reflected rays     = %d\n", stat_reflected);
  fprintf(stderr, "Refracted rays     = %d\n", stat_refracted);
#endif
    
}
