/*
	$Id: provider_magick.cpp,v 1.4 2000/05/09 21:06:44 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

#ifdef WIN32
#define _VISUALC_
#endif

#include "API/Magick/provider_magick.h"

#include <magick/magick.h>
#undef class

CL_Surface *CL_MagickProvider::create(
	std::string name,
	CL_InputSourceProvider *provider,
	EPixelFormat pixelformat,
	bool uses_transcol,
	int transcol)
{
	cl_assert(pixelformat!=PAL8); // no chance pal ;-)

	return CL_Surface::create(
		new CL_MagickProvider(
			name, 
			provider, 
			pixelformat,
			uses_transcol, 
			transcol),
		true);
}

CL_Surface *CL_MagickProvider::create(
	std::string name,
	CL_InputSourceProvider *provider,
	int red_mask,
	int green_mask,
	int blue_mask,
	int alpha_mask,
	bool uses_transcol,
	int transcol)
{
	return CL_Surface::create(
		new CL_MagickProvider(
			name,
			provider,
			red_mask,
			green_mask,
			blue_mask,
			alpha_mask,
			uses_transcol,
			transcol),
		true);
}

CL_MagickProvider::CL_MagickProvider(
	std::string name,
	CL_InputSourceProvider *provider,
	EPixelFormat pixelformat,
	bool uses_transcol,
	int transcol)
{
	cl_assert(pixelformat != PAL8); // no chance pal ;-)
	init(
		name,
		provider,
		CL_Color::get_red_mask(pixelformat),
		CL_Color::get_green_mask(pixelformat),
		CL_Color::get_blue_mask(pixelformat),
		CL_Color::get_alpha_mask(pixelformat),
		uses_transcol,
		transcol);
}

CL_MagickProvider::CL_MagickProvider(
	std::string name,
	CL_InputSourceProvider *provider,
	int red_mask,
	int green_mask,
	int blue_mask,
	int alpha_mask,
	bool uses_transcol,
	int trans_col)
{
	init(
		name,
		provider,
		red_mask,
		green_mask,
		blue_mask,
		alpha_mask,
		uses_transcol,
		transcol);
}

void CL_MagickProvider::init(
	std::string name,
	CL_InputSourceProvider *provider,
	int red_mask,
	int green_mask,
	int blue_mask,
	int alpha_mask,
	bool uses_transcol,
	int transcol)
{
	this->name = name;
	this->transcol = transcol;	
	this->uses_transcol = uses_transcol;
	rmask = red_mask;
	gmask = green_mask;
	bmask = blue_mask;
	amask = alpha_mask;
}

void CL_MagickProvider::perform_lock()
{
	Image *image;
	ImageInfo image_info;

	GetImageInfo(&image_info);
	strcpy(image_info.filename, name.c_str());
	image = ReadImage(&image_info);
//		if (!image)
//			throw CL_Error("error while trying to open file with the Magick provider");
	cl_assert(image != NULL);

	width = image->columns;
	height = image->rows;
	pitch = width * get_bytes_per_pixel();
	
	float *rdata = new float[width*height];
	float *gdata = new float[width*height];
	float *bdata = new float[width*height];
	float *adata = new float[width*height];
	
	GetPixels(image, rdata, gdata, bdata, adata);
	DestroyImage(image);
	
	// here comes the tricky part ... hmhmhm... Quicky Mart
	data = new unsigned char[pitch*height];
	
	unsigned int rshift, gshift, bshift, ashift, tmp;
	unsigned int rtmp, gtmp, btmp, atmp;
	
	rshift = 32; tmp = rmask;
	while (tmp!=0)
	{
		tmp = (tmp << 1); rshift--;
	}
	
	gshift = 32; tmp = gmask;
	while (tmp!=0)
	{
		tmp = (tmp << 1); gshift--;
	}
	 
	bshift = 32; tmp = bmask;
	while (tmp!=0)
	{
		tmp = (tmp << 1); bshift--;
	}
	
	ashift = 32; tmp = amask;
	while (tmp!=0)
	{
		tmp = (tmp << 1); ashift--;
	}
	
	rtmp = rmask >> rshift;
	gtmp = gmask >> gshift;
	btmp = bmask >> bshift;
	atmp = amask >> ashift;

// BUG: this is not endian clean - but I don't mind! If you have a big endian machine you'd have to change
// BUG: the below part.
	for (int y=0; y<height; y++)
		for (int x=0; x<width; x++)
		{
			tmp = (((unsigned int) (rdata[y*width+x] * rtmp)) << rshift) |
			      (((unsigned int) (gdata[y*width+x] * gtmp)) << gshift) |
			      (((unsigned int) (bdata[y*width+x] * btmp)) << bshift) |
			      (((unsigned int) (atmp - adata[y*width+x] * atmp)) << ashift);

			switch (get_bytes_per_pixel())
			{
				case 1 : data[y*pitch+1*x+0] = (unsigned char) (tmp & 0xFF);
					 break;
				case 2 : data[y*pitch+2*x+0] = (unsigned char) (tmp & 0xFF);
					 data[y*pitch+2*x+1] = (unsigned char) ((tmp >> 8) & 0xFF);
					 break;
				case 3 : data[y*pitch+3*x+0] = (unsigned char) (tmp & 0xFF);
					 data[y*pitch+3*x+1] = (unsigned char) ((tmp >> 8) & 0xFF);
					 data[y*pitch+3*x+2] = (unsigned char) ((tmp >> 16) & 0xFF);
					 break;
				case 4 : data[y*pitch+4*x+0] = (unsigned char) (tmp & 0xFF);
					 data[y*pitch+4*x+1] = (unsigned char) ((tmp >> 8) & 0xFF);
					 data[y*pitch+4*x+2] = (unsigned char) ((tmp >> 16) & 0xFF);
					 data[y*pitch+4*x+3] = (unsigned char) ((tmp >> 24) & 0xFF);
					 break;
				default :
					 cl_assert(false);
			}
		}

	delete[] rdata;
	delete[] gdata;
	delete[] bdata;
	delete[] adata;
}		

void CL_MagickProvider::perform_unlock()
{
	delete[] data;
	data = NULL;
}

CL_MagickProvider::~CL_MagickProvider()
{
}

unsigned int CL_MagickProvider::get_pitch() const { return pitch; }
unsigned int CL_MagickProvider::get_width() const { return width; }
unsigned int CL_MagickProvider::get_height() const { return height; }
unsigned int CL_MagickProvider::get_num_frames() const { return 1; }
unsigned int CL_MagickProvider::get_red_mask() const { return rmask; }
unsigned int CL_MagickProvider::get_green_mask() const { return gmask; }
unsigned int CL_MagickProvider::get_blue_mask() const { return bmask; }
unsigned int CL_MagickProvider::get_alpha_mask() const { return amask; }
CL_Palette *CL_MagickProvider::get_palette() const { return NULL; }
unsigned int CL_MagickProvider::get_src_colorkey() const { return transcol; }
bool CL_MagickProvider::is_indexed() const { return false; }
bool CL_MagickProvider::uses_src_colorkey() const { return uses_transcol; }
void *CL_MagickProvider::get_data() const { return data; }


// Resource support:

class CL_Magick_ResourceSource : public CL_ResourceSource_Surface
{
public:
	virtual const char *get_name() { return "magick"; }

	virtual bool can_create(
		std::string ext,
		CL_ResourceOptions *options)
	{
		// TODO: Add all the extensions image magick can read...

		if (options->exists("magick")) return true;
		if (ext == ".png") return true;
		if (ext == ".bmp") return true;
		if (ext == ".gif") return true;
		if (ext == ".jpg") return true;

		return false;
	}

	virtual CL_SurfaceProvider *create(
		std::string filename,
		CL_ResourceOptions *options,
		CL_ResourceManager *parent)
	{
		return new CL_MagickProvider(filename.c_str(), NULL);
	}

};

static CL_Magick_ResourceSource *res_source_magick = NULL;

void CL_SetupMagick::init()
{
	res_source_magick = new CL_Magick_ResourceSource;
}

void CL_SetupMagick::deinit()
{
	delete res_source_magick;
	res_source_magick = NULL;
}
