
#include "Core/precomp.h"
#include <API/Core/System/error.h>

#include "cdaudio_linux.h"


static std::vector<CL_CDDrive *> CL_CDAudio::cd_drives;

void CL_CDAudio::init_cdaudio()
{
	// Make sure init only happen once:

	static bool already_inited = false;
	if (already_inited) return;
	already_inited = true;
	
	// Locate all CDROM drives available on the system.
	
	// We start by looking the mounttab:
	
	FILE *mount_file = setmntent("/etc/fstab", "r");
	if (mount_file != NULL)
	{
		bool found_in_mount_file = false;
		while (true)
		{
			mntent *entry = getmntent(mount_file);
			if (entry == NULL) break; // no more entries.
			
			if (strcmp(mntent->mnt_type, MNTTYPE_CDROM) == 0)
			{
				try
				{
					CL_CDAudio::cd_drives.push_back(
						new CL_CDDrive_Linux(mntent->mnt_fsname));
					
					found_in_mount_file = true;
				}
				catch (CL_Error err)
				{
					// Damn.
					cl_info(info_sound, err.message.c_str());
				}
			}
		}

		endmntent(mount_file);
		if (found_in_mount_file) return; // success! :)
	}
	
	// Ok. None there - assume he linked it from /dev/cdrom.
	
	try
	{
		cd_drives.push_back(new CL_CDDrive_Linux();
	}
	catch (CL_Error)
	{
		std::cout << "Now CDROM drives found." << std::endl;
		std::cout << "Either symlink it from /dev/cdrom, or add it to /etc/fstab." << std::endl;
	}
}

/**************************************************************/

CL_CDDrive_Linux::CL_CDDrive_Linux(const char *filename)
{
	this->filename = filename;
	
	// Is this really a cdrom drive??
	
	stat status;
	int err = stat(filename, &status);
	if (err <= -1 ||
	{
		// File doesn't exist! (or something else failed.)
		strstream err;
		err << "Device " << filename << " not found!";
		throw CL_Error(err.str());
	}
	
	if (!S_ISCHR(status.st_mode) && !S_ISBLK(status.st_mode))
	{
		strstream err;
		err << "Device " << filename << " is not a block device!";
		throw CL_Error(err.str());
	}
	
	handle = open(filename, (O_RDONLY|O_EXCL|O_NONBLOCK), 0);
	if (handle <= 0)
	{
		strstream err;
		err << "Device " << filename << " could not be opened!";
		throw CL_Error(err.str());
	}
	
	cdrom_subchnl info;
	if (ioctl(handle, CDROMSUBCHNL, &info) == 0 ||
		/* is the disk not present?: */
		errno == EIO || 
		errno == ENOENT ||
		errno == EINVAL)
	{
		// Hurray. Found a cdrom drive.
		cl_info_debug(info_sound, "HURRAY! We found a cdrom drive.");
	}
	else
	{
		close(handle);
	
		strstream err;
		err << "Device " << filename << " is not a cdrom device!";
		throw CL_Error(err.str());
	}
	
	close(handle);
	handle = -1;
}

CL_CDDrive_Linux::~CL_CDDrive_Linux()
{
	if (handle != -1) close(handle);
}

bool CL_CDDrive_Linux::init()
{
	handle = open(filename, (O_RDONLY|O_EXCL|O_NONBLOCK), 0);
	if (handle < 0)
	{
		handle = -1;
		return false;
	}
	
	return true;
}

std::string CL_CDDrive_Linux::get_drive_path()
{
	return std::string("HUH?");
}

std::string CL_CDDrive_Linux::get_cd_name()
{
	return std::string("Fisk");
}

int CL_CDDrive_Linux::get_num_tracks()
{
	return 0;
}

bool CL_CDDrive_Linux::is_playing()
{
	return false;
}

int CL_CDDrive_Linux::get_cur_track()
{
	return 0;
}

int CL_CDDrive_Linux::get_cur_frame()
{
	return 0;
}

bool CL_CDDrive_Linux::play_tracks(int track, int end_track)
{
	return true;
}

bool CL_CDDrive_Linux::play_frames(int frame, int end_frame)
{
	return true;
}

bool CL_CDDrive_Linux::play_track(int track)
{
	return true;
}

void CL_CDDrive_Linux::stop()
{
}

void CL_CDDrive_Linux::pause()
{
}

void CL_CDDrive_Linux::resume()
{
}
