/*
	$Id: resource_manager_net_generic.cpp,v 1.1.1.1 2000/04/09 12:18:01 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

#include "Core/precomp.h"
#include "resource_manager_net_generic.h"
#include <API/Core/Network/netmessage.h>
#include <API/Core/Network/netgame.h>
#include <API/Core/Network/netgroup.h>
#include <API/Core/Network/netcomputer.h>
#include <API/Core/System/error.h>
#include <API/Core/System/cl_assert.h>
#include <API/Core/System/clanstring.h>
#include <API/Core/IOData/outputsource_memory.h>

CL_ResourceManager_Net *CL_ResourceManager_Net::create()
{
	return new CL_ResourceManager_Net_Generic;
}

CL_ResourceManager_Net_Generic::CL_ResourceManager_Net_Generic()
{
}

CL_ResourceManager_Net_Generic::~CL_ResourceManager_Net_Generic()
{
	for (
		std::list<CL_Resource *>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		delete *it;
	}

	for (
		std::list<std::pair<std::string, CL_InputSource_Memory *> >::iterator it2 = sources.begin();
		it2 != sources.end();
		it2++)
	{
		delete (*it2).second;
	}
}

bool CL_ResourceManager_Net_Generic::receive(CL_NetGame *net, int netchannel)
{
	CL_NetMessage msg;

	try
	{
		msg = net->receive(netchannel, 0);
	}
	catch (CL_Error err)
	{
		return false;
	}

	CL_InputSource_Memory input(msg.data);

	int num_sources = input.read_int32();
	for (int s=0; s<num_sources; s++)
	{
		std::string name = input.read_string();
		int size = input.read_int32();
		char *data = new char[size];
		input.read(data, size);
		
		std::string d; d.append(data, size);
			
		sources.push_back(
			std::pair<std::string, CL_InputSource_Memory *>(
				name,
				new CL_InputSource_Memory(d)));
	}
		
	int num_resources = input.read_int32();
	for (int r=0; r<num_resources; r++)
	{
		std::string full_path = input.read_string();
		std::string type = input.read_string();
			
		CL_ResourceOptions options(this);
		CL_Resource *res = NULL;
		for (
			std::list<CL_ResourceType*>::iterator it = CL_ResourceType::resource_types.begin();
			it != CL_ResourceType::resource_types.end();
			it++)
		{
			if ((*it)->get_type() == type)
			{
				res = (*it)->create_from_serialization(full_path, this);
				break;
			}
		}

		if (res == NULL)
		{
			CL_String err;
			err << "Unknown resource type: " << type;
			throw CL_Error(err.get_string());
		}
		else
		{
			resources.push_back(res);
		}
	}

	return true;
}

void CL_ResourceManager_Net_Generic::send(
	CL_NetGame *net,
	int netchannel,
	const CL_NetComputer *comp)
{
	CL_NetGroup group;
	group.computers.push_back((CL_NetComputer*) comp);
	send(net, netchannel, &group);
}

void CL_ResourceManager_Net_Generic::send(
	CL_NetGame *net,
	int netchannel,
	const CL_NetGroup *group)
{
	CL_OutputSource_Memory output;
	output.write_int32(sources.size());

	// Save input sources:	
	{
		for (
			std::list<std::pair<std::string, CL_InputSource_Memory *> >::iterator it = sources.begin();
			it != sources.end();
			it++)
		{
			output.write_string((*it).first.c_str());
			output.write_int32((*it).second->size());
			output.write((*it).second->get_data().data(), (*it).second->size());
		}
	}

	output.write_int32(resources.size());
	
	// Save resource descriptions:
	{
		for (
			std::list<CL_Resource*>::iterator it = resources.begin();
			it != resources.end();
			it++)
		{
			output.write_string((*it)->get_name().c_str());
			output.write_string((*it)->get_type().c_str());
		}
	}
	
	if (group == NULL) group = net->get_all();
	
	net->send(
		netchannel,
		group,
		CL_NetMessage(output.get_data()));
}

void CL_ResourceManager_Net_Generic::add(CL_Resource *resource)
{
	resources.push_back(resource);
	
	CL_OutputSource_Memory output;
	resource->serialize_save(&output);
	
	CL_InputSource_Memory *memory_stream =
		new CL_InputSource_Memory(
			output.get_data());

	sources.push_back(
		std::pair<std::string, CL_InputSource_Memory *>(
			resource->get_name(),
			memory_stream));
}


void CL_ResourceManager_Net_Generic::load_all_resources()
{
	for (
		std::list<CL_Resource*>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		(*it)->load();
	}
}

void CL_ResourceManager_Net_Generic::unload_all_resources()
{
	for (
		std::list<CL_Resource*>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		(*it)->unload();
	}
}

void CL_ResourceManager_Net_Generic::load_section(const char *section_name)
{
	CL_String prefix = section_name;
	prefix << "/";
	
	int prefix_len = prefix.get_length();

	for (
		std::list<CL_Resource*>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		CL_String name = (*it)->get_name();
		if (name.mid(0, prefix_len) == prefix)
		{
			(*it)->load();
		}
	}
}

void CL_ResourceManager_Net_Generic::unload_section(const char *section_name)
{
	CL_String prefix = section_name;
	prefix << "/";
	
	int prefix_len = prefix.get_length();

	for (
		std::list<CL_Resource*>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		CL_String name = (*it)->get_name();
		if (name.mid(0, prefix_len) == prefix)
		{
			(*it)->unload();
		}
	}
}

CL_Resource *CL_ResourceManager_Net_Generic::get_resource(std::string res_id)
{
	for (
		std::list<CL_Resource*>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		if ((*it)->get_name() == res_id) return *it;
	}
	
	CL_String err;
	err << "Resource " << res_id << " not found.";
	throw CL_Error(err.get_string());
	
	return NULL;
}

std::list<std::string> *CL_ResourceManager_Net_Generic::get_all_resources()
{
	std::list<std::string> *retval = new std::list<std::string>;

	for (
		std::list<CL_Resource*>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		retval->push_back((*it)->get_name());
	}
	
	return retval;
}

std::list<std::string> *CL_ResourceManager_Net_Generic::get_resources_of_type(std::string type_id)
{
	std::list<std::string> *retval = new std::list<std::string>;

	for (
		std::list<CL_Resource*>::iterator it = resources.begin();
		it != resources.end();
		it++)
	{
		if ((*it)->get_type() == type_id)
			retval->push_back((*it)->get_name());
	}

	return retval;
}

CL_InputSourceProvider *CL_ResourceManager_Net_Generic::get_resource_provider()
{
	return this;
}

CL_InputSource *CL_ResourceManager_Net_Generic::open_source(const char *handle)
{
	for (
		std::list<std::pair<std::string, CL_InputSource_Memory *> >::iterator it = sources.begin();
		it != sources.end();
		it++)
	{
		if ((*it).first == handle) return (*it).second->clone();
	}
	
	CL_String err;
	err << "Cannot open source " << handle;
	throw CL_Error(err.get_string());
	return NULL;
}

CL_InputSourceProvider *CL_ResourceManager_Net_Generic::clone()
{
	cl_assert(false); // would anyone want to do this? I hope not.
	return NULL;
}
