/*
	$Id: network_delivery_socket.cpp,v 1.1.1.1 2000/04/09 12:18:01 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

#include "Core/precomp.h"

#ifdef USE_NETWORK

#include <API/Core/System/error.h>
#include "API/Core/System/cl_assert.h"
#include <Core/Network/Generic/network_delivery_impl.h>
#include <Core/Network/Generic/network_delivery_socket.h>

#include <string.h>
#include <errno.h>

#define NETWORK_MAGIC 0x16042104

#ifdef WIN32
#define CL_INVALID_SOCKET INVALID_SOCKET
#else
#define CL_INVALID_SOCKET -1
#endif

/******************************************************************************
                         Unix Socket implementation
******************************************************************************/

CL_UniformSocket::CL_UniformSocket(CL_ConnectionProvider *_provider)
{
	provider = _provider;
	is_connection_lost = false;
	sock = CL_INVALID_SOCKET;

	recv_state = expect_magic;
	cur_message = NULL;
	cur_message_size = -1;
}

CL_UniformSocket::~CL_UniformSocket()
{
#ifdef WIN32
	if (sock != CL_INVALID_SOCKET) closesocket(sock);
#else
	if (sock != CL_INVALID_SOCKET) close(sock);
#endif

//	recv_buffer.clear();
	provider->remove_connection(this);
}

bool CL_UniformSocket::read_avail()
{
	fd_set rfds;

	FD_ZERO(&rfds);
	FD_SET(sock, &rfds);

	timeval timeout;
	memset(&timeout, 0, sizeof(timeval));

	int retval = select(sock+1, &rfds, NULL, NULL, &timeout);
	if (retval == -1 || retval == 0) 
	{
		return false;
	}

	char buf[1024]; // why use a deque here, starch?

	int data_received = ::recv(
		sock,
		buf,
		1024,
		0);

	if (data_received == 0 || data_received == -1)
	{
		is_connection_lost = true;
		return false;
	}
/*
	recv_buffer.insert(
		recv_buffer.end(),
		data_received,
		buf[0]);
*/
	recv_buffer.append(buf, data_received);

	return true;
}

bool CL_UniformSocket::require_avail(unsigned int size)
{
	return recv_buffer.size() >= size;
}

bool CL_UniformSocket::get_avail(void *buf, unsigned int size)
{
	if (recv_buffer.size() < size) return false;

	memcpy(buf, recv_buffer.data(), size);
	recv_buffer.erase(0, size);

	return true;
}

bool CL_UniformSocket::init_socket(int init_socket)
{
	if (init_socket != CL_INVALID_SOCKET)
	{
		sock = init_socket;
		write_int(NETWORK_MAGIC);
		return true;
	}
	else
	{
		sock = socket(AF_INET, SOCK_STREAM, 0);
		return (sock != CL_INVALID_SOCKET);
	}
}

bool CL_UniformSocket::peek()
{
	if (sock == CL_INVALID_SOCKET) return false;

	read_avail();

	switch (recv_state)
	{
	case expect_magic:
		{
			int found_magic;
			if (get_avail(&found_magic, sizeof(int)) == false) return false;
			found_magic = ntohl(found_magic);
			if (found_magic != NETWORK_MAGIC)
			{
				is_connection_lost=true;
			}
			else recv_state = expect_packet_size;
		}

	case expect_packet_size:
		{
			if (get_avail(&cur_message_size, sizeof(int)) == false) return false;
			cur_message_size = ntohl(cur_message_size);
			recv_state = expect_packet_data;
		}

	case expect_packet_data:
		{
			if (!require_avail(cur_message_size)) return false;
			cur_message = new char[cur_message_size];
			get_avail(cur_message, cur_message_size);
			recv_state = packet_finished;
		}

	case packet_finished:
		return true;

	default:
		throw CL_Error("Network protocol error.");
		return false;
	};

	return false;
}

CL_ConnectionPacket CL_UniformSocket::receive()
{
	if (peek() == false)
	{
		CL_ConnectionPacket ret;
		ret.size = 0;
		ret.data = NULL;
		return ret;
	}

	CL_ConnectionPacket ret;
	ret.size = cur_message_size;
	ret.data = cur_message;

	recv_state = expect_packet_size;

	return ret;
}

void CL_UniformSocket::send(CL_ConnectionPacket message)
{
	write_int(message.size);
	write_data(message.data, message.size);
}

bool CL_UniformSocket::connection_lost()
{
	return is_connection_lost;
}

unsigned long CL_UniformSocket::read_int()
{
	unsigned char data[4];
	int read = 0;
	while (read < 4)
	{
		int val = recv(sock, (char *) data+read, 4-read, 0);
		if (val == 0 || val == -1) 
		{
			is_connection_lost = true;
			return 0;
		}

		read += val;
	}

	return ntohl(*((unsigned long *) data));
}

void CL_UniformSocket::write_int(unsigned long value)
{
	unsigned long write_num = htonl(value);

	write_data(&write_num, sizeof(unsigned long));
}

void CL_UniformSocket::write_data(void *data, unsigned int size)
{
	unsigned int bytes_send = 0;
	while (bytes_send < size)
	{
		int b = ::send(sock, ((char *) data)+bytes_send, size-bytes_send, 0);
#ifdef WIN32
		if (b == SOCKET_ERROR)
#else
		if (b == -1)
#endif
		{
#ifdef WIN32
			std::cout << "Error: " << WSAGetLastError() << std::endl;
			std::cout << "sock is: " << sock << std::endl;
#endif
			is_connection_lost = true;
			return;
		}

		bytes_send += b;
	}
}

bool CL_UniformSocket::try_connect(unsigned long remote_ip_network_format, int port)
{
	sockaddr_in sock_addr;
	sock_addr.sin_family = AF_INET;
	sock_addr.sin_addr.s_addr = remote_ip_network_format;
	sock_addr.sin_port = htons(port);
	
	int res = ::connect(sock, (sockaddr *) &sock_addr, sizeof(sockaddr_in));
	if (!res)
	{
		write_int(NETWORK_MAGIC);

		return true;
	}
	else 
	{
		return false;
	}
}

/******************************************************************************
					CL_UniformAcceptSocket implementation
******************************************************************************/

CL_UniformAcceptSocket::CL_UniformAcceptSocket(CL_ConnectionProvider *provider)
: CL_UniformSocket(provider)
{
	port = -1;
}

CL_UniformAcceptSocket::~CL_UniformAcceptSocket()
{
}

bool CL_UniformAcceptSocket::bind(int _port)
{
	if (!init_socket()) return false;

	sockaddr_in addr;
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_port = htons(_port);

	// Bind the socket to the specified port
	int err = ::bind(sock, (sockaddr *) &addr, sizeof(sockaddr_in));	
	if (err == -1) return false;

	// Call getsockname to determine allocated port-number,
	// since socket might have been created with _port==0 (any)
#ifdef WIN32
	int len = sizeof(sockaddr_in);
	err = getsockname(sock, (sockaddr *) &addr, &len);
#else
	unsigned int len = sizeof(sockaddr_in);
	err = getsockname(sock, (sockaddr *) &addr, (socklen_t*) &len);
#endif
	if (err == -1) return false;

	port = ntohs(addr.sin_port);
	
	// Make the socket listen for incoming connection attempts
	err = listen(sock, 64);
	if (err == -1) return false;

	return true;
}

bool CL_UniformAcceptSocket::peek()
{
	fd_set rfds;

	FD_ZERO(&rfds);
	FD_SET(sock, &rfds);

	timeval timeout;
	memset(&timeout, 0, sizeof(timeval));

	int retval = select(sock+1, &rfds, NULL, NULL, &timeout);
	cl_assert(retval != -1);

	if (retval == 0) return false;
	else return true;
}

CL_UniformSocket *CL_UniformAcceptSocket::accept()
{
	fd_set rfds;
	FD_ZERO(&rfds);
	FD_SET(sock, &rfds);

	timeval timeout;
	memset(&timeout, 0, sizeof(timeval));

	int retval = select(sock+1, &rfds, NULL, NULL, &timeout);
	if (retval == -1 || retval == 0) 
	{
		return NULL;
	}

	int res_sock = ::accept(sock, NULL, NULL);
	cl_assert(res_sock != CL_INVALID_SOCKET);

	CL_UniformSocket *ret = new CL_UniformSocket(provider);
	ret->init_socket(res_sock);

	return ret;
}

bool CL_UniformAcceptSocket::try_connect(unsigned long remote_ip_network_format, int port)
{
	cl_assert(false);

	return false;
}

CL_ConnectionPacket CL_UniformAcceptSocket::receive()
{
	cl_assert(false);
	CL_ConnectionPacket ret;
	return ret;
}

void CL_UniformAcceptSocket::send(CL_ConnectionPacket message)
{
	cl_assert(false);
}

bool CL_UniformAcceptSocket::connection_lost()
{
	cl_assert(false);
	return false;
}

/******************************************************************************
					CL_UniformUDPConnection implementation
******************************************************************************/

CL_UniformUDPConnection::CL_UniformUDPConnection()
{
	sock = CL_INVALID_SOCKET;
	port = -1;
}

CL_UniformUDPConnection::~CL_UniformUDPConnection()
{
#ifdef WIN32
	if (sock != CL_INVALID_SOCKET) closesocket(sock);
#else
	if (sock != CL_INVALID_SOCKET) close(sock);
#endif
}

bool CL_UniformUDPConnection::bind(unsigned int _port)
{
	int res;

	sock = socket(AF_INET, SOCK_DGRAM, 0);
	if (sock == CL_INVALID_SOCKET) return false;

	sockaddr_in addr;
	memset(&addr, 0, sizeof(sockaddr_in));
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_port = htons(_port);

	if (_port != 0)
	{
		res = ::bind(sock, (sockaddr *) &addr, sizeof(addr));
		if (res == -1)
		{
#ifdef WIN32
			closesocket(sock);
#else
			close(sock);
#endif
			return false;
		}
	}

	int enable = 1;
	res = setsockopt(
		sock,
		SOL_SOCKET,
		SO_BROADCAST,
		(const char *) &enable,
		sizeof(int));

	if (res == -1)
	{
		std::cout << "ClanLib Network: Could not allow broadcast!" << std::endl;
#ifdef WIN32
		closesocket(sock);
#else
		close(sock);
#endif
		return false;
	}

	// Call getsockname to determine allocated port-number,
	// since socket might have been created with _port==0 (any)
#ifdef WIN32
	int len = sizeof(sockaddr_in);
	res = getsockname(sock, (sockaddr *) &addr, &len);
#else
	unsigned int len = sizeof(sockaddr_in);
	res = getsockname(sock, (sockaddr *) &addr, (socklen_t*) &len);
#endif
	if (res == -1)
	{
		std::cout << "ClanLib Network: Could not determine allocated port number!" << std::endl;
#ifdef WIN32
		closesocket(sock);
#else
		close(sock);
#endif
		return false;
	}

	port = ntohs(addr.sin_port);

	return true;
}

bool CL_UniformUDPConnection::peek()
{
	if (sock == CL_INVALID_SOCKET) return false;

	fd_set rfds;

	FD_ZERO(&rfds);
	FD_SET(sock, &rfds);

	timeval timeout;
	memset(&timeout, 0, sizeof(timeval));

	int retval = select(sock+1, &rfds, NULL, NULL, &timeout);
	if (retval == -1) return false;

	return (retval > 0);
}

CL_UDPConnectionPacket CL_UniformUDPConnection::receive()
{
	char *receive_buf = new char[8096];

	sockaddr_in addr;
	memset(&addr, 0, sizeof(sockaddr_in));
	addr.sin_family = AF_INET;
#ifdef WIN32
	int len_addr = sizeof(addr);
#else
	unsigned int len_addr = sizeof(addr);
#endif
	int res = ::recvfrom(sock, receive_buf, 8096, 0, (sockaddr *) &addr, &len_addr);
	cl_assert(res != -1);

	CL_UDPConnectionPacket ret;
	ret.data = receive_buf;
	ret.size = res;
	ret.ip_addr = addr.sin_addr.s_addr;
	ret.port = ntohs(addr.sin_port);

	return ret;
}

void CL_UniformUDPConnection::send(CL_UDPConnectionPacket message)
{
	sockaddr_in addr;
	memset(&addr, 0, sizeof(sockaddr_in));
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_BROADCAST /*message.ip_addr*/;
	addr.sin_port = htons(message.port);

	cl_assert(sock != CL_INVALID_SOCKET);
	cl_assert(message.size < 2000);

	int res = ::sendto(
		sock,
		(char *) message.data,
		message.size,
		0,
		(sockaddr *) &addr,
		sizeof(sockaddr_in));

	if (res == -1)
	{
		std::cout << "sendto failed: " << strerror(errno) << std::endl;
	}

	cl_assert(res != -1);
}

void CL_UniformUDPConnection::broadcast(CL_UDPConnectionPacket message)
{
	cl_assert(sock != CL_INVALID_SOCKET);
	cl_assert(message.size < 2000);

	sockaddr_in addr;
	memset(&addr, 0, sizeof(sockaddr_in));
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_BROADCAST;
	addr.sin_port = htons(message.port);

	int res = ::sendto(
		sock,
		(char *) message.data,
		message.size,
		0,
		(sockaddr *) &addr,
		sizeof(sockaddr_in));
		
	if (res == -1)
	{
		std::cout << "sendto failed: " << strerror(errno) << std::endl;
	}

	cl_assert(res != -1);
}

unsigned int CL_UniformUDPConnection::get_port()
{
	return port;
}

#endif // USE_NETWORK


