/*
	$Id: font.cpp,v 1.1.1.1 2000/04/09 12:18:00 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------

	File purpose:
		Simple Font class.

*/

#include "Core/precomp.h"
#include <API/Core/System/error.h>
#include <API/Core/Font/font.h>
#include <API/Core/Font/font_description.h>
#include <Core/Font/resourcetype_font.h>
#include <API/Core/Resources/resource.h>
#include <API/Core/Resources/resource_manager.h>
#include <API/Core/Display/surface.h>
#include <API/Core/System/cl_assert.h>
#include <API/Core/Display/surfaceprovider.h>

CL_Font *CL_Font::load(
	const char *res_id,
	CL_ResourceManager *manager)
{
	CL_Resource *res = manager->get_resource(std::string(res_id));
	cl_assert(res != NULL);

	if (res->get_type() != "font")
		throw CL_Error("Resource is not a font!!!");

	res->load();

	CL_Font_Resource *res2 = (CL_Font_Resource *) res;
	cl_assert(res2->get_font_desc() != NULL);

	return new CL_Font(res2->get_font_desc());
}


/*******************
   CL_Font class
*******************/

CL_Font::CL_Font(
	CL_Font_Description *font_desc)
{
	space_len = font_desc->space_len;
	subtract_width = font_desc->subtract_width;

	int len = font_desc->letters.length();
	for (int j=0; j<256; j++) chars[j] = NULL;

	for (int i=0; i<len; i++)
	{
		if (i == 0) height = font_desc->letter_providers[i]->get_height();
	
		chars[(unsigned char) font_desc->letters[i]] = CL_Surface::create(
			font_desc->letter_providers[i],
			false);
	}
}

CL_Font::~CL_Font()
{
	for (int i=0; i<256; i++)
	{
		if (chars[i]!=NULL) delete chars[i];
	}
}

int CL_Font::get_text_width(const char *_text)
{
	int len = 0;

	unsigned char *text = (unsigned char *) _text;

	while (*text != 0)
	{
		if (chars[*text] != NULL)
		{
			len += chars[*text]->get_width()-subtract_width;
		}
		else
		{
			len += space_len;
		}

		text++;
	}

	return len;
}

void CL_Font::print_left(int x, int y, const char *_text)
{
	int pos_x = x;

	unsigned char *text = (unsigned char *) _text;

	while (*text != 0)
	{
		if (chars[*text] != NULL)
		{
			int width = chars[*text]->get_width()-subtract_width;

			chars[*text]->put_screen(pos_x, y);

			pos_x += width;
		}
		else
		{
			pos_x += space_len;
		}

		text++;
	}
}

void CL_Font::print_left(int x, int y, float sx, float sy, const char *_text)
{
	int pos_x = x;

	unsigned char *text = (unsigned char *) _text;

	while (*text != 0)
	{
		if (chars[*text] != NULL)
		{
			int width = (int) ((chars[*text]->get_width()-subtract_width)*sx);

			chars[*text]->put_screen(pos_x, y, sx, sy);

			pos_x += width;
		}
		else
		{
			pos_x += (int) (space_len*sx);
		}

		text++;
	}
}

void CL_Font::print_center(int x, int y, const char *text)
{
	print_left(x - get_text_width(text)/2, y, text);
}

void CL_Font::print_right(int x, int y, const char *text)
{
	print_left(x - get_text_width(text), y, text);
}

void CL_Font::put_target(int x, int y, const char *_text, CL_Target *target, int alignment)
{
	int pos_x;
	unsigned char *text = (unsigned char *) _text;
		
	switch(alignment)
	{
		case ALIGN_CENTER:
			pos_x = x - get_text_width(_text)/2;
			break;
		case ALIGN_RIGHT:
			pos_x = x - get_text_width(_text);
			break;
		default:
			pos_x = x;
	}	

	
	while (*text != 0)
	{
		if (chars[*text] != NULL)
		{
			int width = chars[*text]->get_width()-subtract_width;

			chars[*text]->put_target(pos_x, y, 0, target);

			pos_x += width;
		}
		else
		{
			pos_x += space_len;
		}

		text++;
	}
}



