/*
	$Id: inputsource.h,v 1.1.1.1 2000/04/09 12:17:59 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

//! component="I/O Data"	

#ifndef header_inputsource
#define header_inputsource

#include "../../stl.h"

class CL_InputSource
//: Interface to read data.
// CL_InputSource is used to read data from different types of input sources.
// In most cases, this is regular files and ClanLib zlib compressed datafiles.
// <br>
// To access some system specific input source types, such as datafiles and
// regular ones, use the static create functions in the CL_InputSourceProvider
// class.
//!also: CL_InputSourceProvider - Interface to open input sources with.
{
public:
	virtual ~CL_InputSource() { return; }

	virtual void set_system_mode()=0;
	//: Changes input data endianess to the local systems mode.

	virtual void set_big_endian_mode()=0;
	//: Changes input data endianess to big endian mode.

	virtual void set_little_endian_mode()=0;
	//: Changes input data endianess to little endian mode. This is the 
	//: default setting.

	virtual int read_int32()=0;
	//: Reads a signed 32 bit integer from input source.
	//!retval: The integer read.

	virtual unsigned int read_uint32()=0;
	//: Reads a unsigned 32 bit integer from input source.
	//!retval: The unsigned integer read.

	virtual short read_short16()=0;
	//: Reads a signed 16 bit integer (short) from input source.
	//!retval: The short read.

	virtual unsigned short read_ushort16()=0;
	//: Reads a unsigned 16 bit integer (short) from input source.
	//!retval: The unsigned short read.

	virtual char read_char8()=0;
	//: Reads a signed 8 bit integer (char) from input source.
	//!retval: The char read.

	virtual unsigned char read_uchar8()=0;
	//: Reads a unsigned 8 bit integer (char) from input source.
	//!retval: The unsigned char read.

	virtual float read_float32()=0;
	//: Reads an ieee format floating point from input source.
	//!retval: The float read.

	virtual int read(void *data, int size)=0;
	//: Reads larger amounts of data (no endian and 64 bit conversion).
	//!param: data - Points to an array where the read data is stored.
	//!param: size - Number of bytes to read.
	//!retval: Num bytes actually read.
	
	virtual void open()=0;
	//: Opens the input source. By default, it is open.
	
	virtual void close()=0;
	//: Closes the input source.

	virtual CL_InputSource *clone() const =0;
	//: Make a copy of the current InputSource, standing at the same position.
	//!retval: The clone of the input source.

	enum SeekEnum
	{
		seek_set,	 // from beginning of index
		seek_cur,	 // from current position
		seek_end	 // from end of index
	};

	virtual int tell() const =0;
	//: Returns current position in input source.
	//!retval: Current position in input source.

	virtual void seek(int pos, SeekEnum seek_type)=0;
	//: Seeks to the specified position in the input source.
	//!param: pos - Position relative to 'seek_type'.
	//!param: seek_type - Defines what the 'pos' is relative to. Can be either seek_set, seek_cur og seek_end.

	virtual int size() const =0;
	//: Returns the size of the input source
	//!retval: Size of the input source.

	virtual std::string read_string()=0;
	// Reads a string from the input source.
	// <br>
	// The binary format expected in the input source is first an uint32 telling the length of the
	// string, and then the string itself.
	//!retval: The string read.

	virtual void push_position()=0;
	//: Pushes the current input source position. The position can be restored again with pop_position.
	
	virtual void pop_position()=0;
	//: Pops a previous pushed input source position (returns to the position).
};

#endif
