;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;; $Id: form-elements.lisp,v 1.134 2002/03/11 16:43:41 jesse Exp $
;;;
;;; Copyright (c) 1999, 2000, 2001 onShore Development, Inc.
;;;
;;; See the file 'COPYING' in this directory for terms.

(in-package :imho)

;; ------------------------------------------------------------
;; framework class: html-form
;;
;; A form html-element

(defclass html-form (target frame-targetting-mixin html-element)
  ((transient-elements
    :initform nil)
   (submitting
    :accessor submitting
    :initform nil)
   (caller
    :accessor form-caller
    :initarg :caller
    :initform nil)))

(defmethod element-method ((form html-form))
  "process-form-submit")

;; ------------------------------------------------------------
;; framework class: form-submit-element

(defclass form-submit-element ()
  ((action
    :accessor form-action
    :initarg :action
    :initform nil)))

;; form processor that looks at the list of children of this form,
;; finds the element that caused the submission, and runs the method.

(define-wm process-form-submit ((element html-form))
  (flet ((default-processor (element)
             (intern (concatenate 'string "PROCESS-" (symbol-name (type-of element)))
                     (symbol-package (type-of element)))))
    (setf (submitting element) t)
    (let ((action nil))
      (block process
        ;; search children for subtypes of form-submit-element
        ;; retrieve the action of the first one found
        (mapc #'(lambda (elem)
                  (if (and (subtypep (type-of elem) 'form-submit-element)
                           (element-value elem)
                           (form-action elem))
                      (progn
                        ;; (cmsg "Invoking ~A action" (element-internal-name elem))
                        (setq action (form-action elem))
                        (return-from process))))
              (reverse (html-element-all-children element))))
      (apply (etypecase action
               (function (progn
                           (when *warn-deprecated*
                             (cmsg "WARNING: old-style Lambda found in form action"))
                           action))
               (null (let ((wm (lookup-wm (default-processor element))))
                       (if wm
                           (wmethod-body wm)
                           #'ignore)))
               (symbol (if (lookup-wm action)
                           (wmethod-body (lookup-wm action))
                           (progn
                             (when *warn-deprecated*
                               (cmsg "WARNING: old-style function found in form action")
                               (symbol-function action))))))
             (list element)))))

(defmethod render-html :around ((element html-form) stream)
  (with-slots (element-external-name window reference submitting)
    element
    (html-stream
     stream
     ((:form :method :post
             :enctype "multipart/form-data"
             :target (or window "")
             :name element-external-name
             :action (element-url element))
      (call-next-method)))
    (setf submitting nil)))

;; ------------------------------------------------------------
;; framework class: html-form-element

(defclass html-form-element (html-element)
  ((valid
    :accessor validation-state
    :initform t)))

(defun get-form-value (name request)
  "returns the value of form element named NAME in REQUEST"
  (let ((value (cadr (assoc name
                            (request-client-content request)
                            :test #'equal))))
    (if (< 0 (length value)) value nil)))
  
(defmethod take-values-from-request ((element html-form-element) request)
  ;; propogate the values thru our children elements
  (call-next-method)
  ;; snag any value for our external name and stuff it in the  value slot
  (setf (element-value element)
        (get-form-value (element-external-name element)
                        request)))

(defclass labelled ()
  ((label
    :accessor field-label
    :initarg :label
    :initform nil))
  )
  
(defmethod render-html :around ((labelled labelled) stream)
  (with-slots (label)
    labelled
    (if label
        (html-stream
         stream
         ((:table :border 0 :cellspacing 0 :cellpadding 0)
          (:tr
           ((:td :valign :middle)
            (write-string label stream)
            (write-string "&nbsp;" stream))
           ((:td :valign :bottom)
            (call-next-method)))))
        (call-next-method))))

(defclass scriptable-mixin ()
  ((onchange
    :initform nil
    :initarg :onchange)))

;; ------------------------------------------------------------
;; html-element: popup-list
;;
;; If 'rows' is 1, this is drawn as a drop-down list; if > 1, as a
;; conventional list.

(defclass popup-list (html-form-element scriptable-mixin labelled)
  ((items
    :accessor items
    :initform nil
    :initarg :items
    :documentation
    "assoc list, or function returning alist, of items appearing in
the list")
   (rows
    :accessor rows
    :initform 1
    :initarg :rows
    :documentation
    "The number of visible rows")
   (filter
    :initform nil)
   (multiple
    :accessor multiple-selection?
    :initform nil
    :initarg :multiple-select
    :documentation
    "whether or not the widget allows more than one item to be
selected at once")))

(defmethod displayed? ((element popup-list) item)
  (with-slots (filter)
    element
    (or (null filter)
        (funcall filter item))))

(defmethod selected? ((element popup-list) item)
  (member item (listify (element-value element))))

(defmethod render-html ((element popup-list) stream)
  (with-slots ((name element-external-name)
               (size rows)
               value multiple onchange)
    element
    (with-tag (:stream stream :tag "SELECT"
                       :attr `(("SIZE" . ,size)
                               ("NAME" . ,name)
                               ,@(if onchange `(("ONCHANGE" . ,onchange)))
                               ,@(if multiple '(("MULTIPLE" . "Y")))))
      (dolist (item (items element))
        (when (displayed? element item)
          (with-tag (:stream stream :tag "OPTION"
                             :attr `(("VALUE" . ,(car item))
                                     ,@(if (selected? element (car item))
                                           '(("SELECTED" . "Y")))))
            (write-string (cadr item) stream)))))))

;; snag a list of values for our external name and stuff them in the
;; value slot

(defmethod take-values-from-request ((element popup-list) request)
  (with-slots (element-external-name multiple)
    element
    (flet ((is-popup-value? (form-value)
             (string= element-external-name (car form-value)))
           (keywordify (form-value)
		       ;;
		       ;; if value is an integer, leave it as a string
		       ;; else make it a keyword
		       ;;
		       (or (if (ignore-errors (parse-integer (cadr form-value)))
			       (cadr form-value)
			   (intern (cadr form-value) :keyword)))))
      (let ((values (request-client-content request)))
        (setq values (mapcar #'keywordify (filter-list #'is-popup-value? values)))
        (when (not multiple)
          (setq values (car values)))
        (setf (element-value element) values)))))

;; ------------------------------------------------------------

(defclass auto-list (html-form-element scriptable-mixin labelled)
  ((driver :initarg :driver)
   (any    :initarg :any
           :initform nil)
   (none   :initarg :none
           :initform nil)))

(defmethod render-html ((self auto-list) stream)
  (with-slots (value driver any none)
    self
    (with-slots (data key-fn label-fn)
      driver
      (let ((selected-key (and value (funcall key-fn value)))
            (data (etypecase data
                    (list data)
                    (function (funcall data)))))
        (html-stream
         stream
         ((:select :size 1
		   :name (element-external-name self))
          (when any
            (if (equal value :any)
                (html-stream
                 stream
                 ((:option :value "AUTO-LIST-ANY" :selected "Y")
                  (:princ "Any")))
                (html-stream
                 stream
                 ((:option :value "AUTO-LIST-ANY")
                  (:princ "Any")))))
          (when none
            (if (null value)
                (html-stream
                 stream
                 ((:option :value "AUTO-LIST-NONE" :selected "Y")
                  (:princ "None")))
                (html-stream
                 stream
                 ((:option :value "AUTO-LIST-NONE")
                  (:princ "None")))))
          (when (or any none)
            (html-stream
             stream
             ((:option :value "AUTO-LIST-SEP")
              (:princ "  ----  "))))
          (dolist (row data)
            (let ((this-key (funcall key-fn row))
                  (label (funcall label-fn row)))
              (if (equal selected-key this-key)
                  (html-stream
                   stream
                   ((:option :value this-key :selected "Y")
                    (:princ label)))
                  (html-stream
                   stream
                   ((:option :value this-key)
                    (:princ label))))))))))))

(defmethod take-values-from-request ((self auto-list) request)
  (with-slots (value driver element-external-name)
    self
    (with-slots (key-fn data)
      driver
      (let* ((data (typecase data
                     (list data)
                     (function (funcall data))))
             (fvalue (cadar (member element-external-name (request-client-content request)
                                    :key #'car :test #'string=)))
             (evalue (cond ((string= fvalue "AUTO-LIST-NONE")
                            nil)
                           ((string= fvalue "AUTO-LIST-ANY")
                            :any)
                           (t
                            (car (member fvalue data
                                         :key (lambda (x) (princ-to-string (funcall key-fn x)))
                                         :test #'string=))))))
        (setf value evalue)))))

(defclass list-driver ()
  ((data     :initarg :data)
   (key-fn   :initarg :key)
   (label-fn :initarg :label)))

(defun make-list-driver (data &optional key label)
  (make-instance 'list-driver
                 :data data
                 :key (or key #'identity)
                 :label (or label #'identity)))

;; ------------------------------------------------------------
;; html-element: submit-button

(defclass submit-button (html-form-element form-submit-element frame-targetting-mixin)
  ())

(defmethod render-html ((element submit-button) stream)
  (with-slots (element-external-name value window)
    element
    (html-stream stream ((:input :type "submit" :target window
                                 :name element-external-name :value value)))))

;; ------------------------------------------------------------
;; html-element: file-upload

(defclass file-upload (html-form-element)
  ())

(defmethod render-html ((element file-upload) stream)
  (setf (element-value element) nil)
  (with-slots (element-external-name)
    element
    (html-stream stream ((:input :type :file :name element-external-name)))))

;; ------------------------------------------------------------
;; html-element: image-submit-button

(defclass image-submit-button (html-form-element image form-submit-element)
  ())

(defmethod render-html ((element image-submit-button) stream)
  (with-slots (source width height enabled element-external-name)
    element
    (if (not enabled)
        (call-next-method)
        (let ((standard (get-image-url source :default))
              (highlit (get-image-url source :highlit))
              (clicked (get-image-url source :clicked)))
          (scripted-element-init "IMHOLoadImage" element-external-name standard highlit clicked "")
          (with-tag2 ("a" :attr `(("href" . "");; ,(element-url element))
                                  ("name" . ,element-external-name)
                                  ("value" . "clicked")
                                  ("onmouseover" . ,(format nil "return CSIShow('~a',1)" element-external-name))
                                  ("onmouseout" . ,(format nil "return CSIShow('~a',0)" element-external-name))
                                  ("onmousedown" . ,(format nil "document.~A.submit(); CSIShow('~a',2); return CSClickReturn();"
                                                            (target-name element) element-external-name))))
            (call-next-method))))))

;; ------------------------------------------------------------
;; html-element: text-field

(defclass text-field (html-form-element form-submit-element targeted)
  ((submit-on-return
    :initarg :submit-on-return
    :initform nil)
   (visible
    :initform t)
   (columns
    :initarg :cols
    :initform 30)))

(defmethod element-value ((field text-field))
  (or (call-next-method) nil))

(defmethod display-value ((field text-field))
  (element-value field))

(defmethod render-html ((element text-field) stream)
  (with-slots (element-external-name visible columns submit-on-return)
    element
    (let* ((value (display-value element))
           (string (typecase value
                     (function	(funcall value))
                     (string value)
                     (number (format nil "~D" value))
                     (t ""))))
      (html-stream
       stream
       ((:input :type (if visible "TEXT" "PASSWORD")
                :size (format nil "~d" columns)
                :onkeyup (if submit-on-return
                             (format nil "if (CheckReturn(event)) { form.action='~a'; form.submit(); } return false;"
                                     (element-url element))
                             "")
                :name element-external-name
                :value string))))))

;; ------------------------------------------------------------
;; html-element: text-area

(defclass text-area (html-form-element labelled)
  ((wrap-type
    :initarg :wrap
    :initform :hard)
   (columns
    :initarg :cols
    :initform 40)
   (rows
    :initarg :rows
    :initform 5)
   (max-input-length
    :initform 7500)
   (hscroll
    :initarg :hscroll
    :initform nil)
   ))

(defmethod text-area-wrap-attribute ((field text-area))
  (with-slots (wrap-type)
    field
    (case wrap-type
      (:hard
       "HARD")
      (t
       "HARD"))))

(defmethod element-value :around ((self text-area))
  (let ((value (call-next-method)))
    (with-slots (max-input-length)
      self
      (cond ((not value)
             "")
            ((< max-input-length (length value))
             (cmsg "Length of text: ~d: truncating" (length value))
             (subseq value 0 max-input-length))
            (t
             value)))))

(defmethod render-html ((self text-area) stream)
  (with-slots (element-external-name rows columns)
    self
    (let* ((value (element-value self))
           (string (typecase value
                     (string    value)
                     (function  (funcall value))
                     (integer   (format nil "~d" value))
                     (float     (format nil "~f" value))
                     (t         ""))))
      (html-stream
       stream
       ((:textarea :type :textarea
                   :wrap (text-area-wrap-attribute self)
                   :rows rows
                   :cols columns
                   :name element-external-name)
        (write-string string stream))))))
  
;; ------------------------------------------------------------
;; html-element: password-field

(defclass password-field (text-field)
  ())

(defmethod initialize-instance ((self password-field) &rest initargs)
  (declare (ignore initargs))
  (call-next-method)
  (setf (slot-value self 'visible) nil))
   
;; ------------------------------------------------------------
;; html-element: checkbox

(defclass checkbox (html-form-element labelled)
  ())

(defmethod set-element-value ((self checkbox) value)
  (setf (slot-value self 'value)
        (not (null value))))

(defmethod render-html ((self checkbox) stream)
  (with-slots (element-external-name)
    self
    (format stream "<INPUT TYPE='CHECKBOX' NAME='~a' ~@[CHECKED~]>"
            element-external-name
            (element-value self))))

;; ------------------------------------------------------------
;; html-element: radio-button

(defclass radio-button (html-form-element labelled)
  ((group
    :initarg :group
    :initform :radio
    :documentation
    "Group to which a radio button belongs")
   (checked
    :initarg :checked
    :initform nil
    :documentation
    "Whether this button is initially checked in its group"))
  )

;; ------------------------------------------------------------
;; A radio button returns t if it's on, nil otherwise

(defmethod element-value ((self radio-button))
  (let* ((gname          (slot-value self 'group))
         (self           (slot-value self 'element-external-name))
         (received-value (cadr
                          (assoc
                           gname (request-client-content *active-request*)
                           :test #'string-equal))))
    (string-equal received-value self)))

(defmethod render-html ((self radio-button) stream)
  (with-slots (element-external-name group checked)
    self
    (if checked
        (html-stream
         stream
         ((:input :type "radio"
                  :name group
                  :checked t
                  :value element-external-name)))
        (html-stream
         stream
         ((:input :type "radio"
                  :name group
                  :value element-external-name))))))

;; ------------------------------------------------------------
;; html-element: reset-button

(defclass reset-button (html-form-element)
  ())

(defmethod render-html ((self reset-button) stream)
  (html-stream
   stream
   ((:input :type :reset))))

(defclass image-reset-button (html-form-element image)
  ())

(defmethod render-html ((self image-reset-button) stream)
  (with-slots (source width height enabled element-external-name)
    self
    (if (not enabled)
        (call-next-method)
        (let ((standard (get-image-url source :default))
              (highlit (get-image-url source :highlit))
              (clicked (get-image-url source :clicked)))
          (scripted-element-init "IMHOLoadImage" element-external-name standard highlit clicked "")
          (html-stream
           stream
           ((:a :href ""
                :onmouseover (format nil "return CSIShow('~a',1)" element-external-name)
                :onmouseout (format nil "return CSIShow('~a',0)" element-external-name)
                :onmousedown (format nil "document.~A.reset(); CSIShow('~a',0); return CSClickReturn();"
                                     (target-name self) element-external-name))
            (call-next-method)))))))

   
;; ------------------------------------------------------------
;; html-element: file-chooser

(defclass file-chooser (html-form-element)
  ())

(defmethod render-html ((self file-chooser) stream)
  (html-stream
   stream
   ((:input :type :file))))
   
;; ------------------------------------------------------------
;; html-element: image-input

(defclass image-input (html-form-element form-submit-element image)
  ((source
    :initarg :source
    :initform nil)))
  
(defmethod render-html ((self image-input) stream)
  (with-slots (element-external-name source border)
    self
    (html-stream
     stream
     ((:input :type "image"
              :border border
              :name element-external-name
              :src source)))))

(defmethod take-values-from-request ((self image-input) request)
  (setf (element-value self) nil)
  (let ((valuex (cadr (assoc (concatenate 'string
                                          (element-external-name self)
                                          ".x")
                             (request-client-content request)
                             :test #'equal))))
    (if valuex
        (setf (element-value self) t))))

;; ------------------------------------------------------------
;; html-element: button

(defclass button (targeted html-form-element enabling-mixin)
  ()
  (:documentation
   "An HTML form button"))

(defmethod set-element-value ((self button) value)
  (when (stringp value)
    (setf (slot-value self 'value) value)))

(defmethod render-html ((self button) stream)
  (with-slots (element-external-name text);; THINKME: enabled
    self
    (let ((onclick (format nil "form.target = '~a'; form.action='~a'; form.submit();"
                           ;; an actionless button is simply process-form-submit
                           (find-element-target self)
                           (element-url self))))
      (html-stream
       stream
       ((:input :type "button"
                :onclick onclick
                :name element-external-name
                :value (element-value self)))))))

;; repeater

(defclass repeater (html-element)
  ((repeated-elements :initform nil)
   (repeated-children :initform nil)
   (result-values)))

(defmethod render-html :before ((self repeater) stream)
  (declare (ignore stream))
  (with-slots (repeated-elements result-values)
    self
    (setf repeated-elements nil
          result-values nil)))

(defmethod take-values-from-request :after ((self repeater) request)
  (flet ((check-string (string)
           (and string
                (< 0 (length string))
                string)))
    (let ((form-values (request-client-content request)))
      (dolist (elt (slot-value self 'repeated-elements))
        (when-bind (value (car (member (car elt) form-values :key #'car :test #'string=)))
          (when (check-string (cadr value))
            (push (nreverse (cons (cadr value) (cdr elt)))
                  (slot-value self 'result-values))))))))

(defmethod repeat-value ((self repeater) object name &key (test #'eql))
  (flet ((match (value)
           (and (funcall test object (first value))
                (eql name (second value))
                (third value))))
    (dolist (value (slot-value self 'result-values))
      (when-bind (value (match value))
        (return-from repeat-value value)))
    (when-bind (element (gethash name (slot-value self 'children)))
               (element-value element))))

(defmethod repeat-values-by-name ((self repeater) name)
  (flet ((match (value)
           (and (eql name (second value))
                (third value))))
    (let (results)
      (dolist (value (slot-value self 'result-values))
        (when (match value)
          (push (car value) results)))
      results)))

(defmethod repeat-values-by-name-value ((self repeater) name)
  (flet ((match (value)
           (and (eql name (second value))
                (third value))))
    (let (results)
      (dolist (value (slot-value self 'result-values))
        (when (match value)
          (push (cons (first value) (third value)) results)))
      results)))

(defmethod repeat-value-lite ((self repeater) name)
  (flet ((match (value)
           (and (eql name (first value))
                (second value))))
    (dolist (value (slot-value self 'result-values))
      (when-bind (value (match value))
        (return-from repeat-value-lite value)))))

;; This one returns an object reference

(defmethod repeat-element ((self repeater) stream name object form-element-type &rest form-element-attributes)
  (with-slots (repeated-elements)
    self
    (let ((ext-name (symbol-name (gensym))))
      (push (cons ext-name (list name object)) repeated-elements)
      (format stream "<INPUT TYPE=\"~A\" NAME=\"~A\" ~{~A=\"~A\" ~}>"
              (string form-element-type)
              (string ext-name) form-element-attributes))))

(defmethod repeat-element-2 ((self repeater) stream name object element-class element-value &rest element-initargs)
  (with-slots (repeated-elements)
    self
    (let ((child (apply #'make-child `(,self ,element-class ,name ,@element-initargs))))
      (push (cons (element-external-name child) (list name object)) repeated-elements)
      (setf (element-value child)
            element-value)
      (preawake child)
      (awake child)
      (render-html child stream))))

;; this one just returns a name

(defmethod repeat-element-lite ((self repeater) stream name type &rest initargs)
  (with-slots (repeated-elements)
    self
    (let ((ext-name (symbol-name (gensym))))
      (push (cons ext-name (list name)) repeated-elements)
      (format stream "<INPUT TYPE=\"~A\" NAME=\"~A\" ~{~A=\"~A\" ~}>"
              (string type)
              (string ext-name) initargs))))
