#include "cp_types.h"
#include "cp_proto.h"

/*  ==============  Routines for choosing the color settings ============ */
/* Note that calling program must set color_ramp_size, default
   being COLOR_RAMP; this is the number of colors in the blue-red
   color ramp. mid=color_ramp_size/2 is white, 1 to (mid-1) are dark to
   light blue, mid+1 to 2*mid are light to dark red. 

   When CirclePack is executed with the -colorfile <filename> command
   line option, a ramp size can be set and special color codes can
   be assigned to indices. See the read_rgb_file procedure. For
   each color specified, a saturated version (.75 times the rgb value)
   is also recorded so color circle/faces can have a darker edge
   with same basic color as fill. */

extern int color_ramp_size;
extern char next[];

int radius_col_comp(struct p_data *p)
     /* color gradations based on radii. */
{
  double b,t;
  int i,mid;
  struct R_data *pR_ptr;

  pR_ptr=p->packR_ptr;
  mid=(int)(color_ramp_size/2);
  b=t=pR_ptr[1].rad;
  for (i=2;i<=p->nodecount;i++)
    {
      if (pR_ptr[i].rad>t) t=pR_ptr[i].rad;
      if (pR_ptr[i].rad<b) b=pR_ptr[i].rad;
    }
  if (b<0) b=0.0;
  if (t<=0 || fabs(t-b)/t<.005) /* problem of small variation */
    {
      for (i=1;i<=p->nodecount;i++) p->packK_ptr[i].color=mid;
      return 1;
    }
  if (p->hes<0)
    for (i=1;i<=p->nodecount;i++)
      {
	p->packK_ptr[i].color=
	  1+(int)((mid-2)*(pR_ptr[i].rad-b)/(t-b));
      }
  else if (p->hes==0)
    for (i=1;i<=p->nodecount;i++)
      {
	p->packK_ptr[i].color=
	  1+(int)((mid-2)*(pR_ptr[i].rad-t)/(b-t));
      }
  return 1;
} /* radius_col_comp */

int area_col_comp(struct p_data *p)
     /* color gradations, index 1-19 (blues). */
{
  double b,t,*areas;
  int i,flag=1,mid;
  struct R_data *pR_ptr;

  pR_ptr=p->packR_ptr;
  mid=(int)(color_ramp_size/2);
  t=0.0;
  b=h_area(pR_ptr[p->faces[1].vert[0]].rad,
	   pR_ptr[p->faces[1].vert[1]].rad,
	   pR_ptr[p->faces[1].vert[2]].rad);
  if ( !(areas=(double *)malloc((p->facecount+1)*sizeof(double))) ) 
    return 0;
  if (p->hes<0)
    for (i=1;i<=p->facecount;i++)
      {
	areas[i]=h_area(pR_ptr[p->faces[i].vert[0]].rad,
			pR_ptr[p->faces[i].vert[1]].rad,
			pR_ptr[p->faces[i].vert[2]].rad);
	t=(areas[i]>t) ? areas[i]:t;
	b=(areas[i]<b) ? areas[i]:b;
      }
  else if (p->hes==0)
    for (i=1;i<=p->facecount;i++)
      {
	areas[i]=e_area(pR_ptr[p->faces[i].vert[0]].rad,
			pR_ptr[p->faces[i].vert[1]].rad,
			pR_ptr[p->faces[i].vert[2]].rad);
	t=(areas[i]>t) ? areas[i]:t;
	b=(areas[i]<b) ? areas[i]:b;
      }
  if (b==0.0) flag=0;
  if (t==0 || fabs(t-b)/t<.005)
    for (i=1;i<=p->facecount;i++) p->faces[i].color=mid;
  else for (i=1;i<=p->facecount;i++)
    {
      p->faces[i].color=
	1+(int)((mid-2)*(areas[i]-t)/(b-t));
    }
  free(areas);
  return (flag);
} /* area_col_comp */

int h_compare_area(struct p_data *p,struct p_data *q)
     /* compare ratio of hyperbolic areas of faces from p and q. 
In 2-color ramp, lower indices for q larger, uppers for p larger. */
{
  int i,node,mid;
  double b=1.0,ratio,*areas_p,*areas_q;
  struct R_data *pR_ptr,*qR_ptr;

  pR_ptr=p->packR_ptr;
  qR_ptr=q->packR_ptr;
  mid=(int)(color_ramp_size/2);
  node=(p->facecount>q->facecount) ? q->facecount : p->facecount;
  if (!( areas_p=(double *)malloc((node+1)*sizeof(double)) )
      || !( areas_q=(double *)malloc((node+1)*sizeof(double)) ) ) 
    return 0;
  for (i=1;i<=node;i++)
    {
      areas_p[i]=h_area(pR_ptr[p->faces[i].vert[0]].rad,
			pR_ptr[p->faces[i].vert[1]].rad,
			pR_ptr[p->faces[i].vert[2]].rad);
      areas_q[i]=h_area(qR_ptr[q->faces[i].vert[0]].rad,
			qR_ptr[q->faces[i].vert[1]].rad,
			qR_ptr[q->faces[i].vert[2]].rad);
      ratio=areas_q[i]/areas_p[i];
      if (ratio>b) b=ratio;
      if ((1.0/ratio)>b) b=1.0/ratio;
    }
  if (b==1) for (i=1;i<=p->facecount;i++) p->faces[i].color=mid;
  else
    {
      for (i=1;i<=node;i++)
	{
	  if ((ratio=areas_p[i]/areas_q[i])>1.0)
	    p->faces[i].color=
	      (int)(mid+(mid-1)*(ratio-1.0)/(b-1.0));
	  else 
	    p->faces[i].color=
	      1+(int)((mid-2)*(1.0-(1.0/ratio-1.0)/(b-1.0)));
	}
    }
  free(areas_p);free(areas_q);
  if (node<p->facecount) for (i=node+1;i<=p->facecount;i++)
    p->faces[i].color=mid;
  return 1;
} /* h_compare_area */

int h_compare_cl(struct p_data *p,struct p_data *q)
     /* compare ratio of hyp radii of packs p and q, put results into 
colors of pack p. Use 2-color ramp, lower indices for q larger, 
uppers for p larger. Data is scaled. */
{
  int i,node,mid;
  double b=1.0,ratio;
  struct K_data *pK_ptr;
  struct R_data *pR_ptr,*qR_ptr;

  pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
  qR_ptr=q->packR_ptr;
  mid=(int)(color_ramp_size/2);
  node=(p->nodecount>q->nodecount) ? q->nodecount : p->nodecount;
  for (i=1;i<=node;i++)
    {
      if (pR_ptr[i].rad>0.0 && qR_ptr[i].rad>0.0)
	{
	  ratio=log(pR_ptr[i].rad)/log(qR_ptr[i].rad);
	  if (ratio>b) b=ratio;
	  else if ((1.0/ratio)>b) b=1.0/ratio;
	}
    } /* determine b for scaling purposes */
  if (b==1.0) 
    {
      for (i=1;i<=node;i++)
	{
	  if (pR_ptr[i].rad<=0.0 && qR_ptr[i].rad>0.0)
	    pK_ptr[i].color=color_ramp_size;
	  else if (pR_ptr[i].rad>0.0 && qR_ptr[i].rad<=0.0)
	    pK_ptr[i].color=1;
	  else pK_ptr[i].color=mid;
	}
      return 1;
    }
  for (i=1;i<=node;i++)
    {
      if (pR_ptr[i].rad<=0.0 && qR_ptr[i].rad>0.0)
	pK_ptr[i].color=color_ramp_size;
      else if (pR_ptr[i].rad>0.0 && qR_ptr[i].rad<=0.0)
	pK_ptr[i].color=1;
      else
	{
	  ratio=log(pR_ptr[i].rad)/log(qR_ptr[i].rad);
	  if (ratio>1.0) pK_ptr[i].color=
			   (int)(mid+(mid-1)*(1.0-ratio)/(1.0-b));
	  else pK_ptr[i].color=
		 1+(int)((mid-2)*(1.0-(1.0-1.0/ratio)/(1.0-b)));
	}
    }
  if (node<p->nodecount) for (i=node+1;i<=p->nodecount;i++)
    pK_ptr[i].color=mid;
  return 1;
} /* h_compare_cl */

int e_compare_cl(struct p_data *p,struct p_data *q)
     /* compare ratio of eucl radii of packs p and q, put results into 
colors of pack p. Use 2-color ramp, lower indices indicating q larger, 
uppers indicating p larger. Data is scaled. */
{
  int i,node,mid;
  double b=1.0,ratio;
  struct K_data *pK_ptr;
  struct R_data *pR_ptr,*qR_ptr;

  pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
  qR_ptr=q->packR_ptr;

  mid=(int)(color_ramp_size/2);
  node=(p->nodecount>q->nodecount) ? q->nodecount : p->nodecount;
  for (i=1;i<=node;i++)
    {
      ratio=pR_ptr[i].rad/qR_ptr[i].rad;
      if (ratio>b) b=ratio;
      else if ((1/ratio)>b) b=1/ratio;
    } /* determine b for scaling purposes */
  if (b==1.0) 
    {
      for (i=1;i<=node;i++) pK_ptr[i].color=mid;
      return 1;
    }
  for (i=1;i<=node;i++)
    {
      ratio=pR_ptr[i].rad/qR_ptr[i].rad;
      if (ratio>1.0) pK_ptr[i].color=
		       (int)(mid+(mid-1)*(1.0-ratio)/(1.0-b));
      else pK_ptr[i].color=
	     1+(int)((mid-2)*(1.0-(1.0-1.0/ratio)/(1.0-b)));
    }
  if (node<p->nodecount) for (i=node+1;i<=p->nodecount;i++)
    pK_ptr[i].color=mid;
  return 1;
} /* e_compare_cl */

int e_compare_area(struct p_data *p,struct p_data *q)
     /* compare ratio of eucl areas of faces from p and q. Use 2-color 
ramp, lower indices indicating q larger, uppers indicating p larger. */
{
  int i,node,flag=1,mid;
  double b=1.0,ratio,*areas_p,*areas_q;
  struct R_data *pR_ptr,*qR_ptr;

  pR_ptr=p->packR_ptr;
  qR_ptr=q->packR_ptr;
  mid=(int)(color_ramp_size/2);
  node=(p->facecount>q->facecount) ? q->facecount : p->facecount;
  if (!(areas_p=(double *)malloc((node+1)*sizeof(double)) ) 
      || !(areas_q=(double *)malloc((node+1)*sizeof(double)) ) ) 
    return 0;
  for (i=1;i<=node;i++)
    {
      areas_p[i]=e_area(pR_ptr[p->faces[i].vert[0]].rad,
			pR_ptr[p->faces[i].vert[1]].rad,
			pR_ptr[p->faces[i].vert[2]].rad);
      areas_q[i]=e_area(qR_ptr[q->faces[i].vert[0]].rad,
			qR_ptr[q->faces[i].vert[1]].rad,
			qR_ptr[q->faces[i].vert[2]].rad);
      ratio=areas_q[i]/areas_p[i];
      if (ratio>b) b=ratio;
      if (1.0/ratio>b) b=1.0/ratio;
    }
  if (b==1) flag=0;
  else
    {
      for (i=1;i<=node;i++)
	{
	  if ((ratio=areas_p[i]/areas_q[i])>1.0)
	    p->faces[i].color=
	      (int)(mid+(mid-1)*(ratio-1.0)/(b-1.0));
	  else 
	    p->faces[i].color=
	      1+(int)((mid-2)*(1.0-(1.0/ratio-1.0)/(b-1.0)));
	}
    }
  free(areas_p);free(areas_q);
  if (node<p->facecount) for (i=node+1;i<=p->facecount;i++)
    p->faces[i].color=mid;
  return (flag);
} /* e_compare_area */
