/* complex_math.c   file of complex arithmetic and utility routines */
/* cc complex_math.c -o complex_math.o -g -c -lm */

#include <math.h>
#include "complex_math.h"
#define OKERR 0.000001

float cAbs(z) complex z;
{
	float a;
	a=(float)sqrt((z.re)*(z.re)+(z.im)*(z.im));
	return (a);
} 

complex cadd(z1,z2) complex z1,z2;
{
	complex z;
	z.re=z1.re+z2.re;
	z.im=z1.im+z2.im;
	return (z);
}

complex cmult(z1,z2) complex z1,z2;
{
	complex z;
	z.re=z1.re*z2.re-z1.im*z2.im;
	z.im=z1.re*z2.im+z2.re*z1.im;
	return (z);
}

complex csub(z1,z2) complex z1,z2;
{
	complex z;
	z.re=z1.re-z2.re;
	z.im=z1.im-z2.im;
	return (z);
}

complex cconj(z1) complex z1;
{
	complex z;
	z.re=z1.re;
	z.im=(-z1.im);
	return (z);
}

complex cdiv(z1,z2) complex z1,z2;
{
	complex z;
	float av;

	av = 1/(cAbs(z2));
	z=cmult(z1,cconj(z2));
	z.re=av*av*z.re;
	z.im=av*av*z.im;
	return (z);
}

float Arg(w) complex w;
{
	float s;
	if (w.re==0)
	 {
		if (w.im==0) return (0);
		if (w.im>0) return (M_PI_2);
		return (-M_PI_2);
	 }
	else if (w.im==0)
	 {
		if (w.re>0) return (0);
		return (M_PI);
	 }
	s=(float)atan(w.im/w.re);
	if (w.re<0)
	 {
		if (w.im>0) return (M_PI+s);
		return (s-M_PI);
	 }
	return (s);
} 

complex cexp(r,t) /* returns complex r*e^(it) */
float r,t;
{
	complex w;
	w.re=r*cos(t); w.im=r*sin(t);
	return (w);
}

complex cLog(w) complex w;
{
	complex z;
	z.im=Arg(w); z.re=log(cAbs(w));
	return (z);
}

float 
inv_dist(z1,r1,z2,r2) /* compute inversive distance between
two eucl circles. Return negative value if overlap more than PI/2.
Return default 1.0 if situation not yet covered. */
complex z1,z2;
float r1,r2;
{
	float ab;

	if (r1>0 && r2>0)
	 {
		ab=cAbs(csub(z1,z2));
		return ((ab*ab-(r1*r1+r2*r2))/(2.0*r1*r2));
	 }
/* need to handle: one plane, r=0, or one negative r */
	return 1.0;
} /* inv_dist */

/* -------------------------- Mobius transform stuff ------- */

Mobius
zero_inf(z0,z1,r0,r1,t) /* sets up Mobius data for mapping centering
eucl circle (z0,r0) at origin, (z1,r1) at infinity, with ratio t
of spherical radius at infinity to that at 0 (so t=1 makes them same
spherical size). */
complex z0,z1;
float r0,r1,t;
{
	Mobius mob;
	float aa,bb,rho,y,x,gadget,rr0,rr1;
	complex w,tmp,ww;

	w=csub(z1,z0);
	rr0=(float)r0;
	rr1=(float)r1;
	if ((x=cAbs(w))>(.00001))
	 {
		y=(x*x+r0*r0-r1*r1)/(2.0*x);
		rho=sqrt(y*y-rr0*rr0);
		gadget=sqrt(fabs((rr0-y+rho)/(rr0-y-rho))*
			fabs((x-rr1-y+rho)/(x-rr1-y-rho)));
		aa=(rho-y)/x;
		bb=(rho+y)/x;
		ww.re=w.re*aa;ww.im=w.im*aa;
		mob.a.re=1.0;mob.a.im=0.0;
		mob.b=csub(ww,z0);
		mob.c.re=t*gadget;mob.c.im=0.0;
		ww.re=w.re*bb;ww.im=w.im*bb;
		tmp=cadd(ww,z0);
		mob.d.re = tmp.re*(-1.0)*t*gadget;mob.d.im=tmp.im*(-1.0)*t*gadget;
		return (mob);
	 }
	mob.a.re=1.0;mob.a.im=0.0;
	mob.b.re=(-1.0)*z0.re;mob.b.im=(-1.0)*z0.im;
	mob.c.re=mob.c.im=0.0;
	mob.d.re=t*sqrt(r0*r1);mob.d.im=0.0;
	mob.flip=1;
	return (mob);
} /* zero_inf */

complex
mob_trans(z,a) /* return value for (a-z)/(1-z*conj(a)) */
complex z,a;
{
	complex w,COMP_UNIT;

	COMP_UNIT.re=1.0;COMP_UNIT.im=0.0;
	w=cdiv(csub(a,z),csub(COMP_UNIT,cmult(z,cconj(a)) ) );
	return (w);
} /* mob_trans */

Mobius
trans_abAB(a,b,A,B) /* return mobius of unit disc with a->A, b->B,
a,b,A,B must be on unit circle. */
complex a,b,A,B;
{
	complex boaa,Boa,boa,Two,One;
	Mobius M;

	M.a.re=M.d.re=1.0;
	M.a.im=M.d.im=0.0;
	M.b.re=M.b.im=M.c.re=M.c.im=0.0;
	M.flip=0;
	if (cAbs(a)< 1-OKERR || cAbs(b)< 1-OKERR ||
	    cAbs(A)< 1-OKERR || cAbs(B)< 1-OKERR )
		/* not close enough to being on circle */
		return M;
	if (cAbs(csub(a,b))<OKERR || cAbs(csub(A,B))<OKERR) 
		/* too close to work with, just rotate a to A */
	 {
		M.a=A;
		return M;
	 }
	boaa=cdiv(b,cmult(a,a));
	Boa=cdiv(B,a);
	boa=cdiv(b,a);
	One.re=1.0;Two.re=2.0;
	One.im=Two.im=0.0;
	M.a=cadd(cmult(boaa,B),cdiv(csub(A,cmult(Two,B)),a));
	M.b=csub(B,cmult(boa,A));
	M.c=csub(boaa,cdiv(B,cmult(a,A)));
	M.d=cadd(One,cmult(boa,csub(cdiv(B,A),Two)));

	return M;
} /* trans_abAB */
	
complex
mob_rotate(z,a) /* return value of z rotated by angle */
complex z;float a;
{
	complex ea,w;

	ea.re=cos(a);ea.im=sin(a);
	w=cmult(z,ea);
	return (w);
} /* mob_rotate */

complex
mob_norm(z,a,b) /* returns value at z of mobius transform of unit
disc which maps a to zero and b to positive x-axis. */
complex z,a,b;
{
	complex w,v;
	float c;

	v=mob_trans(b,a);
	c=cAbs(v);
	if (c<OKERR) return (mob_trans(z,a));
	w=cdiv(mob_trans(z,a),v);
	w.re*=c;
	w.im*=c;
	return (w);
} /* mob_norm */

complex
mob_norm_inv(w,a,b) /* returns preimage of w under mobius of disc
which maps a to zero and b to positive x-axis */
complex w,a,b;
{
	complex z;
	float c;

	z=mob_trans(b,a);
	c=cAbs(z);
	z.re/=c;
	z.im/=c;
	z=mob_trans(cmult(w,z),a);
	return (z);
} /* mob_norm_inv */

complex
mobius(M,z,flag) /* return M(z) (flag==1) or M^{-1} (flag==-1). 
What to do for infinity? */
int flag;
Mobius M;
complex z;
{
	complex a,b,c,d,det,ans;

	if (flag!=(-1)) /* apply M */
	 {
		if (M.flip) z.im *= (-1.0);
		a=cmult(M.c,z);
		b=cadd(M.d,a);
		c=cmult(M.a,z);
		d=cadd(M.b,c);
		return cdiv( d,b);
	 }
	a=cmult(M.a,M.d);
	b=cmult(M.c,M.b);
	det=csub(a,b);
	a=cdiv(M.d,det); 
	b=cdiv(M.b,det); b.re *= (-1.0); b.im *= (-1.0);
	c=cdiv(M.c,det); c.im *= (-1.0); c.re *= (-1.0);
	d=cdiv(M.a,det); 
	ans=cdiv( cadd(cmult(a,z),b),cadd(cmult(c,z),d) );
	if (M.flip) ans.im *= (-1.0);
	return ans;
} /* mobius */

set_reflection(M,ctr,r) /* set coeff's of Mobius giving reflection
in euclidean circle */
Mobius *M;
complex ctr;
float r;
{
	M->a=ctr;
	M->b.re=r*r-cAbs(ctr);M->b.im=0.0;
	M->c.re=1.0;M->c.im=0.0;
	M->d.re=(-1.0)*ctr.re;M->d.im=ctr.im;
	M->flip=1;
} /* set_reflection */

matrix_conj(M)
Mobius *M;
{
	M->a=cconj(M->a);
	M->b=cconj(M->b);
	M->c=cconj(M->c);
	M->d=cconj(M->d);
} /* matrix_conj */

matrix_transpose(M)
Mobius *M;
{
	complex tmp;

	tmp=M->b;
	M->b=M->c;
	M->c=tmp;
} /* matrix_transpose */

Mobius
matrix_product(A,B)	/* returns AB.  What to do about flip? */
Mobius A,B;
{
	Mobius C;

	C.a=cadd(cmult(A.a,B.a),cmult(A.b,B.c));
	C.b=cadd(cmult(A.a,B.b),cmult(A.b,B.d));
	C.c=cadd(cmult(A.c,B.a),cmult(A.d,B.c));
	C.d=cadd(cmult(A.c,B.b),cmult(A.d,B.d));

	return (C);
} /* matrix_product */

complex
det(M)		/* returns the determinant of M */
Mobius M;
{
	complex answer;
	
	answer=csub(cmult(M.a,M.d),cmult(M.c,M.b));

	return(answer);
} /* det */

Mobius
apply_mobius(M,C,flag) /* C is circle in 2x2 matrix form. This returns 
the image of C under M. flag not yet used: eventually apply M (flag==1)
or M^{-1} (flag==-1) */
Mobius M,C;
int flag;
{
	Mobius G,image,image1;
	complex NEG_ONE;

	if (flag==(-1)) return (C); /* not yet ready for inverses */

	NEG_ONE.re=-1.0;NEG_ONE.im=0.0;
	G.a=M.d;		/* G=M^(-1) * det(M) */
	G.b=cmult(NEG_ONE,M.b);
	G.c=cmult(NEG_ONE,M.c);
	G.d=M.a;

	matrix_transpose(&G);
	image1=matrix_product(G,C);
	matrix_transpose(&G);
	matrix_conj(&G);
	image=matrix_product(image1,G);

	return (image);
} /* apply_mobius */

