/*
 * BrGUI.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  BrGUI.cpp
*/
#include <FL/fl_ask.H>          // fl_message()

#include "br_defs.hpp"          // NO_CINEPAINT
#include "BrGUI.hpp"
#include "StatusLine.hpp"       // `__statusline'
#include "plugin_main.hpp"      // cpaint_show_image()

#include "YS/ys_dbg.hpp"        // debug macros CTOR(), DTOR()



/**=================================================================
 *
 * BracketingGUI  --  class  
 *
 * Contents: see "BracketingGUI.hpp".
 * Further here `show_HDR()', which wraps CinePaint functionality.
 *
 * BracketingGUI could be initialized with a `BracketingCore' reference, 
 *   then no longer references to the global `Br' object!
 *
 * ToDo: Overcome using of the global `__progressinfo' and `__statusline'!
 *
 *==================================================================*/

// default Ctor...
BracketingGUI::BracketingGUI() 
{
    CTOR("")
    pFollowCurveWindow = 0;
    pCCDCurveWindow    = 0;
    pShiftMenu         = 0;
    pHelpAbout         = 0;
    pHelpTutorial      = 0;
}

// Dtor...
BracketingGUI::~BracketingGUI()
{
    DTOR("")
    clear();  
    clear_rest();
}

// Delete (secondary) top windows...
void BracketingGUI::clear() 
{
    delete pFollowCurveWindow;  pFollowCurveWindow = 0;
    delete pCCDCurveWindow;     pCCDCurveWindow    = 0;
    delete pShiftMenu;          pShiftMenu         = 0;
    delete pHelpTutorial;       pHelpTutorial      = 0;
    // (the rest in clear_rest()...)
}    
void BracketingGUI::clear_rest()
{
    delete pHelpAbout;          pHelpAbout         = 0; 
}

/**-------------------------------------------------------------------
 * rest_shown()
 *
 * @return: True if any of the rest windows are shown. 
 * 
 * As we don't know whether such windows have been created at all, we
 *   have to reckon also ==0. Therefore ``if (win && win->shown())''.
 *--------------------------------------------------------------------*/
bool BracketingGUI::rest_shown()
{
    return (pHelpAbout && pHelpAbout->shown() ||
            pHelpTutorial && pHelpTutorial->visible());
}


// Checks (including tips) concerning the global object `Br'...

// Is `Br' ready for camera initialization?...
bool BracketingGUI::check_for_init ()
{
    if (Br.imgVec.size() < 2) {
      fl_message("Please open at least two images first."); 
      return false;
    }
    if (Br.imgVec.size_active() < 2) {
      fl_message("Pleae activate at least two images first."); 
      return false;
    }
    return true;
}

// Is `camera' in `Br' ready for use?...
bool BracketingGUI::check_for_camera () 
{                                       
    return (Br.camera() && Br.camera()->nImages() > 1);
}

// Is `camera' ready for use OR `Br' ready for camera init?...
//   (returns true, if camera() ready or init possible)
bool BracketingGUI::check_for_camera_or_init () 
{                                       
    if (Br.camera() && Br.camera()->nImages() > 1) return true;
    return check_for_init();  
}                             

// Have CCD-curves been calculated?...
bool BracketingGUI::check_for_CCD_curves ()
{
    if (!(Br.camera() && Br.camera()-> ccd_curves_ready())) {
      fl_message("Please start Response computation first."); 
      return false;
    }
    return true;
}

/**---------------------------------------------------------
 * init_camera()  --  Br.init_camera() + checks-with-messages
 *---------------------------------------------------------*/
void BracketingGUI::init_camera ()
{ 
  // Check and tip for the emergency case, that a deactivation failed...
  if (!check_for_init()) return;
    
  Br.init_camera();
}

/**---------------------------------------------------------
 * do_FollowCurveWindow()
 *---------------------------------------------------------*/
void BracketingGUI::do_FollowCurveWindow ()
{
  // Checks from times, as `FollowCurveWindow' was not camera==0)-safe...
  //if (!check_for_camera()) return;
  
  //  Create Window, if not yet exists; show() it...
  if (!pFollowCurveWindow) {
    pFollowCurveWindow = new FollowCurveWindow (600, 500);
  }
  pFollowCurveWindow -> show();
    // Falls !shown() (geschlossen worden), macht ein show() es sichtbar,
    //   falls verdeckt, sollte ein show() es nach oben bringen -- in KDE
    //   blinkt's dann leider nur in Arbeitsleiste. Ein zusaetzliches
    //   redraw() hilft leider auch nicht. 
}

/**----------------------------------------------------------------
 * do_CCDCurveWindow()
 *-----------------------------------------------------------------*/
void BracketingGUI::do_CCDCurveWindow() 
{ 
  printf("BracketingGUI::%s(): visible=%d\n", __func__,
    pCCDCurveWindow ? pCCDCurveWindow->visible() : -1);
  
  // Checks from times, as `CCDCurveWindow' was not NO_CURVES-safe...
  //if (!check_for_CCD_curves()) return;

  if (!pCCDCurveWindow) {
    pCCDCurveWindow = new CCDCurveWindow (400, 400);
  } 
  pCCDCurveWindow -> show();
}

/**----------------------------------------------------------------
 * do_ShiftMenu()
 *-----------------------------------------------------------------*/
void BracketingGUI::do_ShiftMenu() 
{
  // Check and tip for the emergency case, that a deactivation failed...
  if (!check_for_camera_or_init()) return;
  
  // Init camera only, if inexistent... (for emergency case only)  
  if (!Br.camera()) Br.init_camera();  // (Or better in Br.make_CCD_curve()?)
    
  if (!pShiftMenu) {
    pShiftMenu = new ShiftMenu (500, 350, "Shifts (Bracketing to HDR)");
  }
  pShiftMenu -> show();
}


/**----------------------------------------------------------------
 * compute_FollowCurves()  --  wrapper for future use
 *-----------------------------------------------------------------*/
void BracketingGUI::compute_FollowCurves()
{
    Br.compute_FollowCurves();
}

/**----------------------------------------------------------------
 * make_CCD_curve()
 * 
 * Bad: use of the global `__statusline' und `__progressinfo'!
 *-----------------------------------------------------------------*/
void BracketingGUI::make_CCD_curve() 
{
  // Check and tip for the emergency case, that a deactivation failed...
  if (!check_for_camera_or_init()) return;
  
  // Init camera only, if inexistent... (for emergency case only)  
  if (!Br.camera()) Br.init_camera();  // (Or better in Br.make_CCD_curve()?)
  
  __statusline -> out("Compute response curves...");
  __progressinfo -> show();
  
  // Compute CCD curves...
  Br.make_CCD_curve();
  
  __progressinfo -> hide();
  __statusline -> out_default();
}

/**----------------------------------------------------------------
 * complete_HDR()
 *  
 * Completes HDR calculation, i.e. if CCD curves already exist, calc
 *   HDR image only, else calc all. Finally the HDR image is shown.
 *-----------------------------------------------------------------*/
void BracketingGUI::complete_HDR () 
{
  // Check and tip for the emergency case, that a deactivation failed...
  if (!check_for_camera_or_init()) return;
  
  // Init camera only, if inexistent... (for emergency case only)  
  if (!Br.camera()) Br.init_camera();   // or better in Br.complete_HDR()?
  
  __statusline -> out("Complete HDR image computation...");
  __progressinfo -> show();
  
  // Complete HDR image computation and show image...
  if (!Br.camera()-> ccd_curves_ready())
    make_CCD_curve();             // includes GUI update!
  
  show_HDR (Br.calc_HDR());
  
  __progressinfo -> hide();
  __statusline -> out_default();
}

/**-----------------------------------------------------
 * make_HDR()  --  init() + CCD() + HDR()
 *-----------------------------------------------------*/
void BracketingGUI::make_HDR () 
{
  // Check and tip for the emergency case, that a deactivation failed...
  if (!check_for_init()) return;
  
  __statusline -> out("Make HDR image...");
  __progressinfo -> show();

  Br.init_camera();         // --> CAMERA_INIT
  make_CCD_curve();         // --> CCD_UPDATED + GUI-update
  show_HDR (Br.calc_HDR()); // --> HDR_UPDATED
  
  __progressinfo -> hide();
  __statusline -> out_default();
}


/**------------------------------------------------------------------
 * show_HDR()...
 *
 * Camera's resp. BracketingCore's make_HDR() and calc_HDR() return an 
 *   Array2D HDR image, which can be shown herewith. Wraps CinePaint
 *   functionality. 
 *
 * Ein des Fragens wuerdiger Platz fuer diese Funktion.
 *-------------------------------------------------------------------*/
void BracketingGUI::show_HDR (const Array2D< Rgb<float> > & A)
{
#ifdef NO_CINEPAINT
  printf("show_HDR(..) dummy for the NO_CINEPAINT test mode\n");
#else
  cpaint_show_image (A.dim2(), A.dim1(), (uchar*)A[0], PRECISION_FLOAT); 
#endif 
}



// The singular global INSTANCE we name "BrGUI"...
BracketingGUI BrGUI;       // default Ctor

// END OF FILE
