/*
 * Camera.hpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  Camera.hpp  
  - 03.11.2005: naechste Entruemplungsaktion:
     o Alle Shift-Sachen entfernt
  ----------
  - entruempelte Variante von "pre_machine.cpp"
  - aber noch immer vollkommen zugemuellt, bedarf dringender Ueberarbeitung! 
    Insbesondere die Shift-Sache auslagern.
    
  Angestrebte Schnittstelle: 
    Klasse, der ein Feld von Array2D's und ein Feld von Belichtungszeiten
    uebergeben wird - oder, wie jetzt, std::vector'en.
  
  Leistungen (zu viel fuer eine Klasse):
   - Statistik 
   - Bildverschiebungen korrigieren 
   - Z-Matrix generieren (wichtigster Punkt)
   - CCD-Kennlinie mittels class `CCDResponse<Real>' berechnen
   - HDR-Bild berechnen
  
  Die Templateparameter `Unsign' und `Real' scheinen mir ziemlich weit oben
  angesiedelt. Geht das schlanker? Insbesondere die Menues sollten nicht
  dupliziert werden. Aber die sowieso inzwischen weg.
  
  Der Uebersichtlichkeit wegen schlage ich vor, alle Funktionen, die nur
  auf *einem* Array2D operieren, aus der Camera-Klasse auszulagern in eine
  Datei "Camera_util.hpp" o.ae. und hier nur das zu behalten, was substanziell
  Beziehungen ueber *mehrere* Bilder hinweg benoetigt.
  
  NOTE: Seitdem in main.cpp mit std::vector gearbeitet wird, meldet der
    Compiler Mehrdeutigkeit hinsichtlich der Templates 'min' und 'max()',
    ein Konflikt zwischen "TNT/tnt_math_utils.hpp" und 
    "/usr/include/g++/bits/stl_algobase.h", das offenbar von std::vector
    inkludiert wird. 
    Ausweg: Entweder eine der beiden namespaces entfernen oder explizites
    TNT::min() oder std::min().
    
    Nebenbei, in "tnt_math_utils.hpp" ist der Rueckgabetyp von min(), max(), 
    swap() der Typ 'T' selber, in "stl_algobase.h" eine Referenz 'const T&'. 
    Auswirkungen? Im ersten Fall wird nochmal kopiert, im zweiten nicht.
    
    Namensfragen:
     n_zvals() o.  nCCDValues() o. nOutValues...?
     nImages() o. n_Images() o. n_images()...?   
    
  TODO:
  o In z_average() werden die ChSum-Werte zum Schluss auch durch nn[i]
    dividiert. Das setzt Floattypen voraus, sonst ja "div". Wenn Summation
    in Integern offengehalten werden soll (sum_t), dann ist nochmal
    zu kopieren in Floattyp. Am besten mit Abfrage
       if (sum_t == float_t)...   // aber das geht wohl nicht
  
  o Eine abstrakte CameraBase-Klasse vorlagern, die alles Datentyp-unabhaengige
    wie nImages() enthaelt. Erlaubt diese Dinge dann abstrakt abzuhandeln.
    
  o Typnamen mit Endung "_t" (a'la `size_t') koennten im C++-Standard bald
    reserviert werden. Daher `sum_t' besser vermeiden, wenigstens `Sum_t'.
   
  o In den shift_...()-Berechnungsfunktionen mit lokalen Kopien der 
    Ausschnitte arbeiten. Das sichert Lokalitaet und vermeidet eventl.
    Swappen bei grossen Dateien.   
    
  o Fortschrittsinfo z.Z. sehr mangelhaft. Ueberlegen, wie diese z.B. aus
    calc_CCD_curve() heraus fuer verschiedene Anwendungen dieser Fkt. zuge-
    schnitten werden koennen. Etwa zwei Parameter `v_start' und `delte_v'
    uebergeben?    

*/
#ifndef Camera_hpp
#define Camera_hpp


//#include <iostream>
//#include <sstream>    // wozu?
#include <cstdio>       // printf()
#include <cstdlib>      // calloc()
#include <cstring>      // strcmp()
#include <limits>       // numeric_limits<>

/*
 * Index range check in TNT classes? (to define before TNTs are included!)
 */
//#define TNT_BOUNDS_CHECK            

#include "TNT/tnt_math_utils.hpp"       // swap()
#include "TNT/tnt_array2d.hpp"
#include "TNT/tnt_array2d_utils.hpp"    // '<<'-Op for Array2D
#include "TNT/tnt_stopwatch_.hpp"

#include "br_types.hpp"             // types ulong, Real
#include "response.hpp"             // CCDResponse
#include "Rgb.hpp"                  // Rgb<..>-Template
#include "Rgb_util.hpp"             // '<<'-Op fuer Rgb's
#include "Br_Image.hpp"             // Sofern im Konstruktor gebraucht
#include "ProgressInfo.hpp"         // ProgressInfo

#define YS_DEBUG
#undef YS_DEBUG
#include "YS/ys_dbg.hpp"



template <class T>      // no inline, vorgesehen fuer "teure" Objekte
T sqr (const T& x)
{
  return x * x;
}

/**
 * Deklaration der hier definierten Templates
 */
//template <class Unsign, class Real>  class Camera;



typedef Rgb<double> CorrRGB;    // Rueckgabetyp der correl-Funktionen
                                // Oder float od. Real?


/**
 * writeFile():
 *  
 * Schreibt Array2D<T> in eine Textdatei im Matrixformat. Formatierung
 * der T's entsprechend ``<<''-Op.
 */
template <class T>
void
writeFile (Array2D<T> A, const char* fname)
{
  printf ("Schreibe Datei %s...\n", fname);

  std::ofstream f(fname);           // ueberschreibt ohne Nachfrage!
  if (!f) 
  { 
    std::cerr << "Konnte " << fname << " nicht oeffnen\n";
    return;
  }
  
  for (int i=0; i < A.dim1(); i++)
  { 
    for (int j=0; j < A.dim2(); j++)   f << A[i][j] << " ";
    f << "\n";
  }
}
/**
 * wie oben, nur wird jetzt fprintf() benutzt und Format ist explizit
 * anzugeben (inklusive Leerzeichen zur Trennung der Daten!). 
 * Bsp: format = "%d ".
 */
template <class T>
void
writeFile (Array2D<T> A, const char* fname, const char* format)
{
  printf ("Schreibe Datei %s...\n", fname);

  FilePtr f (fname, "w+");              // ueberschreibt ohne Nachfrage!
                                        // Vorsicht: wirft!
  for (int i=0; i < A.dim1(); i++)
  { 
    for (int j=0; j < A.dim2(); j++)   fprintf (f, format, A[i][j]);
    fprintf (f, "\n");
  }
}


template <class T>
void list_buf (T* buf, int what)
{
  switch (what)
  {
  case 0:  //  RGB
    for (int i=0, w=0; i<10; i++) {
      printf ("%3i:  R=%i  G=%i  B=%i\n", i, buf[w], buf[w+1], buf[w+2]);
      w += 3;
    }
    break;

  case 1:  // RGBA
    for (int i=0, w=0; i<10; i++) {
      printf ("%3i:  R=%i  G=%i  B=%i  alpha=%i\n",
              i, buf[w], buf[w+1], buf[w+2], buf[w+3]);
      w += 4;
    }
  }
}

// keine Templates, Definitionen in der *.cpp-Datei...

void list (TNT::Array1D<XYindex> A);
void list (TNT::Array2D<unsigned char> A);



/**-----------------------------------------------
 * Area  -  structur
 *
 * Indexgrenzen eines 2d-Gebietes im Sinne `for (i=x1; i<xe; i++)'
 *------------------------------------------------*/
struct Area 
{
  int x1,xe, y1,ye;     

  Area() {}         // Default notwendig, wenn als Klassenelement
  
  Area (int X1, int XE, int Y1, int YE)
    : x1(X1), xe(XE), y1(Y1), ye(YE) {}

  //Area (const ArrayRgb& A)
  //  : x1(0), xe(A.dim2()), y1(0), ye(A.dim1()) {}
};


/**
 * Die Summation muss in double oder uint64 durchgefuehrt werden, zur
 * Aufbewahrung der Ergebnisse aber reichen float. Der Platz fuer 
 * sizeof(2*float + int) = 12 ist sogar kleiner noch als der von ChSum
 * mit sizeof(2*double) = 16.
 */
struct ExpectationFollowValue {
  float  z;     // expectation (follow) value, EX
  float  dz;    // Streuung, DX^2 = EX^2 - (EX)^2 (o. "Dz2" o. "sigma"?)
  size_t n;     // number of values contributed to z-statistic
  
  ExpectationFollowValue() : z(0), dz(0), n(0) {}
};
typedef Array2D <ExpectationFollowValue>  ChannelFollowValues;
 
// Indes, da `n' (zu einem Hauptwert) fuer alle Bilder eines Kanals identisch,
// ist die bildweise Speicherung von `n' Verschwendung; die bessere Struktur
// waere
//
// struct EFV {
//   float  z [nImages];   Feld der Erwartungsfolgewerte aller Bilder
//   float  dz [nImages];  Feld aller zugehoerigen Varianzen
//   size_t n;             Anzahl beitragender Punkte
// };
// Array1D<EFV> (n_zvals);   Feld dieser Daten fuer alle z-Werte eines Kanals
// 
// oder
//
// Array1D<Rgb<EFV> > (n_zvals)  Alle Kanaele zusammen
//
// Fuer Geschw. waere guenstig, nicht `n_zvals'-mal malloc() fuer z und dz
// zu bemuehen mit womoeglich zerstreuten Speicherbloecken, sondern fuer alle
// n_zvals einen Block zu allokieren und intern ansteuern. Aber sowas erst, 
// wenn das Grundverfahren steht.


/**====================================================================
 *
 * class Camera
 * 
 * Variante, die ohne Ruecksicht auf Effizienz zunaechst nur auf's
 * Inhaltliche sieht. Alle Bilddaten werden gleichzeitig im Speicher
 * gehalten, sozwar in bequemen Array2D-Strukturen.
 *
 *======================================================================*/
template <class Unsign, class Real>
class Camera 
{
public:
  
  typedef Array2D< Rgb<Unsign> > ArrayRgb;
  typedef double sum_t;             // sum type: double or uint64

  struct ChSum {                    // "ChannelSum"
    sum_t  s, s2;                   // Sum(z) und Sum(z^2)
    ChSum(): s(0), s2(0)  {}        // default constructor nullt
  };

private:
  
  int       nlayers;             // number of layers = pictures
  ArrayRgb* arrays;              // Feld der Bilderdaten
  unsigned  CCDbits;             // resolution of CCD
  Unsign    z_max;               // max value of CCD output
  size_t    n_;                  // number of z-values (n_zval)
  XYindex*  shifts;              // Verschieb. der Bilder gegen Bild 0
  Area      area_;               // aktuelles Schnittgebiet, stets gueltig,
                                 // gesetzt in intersection_area()
  size_t    width, height,       
            n_pixel;             // Wirkl. alle Bilder von gleicher Dim.?       
            
  ProgressInfo* progressinfo_;

public:  
  ChannelFollowValues  follow_R, follow_G, follow_B;
  
  Array1D<Real> times, logtimes; // exposure time and log. expos. time

public:
  Array1D<Real> crvX_r, crvX_g, crvX_b;   // CCD-Kennlinien
  Array1D<Real> logX_r, logX_g, logX_b;   // logarithm. CCD-Kennlinien
  
  Rgb<XYindex>* channel_shifts;

private:
  double  brightness (const ArrayRgb &A);
  void    list_env (ArrayRgb &A, int x, int y, char channel);
  void    list_env_row (int x, int y, char channel);
  
  CorrRGB correl    (int Nx, int Ny, int picA, int xA, int yA,
                                     int picB, int xB, int yB);
  inline
  CorrRGB correlsym (int nx, int ny, int picA, int xA, int yA,
                                     int picB, int xB, int yB);
  inline
  CorrRGB correlsym3x3 (int picA, int xA, int yA,
                        int picB, int xB, int yB);

  Rgb<ulong>  smoothness4 (const ArrayRgb &A, int x, int y);
  Rgb<ulong>  smoothness4_cross (int x, int y);
  
  Area    intersection_area ();
  void    write_FollowCurve (const char* fname, const ChSum* h, bool);
  
public:
  Camera (Br_ImageVec &,
          std::vector< ArrayRgb > &,
          unsigned int  CCDbits_,
          ProgressInfo* progress = 0);
  ~Camera();
  
  // Die Index-Grenzen des Zielbildes... 
  int xBegin() const { return area_.x1; }
  int xEnd()   const { return area_.xe; }
  int yBegin() const { return area_.y1; }
  int yEnd()   const { return area_.ye; }
  
  int    nImages() const { return nlayers; }  
  int    n_zvals() const { return n_;      }  // 'n_' eigentl. size_t!
  size_t nPoints() const        // Gesamtpunktzahl im *Ziel*bild
    { return size_t(area_.xe - area_.x1) * (area_.ye - area_.y1); }
  
  // Hack for Br.used(..): return data address of the i-th Array2D
  const Rgb<Unsign>* array_addr (int i) const { return arrays[i][0]; }
    
  void z_average (int pic);  
  void z_average_smooth();
  void z_average_corr();
  
  Array2D<Unsign> create_Z    (ChannelFollowValues &, int pic, int nselect);
  Array2D<Unsign> make_test_Z (int nselect);

  void calc_CCD_curve (int pic, int n_grid, Real smooth);
  
  Array2D<Rgb<float> > calc_HDR ();     // reine Rechnung ohne Vorbereitung
  Array2D<Rgb<float> > calc_logHDR ();
  
  // Rechnung inkl. Vorbereitung...
  void                 make_CCD_curve (int refpic, int ngrid, double smooth);   
  Array2D<Rgb<float> > make_HDR       (int refpic, int ngrid, double smooth);   
  Array2D<Rgb<float> > complete_HDR   (int refpic, int ngrid, double smooth);   
  
  bool ccd_curves_ready();      // true, if curves already computed
  
  void read_knnl (const char* fname, Array1D<Real> &X);
  void read_knnl (const char* basename);
  void read_knnl_menu ();
  
  Array1D<double> get_FollowCurve (int pic, char channel);
  
  void
  get_FollowCurve (
      Array1D<double> & Y,      // O: [n_]-Feld der Folgewerte (Ordinate)
      Array1D<double> & YerrLo, // O: [n_]-Feld der unteren Streuung
      Array1D<double> & YerrHi, // O: [n_]-Feld der obereren Streuung
      int             pic,      // I: wanted picture, \in [0..nImages)
      char            channel,  // I: wanted channel, \in {'R','G','B'}
      bool            bounded=true); // I: Wertegrenzen beruecksichtigen?

  void list (const ChannelFollowValues &) const;
  void list_ChSum (const ChSum* h, char c) const;
  void analyze_ChSum (const ChSum*, char c) const;
  
  int x_shift (int i) { return shifts[i].x; }
  int y_shift (int i) { return shifts[i].y; }
  
};


template <class Unsign, class Real>
void Camera<Unsign, Real>::list (const ChannelFollowValues & F) const
{
  for (size_t i=0; i < n_; i++) {
    std::cout << i << ": n=" << F[0][i].n;  // `n' for all identic.
    for (int p=0; p < nImages(); p++)
      std::cout << "\t (z:" << F[p][i].z << "\t dz:" << F[p][i].dz << ")";
    std::cout << std::endl;
  }
}

template <class Unsign, class Real>
void Camera<Unsign, Real>::list_ChSum (const ChSum* h, char c) const
{
  printf("List ChSum '%c'...\n", c);
  for (size_t i=0; i < n_; i++) {
    std::cout << i << ":";
    for (int p=0; p < nImages(); p++)
      std::cout << "\t (s:" << h[p*n_+ i].s 
                << "\t s2:" << h[p*n_+ i].s2 << ")";
    std::cout << std::endl;
  }
}

//-------------------------------------------------------------
// Analyse (lokal benutzter) ChSum-Summationsfelder...
// NB: Rekonstruieren eingebautes [nlayers][n_]-Feld per Hand.
//--------------------------------------------------------------
template <class Unsign, class Real>
void Camera<Unsign, Real>::analyze_ChSum (const ChSum* h, char c) const
{
  printf("Analyze ChSum '%c': z > z_max? ...\n", c);
  for (size_t i=0; i < n_; i++) {
    bool out = false;
    for (int p=0; p < nImages(); p++) {
      int index = p * n_ + i;
      if (!(h[index].s <= z_max)) {  // z.B. auch 'nan' - Div. durch 0
        if (!out) { printf("i=%d,", i); out=true; }
        printf("  p=%d: s=%f", p, h[index].s);  
      }
    }
    if (out) printf("\n");
  }
  printf("Analyze ChSum '%c': s2 < 0? ...\n", c);
  for (size_t i=0; i < n_; i++) {
    bool out = false;
    for (int p=0; p < nImages(); p++) {
      int index = p * n_ + i;
      if (h[index].s2 < 0.0) {    // h[p][i]
        if (!out) { printf("i=%d,", i); out=true; }
        printf("  p=%d: s2=%f", p, h[index].s2);  
      }
    }
    if (out) printf("\n");
  }
}


/**-----------------------------------
 *
 * Constructor...
 *
 *------------------------------------*/
template <class Unsign, class Real>
Camera<Unsign, Real>::Camera 
  (
    Br_ImageVec & imgVec,
    std::vector< ArrayRgb > & arrVec,
    unsigned int CCDbits_,
    ProgressInfo* progress
  )
  : 
  shifts(NULL),
  progressinfo_ (progress)
{
  CTOR("") 
  assert (CCDbits_ <= sizeof(Unsign)*8); // nur fuer unsigned sicher!
  assert (imgVec.size() == arrVec.size()); // size!=nImages bei inakt. Bildern
  
  CCDbits  = CCDbits_;
  z_max    = (1 << CCDbits) - 1;         // Gibt das nicht temp. Ueberlauf?
  n_       = z_max + 1;                  //   Ja, aber macht nichts (= 0-1)

  nlayers  = imgVec.size_active();
  assert (nlayers > 1);                  // 
  
  // Allokieren...
  arrays   = new ArrayRgb[nlayers];             
  times    = Array1D<Real> (nlayers);    // allokiert
  logtimes = Array1D<Real> (nlayers);    // allokiert
  shifts   = (XYindex*) calloc (nlayers, sizeof(XYindex)); // genullt!
                                         // Deshalb calloc() statt new?
  channel_shifts = (Rgb<XYindex>*) calloc (nlayers, sizeof(Rgb<XYindex>));
  
  follow_R = ChannelFollowValues (nlayers, n_);  // allokiert
  follow_G = ChannelFollowValues (nlayers, n_); 
  follow_B = ChannelFollowValues (nlayers, n_);
  
  // aktive Bilder kopieren...
  int k = 0;                            // active index, 0 <= k < nlayers
  for (size_t i=0; i < imgVec.size(); i++)
    if (imgVec[i].active) 
    { 
      arrays  [k] = arrVec[i];          // copy of handles
      times   [k] = imgVec[i].time;
      logtimes[k] = log( times[k] );
      SPUR(("use [%d] \"%s\", time=%f\n", i, imgVec[i].name(), imgVec[i].time))
      k++;
    }
      
  // alle Bilder haben dieselbe Breite und Hoehe (Fraglich!)
  width   = arrays[0].dim2();
  height  = arrays[0].dim1();
  n_pixel = width * height;

  // Startbelegung von `area_' mit shifts[]=0...
  intersection_area();
}


//-----------
// Dtor()...
//-----------
template <class Unsign, class Real>
Camera<Unsign, Real>::~Camera() 
{ 
  DTOR("");
  
  delete[] arrays;
  if (shifts) free (shifts);
  if (channel_shifts) free (channel_shifts);
}

/**
 * write_FollowCurve()
 *
 * Schreibt Feld von gemittelten z-Folgewerten inkl. Grenzen der mittl.
 * quadratischen Abweichung in gnuplot-lesbarer Form in eine Datei.
 *
 * @param fname: file name
 * @param h: [n_]-C-Feld von ChSum-Folgewerten
 * @param bounded: wenn true, dann zHigh und zLow so justiert, dass die 
 *   Fehlerbalken 0 und z_max nicht ueberschreiten; wenn false, symmetrisch
 *   um z-Wert (koennen dann auch irreale Werte wie > z_max annehmen).
 */
template <class Unsign, class Real>
void
Camera<Unsign,Real>::write_FollowCurve (
    const char* fname, const ChSum* h, bool bounded)
{
  FilePtr f (fname, "w+");          // Ueberschreibt ohne Warnung!

  if (bounded)
    for (size_t z=0; z < n_; z++)
    {
      double ds  = 0.5 * sqrt(h[z].s2);
      double s   = h[z].s;
      double sLo = s - ds;
      double sHi = s + ds;

      // weil z-Werte auf [0,..,z_max] beschraenkt...
      if (sHi > z_max) {
        sLo -= (sHi - z_max);      // ...nach unten verschieben
        sHi = z_max;
      } 
      else if (sLo < 0.0) {
        sHi -= sLo;                // ...nach oben verschieben
        sLo = 0.0;
      }
      fprintf (f, "%i %f %f %f\n", z, s, sLo, sHi);
    }
  else
    for (size_t z=0; z < n_; z++)
      fprintf (f, "%i %f %f\n", z, h[z].s, 0.5*sqrt(h[z].s2));
}

/** 
 * get_FollowCurve  -  Plotkurven aus den ChannelFollowValues-Feldern...
 *
 */
template <class Unsign, class Real> 
void 
Camera<Unsign,Real>::get_FollowCurve (
    
    Array1D<double> & Y,      // O: [n_]-Feld der Folgewerte (Ordinate)
    Array1D<double> & YerrLo, // O: [n_]-Feld der unteren Streuung
    Array1D<double> & YerrHi, // O: [n_]-Feld der obereren Streuung
    int             pic,      // I: wanted picture, \in [0..nlayers)
    char            channel,  // I: wanted channel, \in {'R','G','B'}
    bool            bounded   // I: Wertegrenzen beruecksichtigen?
    )
{
  assert (0 <= pic && pic < nImages());
  Y      = Array1D<double> (n_);    // allokiert
  YerrLo = Array1D<double> (n_);    // allokiert
  YerrHi = Array1D<double> (n_);    // allokiert
  const ExpectationFollowValue* h;               
  
  switch (channel) 
  {
    case 'R': h = follow_R[pic]; break;
    case 'G': h = follow_G[pic]; break;
    default : h = follow_B[pic]; 
  }
  
  for (size_t i=0; i < n_; i++)
  {
    if (h[i].dz < 0) printf("i=%d, DZ^2 = %f < 0!\n", i, h[i].dz);
    double s   = h[i].z;        // bedeutet evntl. Konvertierung
    double ds  = 0.5 * sqrt(h[i].dz);     
    double sLo = s-ds;
    double sHi = s+ds;
    
    if (bounded)    // Schranken beruecksichtigen... 
    {
      if (sHi > z_max) {
        sLo -= (sHi - z_max);    // ...nach unten verschieben
        sHi = z_max;
      } 
      else if (sLo < 0.0) {
        sHi -= sLo;              // ...nach oben verschieben
        sLo = 0.0;
      }
    }
    Y[i]      = s;
    YerrLo[i] = sLo;       
    YerrHi[i] = sHi;
  }
}

/**
 * get_FollowCurve  -  Plotkurven aus den ChannelFollowValues-Feldern...
 *
 * @param pic: (I) wanted picture, \in [0..nlayers)
 * @param channel: (I) wanted color channel, \in {'R','G','B'}
 * @return: [n_]-Array1D of Folgewerte follow_?[pic][].z; hier allokiert
 */
template <class Unsign, class Real>
Array1D<double>                         // double weil...?
Camera<Unsign,Real>::get_FollowCurve (int pic, char channel)
{
  assert (0 <= pic && pic < nlayers);
  Array1D<double> Y(n_);        // allokiert
  const ExpectationFollowValue* h;
  switch (channel) 
  {
    case 'R': h = follow_R[pic]; break;
    case 'G': h = follow_G[pic]; break;
    default : h = follow_B[pic]; 
  }
  for (size_t i=0; i < n_; i++) 
    Y[i] = h[i].z;              // double <-- sum_t
  return Y;
}

/**
 * brightness()  -  Helligkeit bestimmen.
 *
 * Moegl. auch, Kanaele zunaechst getrennt und dann Plausibiltaets-Check.
 * Moeglw. auch Summe mittels uint64-Typ, was genauer und vielleicht
 * schneller als mit double. Naja, schneller nicht, ergab Test.
 */
template <class Unsign, class Real>
double
Camera<Unsign,Real>::brightness (const ArrayRgb & A)
{
  Rgb<uint64> s(0);   // uint64 statt double beschleunigt nicht

  for (int i=0; i < A.dim1(); i++)
  for (int j=0; j < A.dim2(); j++)
  {
    s += A[i][j];
  }
  return double(s.r + s.g + s.b) / (A.dim1() * A.dim2());
}

/** 
 * list_env():
 *
 * Listet 3x3-Umgebung um Punkt (x,y) fuer einen Kanal.
 */
template <class Unsign, class Real>
void
Camera<Unsign,Real>::list_env (ArrayRgb & A, int x, int y, char channel)
{
    int x1 = x-1;
    int x2 = x+1;
    int y1 = y-1;
    int y2 = y+1;

    if (x1 < 0) x1 = 0;
    if (x2 >= A.dim2()) x2 = A.dim2()-1;
    if (y1 < 0) y1 = 0;
    if (y2 >= A.dim1()) y2 = A.dim1()-1;

    for (y=y1; y <= y2; y++)
    {   printf ("[%4i,%4i...] ", y,x1);

        for (x=x1; x <= x2; x++)
            switch (channel)
            {
            case 'R': printf ("%3u ", A[y][x].r); break;
            case 'G': printf ("%3u ", A[y][x].g); break;
            case 'B': printf ("%3u ", A[y][x].b); break;
            }
        printf ("\n");
    }
}


template <class Unsign, class Real>
void
Camera<Unsign,Real>::list_env_row (int x, int y, char channel)
{
    int x1 = x-1;
    int x2 = x+1;
    int y1 = y-1;
    int y2 = y+1;

    // passt nur, wenn alle Bilder gleiche Dimension
    if (x1 < 0) x1 = 0;
    if (x2 >= (int)width)  x2 = width-1;
    if (y1 < 0) y1 = 0;
    if (y2 >= (int)height) y2 = height-1;

    for (y=y1; y <= y2; y++)
    {   printf ("[%4i,%4i...] ", y,x1);

        for (int pic=0; pic < nlayers; pic++)
        {   for (x=x1; x <= x2; x++)
                switch (channel)
                {
                case 'R': printf ("%3u ", arrays[pic][y][x].r); break;
                case 'G': printf ("%3u ", arrays[pic][y][x].g); break;
                case 'B': printf ("%3u ", arrays[pic][y][x].b); break;
                }
            printf ("  ");
        }
        printf ("\n");
    }
}


/**
 * smoothness4()  --  Hilfsfkt
 *
 * @return: Summe der quadrat. Abweichung der 4er-Umgebung des Punktes
 *   (x,y) fuer alle drei Kanaele. (D.h. *kein* Mittelwert, keine Div durch N!)
 *   Ergebnis im Wertebereich [0,..., 4 * zmax^2].
 *
 * Wegen Differenzbildung ist Umwandlung in Vorzeicheninteger notwendig.
 * Integer muss nicht nur Bereich -Unsign,...,+Unsign umfassen, sondern auch
 * das Quadrat, wegen Symmetrie der Vorzeichentypen also -zmax^2,...,+zmax^2.
 *
 * Aufrufer hat zu garantieren, dass 1er-Abstand um (x,y) begehbar.
 */
template <class Unsign, class Real>
Rgb<ulong>
Camera<Unsign,Real>::smoothness4 (const ArrayRgb & A, int x, int y)
{
    Rgb<ulong> s;

    const Rgb<Unsign>& z  = A[ y ][ x ];    // spart Index-Ops
    const Rgb<Unsign>& zL = A[ y ][x-1];    // Left
    const Rgb<Unsign>& zR = A[ y ][x+1];    // Right
    const Rgb<Unsign>& zU = A[y-1][ x ];    // Up
    const Rgb<Unsign>& zD = A[y+1][ x ];    // Down

    s.r = sqr<int>(zL.r - z.r) + sqr<int>(zR.r - z.r)
        + sqr<int>(zU.r - z.r) + sqr<int>(zD.r - z.r);

    s.g = sqr<int>(zL.g - z.g) + sqr<int>(zR.g - z.g)
        + sqr<int>(zU.g - z.g) + sqr<int>(zD.g - z.g);

    s.b = sqr<int>(zL.b - z.b) + sqr<int>(zR.b - z.b)
        + sqr<int>(zU.b - z.b) + sqr<int>(zD.b - z.b);

    //printf ("z=%i, L|R=%i,%i  U|D=%i,%i\n", z.r, zL.r, zR.r, zU.r, zD.r);
    return s;
}


/**
 * z_average()
 *
 * Mittlerer Folgewert in jedem Folgebild zum Wert z im Ref-Bild pic.
 * Hier wird shifts[] beruecksichtigt, in histo_ext() nicht.
 *
 * @param pic: Referenzbild
 *
 * Voraussetzungen: Schnittgebiet `area_' bestimmt
 *
 * VORSCHLAG: Die Summation kann lokale Summations-Strukturen benutzen, die
 *  u.U. auf uint64-Typen basieren und Anzahl-Element enthalten kann, und 
 *  erst anschliessend auf Camera-weit bekannte Strukturen umrechnen. 
 *  Letztere koennten dann platzsparend auch aus floats statt doubles
 *  bestehen. Die lokale Summationsstruktur kann auch das Feld fuer das
 *  Ref-Bild einsparen, z.Z. der Einfachheit halber nicht getan.
 *
 * ChSum h[nlayer][n_zval]: 
 *  h[p][z].{s,s2} beschreibt den Folgewert im Bild `p' zum Hauptwert `z' 
 *  im HauptBild `pic' bzgl. Mittelwert (s) und Streuung (s2). Bsp.:
 *  Sei pic==0; Dann sagt  h[3][56].s == 99.1: der mittlere Folgewert in 
 *  Bild 3 zum Hauptwert z=56 (in Bild 0) ist z=99.1. Seine Streuung ist
 *  h[3][56].s2.
 *
 * NOTE: Bei Summe double notwendig fuer Genauigkeit, floats geben u.U.
 *  negative s2.
 */
template <class Unsign, class Real>
void
Camera<Unsign,Real>::z_average (int pic)
{
  DBG
  // lokale Summations-Felder...
  ChSum h_R [nlayers][n_],    // genullt per Default-Constr.
        h_G [nlayers][n_],    // h[p][z] beschreibt den Folgewert in Bild `p'
        h_B [nlayers][n_];    // zum Hauptwert `z' in Bild `pic'

  Rgb<size_t> nn[n_];         // Anzahl Punkte mit Hauptwert z; nicht genullt
  
  for (size_t i=0; i < n_; i++)     // nullen
    nn[i] = Rgb<size_t>(0);
  
  for (int y=yBegin(); y < yEnd(); y++)         
  for (int x=xBegin(); x < xEnd(); x++)
  {
    // Hauptwert:
    Rgb<Unsign> z = arrays [pic] [y + shifts[pic].y] [x + shifts[pic].x];

    // Anzahl Punkte mit diesem z-Hauptwert feststellen; ist zugleich Anzahl 
    // dann natuerlich auch zugehoeriger Folgewerte. Da fuer alle Bilder
    // identisch, ausserhalb der p-Schleife...
    nn[z.r].r ++;      
    nn[z.g].g ++;      
    nn[z.b].b ++;      

    for (int p=0; p < nlayers; p++)
    {
      if (p==pic) continue;
      Rgb<Unsign> zf = arrays [p] [y + shifts[p].y] [x + shifts[p].x];

      h_R[p][z.r].s  += zf.r;
      h_R[p][z.r].s2 += (sum_t)zf.r * zf.r;

      h_G[p][z.g].s  += zf.g;
      h_G[p][z.g].s2 += (sum_t)zf.g * zf.g;

      h_B[p][z.b].s  += zf.b;
      h_B[p][z.b].s2 += (sum_t)zf.b * zf.b;
    }
  }
  //list_ChSum (h_R[0], 'R');
  //analyze_ChSum (h_R[0],'R');
  //analyze_ChSum (h_G[0],'G');
  //analyze_ChSum (h_B[0],'B');
  
  // Mittelwerte und Streuung berechnen und in die globalen Felder
  // `follow_?' eintragen. 
  // NOTE: Die nn[]-Daten sind fuer alle Bilder gleich (Verschwendung!)
  // NOTE: `follow' koennte durchaus auch Rgb-Feld hier sein.
  for (int p=0; p < nlayers; p++)
  {
    if (p==pic)             // Hauptwerte eintragen...
      for (size_t i=0; i < n_; i++) 
      {
        follow_R[p][i].z = i; follow_R[p][i].n = nn[i].r; follow_R[p][i].dz= 0;
        follow_G[p][i].z = i; follow_G[p][i].n = nn[i].g; follow_G[p][i].dz= 0;
        follow_B[p][i].z = i; follow_B[p][i].n = nn[i].b; follow_B[p][i].dz= 0;
      }
    else 
      for (size_t i=0; i < n_; i++)
      {
        int N;  
        double EX, EX2;  // fuer Summe EX^2-EX*EX double, nicht float!
                         // auch nicht (u)llong, wegen Div durch N
        N = nn[i].r;
        if (N) {
          EX  = double(h_R[p][i].s) / N;        // EX
          EX2 = double(h_R[p][i].s2) / N;       // E(X^2)
        } else 
          EX  = EX2 = 0;
        
        follow_R[p][i].n  = N;
        follow_R[p][i].z  = EX;
        follow_R[p][i].dz = EX2 - EX*EX;        // DX^2 = E(X^2) - (EX)^2
      
        N = nn[i].g;
        if (N) {
          EX  = double(h_G[p][i].s) / N;
          EX2 = double(h_G[p][i].s2) / N;
        } else
          EX  = EX2 = 0;
        
        follow_G[p][i].n  = N;
        follow_G[p][i].z  = EX;
        follow_G[p][i].dz = EX2 - EX*EX;
        
        N = nn[i].b;
        if (N) {
          EX  = double(h_B[p][i].s) / N;
          EX2 = double(h_B[p][i].s2) / N;
        } else
          EX  = EX2 = 0;
          
        follow_B[p][i].n  = N;
        follow_B[p][i].z  = EX;
        follow_B[p][i].dz = EX2 - EX*EX;
      }
  }
 
#if 0
  // gnuplot-Kurven schreiben...
  // Hier noch direkt aus den ChSum-Strukturen, die jetzt aber keine
  // Mittelwerte mehr enthalten (nur noch Summen). Waere zu aktuallisieren,
  // am bestenn 1D-Kurven aus follow_?-Feldern mittels get_FollowCurve()... 
  for (int p=0; p < nlayers; p++)
  {
    if (p==pic) continue;
    char  fname[256];
    char* base = "aver";
    bool  bounded = true;

    sprintf (fname, "%s%i_%iR.crv", base, pic, p);  // R-Kanal
    write_FollowCurve (fname, h_R[p], bounded);

    sprintf (fname, "%s%i_%iG.crv", base, pic, p);  // G-Kanal
    write_FollowCurve (fname, h_G[p], bounded);

    sprintf (fname, "%s%i_%iB.crv", base, pic, p);  // B-Kanal
    write_FollowCurve (fname, h_B[p], bounded);
  }
#endif
}

/**
 * z_average_smooth()
 *   
 * Wie z_average(), aber gewichtet mit der Streuung der Punktumgebungen.
 * Weil jetzt jeder Folgewert entsprechend seiner Wichtung eigenen
 * (gebrochenen) Anzahl-Wert hat, in Histo-Struktur Anzahlwert hinzu.
 */
template <class Unsign, class Real>
void
Camera<Unsign,Real>::z_average_smooth()
{
    DBG
    struct Histo 
    {
      double n;
      sum_t  s;
      sum_t  s2;
      int    n_bad;   // nur zur Info

      Histo() : n(0), s(0), s2(0), n_bad(0) {}  // default constr., nullt
    };

    Histo h_R [nlayers-1][n_],      // genullt per Default-Constr.
          h_G [nlayers-1][n_],      // h[p][z] enthaelt den Folgewert von Bild
          h_B [nlayers-1][n_];      // p+1 zum Hauptwert z (in Bild 0)

    //ulong sig_max = 2 * 4 * (ulong)z_max * z_max; // absolutes Maximum
    ulong  sig_max   = 100;                 // willkuerlich gesetzt
    double sig_max_d = double(sig_max);
    //std::cout << "sig_max = " << sig_max << "\n";

    for (int y=yBegin()+1; y < yEnd()-1; y++)
    for (int x=xBegin()+1; x < xEnd()-1; x++)
    {
        Rgb<Unsign> z      = arrays[0][y][x];   // Hauptwert
        Rgb<ulong>  sigma0 = smoothness4 (arrays[0], x,y);

        for (int p=0; p < nlayers-1; p++)
        {
            int xx = x + shifts[p+1].x;
            int yy = y + shifts[p+1].y;

            Rgb<Unsign> zf     = arrays [p+1][yy][xx];
            Rgb<ulong>  sigma1 = smoothness4 (arrays[p+1], xx, yy);
            Rgb<ulong>  sig;
            Rgb<double> w;    // weight

            sig    = sigma0 + sigma1;
            sigma0 = sigma1;            // fuer naechstes p

            if (sig.r < sig_max)
            {   w.r = (sig_max_d - sig.r) / sig_max_d;
                h_R[p][z.r].n  += w.r;
                h_R[p][z.r].s  += w.r * zf.r;
                h_R[p][z.r].s2 += w.r * zf.r * zf.r;
                //printf ("s0=%lu,   s1=%lu,  w.r = %f\n", sigma0.r, sigma1.r, w.r);
            }
            else
                h_R[p][z.r].n_bad ++;

            if (sig.g < sig_max)
            {   w.g = (sig_max_d - sig.g) / sig_max_d;
                h_G[p][z.g].n  += w.g;
                h_G[p][z.g].s  += w.g * zf.g;
                h_G[p][z.g].s2 += w.g * zf.g * zf.g;
            }
            else
                h_G[p][z.g].n_bad ++;

            if (sig.b < sig_max)
            {   w.g = (sig_max_d - sig.g) / sig_max_d;
                h_B[p][z.b].n  += w.b;
                h_B[p][z.b].s  += w.b * zf.b;
                h_B[p][z.b].s2 += w.b * zf.b * zf.b;
            }
            else
                h_B[p][z.b].n_bad ++;
        }
    }

    ChSum c_r [nlayers-1][n_],      // genullt per Default-Constr.
          c_g [nlayers-1][n_],
          c_b [nlayers-1][n_];

    // Mittelwerte und Streuung (TODO: nn[i]==0 ungesichert!!)
    for (int p=0; p < nlayers-1; p++)
    {   for (size_t z=0; z < n_; z++)
        {
            double n = h_R[p][z].n;
            c_r[p][z].s  = h_R[p][z].s  / n;    // EX
            c_r[p][z].s2 = h_R[p][z].s2 / n;    // E(X^2)
            c_r[p][z].s2 -= sqr(c_r[p][z].s);   // E(X^2) - (EX)^2

            n = h_G[p][z].n;
            c_g[p][z].s  = h_G[p][z].s  / n;    // EX
            c_g[p][z].s2 = h_G[p][z].s2 / n;    // E(X^2)
            c_g[p][z].s2 -= sqr(c_g[p][z].s);   // E(X^2) - (EX)^2

            n = h_B[p][z].n;
            c_b[p][z].s  = h_B[p][z].s  / n;    // EX
            c_b[p][z].s2 = h_B[p][z].s2 / n;    // E(X^2)
            c_b[p][z].s2 -= sqr(c_b[p][z].s);   // E(X^2) - (EX)^2
        }
    }
    printf ("z und mittlere Folgewerte (R-Kanal):\n");
    for (size_t z=0; z < n_; z++)
    {   printf ("%3u",z);
        for (int p=0; p < nlayers-1; p++)
            printf ("  %6.1f (%6.2f) n=%.2f  n_bad=%i", c_r[p][z].s,
                sqrt(c_r[p][z].s2), h_R[p][z].n, h_R[p][z].n_bad);
        printf ("\n");
    }
    // gnuplot-Kurven schreiben...
    for (int p=0; p < nlayers-1; p++)
    {   char fname[256];
        char* base = "smooth";

        sprintf (fname, "%s_%iR.crv", base, p+1);   // R-Kanal
        write_FollowCurve (fname, c_r[p], true);

        sprintf (fname, "%s_%iG.crv", base, p+1);   // G-Kanal
        write_FollowCurve (fname, c_g[p], true);

        sprintf (fname, "%s_%iB.crv", base, p+1);   // B-Kanal
        write_FollowCurve (fname, c_b[p], true);
    }
}

/**
 * z_average_corr()
 *   
 * Wie z_average(), aber gewichtet mit der Korrelation der 3x3-Umgebungen
 *   benachbarter Bilder.
 * Grosses Potential zur Effizienzsteigerung: Vorsummen beim Bewegen des
 *   3x3-Fensters ueber's Bild als wohl auch in der p-Schleife ueber alle
 *   Bilder.
 *
 * WICHTIG: Funktioniert nur mit *ausgeschaltetem* HDR_WORKING_RANGE, denn
 *   ein correl()-Ergebnis -2 ("nicht berechenbar") hier nicht interpretiert.
*/
template <class Unsign, class Real>
void
Camera<Unsign,Real>::z_average_corr()
{
    DBG
    struct Histo 
    {
      double  n;            // "gebrochene Anzahl", weil gewichtet
      sum_t   s, s2;

      Histo() : n(0), s(0), s2(0) {}  // defaullt constr. (nullt)
    };

    Histo h_R [nlayers-1][n_],      // genullt per Default-Constr.
          h_G [nlayers-1][n_],      // h[p][z] enthaelt den Folgewert von Bild
          h_B [nlayers-1][n_];      // p+1 zum Hauptwert z (in Bild 0)

    for (int y=yBegin()+1; y < yEnd()-1; y++)
    for (int x=xBegin()+1; x < xEnd()-1; x++)
    {
        Rgb<Unsign> z = arrays[0][y][x];    // Hauptwert

        for (int p=0; p < nlayers-1; p++)
        {
            int xx     = x + shifts[p+1].x;
            int yy     = y + shifts[p+1].y;
            int xx_pre = x + shifts[p].x;   // Koordinaten im Vorgaengerbild
            int yy_pre = y + shifts[p].y;

            Rgb<Unsign> zf   = arrays [p+1][yy][xx];
            Rgb<double> corr = correlsym3x3 (p, xx_pre, yy_pre, p+1,xx,yy);

            Rgb<double> w;      // weight
            w.r = sqr(1.0 + corr.r);        // corr\in [-1,+1] ==> w\in [0,2]
            w.g = sqr(1.0 + corr.g);
            w.b = sqr(1.0 + corr.b);

            h_R[p][z.r].n  += w.r;
            h_R[p][z.r].s  += w.r * zf.r;
            h_R[p][z.r].s2 += w.r * zf.r * zf.r;

            h_G[p][z.g].n  += w.g;
            h_G[p][z.g].s  += w.g * zf.g;
            h_G[p][z.g].s2 += w.g * zf.g * zf.g;

            h_B[p][z.b].n  += w.b;
            h_B[p][z.b].s  += w.b * zf.b;
            h_B[p][z.b].s2 += w.b * zf.b * zf.b;
#ifdef HDR_DEBUG
            printf ("Pkt_pre=(%i,%i)   Pkt=(%i,%i)\n", xx_pre,yy_pre, xx,yy);
            printf ("corr = (%f, %f, %f)\n", corr.r, corr.g, corr.b);
            printf ("   w = (%f, %f, %f)\n", w.r, w.g, w.b);
            if (x==200) return;
#endif
        }
    }

    ChSum c_r [nlayers-1][n_],      // genullt per Default-Constr.
          c_g [nlayers-1][n_],
          c_b [nlayers-1][n_];

    // Mittelwerte und Streuung (TODO: nn[i]==0 ungesichert!!)
    for (int p=0; p < nlayers-1; p++)
    {   for (size_t z=0; z < n_; z++)
        {
            double n = h_R[p][z].n;
            c_r[p][z].s  = h_R[p][z].s  / n;    // EX
            c_r[p][z].s2 = h_R[p][z].s2 / n;    // E(X^2)
            c_r[p][z].s2 -= sqr(c_r[p][z].s);   // E(X^2) - (EX)^2

            n = h_G[p][z].n;
            c_g[p][z].s  = h_G[p][z].s  / n;    // EX
            c_g[p][z].s2 = h_G[p][z].s2 / n;    // E(X^2)
            c_g[p][z].s2 -= sqr(c_g[p][z].s);   // E(X^2) - (EX)^2

            n = h_B[p][z].n;
            c_b[p][z].s  = h_B[p][z].s  / n;    // EX
            c_b[p][z].s2 = h_B[p][z].s2 / n;    // E(X^2)
            c_b[p][z].s2 -= sqr(c_b[p][z].s);   // E(X^2) - (EX)^2
        }
    }
    printf ("z und mittlere Folgewerte (R-Kanal):\n");
    for (size_t z=0; z < n_; z++)
    {   printf ("%3u",z);
        for (int p=0; p < nlayers-1; p++)
            printf ("  %6.1f (%6.2f) n=%.2f", c_r[p][z].s,
                sqrt(c_r[p][z].s2), h_R[p][z].n);
        printf ("\n");
    }

    // gnuplot-Kurven schreiben...
    for (int p=0; p < nlayers-1; p++)
    {   char fname[256];
        char* base = "corr";

        sprintf (fname, "%s_%iR.crv", base, p+1);   // R-Kanal
        write_FollowCurve (fname, c_r[p], true);

        sprintf (fname, "%s_%iB.crv", base, p+1);   // G-Kanal
        write_FollowCurve (fname, c_g[p], true);

        sprintf (fname, "%s_%iB.crv", base, p+1);   // B-Kanal
        write_FollowCurve (fname, c_b[p], true);
    }
}

/**
 * calc_CCD_curve()  
 *
 * Berechnet aus gegebenen `follow_?'-Feldern die drei CCD-Kennlinien und
 * hinterlegt diese in Camera-weiten Array1D's `crvX_?' und `logX_?'.
 *
 * @param pic: Was war Ref-Bild bei der Best. der Folgewerte in histo_?
 * @param nselect: Anzahl selekt. Pixel (Stuetzstellen, Gitterpunkte)
 * @param lambda: Glaettungskoeffizient 
 */  
template <class Unsign, class Real>
void 
Camera<Unsign,Real>::calc_CCD_curve (int pic, int nselect, Real lambda)
{
  bool save = false;   // Kennlinien in Datei schreiben?
  char crv_name[64];

  Array2D<Unsign> Z = create_Z (follow_R, pic, nselect);
  //list (Z);
  //write (Z, "ZmatrixR.dat", "%d ");
  CCDResponse<Unsign,Real> ccd_resp (Z, logtimes, CCDbits);
  ccd_resp.calc (lambda, false);
  logX_r = ccd_resp.get_logXcrv();
  crvX_r = ccd_resp.get_Xcrv();
  std::cout << "dim of logX_r = " << logX_r.dim() << '\n'; 
  if (save) {
    sprintf (crv_name, "%d_%d_%.0fR.knn", pic, nselect, lambda);
    ccd_resp.write_Xcrv (crv_name);         // nicht logarithmiert
    ccd_resp.write_Xcrv (crv_name, true);   // logarithmiert
  }
  if (progressinfo_) progressinfo_ -> value(0.33);

  Z = create_Z (follow_G, pic, nselect); 
  //list (Z);
  //write (Z, "ZmatrixG.dat", "%d ");
  ccd_resp.exchange_Z (Z);
  ccd_resp.calc (lambda, false);
  logX_g = ccd_resp.get_logXcrv();
  crvX_g = ccd_resp.get_Xcrv();
  if (save) {
    sprintf (crv_name, "%d_%d_%.0fG.knn", pic, nselect, lambda);
    ccd_resp.write_Xcrv (crv_name);         // nicht logarithmiert
    ccd_resp.write_Xcrv (crv_name, true);   // logarithmiert
  }
  if (progressinfo_) progressinfo_ -> value(0.66);

  Z = create_Z (follow_B, pic, nselect);
  //list (Z);
  //write (Z, "ZmatrixB.dat", "%d ");
  ccd_resp.exchange_Z (Z);
  ccd_resp.calc (lambda, false);
  logX_b = ccd_resp.get_logXcrv();
  crvX_b = ccd_resp.get_Xcrv();
  if (save) {
    sprintf (crv_name, "%d_%d_%.0fB.knn", pic, nselect, lambda);
    ccd_resp.write_Xcrv (crv_name);         // nicht logarithmiert
    ccd_resp.write_Xcrv (crv_name, true);   // logarithmiert
  }
  if (progressinfo_) progressinfo_ -> value(1.0);
}



/**
 * calc_HDR() & calc_logHDR():
 *
 * @return: [logarithm.] HDR<float>-Bild, derzeit einfach die primaer gemaess
 *   log(E) = log(X) - log(t) aus der Rechnung kommenden log(E)-Daten.
 *
 * Voraussetzungen: CCD-Kennlinien logX... bereits berechnet!
 *
 * NOTE: Zur Effizienzsteigerung koennte wght() tabelliert werden.
 *  Aber dann am besten schon bei der Kennlinienberechnung. Das ist aber
 *  schon eine andere Klasse, dann also global verfuegbare, sprich,
 *  separate Wichtungsklasse.
 *
 * NOTE: Wenn ein Pixel in allen Bildern auf einem Randwert (0 oder zmax),
 *  dann wegen Wichtung sum_w=0 und sum_e=0 und wir bekaemen "0/0", also
 *  numerisch Ueberlauf. Tatsaechlich wollen wir dann minimalen
 *  HDR-Wert (E) fuer z=0 und maximalen fuer z=zmax. Der kleinste reale
 *  HDR-Wert (den diese Belichtungsreihe erfassen koennte) ist der zu z=0
 *  im Bild mit der *laengsten* Belichtungszeit, der groesste noch erfassbare
 *  der zu z=z_max im Bild mit der *kuerzesten* Bel-Zeit.
 *  Gemaess logE = logX - logtimes und dem Umstand, dass die Bilder in
 *  aufsteigender Bel-Zeit geordnet, also
 *      logE_min = logX[0] - logtimes[nlayers-1]
 *      logE_max = logX[z_max] - logtimes[0]
 */
template <class Unsign, class Real>
Array2D< Rgb<float> >
Camera<Unsign,Real>::calc_logHDR()
{
  if (progressinfo_) progressinfo_ -> text("Create log HDR image...");
  if (progressinfo_) progressinfo_ -> value(0.1);
  
  zWeighting<Unsign> wght (0, z_max); // Wichtungsfunktionsobjekt.

  int dim_x = xEnd() - xBegin();    // dimension of the HDR-image
  int dim_y = yEnd() - yBegin();

  Array2D< Rgb<float> > hdr (dim_y, dim_x);   // allokiert

  Rgb<float> val_min (logX_r[0] - logtimes[nlayers-1],
                      logX_g[0] - logtimes[nlayers-1],
                      logX_b[0] - logtimes[nlayers-1]);
  Rgb<float> val_max (logX_r[z_max] - logtimes[0],
                      logX_g[z_max] - logtimes[0],
                      logX_b[z_max] - logtimes[0]);
  // Mit ``Rgb<T> - T''-Operator auch zu schreiben
  // val_min (Rgb<float>(logX_r[0],...,...) - (float)logtimes[..]);

  float max_float = std::numeric_limits<float>::max();
  Rgb<float> h_min ( max_float);  // groesstmoegl. Wert
  Rgb<float> h_max (-max_float);  // kleinstmoegl. Wert

  Stopwatch uhr;
  double zeit;
  uhr.start();

  for (int y=yBegin(); y < yEnd(); y++)
  for (int x=xBegin(); x < xEnd(); x++)
  {
    Rgb<double> sum_w (0.0);
    Rgb<double> sum_e (0.0);
    for (int p=0; p < nlayers; p++)
    {
      Rgb<Unsign> z = arrays [p] [y + shifts[p].y] [x + shifts[p].x];
      Rgb<double> w (wght(z.r), wght(z.g), wght(z.b));
      sum_w += w;
      // sum_e += w * Rgb<double>(..., ..., ...)
      sum_e.r += w.r * (logX_r[z.r] - logtimes[p]);
      sum_e.g += w.g * (logX_g[z.g] - logtimes[p]);
      sum_e.b += w.b * (logX_b[z.b] - logtimes[p]);
    }
    Rgb<float>& h = hdr [y-yBegin()][x-xBegin()];

//  h = sum_e / sum_w;                // float <- double
//  ^^^^^^^^^^^^^^^^^^
//  So, falls sum_w != 0 garantiert. Falls nicht, wie folgt. Ob
//  sum_w=0 dabei durch z=0 oder z=zmax verursacht worden war, wird
//  festgestellt, indem der Wert in (irgend)einem Bild angeschaut wird,
//  vorzugsweise in einem, fuer das das besonders billig (shifts[p]=0).
//  Theoretisch ist sum_w=0 auch in Kombination von z=0 UND z=zmax
//  moeglich. Aber pathologisch und hier nicht beruecksichtigt.

    if (sum_w.r != 0.0)
        h.r = sum_e.r / sum_w.r;        // float <- double
    else 
    {   if (arrays[0][y][x].r == 0)     // p=0, damit shifts[p]=0, um
            h.r = val_min.r;            // `+ shifts[p].y' zu sparen.
        else                            // Also z=zmax
            h.r = val_max.r;
    }

    if (sum_w.g != 0.0)
        h.g = sum_e.g / sum_w.g;        // float <- double
    else 
    {   if (arrays[0][y][x].g == 0)     // p, fuer das shifts[p]=0
            h.g = val_min.g;
        else                            // also z=zmax
            h.g = val_max.g;
    }

    if (sum_w.b != 0.0)
        h.b = sum_e.b / sum_w.b;        // float <- double
    else 
    {   if (arrays[0][y][x].b == 0)     // p, fuer das shifts[p]=0
            h.b = val_min.b;
        else                            // also z=zmax
            h.b = val_max.b;
    }

    if (h.r < h_min.r) h_min.r = h.r; else
    if (h.r > h_max.r) h_max.r = h.r;

    if (h.g < h_min.g) h_min.g = h.g; else
    if (h.g > h_max.g) h_max.g = h.g;

    if (h.b < h_min.b) h_min.b = h.b; else
    if (h.b > h_max.b) h_max.b = h.b;
/*
    if (sum_w.r == 0.0)
    { printf ("pixel(%d,%d),  sum_e.r=%f", x,y, sum_e.r);
      for (int p=0; p < nlayers; p++)
        printf ("  %d", arrays [p] [y + shifts[p].y] [x + shifts[p].x].r);
      printf ("\n");
    }
*/
  }
  // Weil bei streng fallenden Folgen obige else-Zweige nie erreicht
  // wuerden, pruefen, ob h_max geaendert wurde; wenn nicht, ist
  // Maximalwert der erste, denn wie gesagt: streng fallend...
  
  if (h_max.r == -max_float) h_max.r = hdr[0][0].r;
  if (h_max.g == -max_float) h_max.g = hdr[0][0].g;
  if (h_max.b == -max_float) h_max.b = hdr[0][0].b;

  zeit = uhr.stop();
  printf ("logHDR-Zeit: %f sec\n", zeit);

  std::cout << "val_min = " << val_min << '\n';
  std::cout << "val_max = " << val_max << '\n';
  std::cout << "min = " << h_min << '\n';
  std::cout << "max = " << h_max << '\n';

  // logarithm. Wertebereich (-infty, +infty). Normieren auf [0,1]...
  
  Rgb<float> fac = Rgb<float>(1.0) / (h_max - h_min);

  for (int y=0; y < dim_y; y++)
  for (int x=0; x < dim_x; x++)
    hdr[y][x] = fac * (hdr[y][x] - h_min);    // --> \in [0...1]

  if (progressinfo_) progressinfo_ -> value(1.0);
  return hdr;
}

//----------------------------------------------------------------------
// Auch hier wird mit log. Kennlinien gearbeitet. Warum gleich noch mal?
//----------------------------------------------------------------------
template <class Unsign, class Real>
Array2D< Rgb<float> >
Camera<Unsign,Real>::calc_HDR()
{
  if (progressinfo_) progressinfo_ -> text("Create HDR image...");
  if (progressinfo_) progressinfo_ -> value(0.1);
  
  zWeighting<Unsign> wght (0, z_max); // Wichtungsfunktionsobjekt

  int dim_x = xEnd() - xBegin();      // dimension of the HDR-image
  int dim_y = yEnd() - yBegin();

  Array2D< Rgb<float> > hdr (dim_y, dim_x);     // allokiert

  Rgb<float> val_min (exp (logX_r[0] - logtimes[nlayers-1]),
                      exp (logX_g[0] - logtimes[nlayers-1]),
                      exp (logX_b[0] - logtimes[nlayers-1]));
  Rgb<float> val_max (exp (logX_r[z_max] - logtimes[0]),
                      exp (logX_g[z_max] - logtimes[0]),
                      exp (logX_b[z_max] - logtimes[0]));

  float max_float = std::numeric_limits<float>::max();
  Rgb<float> h_min ( max_float);  // groesstmoegl. Wert
  Rgb<float> h_max (-max_float);  // kleinstmoegl. Wert

  Stopwatch uhr;
  double zeit;
  uhr.start();

  for (int y=yBegin(); y < yEnd(); y++)
  for (int x=xBegin(); x < xEnd(); x++)
  {
    Rgb<double> sum_w (0.0);
    Rgb<double> sum_e (0.0);
    
    for (int p=0; p < nlayers; p++)
    {
      Rgb<Unsign> z = arrays [p] [y + shifts[p].y] [x + shifts[p].x];
      Rgb<double> w (wght(z.r), wght(z.g), wght(z.b));

      sum_w += w;
//    sum_e += w * Rgb<double>(..., ..., ...)
      sum_e.r += w.r * (logX_r[z.r] - logtimes[p]);
      sum_e.g += w.g * (logX_g[z.g] - logtimes[p]);
      sum_e.b += w.b * (logX_b[z.b] - logtimes[p]);
    }
    Rgb<float>& h = hdr [y - yBegin()][x - xBegin()];

//  So koennten wir schreiben, wenn sum_w!=0 garantiert waere:
//  h = sum_e / sum_w;                  // float <- double

    if (sum_w.r != 0.0)
        h.r = exp(sum_e.r / sum_w.r);   // float <- double
    else
    {   if (arrays[0][y][x].r == 0)     // p, fuer das shifts[p]=0, um
            h.r = val_min.r;            // `+shifts[p].y' zu sparen.
        else                            // Also z=zmax
            h.r = val_max.r;
    }

    if (sum_w.g != 0.0)
        h.g = exp(sum_e.g / sum_w.g);   // float <- double
    else
    {   if (arrays[0][y][x].g == 0)     // p, fuer das shifts[p]=0
            h.g = val_min.g;
        else                            // also z=zmax
            h.g = val_max.g;
    }

    if (sum_w.b != 0.0)
        h.b = exp(sum_e.b / sum_w.b);   // float <- double
    else
    {   if (arrays[0][y][x].b == 0)     // p, fuer das shifts[p]=0
            h.b = val_min.b;
        else                            // also z=zmax
            h.b = val_max.b;
    }

    // MinMax feststellen; fakultativ
    if (h.r < h_min.r) h_min.r = h.r; else
    if (h.r > h_max.r) h_max.r = h.r;

    if (h.g < h_min.g) h_min.g = h.g; else
    if (h.g > h_max.g) h_max.g = h.g;

    if (h.b < h_min.b) h_min.b = h.b; else
    if (h.b > h_max.b) h_max.b = h.b;
  }
  // Weil bei streng fallenden Folgen obige else-Zweige nie erreicht
  // wuerden, pruefen, ob h_max geaendert wurde; wenn nicht, ist
  // Maximalwert der erste, denn wie gesagt: streng fallend...

  if (h_max.r == -max_float) h_max.r = hdr[0][0].r;
  if (h_max.g == -max_float) h_max.g = hdr[0][0].g;
  if (h_max.b == -max_float) h_max.b = hdr[0][0].b;

  zeit = uhr.stop();
  printf ("HDR-Zeit: %f sec\n", zeit);

  std::cout << "val_min = " << val_min << '\n';
  std::cout << "val_max = " << val_max << '\n';
  std::cout << "min = " << h_min << '\n';
  std::cout << "max = " << h_max << '\n';

  if (progressinfo_) progressinfo_ -> value(1.0);
  return hdr;
}


/**-------------------------------------------------------------
 * make_CCD_curve()  --  compute CCD curves completely new
 *-------------------------------------------------------------*/
template <class Unsign, class Real>
void
Camera<Unsign,Real>::make_CCD_curve (int refpic, int ngrid, double smooth)
{
  //calc_shifts();         // -> area_
  
  if (progressinfo_) progressinfo_ -> text("Select pixel...");
  z_average (refpic);      // -> follow_*  "calc_FollowValues()"
  if (progressinfo_) progressinfo_ -> value(0.1);
  
  if (progressinfo_) progressinfo_ -> text("Calculate response functions...");
  calc_CCD_curve (refpic, ngrid, smooth); // -> crvX_* und logX_*
}

/**-------------------------------------------------------------
 * make_HDR()  --  compute HDR image completely new
 *-------------------------------------------------------------*/
template <class Unsign, class Real>
Array2D<Rgb<float> >
Camera<Unsign,Real>::make_HDR (int refpic, int ngrid, double smooth)
{
  //calc_shifts();         // -> area_

  make_CCD_curve (refpic, ngrid, smooth);  
  
  return calc_HDR();       // return HDR image
}

/**-------------------------------------------------------------
 * complete_HDR()  --  from "to complete"
 *
 * Calc. only HDR, if CCD curves already exist, else calc all.
 *-------------------------------------------------------------*/
template <class Unsign, class Real>
Array2D<Rgb<float> >
Camera<Unsign,Real>::complete_HDR (int refpic, int ngrid, double smooth)
{
  //calc_shifts();         // -> area_
  
  if (!ccd_curves_ready()) 
    make_CCD_curve (refpic, ngrid, smooth);
  
  return calc_HDR();       // return HDR image
}

/**-------------------------------------------------------------
 * ccd_curves_ready()  --  true if curves already computed
 *
 * Criterion (at present): "all logX-dim()s != 0"?
 *-------------------------------------------------------------*/
template <class Unsign, class Real>
bool
Camera<Unsign,Real>::ccd_curves_ready ()
{  
  return (logX_r.dim() && logX_g.dim() && logX_g.dim());
}

/**
 * read_knnl()
 *
 * SCHLECHT: fscanf() benoetigt explizite Typangabe, die mit 'Real'
 *    nicht zu leisten ist. Entweder logX_* fest auf zB double
 *    umstellen oder mit <iostream> arbeiten.
 * SCHLECHT: Unuebersichtliches Fehlerverhalten, gefaellt mir nicht. 
 */
template <class Unsign, class Real>
void
Camera<Unsign,Real>::read_knnl (const char* fname, Array1D<Real>& X)
{
    int lines = 0;
    FilePtr f (fname, "rt");                    // kann werfen!
    
    while (!feof(f)) 
      if (fgetc(f) == '\n') ++lines;
    
    printf ("%d Zeilen in \"%s\"\n", lines, fname);
    X = Array1D<Real> (lines, 0.0);  // allokiert neu
    rewind (f);
    for (int i=0; i < lines; i++)
        if (fscanf (f, "%*d %lf",  &X[i]) != 1)
        {   printf ("Fehlerhaftes Format in Zeile %d\n", i);
            return;
        }
/*  lines=0;
    while (feof(f))
        if (fscanf (f, "%*d %lf", &X[lines++]) != 1)
        {   printf ("Fehlerhaftes Format in Zeile %d\n", lines);
            return;
        }
*/
    //std::cout << X ;
}

/**
 * read_knnl()
 *
 * Liest die drei Kennlinien logXcrv_r|g|b aus den Dateien
 * log_<basename>@.knn, wobei @ = 'R', 'G', 'B'.
 */
template <class Unsign, class Real>
void
Camera<Unsign,Real>::read_knnl (const char* basename)
{
    char* fname = (char*) malloc (strlen(basename) + 10);

    sprintf (fname, "log_%sR.knn", basename);
    read_knnl (fname, logX_r);

    sprintf (fname, "log_%sG.knn", basename);
    read_knnl (fname, logX_g);

    sprintf (fname, "log_%sB.knn", basename);
    read_knnl (fname, logX_b);

    free (fname);
}

template <class Unsign, class Real>
void
Camera<Unsign,Real>::read_knnl_menu ()
{
    char basename[256];
    printf ("Basisname eingeben (ohne \"log\"): ");
    std::cin >> basename;
    std::cout << basename << '\n';
    read_knnl (basename);
    
    //calc_shifts();
    //make_logHDR();
    make_HDR();
}

/**
 * Wollte 'mal eine Test-Z-Matrix erstellen...
 *   BUG: Matrix hat Rangdefekt
 */
template <class Unsign, class Real>
Array2D<Unsign>
Camera<Unsign,Real>::make_test_Z (int nselect)
{
    Array2D<Unsign> Z (nselect, nlayers);

    double d = (double) z_max / (nselect - 1);

    for (int i=0; i < nselect; i++)
    {   for (int j=0; j < nlayers; j++)
        {   int val = int(i * d + 0.5) + 5 * j;
            if (val > (int)z_max) val = z_max;
            Z[i][j] = val;
        }
    }
    return Z;
}


/**
 * intersection_area():
 *
 * Sucht das Schnittgebiet *aller* Bilder, wobei Bild p um shifts[p] gegen
 * Bild 0 verschoben.
 * NEU: Setzt Camera-weite Area-Struktur `area_'. Zur Rueckwaertskompatibilitaet
 *   vorl. weiter auch noch der Rueckgabewert.
 */
template <class Unsign, class Real>
Area
Camera<Unsign,Real>::intersection_area ()
{
  DBG
  Area ar(0, arrays[0].dim2(), 0, arrays[0].dim1());
  printf ("%s() 0: x=[%i,%i),  y=[%i,%i)\n", __func__, ar.x1, ar.xe, ar.y1, ar.ye);

  // Schnittgebiet...
  // shifts[p] enthaelt Verschiebung des p-ten Bildes gegen Bild 0
  for (int p=1; p < nlayers; p++)
  {
    ar.x1 = TNT::max (ar.x1, -shifts[p].x); // geht effektiver ohne max()
    ar.y1 = TNT::max (ar.y1, -shifts[p].y);
    ar.xe = TNT::min (ar.xe, arrays[p].dim2() - shifts[p].x);
    ar.ye = TNT::min (ar.ye, arrays[p].dim1() - shifts[p].y);
    printf ("%s() %d: x=[%i,%i),  y=[%i,%i)\n", __func__, p, ar.x1, ar.xe, ar.y1, ar.ye);
  }
  area_ = ar;
  return ar;
}


/**------------------------------------------------------------------
 * create_Z() 
 *
 * Erzeugung der Z-Matrizen der Dimension [nselect, nlayers], die
 * Ausgangspunkt der CCD-Kennlinienberechnung sind.
 *
 * @param F: [nImages,n_]-Array2D gemittelter Folgewerte eines Kanals.
 * @param pic: das war Ref-Bild bei Erstellung von F, not used till now
 * @param nselect: Anzahl auszuwaehlender Werte (Stuetzstellen)
 *
 * Von den n_ moegl. Unsign-Werten [0...z_max] sind nselect auszuwaehlen. 
 * Wie? Gleichabstaendig oder auch unter Beruecksichtigung ihrer Streuung
 * oder...?
 *
 * Gegeben [n_]-Feld `F' von Zeilen (oder Spalten). Daraus sind nselect
 * moeglichst gleichabstaendig auszuwaehlen. Dabei koennen Zeilen allerdings 
 * unbrauchbar sein (n==0 oder auch innerhalb einer Zeile keine zwei
 * benachtbarten Bilder im AB).  Diese Zeilen sind auszulassen und
 * stattdessen naechstbenachtbarte zu nehmen. Dazu dient unten das Infofeld
 * `aa[]`. Ok-Kriterium fuer Zeilen ist z.Z. nur n!=0. Spaeter koennte noch
 * "zwei benachtbarte Bilder im AB" gefordert werden.
 *
 * In der Tat sind nur Zeilen brauchbar, darin wenigstens in zwei Bildern
 * die Daten im Arbeitsbereich. Wertkombinationen 0--x oder x--zmax sind
 * wertlos, weil 0 *alles* Unterbelichtete aufsammelt und zmax *alles*
 * Ueberbelichtete.
 *
 * shifts[] war bei der Folgewert-Erstellung zu beruecksichtigen, hier
 * irrelevant. Der Kanal wurde ebenfalls dort gewaehlt, hier irrelevant.
 * 
 * FRAGE: Ist fuer Z der Array2D<Unsign>-Typ wirklich erforderlich (wenn
 *  singulaer im Programm, repliziert das eine Menge Code), oder reicht
 *  auch was Elementareres bzw. ein schon benutzter Array2D-Typ, z.B.
 *  Array2D<Real>?
 *  NB: Die Bilder selbst sind Array2D<Rgb<Unsign>>-Felder!
 *
 * NOTE: Die Impl. von "zwei benachtbarte im AB" unten erkennt zwar
 *  Situation "xx000", aber nicht "0xx00", dh wenn zwei beachtbarte spaeter.
 *  Zudem bisher nur fuer `first' implementiert, `last' noch gemaess
 *  "irgend zwei" ermittelt!
 */
template <class Unsign, class Real>
Array2D<Unsign>
Camera<Unsign,Real>::create_Z (ChannelFollowValues &F, int pic, int nselect)
{
  // Bereich feststellen, darin wenigstens zwei Werte im Arbeitsbereich
  size_t first, last;           // gleicher Typ wie `n_'!
  bool ok;

  ok = false;                   // noch keine zwei Werte im AB
  for (first=0; first < n_; first++)
  {
    bool ok1=false;             // noch kein Wert im AB
    for (int p=0; p < nImages(); p++)
    {
      Unsign z = Unsign(F[p][first].z + 0.5);  // runden fuer > 0

#if 0   
      // zwei AB-Werte in *irgendzwei* Bildern
      if (z > 1 && z < z_max-1)     // im Arbeitsbereich
//    if (z > 0 && z < z_max)       // im Arbeitsbereich
      {   if (ok1)                  // ein Wert war schon im AB
          {   ok = true;            // folglich jetzt ein zweiter
              printf ("ok-break bei first=%i, p=%i\n", first,p);
              break;                // -> p-Schleife verlassen; wegen
          }                         // ok dann auch first-Schleife
          else
              ok1 = true;           // 1. Wert im AB
      }
#else
      // zwei AB-Werte in *benachtbarten* Bildern
      if (ok1)                      // ein Wert war schon im AB
      {   if (z > 1 && z < z_max-1)
          {   ok = true;            // folglich jetzt ein zweiter
              break;                // -> p-Schleife verlassen; wegen
          }                         // ok dann auch first-Schleife
          else
              break;                // p-Schleife verlassen, da nicht
      }                             // mehr benachbart
      else                          // noch kein Wert im AB
          ok1 = (z > 1 && z < z_max-1); // Erster Wert im AB?
#endif
    }
    if (ok) break;
  }

  ok = false;                       // noch keine zwei Werte im AB
  for (last=z_max; last > first; last--)
  {
    bool ok1=false;
    for (int p=0; p < nImages(); p++)
    {
      Unsign z = Unsign(F[p][last].z + 0.5);   // runden fuer > 0
      if (z > 1 && z < z_max-1)     // im Arbeitsbereich
//    if (z > 0 && z < z_max)       // im Arbeitsbereich
      {   if (ok1)                  // ein Wert war schon im AB
          {   ok = true;            // folglich jetzt ein zweiter
              break;                // --> p-Schleife verlassen; wegen
          }                         // ok dann auch last-Schleife
          else
              ok1 = true;           // 1. Wert im AB
      }
    }
    if (ok) break;
  }
  // Das ueberlappende Intervall ist also [first..last].
  printf ("%s(): first = %u, last = %u\n", __func__, first, last);

  // Ein [n_]-Infofeld, darin fuer jede Zeile von `F' markiert, ob brauchbar
  //   (ok==true), welches die vorhergehende und welches die nachfolgende
  //   brauchbare Zeile, und ob Zeile bereits fuer Z-Matrix benutzt...
  struct {bool ok; int pred; int succ; bool used;}  aa[n_];  
  
  // initialization of `aa[]'... 
  // evaluate `ok'...
  for (size_t i=0;      i < first; i++) { aa[i].used = aa[i].ok = false;}
  for (size_t i=last+1; i < n_;    i++) { aa[i].used = aa[i].ok = false;}
  for (size_t i=first; i <= last; i++)
  {
    aa[i].used = false;
    aa[i].ok = true;        
    for (int p=0; p < nImages(); p++)  // ok if n != 0...
    {
      if (F[p][i].n == 0) { aa[i].ok = false;  break; }
    }
  }
  // evaluate `pred' and `succ'...
  int pred = -1;
  for (int i=0; i < int(n_); i++)
  {
    aa[i].pred = pred;
    if (aa[i].ok) pred = i;
  }
  int succ = -1;
  for (int i=n_-1; i >= 0; i--)
  {
    aa[i].succ = succ;
    if (aa[i].ok) succ = i;
  }
    
//   // Liste aa-Feld...
//   for (int i=0; i < int(n_); i++)
//     printf("%3d: ok=%d, pred=% d, succ=% d\n", 
//            i, aa[i].ok, aa[i].pred, aa[i].succ);
      
  // number of useful lines...
  int nuseful = 0;
  for (size_t i=0; i < n_; i++) 
    if (aa[i].ok) nuseful++;
  
  // how many of `nselect' are realy possible...
  int myselect;

//   // Als noch ohne Infofeld...
//   if (first >= last)
//   {
//     printf ("\tKein ueberlappender Bereich! Abbruch.\n");
//     return Array2D<Unsign>();       // Nullarray
//   }
//   else if (last-first+1 < (size_t)nselect)
//   {
//     myselect = last-first+1;        // > 1 garantiert
//     printf ("\tNur %d ueberlappende Werte bei %d auszuwaehlenden\n",
//             myselect, nselect);
//   }
//   else
//     myselect = nselect;             // > 1 garantiert
  
  if (nuseful < 2)
  {
    printf ("%s(): Less than 2 useful lines! Abbort.\n",__func__);
    return Array2D<Unsign>();       // Nullarray
  }
  else if (nuseful < nselect)
  {
    myselect = nuseful;             // > 1 garantiert
    printf ("%s(): Only %d useful lines of %d desired\n", 
      __func__, myselect, nselect);
  }
  else
    myselect = nselect;             // > 1 garantiert

  printf("nselect = %d,  myselect = %d,  nuseful = %d,  n_zval = %d\n", 
    nselect, myselect, nuseful, n_);
    
  Array2D<Unsign> Z (myselect, nImages());          // allokiert

  // jetzt ungefaehr gleichabstaendig auswaehlen...
  double dd = double(last-first) / (myselect-1);    // dd>=1 garantiert
  
  for (int i=0; i < myselect; i++)
  {
    size_t k = size_t(first + i*dd + 0.5);          // runden fuer > 0
    SPUR(("\tgewuenschtes k = %d\n", k));
    
    // if selected line not ok, find the nearest suitable...
    if (! aa[k].ok || aa[k].used)
    {
      // Find nearest predecessor and successor not yet used...
      int pred = aa[k].pred;
      int succ = aa[k].succ;
      while (succ >= 0 && aa[succ].used) succ = aa[succ].succ; 
      while (pred >= 0 && aa[pred].used) pred = aa[pred].pred;
      if (pred < 0) 
      { 
        aa[k].pred = -1;    // Stopper fuer nachfolgende
        if (succ < 0) 
          printf("%s(): both borders reached: Should not happen!\n",__func__);
        else 
          k = succ;
      }
      else if (succ < 0)    // pred >= 0
      {
        k = pred;
      }
      else                  // pred >=0 && succ >= 0
      {
        if (k-pred < succ-k)
          k = pred;
        else
          k = succ;
      }
      printf("\tbenutztes k = %d\n", k);
    }

    for (int p=0; p < nImages(); p++) 
    {
      //if (F[p][k].n == 0) printf("i=%d p=%d: n == 0 !!\n", k,p);
      Z [i][p] = Unsign(F[p][k].z + 0.5);           // runden fuer > 0
      aa[k].used = true;
    }
  }
//   // Liste aa-Feld...
//   for (int i=0; i < int(n_); i++)
//     printf("%3d: ok=%d, pred=% d, succ=% d, used=%d\n", 
//            i, aa[i].ok, aa[i].pred, aa[i].succ, aa[i].used);
  
  //::list (Z);
  return Z;
}


/*  Falls wir histo-Felder *nicht* als Array2D's betreiben, sondern
    als eingebaute...

template <class Unsign, class Real>
Camera<Unsign, Real>::create_histo ()
{
  histo = (ChSum**) malloc (nImages() * sizeof(void*)); 
  for (int i=0; i < n_; i++)
    histo[i] = (ChSum*) malloc (n_ * sizeof(ChSum));
}
template <class Unsign, class Real>
Camera<Unsign, Real>::destroy_histo()
{
  for (int i=0; i < n_; i++)  free (histo[i]);
  free (histo);
  histo = 0;
}
*/


/**
 * Korrelationskoeff. fuer alle drei Kanaele simultan.
 */
template <class Unsign, class Real>
CorrRGB
Camera<Unsign,Real>::correl (int Nx, int Ny,
        int picA, int xA, int yA,
        int picB, int xB, int yB)
{
    Rgb<double> sA(0.0), sB(0.0), s2A(0.0), s2B(0.0), sAB(0.0);
    Rgb <int>   bad(0);

    for (int i=0; i < Ny; i++)
    for (int j=0; j < Nx; j++)
    {
        Rgb<double> a,b;                    // Oder float oder Real?
        a = arrays [picA][yA+i][xA+j];      // double <-- Unsign
        b = arrays [picB][yB+i][xB+j];

        // Warum ueberhaupt hier schon auf double-konvertieren?
        // Dann naemlich sind auch alle Abfragen mit double. Besser
        // doch die Abfragen mit Unsign und erst die Summationen mit
        // double (oder uint64s)!

#ifdef HDR_WORKING_RANGE
        if (a.r==0 || a.r>=z_max || b.r==0 || b.r>=z_max)
            bad.r++;
        else
        {   sA. r += a.r;
            sB. r += b.r;
            s2A.r += a.r * a.r;
            s2B.r += b.r * b.r;
            sAB.r += a.r * b.r;
        }

        if (a.g==0 || a.g>=z_max || b.g==0 || b.g>=z_max)
            bad.g++;
        else
        {   sA. g += a.g;
            sB. g += b.g;
            s2A.g += a.g * a.g;
            s2B.g += b.g * b.g;
            sAB.g += a.g * b.g;
        }

        if (a.b==0 || a.b>=z_max || b.b==0 || b.b>=z_max)
            bad.b++;
        else
        {   sA.b  += a.b;
            sB.b  += b.b;
            s2A.b += a.b * a.b;
            s2B.b += b.b * b.b;
            sAB.b += a.b * b.b;
        }
#else
        sA  += a;
        sB  += b;
        s2A += a * a;
        s2B += b * b;
        sAB += a * b;
#endif
    }

    Rgb<int> n;                     // mit "-="-Op: Rgb<int> n (Nx*Ny);
    n.r = Nx * Ny - bad.r;          //              n -= bad;
    n.g = Nx * Ny - bad.g;          // oder: Rgb<int> n = Nx*Ny - bad;
    n.b = Nx * Ny - bad.b;

    Rgb<double> cov;
    CorrRGB     rho;

#ifndef HDR_WORKING_RANGE           // *ohne* sooo schoen stringent
                                    // (z.B. n.r==n.g==n.b != 0)
    cov = sAB - sA * sB / (double)n.r;  // eigentl. ist das n*cov, nicht cov

    if (cov.r == 0.0)
         rho.r = 0.0; 
    else rho.r = cov.r / sqrt((s2A.r - sA.r*sA.r/n.r)*(s2B.r - sB.r*sB.r/n.r));

    if (cov.g == 0.0)
         rho.g = 0.0; 
    else rho.g = cov.g / sqrt((s2A.g - sA.g*sA.g/n.g)*(s2B.g - sB.g*sB.g/n.g));

    if (cov.b == 0.0)
         rho.b = 0.0; 
    else rho.b = cov.b / sqrt((s2A.b - sA.b*sA.b/n.b)*(s2B.b - sB.b*sB.b/n.b));

#else                           // *mit* klittert's etwas...

    if (n.r==0)                 // all values "bad"
    {   rho.r = CORR_RES_ERR;   // Signum dafuer (irgendein Wert < -1)
        cov.r = 0.0;            // sinnfrei, Zugestaendnis an Debug-Ausgabe
    }
    else
    {   cov.r = sAB.r - sA.r * sB.r / n.r;
        if (cov.r == 0.0)
             rho.r = 0.0; 
        else rho.r = cov.r / sqrt((s2A.r - sA.r*sA.r/n.r)*(s2B.r - sB.r*sB.r/n.r));
    }

    if (n.g==0)     // all values "bad"
    {   rho.g = CORR_RES_ERR;   // Signum dafuer (irgendein Wert < -1)
        cov.g = 0.0;            // sinnfrei, Zugestaendnis an Debug-Ausgabe
    }
    else
    {   cov.g = sAB.g - sA.g * sB.g / n.g;
        if (cov.g == 0.0)
             rho.g = 0.0; 
        else rho.g = cov.g / sqrt((s2A.g - sA.g*sA.g/n.g)*(s2B.g - sB.g*sB.g/n.g));
    }

    if (n.b==0)     // all values "bad"
    {   rho.b = CORR_RES_ERR;   // Signum dafuer (irgendein Wert < -1)
        cov.b = 0.0;            // sinnfrei, Zugestaendnis an Debug-Ausgabe
    }
    else
    {   cov.b = sAB.b - sA.b * sB.b / n.b;
        if (cov.b == 0.0)
             rho.b = 0.0; 
        else rho.b = cov.b / sqrt((s2A.b - sA.b*sA.b/n.b)*(s2B.b - sB.b*sB.b/n.b));
    }
#endif

#ifdef HDR_DEBUG
    if (n.r) cov.r /= n.r;      // die richtige Kovarianz
    if (n.g) cov.g /= n.g;
    if (n.b) cov.b /= n.b;
    printf ("R: n=%4i,  cov(A,B)= %f,  rho(A,B)= %f\n", n.r, cov.r, rho.r);
    printf ("G: n=%4i,  cov(A,B)= %f,  rho(A,B)= %f\n", n.g, cov.g, rho.g);
    printf ("B: n=%4i,  cov(A,B)= %f,  rho(A,B)= %f\n", n.b, cov.b, rho.b);
#endif

    return rho;
}

template <class Unsign, class Real>
inline CorrRGB
Camera<Unsign,Real>::correlsym (int nx, int ny,
                                int picA, int xA, int yA,
                                int picB, int xB, int yB)
{
    return correl (2*nx+1, 2*ny+1, picA, xA-nx, yA-ny,
                                   picB, xB-nx, yB-ny);
}

template <class Unsign, class Real>
inline CorrRGB
Camera<Unsign,Real>::correlsym3x3 (int picA, int xA, int yA,
                                   int picB, int xB, int yB)
{
    return correl (3, 3, picA, xA-1, yA-1,
                         picB, xB-1, yB-1);
}




#endif	// Camera_hpp

// END OF FILE
