;;;; records.scm - Reference-implementation (only slightly adapted to CHICKEN)


(cond-expand
 [hygienic-macros

(define-syntax define-record-type
  (syntax-rules ()
    ((define-record-type type
       (constructor constructor-tag ...)
       predicate
       (field-tag accessor . more) ...)
     (begin
       (define type
         (make-record-type 'type '(field-tag ...)))
       (define constructor
         (record-constructor type '(constructor-tag ...)))
       (define predicate
         (record-predicate type))
       (define-record-field type field-tag accessor . more)
       ...))))

; An auxilliary macro for define field accessors and modifiers.
; This is needed only because modifiers are optional.

(define-syntax define-record-field
  (syntax-rules ()
    ((define-record-field type field-tag accessor)
     (define accessor (record-accessor type 'field-tag)))
    ((define-record-field type field-tag accessor modifier)
     (begin
       (define accessor (record-accessor type 'field-tag))
       (define modifier (record-modifier type 'field-tag))))))

][else

(define-macro (define-record-type type kons pred . fields)
  (let ([constructor (car kons)]
	[ctags (cdr kons)] )
    `(begin
       (define ,type (make-record-type ',type ',(map car fields)))
       (define ,constructor (record-constructor ,type ',ctags))
       (define ,pred (record-predicate ,type))
       ,@(map (lambda (field)
		(let ([tag (car field)]
		      [accessor (cadr field)]
		      [modifier (and (pair? (cddr field)) (caddr field))] )
		  (if modifier
		      `(begin
			 (define ,accessor (record-accessor ,type ',tag))
			 (define ,modifier (record-modifier ,type ',tag)) )
		      `(define ,accessor (record-accessor ,type ',tag)) ) ) )
	      fields) ) ) )

] )


; We define the following procedures:
; 
; (make-record-type <type-name <field-names>)    -> <record-type>
; (record-constructor <record-type<field-names>) -> <constructor>
; (record-predicate <record-type>)               -> <predicate>
; (record-accessor <record-type <field-name>)    -> <accessor>
; (record-modifier <record-type <field-name>)    -> <modifier>
;   where
; (<constructor> <initial-value> ...)         -> <record>
; (<predicate> <value>)                       -> <boolean>
; (<accessor> <record>)                       -> <value>
; (<modifier> <record> <value>)         -> <unspecific>

;----------------
; Record types are themselves records, so we first define the type for
; them.  Except for problems with circularities, this could be defined as:
;  (define-record-type :record-type
;    (make-record-type name field-tags)
;    record-type?
;    (name record-type-name)
;    (field-tags record-type-field-tags))
; As it is, we need to define everything by hand.

(define :record-type (##sys#make-structure 'record #f ':record-type '(name field-tags)))
(##sys#setslot :record-type 1 :record-type)

; Now that :record-type exists we can define a procedure for making more
; record types.

(define (make-record-type name field-tags)
  (##sys#make-structure 'record :record-type name field-tags) )


;----------------
; A utility for getting the offset of a field within a record.

(define (field-index type tag)
  (let loop ((i 2) (tags (##sys#slot type 3)))
    (cond ((null? tags)
           (error "record type has no such field" type tag))
          ((eq? tag (car tags))
           i)
          (else
           (loop (fx+ i 1) (cdr tags))))))

;----------------
; Now we are ready to define RECORD-CONSTRUCTOR and the rest of the
; procedures used by the macro expansion of DEFINE-RECORD-TYPE.

(define (record-constructor type tags)
  (let ((size (length (##sys#slot type 3)))
        (arg-count (length tags))
        (indexes (map (lambda (tag) (field-index type tag)) tags)))
    (lambda args
      (if (= (length args)
             arg-count)
          (let ((new (make-vector (fx+ size 2))))
	    (##sys#vector->structure! new)
	    (##sys#setslot new 0 'record)
            (##sys#setslot new 1 type)
            (for-each (lambda (arg i) (##sys#setslot new i arg))
                      args
                      indexes)
            new)
          (error "wrong number of arguments to constructor" type args)))))

(define (record-predicate type)
  (lambda (thing)
    (and (##sys#structure? thing 'record)
         (eq? (##sys#slot thing 1) type))))

(define (record-accessor type tag)
  (let ((index (field-index type tag)))
    (lambda (thing)
      (if (and (##sys#structure? thing 'record) (eq? (##sys#slot thing 1) type))
	  (##sys#slot thing index)
          (error "accessor applied to bad value" type tag thing)))))

(define (record-modifier type tag)
  (let ((index (field-index type tag)))
    (lambda (thing value)
      (if (and (##sys#structure? thing 'record) (eq? (##sys#slot thing 1) type))
          (##sys#setslot thing index value)
          (error "modifier applied to bad value" type tag thing)))))

(eval-when (compile load eval) (register-feature! 'srfi-9))


(cond-expand [(not use-modules)])

(define-module srfi-9
  (unqualified)
  (export define-record-type) )
