;;;; moremacros.scm - More syntax extensions - felix

(cond-expand [hygienic-macros
	      (eval-when (compile load eval) 
		(error "can not define macros - use low-level macro system instead") ) ]
	     [(not srfi-1)
	      (eval-when (compile load eval)
		(error "this package needs the SRFI-1 module") ) ] 
	     [else] )


;;; Optional argument handling:

;;; Copyright (C) 1996 by Olin Shivers.
;;;
;;; This file defines three macros for parsing optional arguments to procs:
;;; 	(LET-OPTIONALS  arg-list ((var1 default1) ...) . body)
;;; 	(LET-OPTIONALS* arg-list ((var1 default1) ...) . body)
;;; 	(:OPTIONAL rest-arg default-exp)
;;;
;;; The LET-OPTIONALS macro is defined using the Clinger/Rees
;;; explicit-renaming low-level macro system. You'll have to do some work to
;;; port it to another macro system.
;;;
;;; The LET-OPTIONALS* and :OPTIONAL macros are defined with simple
;;; high-level macros, and should be portable to any R4RS system.
;;;
;;; These macros are all careful to evaluate their default forms *only* if
;;; their values are needed.
;;;
;;; The only non-R4RS dependencies in the macros are ERROR 
;;; and CALL-WITH-VALUES.
;;; 	-Olin

;;; (LET-OPTIONALS arg-list ((var1 default1) ...) 
;;;   body
;;;   ...)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; This form is for binding a procedure's optional arguments to either
;;; the passed-in values or a default.
;;;
;;; The expression takes a rest list ARG-LIST and binds the VARi to
;;; the elements of the rest list. When there are no more elements, then
;;; the remaining VARi are bound to their corresponding DEFAULTi values.
;;; It is an error if there are more args than variables.
;;;
;;; - The default expressions are *not* evaluated unless needed.
;;;
;;; - When evaluated, the default expressions are carried out in the *outer*
;;;   environment. That is, the DEFAULTi forms do *not* see any of the VARi
;;;   bindings.
;;;
;;;   I originally wanted to have the DEFAULTi forms get eval'd in a LET*
;;;   style scope -- DEFAULT3 would see VAR1 and VAR2, etc. But this is
;;;   impossible to implement without side effects or redundant conditional
;;;   tests. If I drop this requirement, I can use the efficient expansion
;;;   shown below. If you need LET* scope, use the less-efficient 
;;;   LET-OPTIONALS* form defined below.
;;;
;;; Example:
;;; (define (read-string! str . maybe-args)
;;;   (let-optionals maybe-args ((port (current-input-port))
;;;                              (start 0)
;;;                              (end (string-length str)))
;;;     ...))
;;;
;;; expands to:
;;; 
;;; (let* ((body (lambda (port start end) ...))
;;;        (end-def (lambda (%port %start) (body %port %start <end-default>)))
;;;        (start-def (lambda (%port) (end-def %port <start-default>)))
;;;        (port-def  (lambda () (start-def <port-def>))))
;;;   (if (null? rest) (port-def)
;;;       (let ((%port (car rest))
;;; 	        (rest (cdr rest)))
;;; 	  (if (null? rest) (start-def %port)
;;; 	      (let ((%start (car rest))
;;; 		    (rest (cdr rest)))
;;; 	        (if (null? rest) (end-def %port %start)
;;; 		    (let ((%end (car rest))
;;; 			  (rest (cdr rest)))
;;; 		      (if (null? rest) (body %port %start %end)
;;; 			  (error ...)))))))))


;;; (LET-OPTIONALS args ((var1 default1) ...) body1 ...)

(define-macro (let-optionals arg-list var/defs . body)

  ;; This guy makes the END-DEF, START-DEF, PORT-DEF definitions above.
  ;; I wish I had a reasonable loop macro.

  (define (make-default-procs vars body-proc defaulter-names defs rename)
    (let recur ((vars (reverse vars))
		(defaulter-names (reverse defaulter-names))
		(defs (reverse defs))
		(next-guy body-proc))
      (if (null? vars) '()
	  (let ((vars (cdr vars)))
	    `((,(car defaulter-names)
	       (lambda ,(reverse vars)
		 (,next-guy ,@(reverse vars) ,(car defs))))
	      . ,(recur vars
			(cdr defaulter-names)
			(cdr defs)
			(car defaulter-names)))))))


    ;; This guy makes the (IF (NULL? REST) (PORT-DEF) ...) tree above.

  (define (make-if-tree vars defaulters body-proc rest rename)
    (let recur ((vars vars) (defaulters defaulters) (non-defaults '()))
      (if (null? vars)
	  `(if (null? ,rest) (,body-proc . ,(reverse non-defaults))
	       (error (##core#immutable '"too many optional arguments.") ,rest))
	  (let ((v (car vars)))
	    `(if (null? ,rest)
		 (,(car defaulters) . ,(reverse non-defaults))
		 (let ((,v (car ,rest))
		       (,rest (cdr ,rest)))
		   ,(recur (cdr vars)
			   (cdr defaulters)
			   (cons v non-defaults))))))))

  (##sys#check-syntax 'let-optionals var/defs '#((symbol _) 0))
  (##sys#check-syntax 'let-optionals body '#(_ 1))
  (let* ((vars (map car var/defs))
	 (prefix-sym (lambda (prefix sym)
		       (string->symbol (string-append prefix (symbol->string sym)))))

	 ;; Private vars, one for each user var.
	 ;; We prefix the % to help keep macro-expanded code from being
	 ;; too confusing.
	 (vars2 (map (lambda (v) (gensym (prefix-sym "%" v)))
		     vars))

	 (defs (map cadr var/defs))
	 (body-proc (gensym 'body))

	 ;; A private var, bound to the value of the ARG-LIST expression.
	 (rest-var (gensym '%rest))

	 (defaulter-names (map (lambda (var) (gensym (prefix-sym "def-" var)))
			       vars))

	 (defaulters (make-default-procs vars2 body-proc
					 defaulter-names defs gensym))
	 (if-tree (make-if-tree vars2 defaulter-names body-proc
				rest-var gensym)))

    `(let* ((,rest-var ,arg-list)
	    (,body-proc (lambda ,vars . ,body))
	    . ,defaulters)
       ,if-tree) ) )


;;; (:optional rest-arg default-exp)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; This form is for evaluating optional arguments and their defaults
;;; in simple procedures that take a *single* optional argument. It is
;;; a macro so that the default will not be computed unless it is needed.
;;; 
;;; REST-ARG is a rest list from a lambda -- e.g., R in
;;;     (lambda (a b . r) ...)
;;; - If REST-ARG has 0 elements, evaluate DEFAULT-EXP and return that.
;;; - If REST-ARG has 1 element, return that element.
;;; - If REST-ARG has >1 element, error.

(define-macro (:optional rest default-exp)
  (let ([var (gensym)])
    `(let ((,var ,rest))
       (cond ((null? ,var) ,default-exp)
	     ((null? (cdr ,var)) (car ,var))
	     (else (error (##core#immutable '"too many optional arguments") ,var))))))


;;; (LET-OPTIONALS* args ((var1 default1) ... [rest]) body1 ...)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; This is just like LET-OPTIONALS, except that the DEFAULTi forms
;;; are evaluated in a LET*-style environment. That is, DEFAULT3 is evaluated
;;; within the scope of VAR1 and VAR2, and so forth.
;;;
;;; - If the last form in the ((var1 default1) ...) list is not a 
;;;   (VARi DEFAULTi) pair, but a simple variable REST, then it is
;;;   bound to any left-over values. For example, if we have VAR1 through
;;;   VAR7, and ARGS has 9 values, then REST will be bound to the list of
;;;   the two values of ARGS. If ARGS is too short, causing defaults to
;;;   be used, then REST is bound to '().
;;; - If there is no REST variable, then it is an error to have excess
;;;   values in the ARGS list.

(define-macro (let-optionals* args var/defs . body)
  (##sys#check-syntax 'let-optionals* var/defs '#(_ 0))
  (##sys#check-syntax 'let-optionals* body '#(_ 1))
  (let ([rvar (gensym)])
    `(let ((,rvar ,args))
       ,(let loop ([args rvar] [vardefs var/defs])
	  (if (null? vardefs)
	      `(if (null? ,args)
		   (let () ,@body)
		   (error (##core#immutable '"too many optional arguments.") ,args) )
	      (let ([head (car vardefs)])
		(if (pair? head)
		    (let ([rvar2 (gensym)])
		      `(let ((,(car head) (if (null? ,args) ,(cadr head) (car ,args)))
			     (,rvar2 (if (null? ,args) '() (cdr ,args))) )
			 ,(loop rvar2 (cdr vardefs)) ) )
		    `(let ((,head ,args)) ,@body) ) ) ) ) ) ) )


;;; case-lambda (SRFI-16):

(define-macro (case-lambda . clauses)
  (##sys#check-syntax 'case-lambda clauses '#(_ 0))
  (let* ((mincount (apply min (map (lambda (c)
				     (##sys#decompose-lambda-list 
				      (car c)
				      (lambda (vars argc rest) argc) ) )
				   clauses) ) ) 
	 (minvars (list-tabulate mincount (lambda (z) (gensym))))
	 (rvar (gensym)) 
	 (lvar (gensym)) )
    `(lambda ,(append minvars rvar)
       (let ((,lvar (length ,rvar)))
	 ,(fold-right
	   (lambda (c body)
	     (##sys#decompose-lambda-list
	      (car c)
	      (lambda (vars argc rest)
		(##sys#check-syntax 'case-lambda (car c) 'lambda-list)
		`(if (,(if rest 'fx>= 'fx=) ,lvar ,(fx- argc mincount))
		     ,(receive
		       (vars1 vars2) (split-at (take vars argc) mincount)
		       (let ((bindings
			      (let build ((vars2 vars2) (vrest rvar))
				(if (null? vars2)
				    (cond (rest `(let ((,rest ,vrest)) ,@(cdr c)))
					  ((null? (cddr c)) (cadr c))
					  (else `(let () ,@(cdr c))) )
				    (let ((vrest2 (gensym)))
				      `(let ((,(car vars2) (car ,vrest))
					     (,vrest2 (cdr ,vrest)) )
					 ,(if (pair? (cdr vars2))
					      (build (cdr vars2) vrest2)
					      (build '() vrest2) ) ) ) ) ) ) )
			 (if (null? vars1)
			     bindings
			     `(let ,(zip vars1 minvars) ,bindings) ) ) )
		     ,body) ) ) )
	   '(error (##core#immutable '"no matching clause in call to 'case-lambda' form"))
	   clauses) ) ) ) )

(eval-when (compile load eval)
  (register-feature! 'srfi-16) )


;;; For SRFI-13:

(define-macro (let-string-start+end s-e-r proc s-exp args-exp . body)
  (if (pair? (cddr s-e-r))
      `(receive (,(caddr s-e-r) ,(car s-e-r) ,(cadr s-e-r))
	   (string-parse-start+end ,proc ,s-exp ,args-exp)
	 ,@body)
      `(receive ,s-e-r
	   (string-parse-final-start+end ,proc ,s-exp ,args-exp)
	 ,@body) ) )


;;; Module definitions:

(define-macro (define-module . form)
  (##sys#check-syntax 'define-module form '(symbol . #(_ 0)))
  `(##core#define-module ',(##sys#slot form 0) ',(##sys#slot form 1)) )

(define-macro (define-library-interface . form)
  (##sys#check-syntax 'define-library-interface form '#(symbol 1))
  (let ([name (##sys#slot form 0)])
    `(##core#define-module ',name '((unit ,name) (export ,@(##sys#slot form 1)))) ) )

(define-macro (define-library-implementation . form)
  (##sys#check-syntax 'define-library-implementation form '(symbol . #(_ 0)))
  (let ([name (##sys#slot form 0)]
	[r (##sys#slot form 1)] )
    `(##core#define-module ',name '((unit ,name) ,@r)) ) )


;;; Record printing:

(define-macro (define-record-printer head . body)
  (cond [(pair? head)
	 (##sys#check-syntax 'define-record-printer (cons head body) '((symbol symbol symbol) . #(_ 1)))
	 `(##sys#register-record-printer ',(##sys#slot head 0) (lambda ,(##sys#slot head 1) ,@body)) ]
	[else
	 (##sys#check-syntax 'define-record-printer (cons head body) '(symbol _))
	 `(##sys#register-record-printer ',head ,body) ] ) )


;;; TinyCLOS:

(define-macro (define-class name supers slots . meta)
  (##sys#check-syntax 'define-class name 'symbol)
  (##sys#check-syntax 'define-class supers '#(_ 0))
  (##sys#check-syntax 'define-class slots '#(_ 0))
  (##sys#check-syntax 'define-class meta '#(_ 0 1))
  `(set! ,name
     (make ,(if (pair? meta) (##sys#slot meta 0) '<class>)
       'name ',name
       'direct-supers (list ,@(if (null? supers) '(<object>) supers))
       'direct-slots (list ,@(map (lambda (s) `',s) slots)) ) ) )

(define-macro (define-generic name)
  (##sys#check-syntax 'define-generic name 'symbol)
  `(set! ,name (make-generic ',name)) )

(define-macro (define-method head . body)
  (##sys#check-syntax 'define-method head '(symbol . _))
  (##sys#check-syntax 'define-method body '#(_ 1))
  (let gather ([args (##sys#slot head 1)]
	       [specs '()]
	       [vars '()] )
    (if (or (not (pair? args)) (not (pair? (##sys#slot args 0))))
	(let ([name (##sys#slot head 0)])
	  `(##tinyclos#add-global-method
	    ',name
	    (list ,@(reverse specs))
	    (##core#named-lambda ,name (call-next-method ,@(reverse vars) ,@args) ,@body) ) )
	(let ([arg (##sys#slot args 0)])
	  (##sys#check-syntax 'define-method arg '(_ symbol))
	  (gather (##sys#slot args 1)
		  (cons (##sys#slot (##sys#slot arg 1) 0) specs)
		  (cons (##sys#slot arg 0) vars) ) ) ) ) )


;;; Exceptions:

(define-macro (handle-exceptions var handler . body)
  (let ([k (gensym)]
	[args (gensym)] )
    `((call-with-current-continuation
       (lambda (,k)
	 (with-exception-handler
	  (lambda (,var) (,k (lambda () ,handler)))
	  (lambda ()
	    (##sys#call-with-values
	     (lambda () ,@body)
	     (lambda ,args (,k (lambda () (##sys#apply ##sys#values ,args)))) ) ) ) ) ) ) ) )
