;;;; lolevel.scm - Low-level routines for CHICKEN
;
; Copyright (c) 2000-2002, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany


(declare
  (unit lolevel)
  (uses srfi-4 extras)
  (standard-bindings)
  (extended-bindings) 
  (no-bound-checks)
  (bound-to-procedure ##sys#vector->closure! ##sys#error ##sys#signal-hook)
  (foreign-declare "
#if defined(__i386__) && !defined(C_NONUNIX) && !defined(__CYGWIN__)
# if defined(__FreeBSD__) || defined(__NetBSD__)
#  include <sys/types.h>
# endif
# include <sys/mman.h>
# define C_valloc(n)               valloc(n)
# define C_makeexecutable(a, n)    mprotect(a, n, PROT_READ | PROT_WRITE | PROT_EXEC)
#elif defined(__i386__) || defined(_M_IX86)
# define C_valloc(n)               malloc(n)
# define C_makeexecutable(a, n)    0
#else
# define C_valloc(n)               NULL
# define C_makeexecutable(a, n)    -1
#endif
") )


(cond-expand
 [unsafe
  (eval-when (compile)
    (define-macro (##sys#check-structure x y) '(##core#undefined))
    (define-macro (##sys#check-range x y z) '(##core#undefined))
    (define-macro (##sys#check-pair x) '(##core#undefined))
    (define-macro (##sys#check-list x) '(##core#undefined))
    (define-macro (##sys#check-symbol x) '(##core#undefined))
    (define-macro (##sys#check-string x) '(##core#undefined))
    (define-macro (##sys#check-char x) '(##core#undefined))
    (define-macro (##sys#check-exact x) '(##core#undefined))
    (define-macro (##sys#check-port x) '(##core#undefined))
    (define-macro (##sys#check-number x) '(##core#undefined))
    (define-macro (##sys#check-byte-vector x) '(##core#undefined)) ) ]
 [else] )


;;; Serialization and de-serialization:

(let ([fixnum-tag 0]
      [boolean-tag 1]
      [end-of-file-tag 2]
      [undefined-tag 3]
      [end-of-list-tag 4]
      [vector-tag 5]
      [pair-tag 6]
      [flonum-tag 7]
      [structure-tag 8]
      [port-tag 9]
      [symbol-tag 10]
      [string-tag 11]
      [closure-tag 12]
      [pointer-tag 13]
      [loop-tag 14] 
      [character-tag 15] 
      [bytevector-tag 16]
      [make-string make-string]
      [vector-ref vector-ref] )
  (let ([tagnames (vector "fixnum" "boolean" "end-of-file" "undefined" "end-of-list" "vector" "pair" "flonum"
			  "structure" "port" "symbol" "string" "closure" "pointer" "<loop>" "character") ] )

    (define (tag->name tag) (vector-ref tagnames tag))

    (set! serialize
      (lambda (x)
	(let ([buffer (make-u32vector 32)]
	      [index 0] 
	      [walked (make-hash-table)] )

	  (define (growbuffer)
	    (let ([old buffer])
	      (set! buffer (make-u32vector (fx+ (u32vector-length old) 256)))
	      (move-memory! old buffer) ) )

	  (define (out x)
	    (when (fx>= index (u32vector-length buffer)) (growbuffer))
	    (##sys#u32vector-set! buffer index x) 
	    (set! index (fx+ index 1)) )

	  (define (outn . xs)
	    (let* ([n (length xs)]
		   [i2 (fx+ index n)] )
	      (when (fx>= i2 (u32vector-length buffer)) (growbuffer))
	      (do ([xs xs (cdr xs)]
		   [n n (fx- n 1)] )
		  ((zero? n))
		(##sys#u32vector-set! buffer index (car xs))
		(set! index (fx+ index 1)) ) ) )
	    
	  (define (walkslots x start)
	    (let ([len (##sys#size x)])
	      (do ([i start (fx+ i 1)])
		  ((fx>= i len))
		(walk (##sys#slot x i)) ) ) )

	  (define (walkbytes x)
	    (let ([len (arithmetic-shift (fx+ 3 (##sys#size x)) -2)])
	      (do ([i 0 (fx+ i 1)])
		  ((fx>= i len))
		(out (##sys#peek-unsigned-integer x i)) ) ) )

	  (define (walkspecial x tag)
	    (outn tag (##sys#size x) (##sys#peek-unsigned-integer x 0))
	    (walkslots x 1) )

	  (define (walk x)
	    (let ([p index]
		  [a (hash-table-ref walked x)] )
	      (flush-output)
	      (cond [a (outn loop-tag a)]
		    [else
		     (when (and (##core#inline "C_blockp" x) (not (##core#inline "C_byteblockp" x)) (not (symbol? x)))
		       (hash-table-set! walked x p) )
		     (cond [(boolean? x) (outn boolean-tag (if x 1 0))]
			   [(null? x) (out end-of-list-tag)]
			   [(eof-object? x) (out end-of-file-tag)]
			   [(eq? x (##core#undefined)) (out undefined-tag)]
			   [(char? x) (outn character-tag (char->integer x))]
			   [(number? x)
			    (if (exact? x)
				(outn fixnum-tag x)
				(let* ([fv (f64vector x)]
				       [uv (byte-vector->u32vector (f64vector->byte-vector fv))] )
				  (outn flonum-tag (##sys#u32vector-ref uv 0) (##sys#u32vector-ref uv 1)) ) ) ]
			   [(not (##core#inline "C_blockp" x)) (##sys#error "can not serialize unknown immediate object" x)]
			   [(##sys#bytevector? x)
			    (outn bytevector-tag (##sys#size x))
			    (walkbytes x) ]
			   [(vector? x)
			    (outn vector-tag (##sys#size x))
			    (walkslots x 0) ]
			   [(pair? x)
			    (outn pair-tag)
			    (walk (car x))
			    (walk (cdr x)) ]
			   [(##core#inline "C_structurep" x)
			    (outn structure-tag (##sys#size x))
			    (walkslots x 0) ]
			   [(string? x)
			    (outn string-tag (##sys#size x))
			    (walkbytes x) ]
			   [(symbol? x)
			    (out symbol-tag)
			    (walk (##sys#slot x 1)) ]
			   [(##core#inline "C_portp" x) (walkspecial x port-tag)]
			   [(##core#inline "C_pointerp" x) (walkspecial x pointer-tag)]
			   [(procedure? x) (walkspecial x closure-tag)]
			   [else (##sys#error "can not serialize object" x)] ) ] ) ) )

	  (walk x)
	  (let ([v2 (make-u32vector index)])
	    (move-memory! buffer v2 (arithmetic-shift index 2))
	    v2) ) ) )

    (set! deserialize
      (lambda (buffer . safe)
	(let ([safe (if (pair? safe) (car safe) #f)]
	      [index 0]
	      [gathered (make-hash-table)] )
	  
	  (define (fetch)
	    (let ([x (##sys#u32vector-ref buffer index)])
	      (set! index (fx+ index 1))
	      x) )

	  (define (fetch-signed)
	    (let ([x (##sys#s32vector-ref buffer index)])
	      (set! index (fx+ index 1))
	      x) )

	  (define (inslots n x start)
	    (do ([i start (fx+ i 1)])
		((fx>= i n) x)
	      (##sys#setslot x i (in)) ) )

	  (define (inbytes len words bvec)
	    (let ([s (make-u32vector words)])
	      (do ([i 0 (fx+ i 1)])
		  ((fx>= i words) 
		   (let ([str 
			  (cond [bvec
				 (let ([bv (##sys#allocate-vector len #t 0 #t)])
				   (##core#inline "C_string_to_bytevector" bv)
				   bv) ]
				[else (make-string len)] ) ] )
		     (move-memory! s str len) 
		     str) )
		(##sys#u32vector-set! s i (fetch)) ) ) )

	  (define (inspecial x n tag)
	    (when safe 
	      (##sys#error "can not deserialize object - contains process specific data" (tag->name tag)) )
	    (##sys#poke-integer x 0 (fetch))
	    (let ([n2 (fx- n 1)])
	      (do ([i 1 (fx+ i 1)])
		  ((fx> i n2) x)
		(##sys#setslot x i (in)) ) ) )

	  (define (gather p x)
	    (when (and (##core#inline "C_blockp" x) (not (##core#inline "C_byteblockp" x)) (not (symbol? x)))
	      (hash-table-set! gathered p x) ) )

	  (define (in)
	    (let* ([p0 index]
		   [tag (fetch)]
		   [x (cond [(eq? tag boolean-tag) (eq? 1 (fetch))]
			    [(eq? tag fixnum-tag) (fetch-signed)]
			    [(eq? tag end-of-file-tag) (##sys#fudge 1)]
			    [(eq? tag end-of-list-tag) '()]
			    [(eq? tag undefined-tag) (##core#undefined)]
			    [(eq? tag character-tag) (integer->char (fetch))]
			    [(eq? tag bytevector-tag)
			     (let ([len (fetch)])
			       (inbytes len (arithmetic-shift (fx+ 3 len) -2) #t) ) ]
			    [(eq? tag vector-tag)
			     (let* ([len (fetch)]
				    [v (make-vector len)] )
			       (gather p0 v)
			       (inslots len v 0) ) ]
			    [(eq? tag symbol-tag) (string->symbol (in))]
			    [(eq? tag pair-tag)
			     (let* ([a (in)]
				    [c (cons a #f)] )
			       (gather p0 c)
			       (set-cdr! c (in))
			       c) ]
			    [(eq? tag structure-tag)
			     (let* ([len (fetch)]
				    [v (make-vector len)] )
			       (gather p0 v)
			       (##core#inline "C_vector_to_structure" v)
			       (inslots len v 0) ) ]
			    [(eq? tag string-tag)
			     (let ([len (fetch)])
			       (inbytes len (arithmetic-shift (fx+ 3 len) -2) #f) ) ]
			    [(eq? tag flonum-tag)
			     (let* ([f1 (fetch)]
				    [f2 (fetch)] )
			       (##sys#f64vector-ref
				(byte-vector->f64vector
				 (u32vector->byte-vector (u32vector f1 f2)) ) 
				0) ) ]
			    [(eq? tag port-tag)
			     (let* ([len (fetch)]
				    [port (##sys#make-port #f len #f #f)] )
			       (gather p0 port)
			       (inspecial port len tag) ) ]
			    [(eq? tag pointer-tag)
			     (inspecial (##sys#make-pointer) (fetch) tag) ]
			    [(eq? tag closure-tag)
			     (let* ([len (fetch)]
				    [c (make-vector len)] )
			       (gather p0 c)
			       (##core#inline "C_vector_to_closure" c)
			       (inspecial c len tag) ) ]
			    [(eq? tag loop-tag)
			     (let* ([p (fetch)]
				    [a (hash-table-ref gathered p)] )
			       (or a (##sys#error "can not deserialize circular object - possibly corrupted" p)) ) ]
			    [else (##sys#error "can not deserialize unknown object - possibly corrupted" tag)] ) ] )
	      (gather p0 x)
	      x) ) 

	  (in) ) ) ) ) )


;;; Move arbitrary blocks of memory around:

(define move-memory!
  (let ([memmove1 (foreign-lambda void "C_memmove" c-pointer c-pointer int)]
	[memmove2 (foreign-lambda void "C_memmove" c-pointer pointer int)]
	[memmove3 (foreign-lambda void "C_memmove" pointer c-pointer int)]
	[memmove4 (foreign-lambda void "C_memmove" pointer pointer int)]
	[slot1structs '(mmap u8vector u16vector u32vector s8vector s16vector s32vector f32vector f64vector)] )
    (lambda (from to . n)
      (define (err) (##sys#error "need number of bytes to move" from to))
      (define (xerr x) (##sys#signal-hook #:type-error "invalid argument type" x))
      (define (checkn n nmax)
	(if (cond-expand [unsafe #t] [else (fx<= n nmax)])
	    n
	    (##sys#error "number of bytes to move too large" from to n nmax) ) )
      (define (checkn2 n nmax nmax2)
	(if (cond-expand [unsafe #t] [else (and (fx<= n nmax) (fx<= n nmax2))])
	    n
	    (##sys#error "number of bytes to move too large" from to n nmax nmax2) ) )
      (let move ([from from] [to to])
	(cond [(##sys#generic-structure? from)
	       (if (memq (##sys#slot from 0) slot1structs)
		   (move (##sys#slot from 1) to)
		   (xerr from) ) ]
	      [(##sys#generic-structure? to)
	       (if (memq (##sys#slot to 0) slot1structs)
		   (move from (##sys#slot to 1))
		   (xerr to) ) ]
	      [(##sys#pointer? from)
	       (cond [(##sys#pointer? to) (memmove1 to from (:optional n (err)))]
		     [(or (##sys#bytevector? to) (string? to))
		      (memmove3 to from (checkn (:optional n (err)) (##sys#size to))) ]
		     [else (xerr to)] ) ]
	      [(or (##sys#bytevector? from) (string? from))
	       (let ([nfrom (##sys#size from)])
		 (cond [(##sys#pointer? to) (memmove2 to from (checkn (:optional n nfrom) nfrom))]
		       [(or (##sys#bytevector? to) (string? to))
			(memmove4 to from (checkn2 (:optional n nfrom) nfrom (##sys#size to))) ]
		       [else (xerr to)] ) ) ]
	      [else (xerr from)] ) ) ) ) )


;;; Pointer operations:

(define null-pointer ##sys#null-pointer)

(define (pointer? x)
  (and (##core#inline "C_blockp" x) (##core#inline "C_pointerp" x)) )

(define address->pointer
    (lambda (addr)
      (cond-expand
       [(not unsafe)
	(when (not (integer? addr))
	  (##sys#signal-hook #:type-error "bad argument type - not an integer" addr) ) ]
       [else] )
      (##sys#address->pointer addr) ) )

(define pointer->address
    (lambda (ptr)
      (cond-expand 
       [(not unsafe)
	(when (not (and (##core#inline "C_blockp" ptr) (##core#inline "C_pointerp" ptr)))
	  (##sys#signal-hook #:type-error "bad argument type - not a pointer" ptr) ) ]
       [else] )
      (##sys#pointer->address ptr) ) )

(define null-pointer?
    (lambda (ptr)
      (cond-expand
       [(not unsafe)
	(when (not (and (##core#inline "C_blockp" ptr) (##core#inline "C_pointerp" ptr)))
	  (##sys#signal-hook #:type-error "bad argument type - not a pointer" ptr) ) ]
       [else] )
      (eq? 0 (##sys#pointer->address ptr) ) ) )


;;; Procedures extended with data:

(define extend-procedure
  (let ([make-vector make-vector] )
    (lambda (proc x)
      (cond-expand 
       [(not unsafe)
	(unless (##core#inline "C_closurep" proc)
	  (##sys#signal-hook #:type-error "bad argument type - not a procedure" proc) ) ]
       [else] )
      (let ([len (##sys#size proc)])
	(if (and (fx> len 1) (eq? ##sys#snafu (##sys#slot proc (fx- len 1)))) 
	    proc
	    (let* ([len2 (fx+ len 2)]
		   [p2 (make-vector len2)] )
	      (do ([i 1 (fx+ i 1)])
		  ((fx>= i len)
		   (##sys#setslot p2 i x)
		   (##sys#setslot p2 (fx+ i 1) ##sys#snafu)
		   (##sys#vector->closure! p2 (##sys#peek-unsigned-integer proc 0))
		   p2)
		(##sys#setslot p2 i (##sys#slot proc i)) ) ) ) ) ) ) )

(define (extended-procedure? x)
  (and (##core#inline "C_blockp" x)
       (##core#inline "C_closurep" x)
       (let ([len (##sys#size x)])
	 (and (fx> len 1)
	      (eq? ##sys#snafu (##sys#slot x (fx- len 1))) ) ) ) )

(define procedure-data 
  (let ([extended-procedure? extended-procedure?] )
    (lambda (proc)
      (cond-expand 
       [(not unsafe)
	(unless (extended-procedure? proc)
	  (##sys#signal-hook #:type-error "bad argument type - not an extended procedure") ) ]
       [else] )
      (##sys#slot proc (fx- (##sys#size proc) 2)) ) ) )

(define set-procedure-data!
  (let ([extended-procedure? extended-procedure?] )
    (lambda (proc x)
      (cond-expand
       [(not unsafe)
	(unless (extended-procedure? proc) 
	  (##sys#signal-hook #:type-error "bad argument type - not an extended procedure") ) ]
       [else] )
      (##sys#setslot proc (fx- (##sys#size proc) 2) x) ) ) )


;;; Bytevector stuff:

(define (byte-vector? x)
  (and (##core#inline "C_blockp" x)
       (##core#inline "C_bytevectorp" x) ) )

(define (byte-vector-fill! bv n)
  (##sys#check-byte-vector bv)
  (##sys#check-exact n)
  (let ([len (##sys#size bv)])
    (do ([i 0 (fx+ i 1)])
	((fx>= i len))
      (##sys#setbyte bv i n) ) ) )

(define make-byte-vector
  (let ([byte-vector-fill! byte-vector-fill!])
    (lambda (size . init)
      (##sys#check-exact size)
      (let ([bv (##sys#allocate-vector size #t #f #t)])
	(##core#inline "C_string_to_bytevector" bv)
	(when (pair? init) (byte-vector-fill! bv (car init)))
	bv) ) ) )

(define byte-vector
  (let ([make-byte-vector make-byte-vector])
    (lambda bytes
      (let* ([n (length bytes)]
	     [bv (make-byte-vector n)] )
	(do ([i 0 (fx+ i 1)]
	     [bytes bytes (##sys#slot bytes 1)] )
	    ((fx>= i n) bv)
	  (##sys#setbyte bv i (##sys#slot bytes 0)) ) ) ) ) )

(define byte-vector-ref
    (lambda (bv i)
      (##sys#check-byte-vector bv)
      (##sys#check-exact i)
      (let ([n (##sys#size bv)])
	(if (or (fx< i 0) (fx>= i n))
	    (##sys#error "out of range" bv i)
	    (##sys#byte bv i) ) ) ) )

(define byte-vector-set!
    (lambda (bv i x)
      (##sys#check-byte-vector bv)
      (##sys#check-exact i)
      (##sys#check-exact x)
      (let ([n (##sys#size bv)])
	(if (or (fx< i 0) (fx>= i n))
	    (##sys#error "out of range" bv i)
	    (##sys#setbyte bv i x) ) ) ) )

(define (byte-vector->list bv)
  (##sys#check-byte-vector bv)
  (let ([len (##sys#size bv)])
    (let loop ([i 0])
      (if (fx>= i len)
	  '()
	  (cons (##sys#byte bv i) 
		(loop (fx+ i 1)) ) ) ) ) )

(define list->byte-vector
  (let ([make-byte-vector make-byte-vector])
    (lambda (lst)
      (##sys#check-list lst)
      (let* ([n (length lst)]
	     [v (make-byte-vector n)] )
	(do ([p lst (##sys#slot p 1)]
	     [i 0 (fx+ i 1)] )
	    ((eq? p '()) v)
	  (if (pair? p)
	      (let ([b (##sys#slot p 0)])
		(##sys#check-exact b)
		(##sys#setbyte v i b) )
	      (##sys#not-a-proper-list-error lst) ) ) ) ) ) )

(define (byte-vector-length bv)
  (##sys#check-byte-vector bv)
  (##sys#size bv) )

(define-foreign-variable _c_header_size_mask int "C_HEADER_SIZE_MASK")

(let ([byte-vector-fill! byte-vector-fill!]
      [malloc
       (foreign-lambda* scheme-object ((int size))
	 "char *bv;
           if((bv = (char *)malloc(size + 3)) == NULL) return(C_SCHEME_FALSE);
           bv = (char *)C_align((C_word)bv);
           ((C_SCHEME_BLOCK *)bv)->header = C_BYTEVECTOR_TYPE | size;
           return((C_word)bv);") ]
      [xalloc
       (foreign-lambda* scheme-object ((int size))
	 "char *bv;
           if((bv = (char *)C_valloc(size)) == NULL) return(C_SCHEME_FALSE);
           if(C_makeexecutable(bv, size) == -1) return(C_SCHEME_FALSE);
           ((C_SCHEME_BLOCK *)bv)->header = C_BYTEVECTOR_TYPE | size;
           return((C_word)bv);") ] )
  (define (make size init alloc)
    (##sys#check-exact size)
    (if (fx> size _c_header_size_mask)
	(##sys#error "out of range" size _c_header_size_mask)
	(let ([bv (alloc size)])
	  (cond [bv
		 (when (pair? init) (byte-vector-fill! bv (##sys#slot init 0)))
		 bv]
		[else (##sys#signal-hook #:runtime-error "can not allocate statically allocated bytevector" size)] ) ) ) )
  (set! make-static-byte-vector (lambda (size . init) (make size init malloc)))
  (set! make-executable-byte-vector (lambda (size . init) (make size init xalloc))) )

(define static-byte-vector->pointer 
    (lambda (bv)
      (##sys#check-byte-vector bv)
      (if (##core#inline "C_permanentp" bv)
	  (let ([p (##sys#make-pointer)])
	    (##core#inline "C_pointer_to_block" p bv)
	    p)
	  (##sys#error "can not coerce non-static bytevector" bv) ) ) )


;;; Accessors for arbitrary block objects:

(let ([check
       (lambda (x i)
	 (when (or (not (##core#inline "C_blockp" x)) 
		   (##core#inline "C_specialp" x)
		   (##core#inline "C_byteblockp" x) ) 
	   (##sys#error "slots not accessible" x) )
	 (when (or (fx< i 0) (fx>= i (##sys#size x)))
	   (##sys#error "slot-reference out of range" x i) ) ) ] )
  (set! block-ref
    (lambda (x i)
      (##sys#check-exact i)
      (check x i)
      (##sys#slot x i) ) )
  (set! block-set!
    (lambda (x i y)
      (##sys#check-exact i)
      (check x i)
      (##sys#setslot x i y) ) ) )

(define number-of-slots 
    (lambda (x)
      (when (or (not (##core#inline "C_blockp" x)) 
		(##core#inline "C_specialp" x)
		(##core#inline "C_byteblockp" x) )
	(##sys#signal-hook #:type-error "slots not accessible" x) )
      (##sys#size x) ) )


;;; Convert executable bytevector into a procedure:

(define executable-byte-vector->procedure
  (lambda (bv)
    (##sys#check-byte-vector bv)
    (if (##core#inline "C_permanentp" bv)
	((foreign-lambda* scheme-object ((scheme-object bv) (scheme-object proc))
	   "C_set_block_item(proc, 0, (C_word)(&C_block_item(bv, 0)));
              return(proc);")
	 bv (lambda () #f) )
	(##sys#signal-hook #:type-error "can not coerce non-static bytevector" bv) ) ) )


;;; Copy arbitrary object:

(define copy
  (let ([make-vector make-vector])
    (lambda (x)
      (define (err x)
	(##sys#signal-hook #:type-error "can not copy object" x) )
      (let copy ([x x])
	(cond [(not (##core#inline "C_blockp" x)) x]
	      [(##core#inline "C_permanentp" x) x]
	      [else
	       (let* ([n (##sys#size x)]
		      [words (if (##core#inline "C_byteblockp" x) (##core#inline "C_words" n) n)]
		      [y (##core#inline "C_copy_block" x (make-vector words))] )
		 (unless (##core#inline "C_byteblockp" x)
		   (do ([i (if (##core#inline "C_specialp" x) 1 0) (fx+ i 1)])
		       ((fx>= i n))
		     (##sys#setslot y i (copy (##sys#slot y i))) ) )
		 y) ] ) ) ) ) )


;;; Evict objects into static memory:

(define (evicted? x) (##core#inline "C_permanentp" x))

(define evict
  (let ([make-hash-table make-hash-table]
	[hash-table-ref hash-table-ref]
	[hash-table-set! hash-table-set!] )
    (lambda (x . allocator)
      (let ([allocator 
	     (if (pair? allocator) 
		 (car allocator)
		 (foreign-lambda c-pointer "C_malloc" int) ) ] 
	    [tab (make-hash-table)] )
	(let evict ([x x])
	  (cond [(not (##core#inline "C_blockp" x)) x]
		[(hash-table-ref tab x)]
		[else
		 (let* ([n (##sys#size x)]
			[bytes (if (##core#inline "C_byteblockp" x) n (##core#inline "C_bytes" n))]
			[y (##core#inline "C_evict_block" x (allocator bytes))] )
		   (hash-table-set! tab x y)
		   (unless (##core#inline "C_byteblockp" x)
		     (do ([i (if (##core#inline "C_specialp" x) 1 0) (fx+ i 1)])
			 ((fx>= i n))
		       (##sys#setslot y i (evict (##sys#slot x i))) ) )
		   y) ] ) ) ) ) ) )

(define release
  (lambda (x . releaser)
    (let ([free (if (pair? releaser) 
		    (car releaser) 
		    (foreign-lambda void "C_free" c-pointer) ) ] )
      (let release ([x x])
	(cond [(not (##core#inline "C_blockp" x)) x]
	      [(not (##core#inline "C_permanentp" x)) x]
	      [else
	       (let ([n (##sys#size x)])
		 (unless (##core#inline "C_byteblockp" x)
		   (do ([i (if (##core#inline "C_specialp" x) 1 0) (fx+ i 1)])
		       ((fx>= i n))
		     (release (##sys#slot x i))) )
		 (free (##sys#address->pointer (##core#inline "C_block_address" x))) ) ] ) ) ) ) )
