;;;; csi.scm - Simple interpreter stub for CHICKEN
;
; Copyright (c) 2000-2002, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany


(declare
  (uses extras srfi-1 syntax-case srfi-4 lolevel srfi-18 match modules tinyclos)
  (usual-integrations)
  (disable-interrupts)
  (foreign-declare #<<EOF
#ifndef C_INSTALL_HOME
# ifdef C_NO_HOME
#  define C_INSTALL_HOME NULL
# else
#  include "c_defaults.h"
# endif
#endif
EOF
) )

(include "moremacros")


#{csi 
  max-history-count print-usage print-banner write-to-echo-port
  make-transcript-port run hexdump}


;;; Parameters:

(define-constant init-file ".csirc")
(define-constant max-history-count 16)

(set! ##sys#repl-print-length-limit 512)

(define-foreign-variable installation-home c-string "C_INSTALL_HOME")


;;; Print all sorts of information:

(define (print-usage)
  (display #<<EOF
Usage: csi {FILENAME | OPTION}

  where OPTION may be one of the following:

    -help                       display this text and exit
    -version                    display version and exit
    -case-sensitive             enable case-sensitive reading
    -eval EXPRESSIONS           evaluate given expressions
    -feature SYMBOL             register feature identifier
    -quiet                      do not print banner
    -no-init                    do not load initialization file `~/.csirc'
    -no-feature SYMBOL          unregister feature identifier
    -batch                      terminate after command-line processing
    -hygienic                   install SYNTAX-CASE macro package
    -no-warnings                disable all warnings
    -script                     use interpreter for shell scripts
    -srfi-7                     process source files as SRFI-7 configuration language
    -strict                     disable non-standard macros
    -strict-srfi-0              disable non-standard macros except `cond-expand'
    --                          ignore all following options

EOF
) )

(define (print-banner)
  (printf "; This is the CHICKEN interpreter - Version ~S, Build ~S ~A~%~
           ; (c)2000-2002 Felix L. Winkelmann~%"
	  build-version build-number (##sys#fudge 3) ) )


;;; Reader for she-bang comment:

(set! ##sys#user-read-hook
  (let ([read-char read-char]
	[read read]
	[old-hook ##sys#user-read-hook] )
    (lambda (char port)
      (if (char=? char #\!)
	  (do ((c (read-char port) (read-char port)))
	      ((char=? c #\newline) (read port))
	    (when (eof-object? c) (##sys#error "unexpected end of file")) )
	  (old-hook char port) ) ) ) )


;;; Add some 'comma'-commands:

(set! ##sys#repl-eval-hook
  (let ((eval eval)
	(load-noisily load-noisily)
	(read read)
	(length length)
	(display display)
	(write write)
	(printf printf)
	(integer? integer?)
	(second second)
	(butlast butlast)
	(take take)
	(list-ref list-ref)
	(values values)
	(history-list '())
	(history-count 0) )
    (lambda (form)

      (define (filename x)
	(cond ((string? x) x)
	      ((symbol? x) (##sys#symbol->string x)) 
	      (else (##sys#error "argument is not a valid filename" x)) ) )

      (define (add-h form results)
	(cond ((fx>= history-count max-history-count)
	       (set! history-list 
		 (cons (cons form results) 
		       (take history-list (fx- history-count 1)) ) ) )
	      (else
	       (set! history-list (cons (cons form results) history-list))
	       (set! history-count (fx+ history-count 1)) ) ) )

      (define (ref-h index)
	(let ((i (fx- (inexact->exact index) 1)))
	  (if (and (fx>= i 0) (fx< i (length history-list)))
	      (list-ref history-list i) 
	      (##sys#error "history entry index out of range" index) ) ) )
	  
      (set! ##sys#trace-indent-level 0)
      (set! $ (lambda i
		(let ((index (if (pair? i) (inexact->exact (##sys#slot i 0)) 1)))
		  (if (integer? index)
		      (car (ref-h index))
		      (##sys#error "invalid history entry index" index) ) ) ) )
      (set! & (lambda i
		(let ((index (if (pair? i) (inexact->exact (##sys#slot i 0)) 1)))
		  (if (integer? index)
		      (apply values (cdr (ref-h index)))
		      (##sys#error "invalid history entry index" index) ) ) ) )
      (write-to-echo-port form)
      (cond ((eof-object? form) (exit))
	    ((and (pair? form)
		  (eq? 'unquote (##sys#slot form 0)) )
	     (let ((cmd (second form)))
	       (case cmd
		 ((x)
		  (let ([x (read)])
		    (add-h `(pretty-print (##sys#secondary-macroexpand (macroexpand ',x))) (list (void)))
		    (pretty-print (macroexpand x))
		    (void) ) )
		 ((x1)
		  (let ([x (read)])
		    (add-h `(pretty-print (##sys#secondary-macroexpand (macroexpand-1 ',x))) (list (void)))
		    (pretty-print (macroexpand-1 x))
		    (void) ) )
		 ((p)
		  (let* ([x (read)]
			 [xe (eval x)] )
		    (add-h `(pretty-print (eval ',x)) (list (void)))
		    (pretty-print xe)
		    (void) ) )
		 ((d)
		  (let* ([x (read)]
			 [xe (eval x)] )
		    (add-h `(describe (eval ',x)) (list (void)))
		    (describe xe) ) )
		 ((q) (exit))
		 ((l)
		  (let ((fn (filename (read))))
		    (add-h `(load ,fn) (list (void)))
		    (load fn)
		    (void) ) )
		 ((l7)
		  (let ((fn (filename (read))))
		    (add-h `(load-srfi-7-program ,fn) (list (void)))
		    (load-srfi-7-program fn)
		    (void) ) )
		 ((ln) 
		  (let ((fn (filename (read))))
		    (add-h `(load-noisily ,fn) (list (void)))
		    (load-noisily fn)
		    (void) ) )
		 ((t)
		  (let ((x (read)))
		    (receive rs (time (eval x))
		      (add-h `(time (eval ',x)) rs)
		      (apply values rs) ) ) )
		 ((ts)
		  (for-each
		   (lambda (t)
		     (printf "~A: ~A~%" (##sys#slot t 6) (##sys#slot t 3)) )
		   (##sys#all-threads) ) )
		 ((s)
		  (let* ((x (read))
			 (str (if (symbol? x) (##sys#symbol->string x) x))
			 (r (system str)) )
		    (add-h `(system ,str) (list r))
		    r) )
		 ((h)
		  (fluid-let ([##sys#print-qualifiers #t])
		    (do ((i 1 (fx+ i 1))
			 (hl history-list (##sys#slot hl 1)) )
			((eq? hl '()))
		      (let ((entry (##sys#slot hl 0)))
			(printf "~s~a: ~s" i (if (fx< i 10) #\space "") (car entry))
			(if (not (equal? (list (void)) (cdr entry)))
			    (begin
			      (display " ->")
			      (for-each 
			       (lambda (x)
				 (##sys#with-print-length-limit
				  60
				  (lambda ()
				    (display #\space)
				    (write x) ) ) )
			       (cdr entry) ) ) )
			(newline) ) )
		    (void) ) )
		 ((?)
		  (display 
		   "Toplevel commands:

 ,?            Show this text
 ,h            List history of last commands
 ,NUM          Evaluate NUMth command in history-list
 ,p EXP        Pretty print evaluated expression EXP
 ,d EXP        Describe result of evaluated expression EXP
 ,q            Quit interpreter
 ,l FILENAME   Load file with given filename (may be symbol or string)
 ,ln FILENAME  Load file and print result of each top-level expression
 ,l7 FILENAME  Process file as SRFI-7 configuration language
 ,s STRING     Execute shell-command
 ,t EXP        Evaluate form and print elapsed time
 ,ts           List threads that are ready for scheduling
 ,x EXP        Pretty print macroexpanded expression EXP
 ,x1 EXP       Pretty print expression EXP macroexpanded once\n") 
		  (void) )
		 (else
		  (cond ((fixnum? cmd)
			 (let* ((item (ref-h cmd))
				(exp (car item)) )
			   (receive rs (eval exp)
			     (add-h exp rs)
			     (apply values rs) ) ) )
			(else
			 (printf "Undefined toplevel command ~s - enter ',?' for help~%" form) 
			 (void) ) ) ) ) ) )
	    (else
	     (receive rs (eval form)
	       (add-h form rs) 
	       (apply values rs) ) ) ) ) ) )


;;; Echo-port for transcript:

(let ((open-output-port open-output-port)
      (close-output-port close-output-port)
      (flush-output flush-output)
      (write-char write-char)
      (newline newline)
      (echo #f) )
  (set! make-transcript-port
    (lambda (out)
      (let ((port (##sys#make-port #t 6 #f #f)))
	(##sys#setslot port 1 #t)
	(##sys#setslot port 3 "(transcript)")
	(##sys#setslot port 4 0)
	(##sys#setslot port 5 0)
	(##sys#setslot
	 port 2
	 (lambda (op p args)
	   (case op
	     ((#:close-output-port)
	      (close-output-port out)
	      (if echo (close-output-port echo)) )
	     ((#:flush-output)
	      (flush-output out)
	      (if echo (flush-output echo)) )
	     ((#:write-char)
	      (write-char args out)
	      (if echo (write-char args echo)) )
	     ((#:write-string)
	      (let ((plen ##sys#current-print-length)) ; Save current print-length (or it will be added twice)
		(##sys#print args #f out)
		(set! ##sys#current-print-length plen)
		(when echo 
		  (##sys#print args #f echo)
		  (set! ##sys#current-print-length plen) ) ) )
	     ((#:resolve) out)
	     (else ##sys#snafu) ) ) )
	port) ) )
  (set! write-to-echo-port
    (lambda (x) 
      (when echo
	(write x echo)
	(newline echo) ) ) )
  (set! transcript-on
    (lambda (fname)
      (##sys#check-string fname)
      (when echo (close-output-port echo))
      (set! echo (open-output-file fname)) ) )
  (set! transcript-off
    (lambda ()
      (when echo (close-output-port echo))
      (set! echo #f) ) ) )


;;; Display prompt if stdin is a terminal:

(set! ##sys#read-prompt-hook
  (let ([display display])
    (lambda ()
      (when (or (##sys#fudge 12) (##sys#tty-port? (current-input-port)))
	(display ">>> ") 
	(flush-output) ) ) ) )


;;; Tracing utilities:

(define ##sys#trace-indent-level 0)

(define ##sys#trace-indent
  (lambda ()
    (write-char #\|)
    (do ((i ##sys#trace-indent-level (sub1 i)))
	((<= i 0))
      (write-char #\space) ) ) )

(define ##sys#traced-procedure-entry
  (lambda (name args)
    (##sys#trace-indent)
    (set! ##sys#trace-indent-level (add1 ##sys#trace-indent-level))
    (fluid-let ([##sys#print-qualifiers #t])
      (write (cons name args)) )
    (newline) 
    (flush-output) ) )

(define ##sys#traced-procedure-exit
  (lambda (name results)
    (set! ##sys#trace-indent-level (sub1 ##sys#trace-indent-level))
    (##sys#trace-indent)
    (fluid-let ([##sys#print-qualifiers #t])
      (write name)
      (display " -> ")
      (for-each
       (lambda (x)
	 (write x)
	 (write-char #\space) )
       results) )
    (newline) 
    (flush-output) ) )

(define ##sys#traced-procedures (make-vector 301 '()))

(define-macro (trace . names)
  `(let ((apply apply) 
	 (values values)
	 (##sys#hash-table-set! ##sys#hash-table-set!)
	 (call-with-values call-with-values) )
     (##core#undefined)			; so '(trace)' works
     ,@(map (lambda (s)
	      (let ((name (gensym))
		    (old (gensym)) )
		`(let* ((,name ',s)
			(,old ,s) )
		   (##sys#hash-table-set! ##sys#traced-procedures ,name ,old)
		   (set! ,s (lambda args
			      (##sys#traced-procedure-entry ,name args)
			      (call-with-values (lambda () (apply ,old args))
				(lambda results
				  (##sys#traced-procedure-exit ,name results) 
				  (apply values results) ) ) ) ) ) ) )
	    names) ) )

(define-macro (untrace . names)
  `(begin
     (##core#undefined)			; so '(untrace)' works
     ,@(map (lambda (s)
	      (let ((var (gensym))
		    (old (gensym)) )
		`(let* ((,var ',s)
			(,old (##sys#hash-table-ref ##sys#traced-procedures ,var)) )
		   (if ,old
		       (set! ,s ,old)
		       (printf "Procedure '~s' was not traced.~%" ,var) )
		   (##sys#hash-table-set! ##sys#traced-procedures ,var #f) ) ) )
	    names) ) )


;;; Autoloading:

(define-macro (autoload filename symbols)
  (##sys#check-syntax 'autoload symbols '#(symbol 0))
  `(begin
     ,@(map (lambda (s)
	      `(define (,s . args)
		 (load ,filename)
		 (apply ,s args) ) )
	    symbols) ) )

(cond-expand 
 [(not format)
  (autoload 
   (##sys#resolve-include-filename "format.scm")
   (format) ) ]
 [else] )

(autoload 
 (##sys#resolve-include-filename "report.scm")
 (report) )

(autoload
 (##sys#resolve-include-filename "describe.scm")
 (describe dump) )

(cond-expand
 [unistd
  (autoload (##sys#resolve-include-filename "process.scm") (process)) ]
 [else] )


;;; Start interpreting:

(define (run)
  (let* ([extraopts (or (getenv "CSI_OPTIONS") "")]
	 [args (append (string-split extraopts) (cdr (argv)))] )

    (define (chop-separator str)
      (let ([len (sub1 (string-length str))])
	(if (and (> len 0) (char=? (string-ref str len) pathname-directory-separator))
	    (substring str 0 len)
	    str) ) )

    (or (and-let* ([p (member "-script" args)])
	  (set! command-line-arguments (make-parameter (cddr p)))
	  (register-feature! 'script)
	  (when (pair? (cdr p)) (set-cdr! (cdr p) '())) )
	(and-let* ([p (member "--" args)])
	  (set-cdr! p '()) ) )
    (let ([batch (or (member "-batch" args) (member "-script" args))]
	  [quiet (or (member "-quiet" args) (member "-script" args))]
	  [srfi7 (member "-srfi-7" args)]
	  [strict (member "-strict" args)]
	  [strict-srfi0 (member "-strict-srfi-0" args)]
	  [home (chop-separator (or (getenv "CHICKEN_HOME") installation-home ""))]
	  [ipath (map chop-separator (string-split (or (getenv "CHICKEN_INCLUDE_PATH") "") ";"))] )
      
      (define (collect-options opt)
	(let loop ([opts args])
	  (cond [(member opt opts) 
		 => (lambda (p)
		      (if (null? (cdr p))
			  (##sys#error "missing argument to command-line option" opt)
			  (cons (cadr p) (loop (cddr p)))) ) ]
		[else '()] ) ) )

      (define (loadinit)
	(let* ([prefix (chop-separator (or (getenv "HOME") "."))]
	       [fn (string-append prefix (string pathname-directory-separator) init-file)] )
	  (when (file-exists? fn) 
	    (load fn) ) ) )

      (when (member "-help" args)
	(print-usage)
	(exit 0) )
      (when (member "-version" args)
	(print-banner)
	(exit 0) )
      (when (member "-no-warnings" args)
	(unless quiet (display "; Warnings are disabled\n"))
	(set! ##sys#warnings-enabled #f) )
      (cond [quiet (set! ##sys#load-verbose #f)]
	    [else (print-banner)] )
      (when (member "-case-sensitive" args)
	(unless quiet (display "; Identifiers and symbols are case sensitive\n"))
	(register-feature! 'case-sensitive)
	(##sys#case-sensitive #t) )
      (for-each register-feature! (collect-options "-feature"))
      (for-each unregister-feature! (collect-options "-no-feature"))
      (set! ##sys#include-pathnames 
	(delete-duplicates
	 (append (map chop-separator (collect-options "-include-path"))
		 ##sys#include-pathnames
		 ipath
		 (if home (list home) '()) )
	 string=?) )
      (cond [(member "-hygienic" args) 
	     (unless quiet
	       (display "; Using hygienic macros")
	       (cond [strict (display " - strict mode\n")]
		     [strict-srfi0 (display " - strict mode with SRFI-0\n")]
		     [else (newline)] ) )
	     (##syncase#install-macro-package (or strict strict-srfi0) strict-srfi0) ]
	    [strict
	     (unless quiet (display "; Strict mode\n"))
	     (set! ##sys#strict-mode #t)
	     (unregister-feature! #:srfi-8 #:srfi-2 #:srfi-0 #:srfi-16)
	     (register-feature! #:strict)
	     (##sys#undefine-non-standard-macros '()) ]
	    [strict-srfi0
	     (unless quiet (display "; Strict mode with SRFI-0\n"))
	     (set! ##sys#strict-mode #t)
	     (unregister-feature! #:srfi-8 #:srfi-2 #:srfi-16)
	     (register-feature! #:strict)
	     (##sys#undefine-non-standard-macros '(cond-expand)) ] )
      (unless (member "-no-init" args) (loadinit))
      (set! ##sys#standard-output (make-transcript-port ##sys#standard-output))
      (do ((args args (cdr args)))
	  ((null? args)
	   (unless batch (read-eval-print-loop)) )
	(let* ([arg (car args)]
	       [len (string-length arg)] )
	  (cond ((member 
		  arg 
		  '("--" "-batch" "-quiet" "-no-init" "-hygienic" "-no-warnings" "-script"
		    "-case-sensitive" "-srfi-7" "-strict" "-strict-srfi-0") ) )
		((or (string=? arg "-feature") (string=? arg "-no-feature"))
		 (set! args (cdr args)) )
		((string=? "-include-path" arg) 
		 (set! args (cdr args)) )
		((string=? "-eval" arg)
		 (let ([in (open-input-string (cadr args))])
		   (do ([x (read in) (read in)])
		       ((eof-object? x))
		     (eval x) )
		   (set! args (cdr args)) ) )
		((and (>= len 2) (string=? "-:" (substring arg 0 2))))
		(srfi7
		 (eval 
		  (##sys#expand-srfi-7-program
		   (chop-separator arg)
		   (with-input-from-file arg read)
		   ##sys#error) ) )
		(else (load arg) ) ) ) ) ) ) )
