;;; c-backend.scm - C-generating backend for the CHICKEN compiler
;
; Copyright (c) 2000-2002, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany


(declare (unit backend))


#{compiler
  build-information compiler-arguments process-command-line
  default-analysis-database-size default-standard-bindings default-extended-bindings side-effecting-standard-bindings
  non-foldable-standard-bindings foldable-standard-bindings non-foldable-extended-bindings foldable-extended-bindings
  standard-bindings-that-never-return-false side-effect-free-standard-bindings-that-never-return-false
  installation-home optimization-iterations
  foreign-type-table-size file-io-only nonwinding-call/cc
  unit-name insert-timer-checks used-units inlining external-variables
  foreign-declarations emit-trace-info block-compilation analysis-database-size line-number-database-size
  target-heap-size target-stack-size try-harder default-installation-home 
  default-default-target-heap-size default-default-target-stack-size verbose-mode original-program-size
  current-program-size line-number-database-2 foreign-lambda-stubs immutable-constants
  rest-parameters-promoted-to-vector inline-table inline-table-used constant-table constants-used mutable-constants
  dependency-list broken-constant-nodes inline-substitutions-enabled
  direct-call-ids foreign-type-table first-analysis block-variable-literal?
  initialize-compiler canonicalize-expression expand-foreign-lambda update-line-number-database scan-toplevel-assignments
  perform-cps-conversion analyze-expression simplifications perform-high-level-optimizations perform-pre-optimization!
  reorganize-recursive-bindings substitution-table simplify-named-call find-inlining-candidates perform-inlining!
  perform-closure-conversion prepare-for-code-generation compiler-source-file create-foreign-stub expand-foreign-lambda*
  transform-direct-lambdas! target-include-file
  debugging-chicken warnings-enabled bomb check-signature posq stringify symbolify flonum? build-lambda-list
  string->c-identifier c-ify-string words check-and-open-input-file close-checked-input-file fold-inner constant?
  collapsable-literal? immediate? canonicalize-begin-body extract-mutable-constants string->expr get get-all
  put! collect! count! get-line get-line-2 find-lambda-container display-analysis-database varnode qnode 
  build-node-graph build-expression-tree fold-boolean inline-lambda-bindings match-node expression-has-side-effects?
  simple-lambda-node? compute-database-statistics print-program-statistics output gen gen-list 
  pprint-expressions-to-file foreign-type-check estimate-foreign-result-size scan-used-variables scan-free-variables
  topological-sort print-version print-usage initialize-analysis-database
  generate-external-variables
  product copyright compiler-features default-declarations units-used-by-default words-per-flonum
  foreign-string-result-reserve parameter-limit default-output-filename eq-inline-operator optimizable-rest-argument-operators
  membership-test-operators membership-unfold-limit valid-compiler-options valid-compiler-options-with-argument
  default-optimization-iterations generate-foreign-callback-header generate-foreign-callback-stub-prototypes
  generate-code make-variable-list make-argument-list generate-foreign-stubs foreign-type-declaration
  foreign-argument-conversion foreign-result-conversion}

(include "tweaks")


;;; Write atoms to output-port:

(define output #f)

(define (gen . data)
  (for-each
   (lambda (x) 
     (if (eq? #t x)
	 (newline output)
	 (display x output) ) )
   data) )

(define (gen-list lst)
  (for-each
   (lambda (x) (display x output))
   (intersperse lst #\space) ) )


;;; Generate target code:

(define (generate-code literals lambdas out source-file)

  (define (find-lambda id)
    (or (find (lambda (ll) (eq? id (lambda-literal-id ll))) lambdas)
	(bomb "can't find lambda" id) ) )
  
  (define (expression node temps ll)

    (define (expr n i)
      (let ((subs (node-subexpressions n))
	    (params (node-parameters n)) )
	(case (node-class n)

	  ((##core#immediate)
	   (case (first params)
	     ((bool) (gen (if (second params) "C_SCHEME_TRUE" "C_SCHEME_FALSE")))
	     ((char) (gen "C_make_character(" (char->integer (second params)) #\)))
	     ((nil) (gen "C_SCHEME_END_OF_LIST"))
	     ((fix) (gen "C_fix(" (second params) #\)))
	     (else (bomb "bad immediate")) ) )

	  ((##core#literal) (gen "lf[" (first params) #\]))

	  ((if)
	   (gen #t "if(C_truep(")
	   (expr (car subs) i)
	   (gen ")){")
	   (expr (cadr subs) i)
	   (gen #\} #t "else{")
	   (expr (caddr subs) i)
	   (gen #\}) )

	  ((##core#proc) (gen "(C_word)" (first params)))

	  ((##core#bind) 
	   (let loop ((bs subs) (i i) (count (first params)))
	     (cond [(> count 0)
		    (gen #t #\t i #\=)
		    (expr (car bs) i)
		    (gen #\;) 
		    (loop (cdr bs) (add1 i) (sub1 count)) ]
		   [else (expr (car bs) i)] ) ) )

	  ((##core#ref) 
	   (gen "((C_word*)")
	   (expr (car subs) i)
	   (gen ")[" (+ (first params) 1) #\]) )

	  ((##core#unbox) 
	   (gen "((C_word*)")
	   (expr (car subs) i)
	   (gen ")[1]") )

	  ((##core#update_i)
	   (gen "C_set_block_item(")
	   (expr (car subs) i)
	   (gen #\, (first params) #\,)
	   (expr (cadr subs) i) 
	   (gen #\)) )

	  ((##core#update)
	   (gen "C_mutate(((C_word *)")
	   (expr (car subs) i)
	   (gen ")+" (+ (first params) 1) ",")
	   (expr (cadr subs) i) 
	   (gen #\)) )

	  ((##core#updatebox_i)
	   (gen "C_set_block_item(")
	   (expr (car subs) i)
	   (gen ",0,")
	   (expr (cadr subs) i) 
	   (gen #\)) )

	  ((##core#updatebox)
	   (gen "C_mutate(((C_word *)")
	   (expr (car subs) i)
	   (gen ")+1,")
	   (expr (cadr subs) i) 
	   (gen #\)) )

	  ((##core#closure)
	   (let ([n (first params)])
	     (gen "(*a=C_CLOSURE_TYPE|" n #\,)
	     (for-each 
	      (lambda (x j)
		(gen "a[" j "]=")
		(expr x i)
		(gen #\,) )
	      subs (iota n 1 1) )
	     (gen "tmp=(C_word)a,a+=" (add1 n) ",tmp)") ) )

	  ((##core#box) 
	   (gen "(*a=C_VECTOR_TYPE|1,a[1]=")
	   (expr (car subs) i)
	   (gen ",tmp=(C_word)a,a+=2,tmp)") )

	  ((##core#local) (gen #\t (first params)))

	  ((##core#setlocal) 
	   (gen #\t (first params) #\=)
	   (expr (car subs) i) )

	  ((##core#global)
	   (let ([index (first params)]
		 [safe (second params)] 
		 [block (third params)] ) ; block implies safe
	     (cond [block (gen "lf[" index "]")]
		   [safe (gen "C_retrieve(lf[" index "])")]
		   [else (gen "*((C_word*)lf[" index "]+1)")] ) ) )

	  ((##core#setglobal)
	   (let ([index (first params)]
		 [block (second params)] )
	     (if block
		 (gen "C_mutate(&lf[" index "],")
		 (gen "C_mutate((C_word*)lf[" index "]+1,") )
	     (expr (car subs) i)
	     (gen #\)) ) )

	  ((##core#setglobal_i)
	   (let ([index (first params)]
		 [block (second params)] )
	     (cond [block
		    (gen "lf[" index "]=")
		    (expr (car subs) i)
		    (gen #\;) ]
		   [else
		    (gen "C_set_block_item(lf[" index "],0,")
		    (expr (car subs) i)
		    (gen #\)) ] ) ) )

	  ((##core#undefined) (gen "C_SCHEME_UNDEFINED"))

	  ((##core#call) 
	   (let* ((args (cdr subs))
		  (n (length args))
		  (nc i)
		  (nf (add1 n)) 
		  (p2 (pair? (cdr params)))
		  (name (and p2 (second params)))
		  (call-id (and p2 (pair? (cddr params)) (third params))) 
		  (customizable (and call-id (fourth params)))
		  (empty-closure (and customizable (zero? (lambda-literal-closure-size (find-lambda call-id)))))
		  (fn (car subs)) )
	     (when (and emit-trace-info name) (gen #t "C_trace(\"" name "\");"))
	     (cond ((eq? '##core#proc (node-class fn))
		    (gen #t (first (node-parameters fn)) #\( nf ",0,") 
		    (expr-args args i)
		    (gen ");") )
		   (call-id
		    (cond ((and (eq? call-id (lambda-literal-id ll))
				(lambda-literal-looping ll) )
			   (let* ([temps (lambda-literal-temporaries ll)]
				  [ts (iota n (+ temps nf) 1)] )
			     (for-each
			      (lambda (arg tr)
				(gen #t #\t tr #\=)
				(expr arg i) 
				(gen #\;) )
			      args ts)
			     (for-each
			      (lambda (from to) (gen #t #\t to "=t" from #\;))
			      ts (iota n 1 1) )
			     (unless customizable (gen #t "c=" nf #\;))
			     (gen #t "goto loop;") ) )
			  (else
			   (unless empty-closure
			     (gen #t #\t nc #\=)
			     (expr fn i)
			     (gen #\;) )
			   (gen #t call-id #\()
			   (unless customizable (gen nf #\,))
			   (unless empty-closure (gen #\t nc #\,))
			   (expr-args args i)
			   (gen ");") ) ) )
		   (else
		    (gen #t #\t nc #\=)
		    (expr fn i)
		    (gen #\; #t
			 "((C_proc" nf ")")
		    (if (or unsafe no-procedure-checks (first params))
			(gen "(void*)(*((C_word*)t" nc "+1))")
			(gen "C_retrieve_proc(t" nc ")") )
		    (gen ")(" nf ",t" nc #\,)
		    (expr-args args i)
		    (gen ");") ) ) ) )
	  
	  ((##core#recurse) 
	   (let* ([n (length subs)]
		  [nf (add1 n)]
		  [tailcall (first params)]
		  [call-id (second params)] 
		  [emtpy-closure (zero? (lambda-literal-closure-size ll))] )
	     (cond (tailcall
		    (let* ([temps (lambda-literal-temporaries ll)]
			   [ts (iota n (+ temps nf) 1)] )
		      (for-each
		       (lambda (arg tr)
			 (gen #t #\t tr #\=)
			 (expr arg i) 
			 (gen #\;) )
		       subs ts)
		      (for-each
		       (lambda (from to) (gen #t #\t to "=t" from #\;))
		       ts (iota n 1 1) )
		      (gen #t "goto loop;") ) )
		   (else
		    (gen call-id #\()
		    (unless empty-closure (gen "t0,"))
		    (expr-args subs i)
		    (gen #\)) ) ) ) )

	  ((##core#direct_call) 
	   (let* ((args (cdr subs))
		  (n (length args))
		  (nf (add1 n)) 
		  (name (second params))
		  (call-id (third params))
		  (demand (fourth params))
		  (allocating (not (zero? demand)))
		  (empty-closure (zero? (lambda-literal-closure-size (find-lambda call-id))))
		  (fn (car subs)) )
	     (when allocating (gen "(a+=" demand #\,))
	     (gen call-id #\()
	     (when allocating 
	       (gen "a-" demand)
	       (when (or (not empty-closure) (pair? args)) (gen #\,)) )
	     (unless empty-closure
	       (expr fn i)
	       (when (pair? args) (gen #\,)) )
	     (when (pair? args) (expr-args args i))
	     (gen #\))
	     (when allocating (gen #\))) ) )

	  ((##core#callunit)
	   ;; The code generated here does not use the extra temporary needed for standard calls, so we have
	   ;;  one unused varable (but, who cares!):
	   (let* ((n (length subs))
		  (nf (+ n 1)) )
	     (gen #t "C_" (first params) "_toplevel(" nf ",C_SCHEME_UNDEFINED,")
	     (expr-args subs i)
	     (gen ");") ) )

	  ((##core#return)
	   (gen #t "return(")
	   (expr (first subs) i)
	   (gen ");") )

	  ((##core#inline)
	   (gen "(C_word)" (first params) #\()
	   (expr-args subs i)
	   (gen #\)) )

	  ((##core#inline_allocate)
	   (gen "(C_word)" (first params) "(&a," (length subs))
	   (if (pair? subs)
	       (begin
		 (gen #\,)
		 (expr-args subs i) ) )
	   (gen #\)) )

	  ((##core#inline_ref)
	   (gen (foreign-result-conversion (second params) "a") (first params) #\)) )

	  ((##core#inline_update)
	   (gen #\( (first params) #\= (foreign-argument-conversion (second params)))
	   (expr (first subs) i)
	   (gen "),C_SCHEME_UNDEFINED)") )

	  ((##core#switch)
	   (gen #t "switch(")
	   (expr (first subs) i)
	   (gen "){")
	   (do ([j (first params) (sub1 j)]
		[ps (cdr subs) (cddr ps)] )
	       ((zero? j)
		(gen #t "default:")
		(expr (car ps) i)
		(gen #\}) )
	     (gen #t "case ")
	     (expr (car ps) i)
	     (gen #\:)
	     (expr (cadr ps) i) ) )

	  ((##core#cond)
	   (gen "(C_truep(")
	   (expr (first subs) i)
	   (gen ")?")
	   (expr (second subs) i)
	   (gen #\:)
	   (expr (third subs) i)
	   (gen #\)) )

	  (else (bomb "bad form")) ) ) )
    
    (define (expr-args args i)
      (pair-for-each
       (lambda (xs)
	 (if (not (eq? xs args)) (gen #\,))
	 (expr (car xs) i) )
       args) )

    (expr node temps) )
    
  (define (header)
    (gen "/* Generated from " source-file " by " product ", " build-information #t
	 "   Command line: ")
    (gen-list compiler-arguments)
    (gen #t)
    (cond [unit-name (gen "   Unit: " unit-name)]
	  [else 
	   (gen "   Used units: ")
	   (gen-list used-units) ] )
    (unless unit-name
      (gen #t "   default installation home: " default-installation-home #t)
      (gen "   default heap size: " default-default-target-heap-size #t)
      (gen "   default nursery (stack) size: " default-default-target-stack-size) )
    (gen #t "*/" #t #t "#include \"" target-include-file "\"")
    (when (pair? foreign-declarations)
      (gen #t)
      (for-each (lambda (decl) (gen #t decl)) foreign-declarations) ) )
  
  (define (trailer)
    (gen #t "/* END OF FILE */" #t) )
  
  (define (declarations)
    (let ((n (length literals)))
      (gen #t)
      (for-each 
       (lambda (uu) (gen #t "C_extern void C_" uu "_toplevel(int c,C_word d,C_word k) C_noret;"))
       used-units)
      (unless (zero? n) (gen #t #t "static C_word lf[" n "];")) ) )
  
  (define (prototypes)
    (let ([large-signatures '()])
      (gen #t)
      (for-each
       (lambda (ll)
	 (let* ([n (lambda-literal-argument-count ll)]
		[customizable (lambda-literal-customizable ll)] 
		[empty-closure (and customizable (zero? (lambda-literal-closure-size ll)))]
		[varlist (intersperse (make-variable-list (if empty-closure (sub1 n) n) "t") #\,)]
		[id (lambda-literal-id ll)]
		[rest (lambda-literal-rest-argument ll)]
		[rest-mode (lambda-literal-rest-argument-mode ll)]
		[direct (lambda-literal-direct ll)] 
		[allocated (lambda-literal-allocated ll)] )
	   (when (>= n small-parameter-limit)
	     (set! large-signatures (lset-adjoin = large-signatures (add1 n))) )
	   (gen #t)
	   (for-each
	    (lambda (s) 
	      (when (>= s small-parameter-limit)
		(set! large-signatures (lset-adjoin = large-signatures (add1 s))) ) )
	    (lambda-literal-callee-signatures ll) )
	   (cond [(not (eq? 'toplevel id))
		  (gen "static " (if direct "C_word " "void "))
		  (when customizable (gen "C_fcall "))
		  (gen id) ]
		 [else
		  (gen "C_extern void ")
		  (gen "C_" (if unit-name (string-append unit-name "_toplevel") "toplevel")) ] )
	   (gen #\()
	   (unless customizable (gen "int c,"))
	   (when (and direct (not (zero? allocated)))
	     (gen "C_word *a")
	     (when (pair? varlist) (gen #\,)) )
	   (apply gen varlist)
	   (cond [rest
		  (gen ",...) C_noret;")
		  (if (not (eq? rest-mode 'none))
		      (begin
			(gen #t "static void " id "r(")
			(apply gen varlist)
			(gen ",C_word t" (+ n 1) ") C_noret;") ) ) ]
		 [else 
		  (gen #\))
		  (when customizable (gen " C_c_regparm"))
		  (unless direct (gen " C_noret"))
		  (gen #\;) ] ) ) )
       lambdas) 
      (for-each
       (lambda (s)
	 (gen #t "typedef void (*C_proc" s ")(int")
	 (for-each gen (make-list s ",C_word"))
	 (gen ") C_noret;") )
       large-signatures) ) )
  
  (define (trampolines)
    (let ([ns '()]
          [nsr '()] 
	  [nsrv '()] )

      (define (restore n)
        (do ((i (- n 1) (- i 1))
	     (j 0 (+ j 1)) )
            ((negative? i))
          (gen #t "C_word t" i "=C_pick(" j ");") ) 
	(gen #t "C_adjust_stack(-" n ");") )

      (define (emitter vflag)
	(lambda (n)
	  (gen #t #t "static void C_fcall tr" n #\r)
	  (if vflag (gen #\v))
	  (gen "(C_proc" n " k) C_regparm C_noret;")
	  (gen #t "static void C_fcall tr" n #\r)
	  (if vflag (gen #\v))
	  (gen "(C_proc" n " k){"
	       #t "int n;"
	       #t "C_word *a,t" n #\;)
	  (restore n)
	  (gen #t "n=C_rest_count(0);")
	  (gen #t "a=C_alloc(n*3);")
	  (gen #t #\t n "=C_restore_rest")
	  (if vflag (gen "_vector"))
	  (gen "(a,n);")
	  (gen #t "(k)(")
	  (apply gen (intersperse (make-argument-list (+ n 1) "t") #\,))
	  (gen ");}") ) )

      (for-each
       (lambda (ll)
	 (let* ([argc (lambda-literal-argument-count ll)]
		[rest (lambda-literal-rest-argument ll)]
		[rest-mode (lambda-literal-rest-argument-mode ll)]
		[id (lambda-literal-id ll)]
		[customizable (lambda-literal-customizable ll)]
		[empty-closure (and customizable (zero? (lambda-literal-closure-size ll)))] )
	   (when empty-closure (set! argc (sub1 argc)))
	   (unless (lambda-literal-direct ll)
	     (cond [customizable
		    (gen #t #t "static void C_fcall tr" id "(void *dummy) C_regparm C_noret;")
		    (gen #t "static void C_fcall tr" id "(void *dummy){")
		    (restore argc)
		    (gen #t id #\()
		    (let ([al (make-argument-list argc "t")])
		      (apply gen (intersperse al #\,)) )
		    (gen ");}") ]
		   [(or rest (> (lambda-literal-allocated ll) 0) (lambda-literal-external ll))
		    (if (and rest (not (eq? rest-mode 'none)))
			(if (eq? rest-mode 'vector)
			    (set! nsrv (lset-adjoin = nsrv argc))
			    (set! nsr (lset-adjoin = nsr argc)) ) 
			(set! ns (lset-adjoin = ns argc)) ) ] ) ) ) )
       lambdas)
      (for-each
       (lambda (n)
	 (gen #t #t "static void C_fcall tr" n "(C_proc" n " k) C_regparm C_noret;")
	 (gen #t "static void C_fcall tr" n "(C_proc" n " k){")
	 (restore n)
	 (gen #t "(k)(" n #\,)
	 (apply gen (intersperse (make-argument-list n "t") #\,))
	 (gen ");}") )
       ns)
      (for-each (emitter #f) nsr)
      (for-each (emitter #t) nsrv) ) )
  
  (define (literal-frame)
    (do ([i 0 (+ i 1)]
	 [lits literals (cdr lits)] )
	((null? lits))
      (gen-lit (car lits) (sprintf "lf[~s]" i)) ) )

  (define (bad-literal lit)
    (bomb "type of literal not supported" lit) )

  (define (literal-size lit)
    (cond [(immediate? lit) 0]
	  [(string? lit) (+ 2 (words (string-length lit)))]
	  [(number? lit) words-per-flonum]
	  [(symbol? lit) (+ 9 (literal-size (##sys#slot lit 1)))]
	  [(pair? lit) (+ 3 (literal-size (car lit)) (literal-size (cdr lit)))]
	  [(vector? lit) (+ 2 (vector-length lit) (reduce + 0 (map literal-size (vector->list lit))))]
	  [(block-variable-literal? lit) 0]
	  [(##sys#immediate? lit) (bad-literal lit)]
	  [(##sys#bytevector? lit)
	   (if (##sys#permanent? lit)
	       0
	       (+ 2 (words (##sys#size lit))) ) ]
	  [(##sys#generic-structure? lit)
	   (let ([n (##sys#size lit)])
	     (let loop ([i 0] [s (+ 2 n)])
	       (if (>= i n)
		   s
		   (loop (add1 i) (+ s (literal-size (##sys#slot lit i)))) ) ) ) ]
	  [else (bad-literal lit)] ) )

  (define (gen-lit lit to)
    (cond ((fixnum? lit)
	   (if (eq? 'flonum number-type)
	       (gen #t to "=C_flonum(C_heaptop," lit ");")
	       (gen #t to "=C_fix(" lit ");") ) )
	  ((or (eq? lit (void)) (block-variable-literal? lit))
	   (gen #t to "=C_SCHEME_UNDEFINED;") )
	  ((number? lit)
	   (cond ((eq? 'fixnum number-type)
		  (warning "coerced inexact literal number '~s' to fixnum" lit)
		  (gen #t to "=C_fix(" lit ");") )
		 (else (gen #t to "=C_flonum(C_heaptop," lit ");")) ) )
          ((boolean? lit) 
           (gen #t to #\= (if lit "C_SCHEME_TRUE" "C_SCHEME_FALSE") #\;) )
          ((char? lit)
           (gen #t to "=C_make_character(" (char->integer lit) ");") )
	  ((null? lit) 
	   (gen #t to "=C_SCHEME_END_OF_LIST;") )
          ((string? lit) (gen-string-like-lit to lit "C_string" #t))
	  ((pair? lit)
	   (cond ((and (proper-list? lit) (pair? (cdr lit)))
		  (do ((len 0 (add1 len))
		       (lst lit (cdr lst)) )
		      ((null? lst)
		       (gen #t to "=C_a_i_list(C_heaptop," len)
		       (do ((k (sub1 len) (sub1 k)))
			   ((< k 0) (gen ");" #t "C_drop(" len ");"))
			 (gen ",C_pick(" k #\)) ) )
		    (gen-lit (car lst) "tmp")
		    (gen #t "C_save(tmp);") ) )
		 (else
		  (gen-lit (car lit) "tmp")
		  (gen #t "C_save(tmp);")
		  (gen-lit (cdr lit) "tmp")
		  (gen #t to "=C_pair(C_heaptop,C_restore,tmp);") ) ) )
	  ((vector? lit) (gen-vector-like-lit to lit "C_vector"))
          ((symbol? lit)
	   (let* ([str (##sys#slot lit 1)]
		  [cstr (c-ify-string str)]
		  [len (##sys#size str)] )
	     (gen #t to "=C_intern(C_heaptop," len #\, cstr ");") ) )
	  ((##sys#immediate? lit) (bad-literal lit))
	  ((##sys#bytevector? lit)
	   (if (##sys#permanent? lit)
	       (gen-string-like-lit to lit "C_pbytevector" #f)
	       (gen-string-like-lit to lit "C_bytevector" #t) ) )
	  ((##sys#generic-structure? lit) (gen-vector-like-lit to lit "C_structure"))
          (else (bad-literal lit)) ) )

  (define (gen-string-like-lit to lit conser top)
    (let* ([len (##sys#size lit)]
	   [ns (fx/ len 80)]
	   [srest (modulo len 80)] )
      (gen #t to #\= conser #\()
      (when top (gen "C_heaptop,"))
      (gen len #\,)
      (do ([i ns (sub1 i)]
	   [offset 0 (+ offset 80)] )
	  ((zero? i)
	   (when (or (zero? len) (not (zero? srest)))
	     (gen (c-ify-string (string-like-substring lit offset len))) )
	   (gen ");") )
	(gen (c-ify-string (string-like-substring lit offset (+ offset 80))) #t) ) ) )
  
  (define (string-like-substring s start end)
    (let* ([len (- end start)]
	   [s2 (make-string len)] )
      (##sys#copy-bytes s s2 start 0 len)
      s2) )

  (define (gen-vector-like-lit to lit conser)
    (let ([len (##sys#size lit)])
      (do ([j 0 (+ j 1)]
	   [n len (- n 1)] )
	  ((zero? n)
	   (gen #t to #\= conser "(C_heaptop," len)
	   (do ([j (- len 1) (- j 1)])
	       ((< j 0) (gen ");" #t "C_drop(" len ");"))
	     (gen ",C_pick(" j #\)) ) )
	(gen-lit (##sys#slot lit j) "tmp")
	(gen #t "C_save(tmp);") ) ) )

  (define (procedures)
    (for-each
     (lambda (ll)
       (let* ([n (lambda-literal-argument-count ll)]
	      [id (lambda-literal-id ll)]
	      [demand (lambda-literal-allocated ll)]
	      [rest (lambda-literal-rest-argument ll)]
	      [customizable (lambda-literal-customizable ll)]
	      [empty-closure (and customizable (zero? (lambda-literal-closure-size ll)))]
	      [nec (- n (if empty-closure 1 0))]
	      [vlist0 (make-variable-list n "t")]
	      [varlist (intersperse (if empty-closure (cdr vlist0) vlist0) #\,)]
	      [external (lambda-literal-external ll)]
	      [looping (lambda-literal-looping ll)]
	      [direct (lambda-literal-direct ll)]
	      [rest-mode (lambda-literal-rest-argument-mode ll)]
	      [temps (lambda-literal-temporaries ll)]
	      [topname (if unit-name
			   (string-append unit-name "_toplevel")
			   "toplevel") ] )
	 (when empty-closure (debugging 'o "dropping unused closure argument" id))
	 (gen #t #t)
	 (cond [(not (eq? 'toplevel id)) 
		(gen "static " (if direct "C_word " "void "))
		(when customizable (gen "C_fcall "))
		(gen id) ]
	       [else
		(gen "static int toplevel_initialized=0;"
		     #t "static void C_fcall toplevel_trampoline(void *dummy) C_regparm C_noret;"
		     #t "static void C_fcall toplevel_trampoline(void *dummy){"
		     #t "C_" topname "(2,C_SCHEME_UNDEFINED,C_restore);}"
		     #t #t "void C_" topname) ] )
	 (gen #\()
	 (unless customizable (gen "int c,"))
	 (when (and direct (not (zero? demand))) 
	   (gen "C_word *a")
	   (when (pair? varlist) (gen #\,)) )
	 (apply gen varlist)
	 (when rest (gen ",..."))
	 (gen "){")
	 (when (eq? rest-mode 'none) (set! rest #f))
	 (gen #t "C_word tmp;")
	 (if rest
	     (gen #t "C_word t" n #\;)	; To hold rest-list if demand is met
	     (do ([i n (add1 i)]
		  [j (+ temps (if looping (sub1 n) 0)) (sub1 j)] )
		 ((zero? j))
	       (gen #t "C_word t" i #\;) ) )
	 (cond [(eq? 'toplevel id) 
		(let ([ldemand (fold (lambda (lit n) (+ n (literal-size lit))) 0 literals)]
		      [llen (length literals)] )
		  (gen #t "C_word *a;"
		       #t "if(toplevel_initialized) C_kontinue(t1,C_SCHEME_UNDEFINED);"
		       #t "else C_toplevel_entry(C_text(\"" topname "\"));")
		  (when (and target-heap-size (not unit-name))
		    (gen #t "C_set_or_change_heap_size(" target-heap-size ",1);") )
		  (when (and target-stack-size (not unit-name))
		    (gen #t "C_resize_stack(" target-stack-size ");") )
		  (gen #t "if(!C_demand(" demand ")){"
		       #t "C_save(t1);"
		       #t "C_reclaim(toplevel_trampoline,NULL);}"
		       #t "toplevel_initialized=1;")
		  (gen #t "if(!C_demand_2(" ldemand ")) C_bad_memory_2();")
		  (gen #t "a=C_alloc(" demand ");")
		  (when (not (zero? llen))
		    (literal-frame)
		    (gen #t "C_register_lf(lf," llen ");") ) ) ]
	       [rest
		(gen #t "va_list v;")
		(gen #t "C_word *a,c2=c;")
		(gen #t "C_save_rest(")
		(if (> n 0)
		    (gen #\t (- n 1))
		    (gen "c") )
		(gen ",c2," n ");")
		(when (and (not unsafe) (not no-argc-checks) (> n 2))
		  (gen #t "if(c<" n ") C_bad_min_argc(c," n ");") )
		(when insert-timer-checks (gen #t "C_check_for_interrupt;"))
		(gen #t "if(!C_stack_probe(&a-" (+ n demand) ")){") ] ; (rough approximation)
	       [else
		(cond [(and (not direct) (> demand 0))
		       (if looping
			   (gen #t "C_word *a;"
				#t "loop:"
				#t "a=C_alloc(" demand ");")
			   (gen #t "C_word ab[" demand "],*a=ab;") ) ]
		      [else
		       (unless direct (gen #t "C_word *a;"))
		       (when looping (gen #t "loop:")) 
		       (when (and direct (not unsafe)) (gen #t "C_stack_check;")) ] )
		(when (and external (not unsafe) (not no-argc-checks) (not customizable))
		  (if (eq? rest-mode 'none)
		      (when (> n 2) (gen #t "if(c<" n ") C_bad_min_argc(c," n ");"))
		      (gen #t "if(c!=" n ") C_bad_argc(c," n ");") ) )
		(when (and (not direct) (or external (> demand 0)))
		  (when insert-timer-checks (gen #t "C_check_for_interrupt;"))
		  (if (and looping (> demand 0))
		      (gen #t "if(!C_stack_probe(a)){")
		      (gen #t "if(!C_stack_probe(&a)){") ) ) ] )
	 (when (and (not (eq? 'toplevel id))
		    (not direct)
		    (or rest external (> demand 0)) )
	   (cond [(> nec 1)
		  (gen #t "C_adjust_stack(" nec ");")
		  (do ([i (if empty-closure 1 0) (+ i 1)])
		      ((>= i n))
		    (gen #t "C_rescue(t" i #\, (- n i 1) ");") ) ]
		 [(= nec 1) (gen #t "C_save(" (if empty-closure "t1" "t0") ");")] )
	   (cond [rest
		  (gen #t "C_reclaim(tr" n #\r)
		  (when (eq? rest-mode 'vector) (gen #\v))
		  (gen #\, id "r);}"
		       #t "else{"
		       #t "a=C_alloc((c-" n ")*3);")
		  (case rest-mode
		    [(list #f) (gen #t "t" n "=C_restore_rest(a,C_rest_count(0));")]
		    [(vector) (gen #t "t" n "=C_restore_rest_vector(a,C_rest_count(0));")] )
		  (gen #t id "r(")
		  (apply gen (intersperse (make-argument-list n "t") #\,))
		  (gen ",t" n ");}}")
		  ;; Create secondary routine (no demand-check or argument-count-parameter):
		  (gen #t #t "static void " id "r(")
		  (apply gen varlist)
		  (gen ",C_word t" n "){")
		  (gen #t "C_word tmp;")
		  (do ([i (+ n 1) (+ i 1)]
		       [j temps (- j 1)] )
		      ((zero? j))
		    (gen #t "C_word t" i #\;) )
		  (when (> demand 0) (gen #t "C_word *a=C_alloc(" demand ");")) ]
		 [else 
		  (gen #t "C_reclaim(tr")
		  (if customizable 
		      (gen id ",NULL")
		      (gen n #\, id) )
		  (gen ");}") ] ) )
	 (expression
	  (lambda-literal-body ll)
	  (if rest
	      (add1 n)	; One temporary is needed to hold the rest-list
	      n)
	  ll)
	 (gen #\}) ) )
     lambdas) )

  (debugging 'p "code generation phase...")
  (set! output out)
  (header)
  (declarations)
  (generate-external-variables external-variables)
  (generate-foreign-callback-stub-prototypes foreign-callback-stubs)
  (generate-foreign-stubs foreign-lambda-stubs)
  (prototypes)
  (generate-foreign-callback-stubs foreign-callback-stubs)
  (trampolines)
  (procedures)
  (trailer) )


;;; Create list of variables/parameters, interspersed with a special token:

(define (make-variable-list n prefix)
  (list-tabulate
   n
   (lambda (i) (string-append "C_word " prefix (number->string i))) ) )
  
(define (make-argument-list n prefix)
  (list-tabulate
   n
   (lambda (i) (string-append prefix (number->string i))) ) )


;;; Generate external variable declarations:

(define (generate-external-variables vars)
  (gen #t)
  (for-each
   (lambda (xvar)
     (let ([name (car xvar)]
	   [type (cdr xvar)] )
       (gen #t (foreign-type-declaration type name) #\;) ) )
   vars) )


;;; Generate foreign stubs:

(define (generate-foreign-callback-stub-prototypes stubs)
  (for-each
   (lambda (stub)
     (gen #t)
     (generate-foreign-callback-header "C_extern " stub)
     (gen #\;) )
   stubs) )

(define (generate-foreign-stubs stubs)
  (for-each
   (lambda (stub)
     (let* ([id (foreign-stub-id stub)]
	    [types (foreign-stub-argument-types stub)]
	    [n (length types)]
	    [varlist (intersperse (cons "C_word C_buf" (make-variable-list n "C_a")) #\,)]
	    [rtype (foreign-stub-return-type stub)] 
	    [sname (foreign-stub-name stub)] 
	    [body (foreign-stub-body stub)]
	    [names (or (foreign-stub-argument-names stub) (make-list n #f))]
	    [rconv (foreign-result-conversion rtype "C_a")] 
	    [callback (foreign-stub-callback stub)] )
       (gen #t)
       (when body (gen #t "#define return(x) C_cblock C_r = (" rconv "(x))); goto C_return; C_cblockend"))
       (if callback
	   (gen #t "static void " id "(int c,C_word self,C_word k,")
	   (gen #t "static C_word C_fcall " id #\() )
       (apply gen varlist)
       (if callback
	   (gen ") C_noret;" #t "static void " id "(int c,C_word self,C_word k,")
	   (gen ") C_regparm;" #t "static C_word C_fcall " id #\() )
       (apply gen varlist)
       (gen "){")
       (gen #t "C_word C_r=C_SCHEME_UNDEFINED,*C_a=(C_word*)C_buf;")
       (for-each
	(lambda (type index name)
	  (gen #t 
	       (foreign-type-declaration 
		type
		(if name (symbol->string name) (sprintf "t~a" index)) )
	       #\= (foreign-argument-conversion type) "C_a" index ");") )
	types (iota n) names)
       (when callback (gen #t "int C_dummy=C_save_callback_continuation(&C_a,k);"))
       (cond [body
	      (gen #t body
		   #t "C_return:")
	      (if callback
		  (gen #t "k=C_restore_callback_continuation();"
		       #t "C_kontinue(k,C_r);")
		  (gen #t "return C_r;") )
	      (gen #t "#undef return" #t) ]
	     [else
	      (if (not (eq? rtype 'void))
		  (gen #t "C_r=" rconv)
		  (gen #t) )
	      (gen sname #\()
	      (apply gen (intersperse (make-argument-list n "t") #\,))
	      (unless (eq? rtype 'void) (gen #\)))
	      (gen ");")
	      (if callback
		  (gen #t "k=C_restore_callback_continuation();"
		       #t "C_kontinue(k,C_r);")
		  (gen #t "return C_r;") ) ] )
       (gen #\}) ) )
   stubs) )

(define (generate-foreign-callback-stubs stubs)
  (for-each
   (lambda (stub)
     (let* ([id (foreign-callback-stub-id stub)]
	    [rtype (foreign-callback-stub-return-type stub)]
	    [argtypes (foreign-callback-stub-argument-types stub)]
	    [n (length argtypes)]
	    [vlist (make-argument-list n "t")] )

       (define (compute-size type var ns)
	 (case type
	   [(char int short bool void unsigned-short scheme-object unsigned-char unsigned-int) ns]
	   [(float double c-pointer unsigned-integer long integer unsigned-long)
	    (string-append ns "+3") ]
	   [(c-string)
	    (string-append ns "+2+(" var "==NULL?1:C_bytestowords(C_strlen(" var ")))") ]
	   [else
	    (cond [(and (symbol? type) (##sys#hash-table-ref foreign-type-table type)) 
		   => (lambda (t)
			(compute-size (if (vector? t) (vector-ref t 0) t) var ns) ) ]
		  [(pair? type)
		   (case (car type)
		     [(pointer function) (string-append ns "+3")]
		     [else ns] ) ]
		  [else ns] ) ] ) )

       (let ([sizestr (fold compute-size "0" argtypes vlist)])
	 (gen #t)
	 (generate-foreign-callback-header "" stub)
	 (gen #\{ #t "C_word x, *a=C_alloc(" sizestr ");")
	 (for-each
	  (lambda (v t)
	    (gen #t "x=" (foreign-result-conversion t "a") v ");"
		 #t "C_save(x);") )
	  vlist 
	  argtypes)
	 (unless (eq? 'void rtype)
	   (gen #t "return " (foreign-argument-conversion rtype)) )
	 (gen "C_callback_wrapper((void *)" id #\, n #\))
	 (unless (eq? 'void rtype) (gen #\)))
	 (gen ";}") ) ) )
   stubs) )

(define (generate-foreign-callback-header cls stub)
  (let* ([name (foreign-callback-stub-name stub)]
	 [quals (foreign-callback-stub-qualifiers stub)]
	 [rtype (foreign-callback-stub-return-type stub)]
	 [argtypes (foreign-callback-stub-argument-types stub)]
	 [n (length argtypes)]
	 [vlist (make-argument-list n "t")] )
    (gen #t cls #\space (foreign-type-declaration rtype "") quals #\space name #\()
    (pair-for-each
     (lambda (vs ts)
       (gen (foreign-type-declaration (car ts) (car vs)))
       (when (pair? (cdr vs)) (gen #\,)) )
     vlist argtypes)
    (gen #\)) ) )

(define (foreign-type-declaration type target)
  (let ([err (lambda () (quit "illegal foreign type ~s" type))]
	[str (lambda (ts) (string-append ts " " target))] )
    (case type
      [(scheme-object) (str "C_word")]
      [(char) (str "char")]
      [(unsigned-char) (str "unsigned char")]
      [(unsigned-int) (str "unsigned int")]
      [(int integer bool) (str "int")]
      [(short) (str "short")]
      [(long) (str "long")]
      [(unsigned-short) (str "unsigned short")]
      [(unsigned-long) (str "unsigned long")]
      [(float) (str "float")]
      [(double) (str "double")]
      [(unsigned-integer) (str "unsigned int")]
      [(pointer c-pointer) (str "void *")]
      [(c-string) (str "char *")]
      [(void) (str "void")]
      [else
       (cond [(and (symbol? type) (##sys#hash-table-ref foreign-type-table type))
	      => (lambda (t)
		   (foreign-type-declaration (if (vector? t) (vector-ref t 0) t) target)) ]
	     [(string? type) (str type)]
	     [(pair? type)
	      (match type
		[`(pointer ,ptype) (string-append (foreign-type-declaration ptype (string-append "*" target)))]
		[`(struct ,sname) (string-append "struct " sname " " target)]
		[`(union ,uname) (string-append "union " sname " " target)]
		[`(function ,rtype ,argtypes . ,callconv)
		 (string-append
		  (foreign-type-declaration rtype "")
		  (or (and-let* ([(pair? callconv)]
				 [cc (car callconv)]
				 [(string? cc)] )
			cc)
		      "")
		  " (*" target ")("
		  (string-concatenate
		   (map (lambda (at)
			  (if (eq? '... at) 
			      "..."
			      (foreign-type-declaration at "") ) )
			argtypes) 
		   ",")
		  ")" ) ]
		[_ (err)] ) ]
	     [else (err)] ) ] ) ) )

(define (foreign-argument-conversion type)
  (let ([err (lambda () (quit "illegal foreign argument type ~s" type))])
    (case type
      ((scheme-object) "(")
      ((char unsigned-char) "C_character_code((C_word)")
      ((int unsigned-int) "C_unfix(")
      ((short) "C_unfix(")
      ((unsigned-short) "(unsigned short)C_unfix(")
      ((unsigned-long) "(unsigned long)C_num_to_long(")
      ((float) "C_c_double(")
      ((double) "C_c_double(")
      ((integer) "C_num_to_int(")
      ((long) "C_num_to_long(")
      ((unsigned-integer) "C_num_to_unsigned_int(")
      ((pointer) "C_data_pointer_or_null(")
      ((c-pointer) "C_c_pointer_or_null(")
      ((c-string) "C_string_or_null(")
      ((bool) "C_truep(")
      (else
       (cond [(and (symbol? type) (##sys#hash-table-ref foreign-type-table type))
	      => (lambda (t)
		   (foreign-argument-conversion (if (vector? t) (vector-ref t 0) t)) ) ]
	     [(pair? type)
	      (match type
		[`(pointer ,ptype) "C_c_pointer_or_null("]
		[`(function ,rtype ,@argtypes) "C_c_pointer_or_null("]
		[else (err)] ) ]
	     [else (err)] ) ) ) ) )
	    
(define (foreign-result-conversion type dest)
  (let ([err (lambda () (quit "illegal foreign return type ~s" type))])
    (case type
      ((char unsigned-char) "C_make_character((C_word)")
      ((int) "C_fix((C_word)")
      ((unsigned-int) "C_fix(0x3fffffff&(C_word)")
      ((short) "C_fix((short)")
      ((unsigned-short) "C_fix(0xffff&(C_word)")
      ((float double) (sprintf "C_flonum(&~a," dest))
      ((c-string c-pointer) (sprintf "C_mpointer(&~a,(void*)" dest))
      ((integer unsigned-integer) (sprintf "C_int_to_num(&~a,(C_word)" dest))
      ((long unsigned-long) (sprintf "C_long_to_num(&~a,(C_word)" dest))
      ((bool) "C_mk_bool(")
      ((void scheme-object) "((C_word)")
      (else
       (cond [(and (symbol? type) (##sys#hash-table-ref foreign-type-table type))
	      => (lambda (x)
		   (foreign-result-conversion (if (vector? x) (vector-ref x 0) x) dest)) ]
	     [(pair? type)
	      (match type
		[`(pointer ,ptype) (sprintf "C_mpointer(&~a,(void*)" dest)]
		[`(function ,rtype ,@argtypes) (sprintf "C_mpointer(&~a,(void*)" dest)]
		[else (err)] ) ]
	     [else (err)] ) ) ) ) )
