;;;; batch-driver.scm - Driver procedure for the compiler
;
; Copyright (c) 2000-2002, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany


(declare (unit driver))


#{compiler
  build-information compiler-arguments process-command-line
  default-analysis-database-size default-standard-bindings default-extended-bindings side-effecting-standard-bindings
  non-foldable-standard-bindings foldable-standard-bindings non-foldable-extended-bindings foldable-extended-bindings
  standard-bindings-that-never-return-false side-effect-free-standard-bindings-that-never-return-false
  installation-home compiler-cleanup-hook
  foreign-type-table-size file-io-only nonwinding-call/cc
  unit-name insert-timer-checks used-units inlining
  debug-info-vector-name
  foreign-declarations emit-trace-info block-compilation analysis-database-size line-number-database-size
  target-heap-size target-stack-size try-harder default-installation-home 
  default-default-target-heap-size default-default-target-stack-size verbose-mode original-program-size
  current-program-size line-number-database-2 foreign-lambda-stubs immutable-constants foreign-variables
  rest-parameters-promoted-to-vector inline-table inline-table-used constant-table constants-used mutable-constants
  dependency-list broken-constant-nodes inline-substitutions-enabled
  emit-profile profile-lambda-list profile-lambda-index profile-info-vector-name
  direct-call-ids foreign-type-table first-analysis debug-lambda-list debug-variable-list
  initialize-compiler canonicalize-expression expand-foreign-lambda update-line-number-database scan-toplevel-assignments
  perform-cps-conversion analyze-expression simplifications perform-high-level-optimizations perform-pre-optimization!
  reorganize-recursive-bindings substitution-table simplify-named-call find-inlining-candidates perform-inlining!
  perform-closure-conversion prepare-for-code-generation compiler-source-file create-foreign-stub expand-foreign-lambda*
  transform-direct-lambdas!
  debugging-chicken warnings-enabled bomb check-signature posq stringify symbolify flonum? build-lambda-list
  string->c-identifier c-ify-string words check-and-open-input-file close-checked-input-file fold-inner constant?
  collapsable-literal? immediate? canonicalize-begin-body extract-mutable-constants string->expr get get-all
  put! collect! count! get-line get-line-2 find-lambda-container display-analysis-database varnode qnode 
  build-node-graph build-expression-tree fold-boolean inline-lambda-bindings match-node expression-has-side-effects?
  simple-lambda-node? compute-database-statistics print-program-statistics output gen gen-list 
  pprint-expressions-to-file foreign-type-check estimate-foreign-result-size scan-used-variables scan-free-variables
  topological-sort print-version print-usage initialize-analysis-database write-dependency-list
  product copyright compiler-features default-declarations units-used-by-default words-per-flonum default-debugging-declarations
  default-profiling-declarations default-optimization-passes
  foreign-string-result-reserve parameter-limit default-output-filename eq-inline-operator optimizable-rest-argument-operators
  membership-test-operators membership-unfold-limit valid-compiler-options valid-compiler-options-with-argument
  chop-separator chop-extension current-source-file source-file-changed
  generate-code make-variable-list make-argument-list generate-foreign-stubs foreign-type-declaration
  foreign-argument-conversion foreign-result-conversion}


(include "tweaks")
(include "parameters")


;;; Compile a complete source file:

(define (compile-source-file filename . options)
  (initialize-compiler)
  (let ([initforms `((##core#declare
		      ,@(map (lambda (x) `(quote ,x))
			     (append 
			      default-declarations
			      (if (memq 'explicit-use options) 
				  '()
				  `((uses ,@units-used-by-default)) ) ) ) ) ) ]
        [verbose (memq 'verbose options)]
	[outfile (cond [(memq 'output-file options) 
			=> (lambda (node)
			     (let ([oname (cadr node)])
			       (if (symbol? oname)
				   (symbol->string oname)
				   oname) ) ) ]
		       [(memq 'to-stdout options) #f]
		       [else default-output-filename] ) ]
	[home (chop-separator (or (getenv "CHICKEN_HOME") default-installation-home ""))]
	[ipath (map chop-separator (string-split (or (getenv "CHICKEN_INCLUDE_PATH") "") ";"))]
	[ilimit (memq 'inline-limit options)]
	[ipasses (memq 'inline-passes options)]
	[strict (memq 'strict options)]
	[strict-srfi0 (memq 'strict-srfi-0 options)]
	[ipass 1]
	[opasses default-optimization-passes]
	[time0 #f]
	[time-breakdown #f]
	[forms '()]
	[cleanup-forms '(((##sys#implicit-exit-handler)))]
	[profile (memq 'profile options)]
	[dsize (memq 'database-size options)]
	[hsize (memq 'heap-size options)]
	[srfi7 (memq 'srfi-7 options)]
	[ssize (or (memq 'nursery options) (memq 'stack-size options))] )

    (define (print-header mode dbgmode)
      (when verbose (printf "pass: ~a~%~!" mode))
      (and (memq dbgmode debugging-chicken)
	   (begin
	     (printf "[~a]~%" mode)
	     #t) ) )

    (define (print-node mode dbgmode n)
      (when (print-header mode dbgmode)
	(pretty-print (build-expression-tree n)) ) )

    (define (print-db mode dbgmode db pass)
      (when (print-header mode dbgmode)
	(printf "(iteration ~s)~%" pass)
	(display-analysis-database db) ) )

    (define (print mode dbgmode xs)
      (when (print-header mode dbgmode)
	(for-each pretty-print xs) ) )

    (define (infohook class data val)
      (let ([data2 ((or ##sys#default-read-info-hook (lambda (a b c) b)) class data val)])
	(when (and (eq? 'list-info class) (symbol? (car data2)))
	  (##sys#hash-table-set!
	   ##sys#line-number-database
	   (car data2)
	   (alist-cons data2 val
		       (or (##sys#hash-table-ref ##sys#line-number-database (car data2))
			   '() ) ) ) )
	data2) )

    (define (arg-val str)
      (let* ((len (string-length str))
	     (len1 (- len 1)) )
	(or (if (< len 2)
		(string->number str)
		(case (string-ref str len1)
		  ((#\m #\M) (* (string->number (substring str 0 len1)) (* 1024 1024)))
		  ((#\k #\K) (* (string->number (substring str 0 len1)) 1024))
		  (else (string->number str)) ) )
	    (quit "invalid numeric argument ~S" str) ) ) )

    (define (collect-options opt)
      (let loop ([opts options])
	(cond [(memq opt opts) => (lambda (p) (cons (cadr p) (loop (cddr p))))]
	      [else '()] ) ) )

    (define (begin-time)
      (when time-breakdown (set! time0 (current-milliseconds))) )

    (define (end-time pass)
      (when time-breakdown
	(printf "milliseconds needed for ~a: ~s~%" pass (- (current-milliseconds) time0)) ) )

    (define (install-macros)
      (##syncase#install-macro-package (or strict strict-srfi0) strict-srfi0) )

    (define (output-dependency-list out)
      (fprintf out "~A: " outfile)
      (write-dependency-list dependency-list 70 out) )

    (set! debugging-chicken 
      (append-map
       (lambda (do)
	 (map (lambda (c) (string->symbol (string c)))
	      (string->list do) ) )
       (collect-options 'debug) ) )
    (set! try-harder (memq 'optimize options))
    (when dsize (set! analysis-database-size (arg-val (cadr dsize))))
    (when (memq 't debugging-chicken) (##sys#start-timer))
    (when (memq 'b debugging-chicken) (set! time-breakdown #t))
    (when (memq 'no-warnings options) 
      (when verbose (printf "Warnings are disabled~%~!"))
      (set! warnings-enabled #f)
      (set! ##sys#warnings-enabled #f) )
    (when (memq 'inline options) (set! inlining #t))
    (when (memq 'optimize-leaf-routines options) (set! optimize-leaf-routines #t))
    (when (memq 'unsafe options) 
      (set! unsafe #t)
      (##match#set-error-control #:unspecified) )
    (when (memq 'disable-interrupts options) (set! insert-timer-checks #f))
    (when (memq 'fixnum-arithmetic options) (set! number-type 'fixnum))
    (when (memq 'block options) (set! block-compilation #t))
    (when (memq 'no-fancy-ports options) (set! file-io-only #t))
    (when (memq 'no-winding-callcc options) (set! nonwinding-call/cc #t))
    (when (memq 'case-sensitive options) 
      (when verbose (printf "Identifiers and symbols are case sensitive~%~!"))
      (register-feature! 'case-sensitive)
      (##sys#case-sensitive #t) )
    (set! expand-only (memq 'expand-only options))
    (set! write-dependencies (memq 'write-dependencies options))
    (cond [(memq 'dependency-output options) => (lambda (o) (set! dependency-output (cadr o)))])
    (set! verbose-mode verbose)
    (set! ##sys#read-error-with-line-number #t)
    (set! ##sys#load-verbose #f)
    (set! ##sys#include-pathnames
      (append (map chop-separator (collect-options 'include-path))
	      ##sys#include-pathnames
	      ipath
	      (if home (list home) '()) ) )
    (when verbose
      (printf "Nursery size is ~A~%Heap size is ~A~%~!" ssize hsize) )

    ;; Handle feature options:
    (for-each register-feature! (collect-options 'feature))
    (for-each unregister-feature! (collect-options 'no-feature))

    ;; Load extensions:
    (let ([extends (collect-options 'extend)])
      (when verbose
	(printf "loading extensions...~%~!")
	(set! ##sys#load-verbose #t) )
      (for-each load extends) )

    (set! ##sys#features (cons '#:compiling ##sys#features))
    ;; Install hilevel macros:
    (cond [(memq 'hygienic options)
	   (when verbose 
	     (printf "Using hygienic macros~A~!"
		     (cond [strict " - strict mode"]
			   [strict-srfi0 " - strict mode with SRFI-0"]
			   [else ""] ) ) )
	   (install-macros) ]
	  [strict
	   (when verbose (display "Strict mode\n"))
	   (set! ##sys#strict-mode #t)
	   (unregister-feature! #:srfi-8 #:srfi-2 #:srfi-0 #:srfi-16)
	   (register-feature! #:strict)
	   (##sys#undefine-non-standard-macros '()) ]
	  [strict-srfi0
	   (when verbose (display "Strict mode with SRFI-0\n"))
	   (set! ##sys#strict-mode #t)
	   (unregister-feature! #:srfi-8 #:srfi-2 #:srfi-16)
	   (register-feature! #:strict)
	   (##sys#undefine-non-standard-macros '(cond-expand)) ] )

    (when ipasses 
      (set! inline-passes (arg-val (cadr ipasses))) )
    (when ilimit
      (set! inline-limit (arg-val (cadr ilimit))) )
    (set! target-heap-size
      (if hsize
	  (arg-val (cadr hsize))
	  (and-let* ([hsize default-default-target-heap-size]
		     [(not (zero? hsize))] )
	    hsize) ) )
    (set! target-stack-size
      (if ssize
	  (arg-val (cadr ssize))
	  (and-let* ([ssize default-default-target-stack-size]
		     [(not (zero? ssize))] )
	    ssize) ) )
    (set! emit-debug-info (memq 'emit-debug-info options))
    (when (memq 'debug-calls options)
      (set! debug-calls #t)
      (set! emit-debug-info #t) )
    (when (memq 'debug-loops options)
      (set! debug-calls #t)
      (set! debug-loops #t)
      (set! emit-debug-info #t) )
    (set! emit-trace-info (not (or emit-debug-info (memq 'no-trace options))))
    (set! ##sys#print-qualifiers #t)
    (when (memq 'm debugging-chicken) (set-gc-report! #t))
    (when (memq 'usual-integrations options)
      (set! standard-bindings default-standard-bindings)
      (set! extended-bindings default-extended-bindings) )
    (when emit-debug-info
      (set! initforms (append initforms default-debugging-declarations)) )
    (when verbose
      (printf "Debugging info: ~A~%~!"
	      (if emit-debug-info
		  (if debug-calls
		      (if debug-loops
			  "procedures, assignments, calls and loops"
			  "procedures, assignments and calls")
		      "procedures and assignments")
		  (if emit-trace-info
		      "stacktrace"
		      "none") ) ) )
    (when profile
      (when emit-debug-info
	(warning "debugging information is not available in profiled applications - ignored") 
	(set! emit-debug-info #f) )
      (if (pair? (cdr profile))
	  (set! emit-profile (cadr profile))
	  (quit "missing argument to `-profile' option") )
      (set! initforms (append initforms default-profiling-declarations))
      (when verbose
	(printf "Generating profile ~S~%~!" emit-profile) ) )

    (cond ((memq 'version options)
	   (print-version)
	   (newline) )
	  ((memq 'help options) (print-usage))
	  ((not filename)
	   (print-version)
	   (display "\n\nEnter \"chicken -help\" for information on how to use it.\n") )
	  (else

	   ;; Display header:
	   (unless (memq 'quiet options)
	     (printf "compiling file `~a' ...~%" filename) )
	   (debugging 'r "options" options)
	   (debugging 'r "debugging options" debugging-chicken)
	   (debugging 'r "home directory" home)
	   (debugging 'r "target heap size" target-heap-size)
	   (debugging 'r "target stack size" target-stack-size)
	   (debugging 'r "compiler features" compiler-features)

	   ;; Read toplevel expressions:
	   (set! ##sys#read-line-counter 1)
	   (set! ##sys#line-number-database (make-vector line-number-database-size '()))
	   (let ([prelude (collect-options 'prelude)]
		 [postlude (collect-options 'postlude)] 
		 [files (append 
			 (collect-options 'prologue)
			 (list filename)
			 (collect-options 'epilogue) ) ]  )

	     (when write-dependencies
	       (set! dependency-list (append-reverse files dependency-list)) )

	     (let ([proc (user-read-pass)])
	       (cond [proc
		      (when verbose (printf "User read pass...~%~!"))
		      (set! forms (proc prelude files postlude)) ]
		     [else
		      (do ([files files (cdr files)])
			  ((null? files)
			   (set! forms
			     (append (map string->expr prelude)
				     (reverse forms)
				     (map string->expr postlude) ) ) )
			(let* ([f (car files)]
			       [in (check-and-open-input-file f)]
			       [c0 (peek-char in)] )
			  (set! current-source-file f)
			  (set! source-file-changed #t)

			  ;; Check for script header:
			  (when (char=? #\# c0)
			    (read-char in)
			    (unless (char=? #\! (peek-char in))
			      (quit "sorry - file may not begin with `#' character if not a script") )
			    (read-char in)
			    (let ([header (open-output-string)]
				  [cleanup #t] )
			      (do ([c (read-char in) (read-char in)])
				  ((char=? c #\newline)
				   (set! ##sys#read-line-counter (add1 ##sys#read-line-counter)) )
				(if (eof-object? c)
				    (quit "unexpected end of file - incorrect header")
				    (write-char c header) ) )
			      (case (parse-script-header (get-output-string header))
				[(script) 
				 (set! forms (cons '(define command-line-arguments (make-parameter (cdr (argv)))) forms))
				 (set! cleanup #f) ]
				[(srfi7) 
				 (set! srfi7 #t)
				 (install-macros) ]
				[(r5rs srfi0) (install-macros)] 
				[(#f) (warning "unrecognized script header - might be corrupt")] )
			      (when cleanup (set! cleanup-forms (cons '(##sys#script-main '0 '1) cleanup-forms))) ) )

			  (do ([x (##sys#read in infohook) (##sys#read in infohook)])
			      ((eof-object? x) (close-checked-input-file in f))
			    (set! forms (cons x forms)) ) ) ) ] ) ) )

	   ;; Start compilation passes:
	   (let ([proc (user-preprocessor-pass)])
	     (when proc
	       (when verbose (printf "User preprocessing pass...~%~!"))
	       (set! forms (cons (first forms) (map proc (cdr forms)))) ) )

	   (when srfi7
	     (set! forms
	       (list
		(##sys#expand-srfi-7-program 
		 (chop-separator filename)
		 (car forms)
		 ##sys#syntax-error-hook) ) ) )

	   (print "source" '|1| forms)
	   (begin-time)
	   (let* ([exps0 (map canonicalize-expression (append initforms forms))]
		  [pvec (gensym)]
		  [plen (length profile-lambda-list)]
		  [exps (append
			 (map (lambda (ic) `(set! ,(cdr ic) ',(car ic))) immutable-constants)
			 (map (lambda (n) `(##core#callunit ,n)) used-units)
			 (if emit-debug-info
			     `((set! ,debug-info-vector-name 
				 (make-vector ',(+ (length debug-lambda-list) (length debug-variable-list))) ) )
			     '() )
			 (map (lambda (dl) `(##core#inline "C_i_setslot" ,debug-info-vector-name ',(car dl) ',(cdr dl)))
			      (append debug-lambda-list debug-variable-list) )
			 (if emit-debug-info
			     `((##sys#register-debug-info ,debug-info-vector-name ))
			     '() )
			 (if (and emit-debug-info (not unit-name))
			     '((if ##sys#break-on-startup (##sys#break) '#f))
			     '() )
			 (if emit-profile
			     `((set! ,profile-info-vector-name 
				 (##sys#register-profile-info
				  ',plen
				  ',(if unit-name #f emit-profile) ) ) )
			     '() )
			 (map (lambda (pl)
				`(##core#inline 
				  "C_i_setslot"
				  ,profile-info-vector-name
				  ',(* profile-info-entry-size (car pl)) 
				  ',(cdr pl) ) )
			      profile-lambda-list)
			 exps0
			 (if (not unit-name) 
			     cleanup-forms
			     '() ) ) ] )
	     (set! always-bound (cons debug-info-vector-name always-bound))

	     (when (and nonwinding-call/cc emit-debug-info)
	       (warning 
		"`non-winding-callc' option/declaration is not available in combination with debugging information - ignored")
	       (set! nonwinding-call/cc #f) )

	     (when (and nonwinding-call/cc emit-profile)
	       (warning 
		"`non-winding-callc' option/declaration is not available in combination with profiling information - ignored")
	       (set! nonwinding-call/cc #f) )

	     (let ([proc (user-pass)])
	       (when proc
		 (when verbose (printf "User pass...~%~!"))
		 (set! exps (map proc exps)) ) )

	     (set! ##sys#line-number-database line-number-database-2)
	     (set! line-number-database-2 #f)
	     (end-time "canonicalization")
	     (print "canonicalized" '|2| exps)

	     (when write-dependencies
	       (cond [dependency-output
		      (call-with-output-file dependency-output output-dependency-list) ]
		     [else
		      (output-dependency-list (current-output-port)) ] ) )
	     (cond [(memq 'check-syntax options) (exit)]
		   [expand-only
		    (pprint-expressions-to-file 
		     (if (not unit-name)
			 (butlast exps)
			 exps) outfile)
		    (exit) ] )

	     (let ([node0 (make-node
			   'lambda '(())
			   (list (build-node-graph
				  (canonicalize-begin-body exps) ) ) ) ] )
	       (set! ##sys#line-number-database #f)
	       (set! constant-table #f)
	       (set! inline-table #f)
	       (when (and try-harder (not unsafe))
		 (scan-toplevel-assignments (first (node-subexpressions node0))) )

	       (begin-time)
	       (let ([node1 (perform-cps-conversion node0)])
		 (end-time "cps conversion")
		 (print-node "cps" '|3| node1)

		 ;; Optimization loop:
		 (let loop ([i 1] [node2 node1] [progress #t])

		   (begin-time)
		   (let ([db (analyze-expression node2)])
		     (set! first-analysis #f)
		     (end-time "analysis")
		     (print-db "analysis" '|4| db i)

		     (when (pair? debugging-chicken) (print-program-statistics db))

		     (cond [(and progress (or try-harder (<= i default-optimization-passes)))
			    (debugging 'p "optimization pass" i)

			    (begin-time)
			    (receive (node2 progress-flag) (perform-high-level-optimizations node2 db)
			      (end-time "optimization")
			      (print-node "optimized-iteration" '|5| node2)

			      (when (and inlining (not progress-flag) (<= ipass inline-passes))
				(debugging 'p "inlining pass" ipass)
				(begin-time)
				(set! ipass (add1 ipass))
				(let ([ics (find-inlining-candidates node2 db)])
				  (when (pair? ics)
				    (let ([mvars (perform-inlining! ics node2 db)])
				      (set! always-bound (append (unzip1 mvars) always-bound))
				      ;; Mutate body of toplevel lambda node:
				      (set-car!
				       (node-subexpressions node2)
				       (fold
					(lambda (mv r)
					  (make-node 
					   'let
					   (list (car mv))
					   (list (qnode (cdr mv)) r) ) )
					(first (node-subexpressions node2))
					mvars) )
				      (set! progress-flag #t) 
				      (print-node "inlined" '|6| node2) ) ) )
				(end-time "inlining") )

			      (cond [progress-flag (loop (add1 i) node2 #t)]
				    [(not inline-substitutions-enabled)
				     (debugging 'p "rewritings enabled...")
				     (set! inline-substitutions-enabled #t)
				     (loop (add1 i) node2 #t) ]
				    [optimize-leaf-routines

				     (begin-time) ; one more analysis pass
				     (let ([db (analyze-expression node2)])
				       (end-time "analysis")

				       (begin-time)
				       (let ([progress (transform-direct-lambdas! node2 db)])
					 (end-time "leaf routine optimization")
					 (loop (add1 i) node2 progress) ) ) ]
				    [else (loop (add1 i) node2 #f)] ) ) ] 

			   [else
			    (print-node "optimized" '|7| node2)

			    (begin-time)
			    (let ([node3 (perform-closure-conversion node2 db)])
			      (end-time "closure conversion")
			      (print-db "final-analysis" '|8| db i)
			      (print-node "closure-converted" '|9| node3)

			      (begin-time)
			      (receive (node literals lambdas) (prepare-for-code-generation node3 db)
				(end-time "preparation")

				(let ([out (if outfile (open-output-file outfile) (current-output-port))])
				  (begin-time)
				  (generate-code literals lambdas out filename)
				  (end-time "code generation")

				  (when outfile (close-output-port out))
				  (when (memq 't debugging-chicken) (##sys#display-times (##sys#stop-timer)))
				  (compiler-cleanup-hook)
				  (when verbose 
				    (printf "compilation finished.~%~!") ) ) ) ) ] ) ) ) ) ) ) ) ) ) )


;;; Parse script header:

(define (parse-script-header line)
  (let ([len (string-length line)])
    (define (cmp-suffix str)
      (let ([index (- len (string-length str))])
	(and (> index 0)
	     (string=? str (substring line index len))
	     (memq (string-ref line (sub1 index)) '(#\space #\/)) ) ) )
    (cond [(cmp-suffix "-script") 'script]
	  [(cmp-suffix "scheme-r4rs") 'r4rs]
	  [(cmp-suffix "scheme-r5rs") 'r5rs]
	  [(cmp-suffix "scheme-srfi-7") 'srfi7]
	  [(cmp-suffix "scheme-srfi-0") 'srfi0]
	  [(cmp-suffix "scheme-ieee-1178-1990") 'ieee]
	  [else #f] ) ) )
