/*
 * Cheops Network User Interface
 *
 * Copyright (C) 1999, Adtran, Inc.
 * 
 * Distributed under the terms of the GNU GPL
 *
 */

#include <gtk/gtk.h>
#include <netinet/in.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include "cheops.h"

#define HTTP_PORT 80


/* Request to send */
static char *defrequest  = "GET / HTTP/1.0";

/* What to look for */
static char *defsearchstr = "Content-Type:";

static char *service = "www";

static char *label = "Web Server";

struct http_setup {
	/* The request to make */
	char request[80];
	/* What to look for */
	char search[80];
	/* Port to connect on */
	int port;

	GtkWidget *window;
	GtkWidget *requestw;
	GtkWidget *searchw;
	GtkWidget *portw;
};

struct http_monitor {
	int s;			/* Socket for communication */
	int level;		/* level of error we should report */
	struct net_page *np;	/* Page our object is on */
	struct net_object *no;	/* Pointer to the object we're monitoring */
	int io_id;		/* ID for I/O attempts */
	struct http_setup *setup;		/* Setup data */
};

char *plugin_name() {
	return service;
}

char *plugin_label() {
	return label;
}

int plugin_type() {
	return TYPE_MONITOR;
}

int default_level() {
	return MONITOR_CRITICAL;
}

void reset(struct http_monitor *hm)
{
	if (hm->io_id > -1) {
		cancel_data(hm->io_id);
		hm->io_id = -1;
	}
	if (hm->s > -1) {
		close(hm->s);
		hm->s = -1;
	}
}

void stage3(void *data, int s, GdkInputCondition c)
{
	/* Called when data is ready for reading */
	int res=0;
	struct http_monitor *hm = (struct http_monitor *)data;
	char buf[1024];
	char *searchstr;
	hm->io_id = -1;
#if 0
	printf("Stage 3 and condition is %d!\n",no, c);
#endif
	if (hm->setup)
		searchstr = hm->setup->search;
	else
		searchstr = defsearchstr;
	
	read(hm->s, &buf, sizeof(buf));
	
	if (strstr(buf, searchstr)) res++;
	
	if (res) {
		monitor_report(hm->no, hm->np, MONITOR_NOMINAL, service, "Nominal condition");
	} else {
		monitor_report(hm->no, hm->np, hm->level, service, "Did not find expected content");
	}
	reset(hm);
}

void stage2(void *data, int s, GdkInputCondition c)
{
	/* Called when connect() has completed or failed, one or the other */
	struct http_monitor *hm = (struct http_monitor *)data;
	char writestr[256];
	
	int res;
	
	res = get_socket_error(s);
	
#if 0
	printf("Error is %s (%d)\n", strerror(res), res);
#endif
	
	if (hm->setup)
		snprintf(writestr,sizeof(writestr), "%s\r\n\r\n", hm->setup->request);
	else
	snprintf(writestr,sizeof(writestr), "%s\r\n\r\n", defrequest);
#if 0
	printf("Stage 2 and condition is %d!\n",c);
#endif
	
	/* If the socket is connected (i.e. available for write), then
	   we write our request to it.  Assuming both go through, we 
	   then setup a callback to handle the reply, if one is available.
	   If not, we immediately report an error */
	
	hm->io_id = -1;
	if (!res &&
	    (write(s, writestr, strlen(writestr)) == strlen(writestr))) {
		hm->io_id = wait_for_data(s, stage3, data);
	} else {
		snprintf(writestr, sizeof(writestr), "Connection failed: %s", strerror(res));
		monitor_report(hm->no, hm->np, hm->level, service, writestr);
		reset(hm);
	}
}

void monitor(struct net_object *no, int level, void **data, void *setupdata)
{
	struct http_monitor *hm = (struct http_monitor *)(*data);
	struct http_setup *hs = (struct http_setup *)setupdata;
	unsigned short port=HTTP_PORT;
	
	/* While the arguments to our monitoring function are straightforward
	   and the general idea (connect, write a request, read back and look
	   for our magic string) isn't that difficult, the function is somewhat
	   complicated by the requirement that we never make a blocking
	   I/O call.
	
	   This is the first step of the monitoring function.  First, we
	   allocate a monitor stucture to hold some important info like
	   the callback id's, our sockets, and the object and page. */
	
#if 0
	fprintf(stdout, "Monitoring HTTP!\n");
#endif
	if (!*data) {
#if 0
		fprintf(stdout, "Making new monitor structure for %s\n", no->hostname);
#endif
		hm = g_new(struct http_monitor, 1);
		hm->s = -1;
		hm->io_id = -1;
		hm->no = no;
		hm->np = no->np;
		hm->setup = setupdata;
		*data = hm;
	}
	hm->level = level;
	if (hs)
		port = hs->port;

	/* Remember, we're not guaranteed that we have delivered
	   a reply by this time, so we need to close the socket
	   if it isn't already closed after first removing any
	   pending io callbacks */

	reset(hm);
		   
	hm->s = socket(AF_INET, SOCK_STREAM, IPPROTO_IP);
	
	/* Immediately report an error if socket creation failed */
	if (hm->s < 0) {
		monitor_report(no, no->np, hm->level, service, "Unable to create socket (local)");
		return;
	}
	
	/* monitor.c provides this as a convenience function.  We try to asynchronously
	   connect the socket to the given address.  When it is available for writing
	   or an  error has occured, stage2 will be called */
	
	hm->io_id = try_to_connect(hm->s, no->ip_addr, htons(port), hm, stage2);
}

void cleanup(void **data)
{
	/* Here, we clean up our http_monitor structure and 
	   remove any pending inputs we had, since we are no longer
	   being asked to work */
	   
	struct http_monitor *hm = (struct http_monitor *)(*data);
	if (hm->io_id > -1) {
		cancel_data(hm->io_id);
		hm->io_id = -1;
	}
	if (hm->s > -1) {
		close(hm->s);
	}

	g_free(hm);
	*data = NULL;
#if 0
	fprintf(stdout, "Cleaned up!\n");
#endif
}

static void close_setup_window(GtkWidget *w)
{
	struct http_setup *s = (struct http_setup *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	/* When they close the window, grab their info */
	strncpy(s->search, gtk_entry_get_text(GTK_ENTRY(s->searchw)), sizeof(s->search));
	strncpy(s->request, gtk_entry_get_text(GTK_ENTRY(s->requestw)), sizeof(s->request));
	s->port = atoi(gtk_entry_get_text(GTK_ENTRY(s->portw)));
	
	/* Destroy the window itself */
	gtk_widget_hide(s->window);
	gtk_widget_destroy(s->window);
	s->window = NULL;
}

void *setup(void *oldsetup, struct net_object *no)
{
	/* Allocate a http_setup structure, initialize some reasonable
	   defaults, display a window, and then return our structure.
	   
	   It's not important that the structure have final values in 
	   it when we return, since our callbacks will fill them in when
	   the user closes our setup window */
	   
	struct http_setup *s;
	
	GtkWidget *label;
	GtkWidget *bbox;
	GtkWidget *close;
	GtkWidget *vbox;
	
	char buf[256];
	
	if (oldsetup)
		s = oldsetup;
	else {
		s = g_new(struct http_setup, 1);
		memset(s, 0, sizeof(struct http_setup));
		s->port = HTTP_PORT;
		snprintf(s->request, sizeof(s->request), "GET http://%s/ HTTP/1.0", no->hostname);
		strncpy(s->search, defsearchstr, sizeof(s->search));
	}
	
	if (s->window) {
		gtk_widget_show(s->window);
		return s;
	}
	
	/* Make our setup box */
	snprintf(buf, sizeof(buf), "HTTP Monitor: %s", no->hostname);
	s->window = gtk_window_new(GTK_TOPLEVEL);
	gtk_widget_realize(s->window);
	gtk_window_set_title(GTK_WINDOW(s->window), buf);
	fix_icon(s->window->window);

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_widget_show(vbox);

	label = gtk_label_new(buf);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);
	
	label = gtk_label_new("Port Number");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	s->portw = gtk_entry_new();
	snprintf(buf, sizeof(buf), "%d", s->port);
	gtk_entry_set_text(GTK_ENTRY(s->portw), buf);
	gtk_widget_set_usize(s->portw, 50, 0);
	gtk_box_pack_start(GTK_BOX(vbox), s->portw, FALSE, FALSE, 5);
	gtk_widget_show(s->portw);

	label = gtk_label_new("Request to make:");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);
		
	s->requestw = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(s->requestw), s->request);
	gtk_box_pack_start(GTK_BOX(vbox), s->requestw, FALSE, FALSE, 5);
	gtk_widget_show(s->requestw);

	label = gtk_label_new("Nominal response to expect:");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	s->searchw = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(s->searchw), s->search);
	gtk_box_pack_start(GTK_BOX(vbox), s->searchw, FALSE, FALSE, 5);
	gtk_widget_show(s->searchw);

	bbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(bbox);
	
	close = gtk_button_new_with_label("  Close  ");
	gtk_object_set_user_data(GTK_OBJECT(close), s);
	gtk_signal_connect(GTK_OBJECT(close), "clicked", GTK_SIGNAL_FUNC(close_setup_window), NULL);
	gtk_object_set_user_data(GTK_OBJECT(s->window), s);
	gtk_signal_connect(GTK_OBJECT(s->window), "delete_event", GTK_SIGNAL_FUNC(close_setup_window), NULL);
	gtk_widget_show(close);
	
	gtk_box_pack_end(GTK_BOX(bbox), close, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(vbox), bbox, FALSE, FALSE, 5);	

	gtk_container_border_width(GTK_CONTAINER(s->window), 10);
	gtk_container_add(GTK_CONTAINER(s->window), vbox);
	gtk_widget_show(s->window);
	return s;
}

void setup_cleanup(void *data)
{
	struct http_setup *setup = (struct http_setup *)data;
	/* If we have a setup window open, destroy it, then free
	   our structure */
	if (setup->window)
		gtk_widget_destroy(setup->window);
#if 0
	printf("Cleaning up setup structure (%p)!\n", setup);
#endif
	g_free(setup);
}

char *setup2str(void *data)
{
	static char buf[256];
	struct http_setup *s = (struct http_setup *)data;
	
	/* When monitor stuff is saved, the saving code calls this
	   routing to convert the setup structure into a string that it
	   can include in the configuration file. */
	
	snprintf(buf, sizeof(buf), "%s!%s!%d", s->request, s->search, s->port);
	return buf;
}

void *str2setup(char *buf, struct net_object *no)
{

	/* Hre, we read a setup string and return a setup pointer.  We use
	   "!" as delimiters because ^ is already used and ! are unlikely
	   to be found in the request */
	char *c;
	struct http_setup *s;

	s = g_new(struct http_setup, 1);
	s->port = HTTP_PORT;
	snprintf(s->request, sizeof(s->request), "GET http://%s/ HTTP/1.0", no->hostname);
	strncpy(s->search, defsearchstr, sizeof(s->search));
	s->window = NULL;

	c = strtok(buf, "!");
	if (c)
		strncpy(s->request, c, sizeof(s->request));
	c = strtok(NULL, "!");
	if (c)
		strncpy(s->search, c, sizeof(s->search));
	c = strtok(NULL, "!");
	if (c)
		s->port = atoi(c);
	return s;
}

