/*
 * Cheops Network User Interface
 *
 * Copyright (C) 1999, Adtran, Inc.
 * 
 * Distributed under the terms of the GNU GPL
 *
 */

#include <gtk/gtk.h>
#include <netinet/in.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include "cheops.h"

#define FTP_PORT 21


/* Request to send */
static char *defusername  = "ftp";

/* What to look for */
static char *defpasswd = "cheops@";

static char *service = "ftp";

static char *label = "FTP Server";

struct ftp_setup {
	/* The request to make */
	char username[80];
	/* What to look for */
	char passwd[80];
	/* Port to connect on */
	int port;
	int login;

	GtkWidget *window;
	GtkWidget *usernamew;
	GtkWidget *passwdw;
	GtkWidget *portw;
	GtkWidget *loginw;
};

struct ftp_monitor {
	int s;			/* Socket for communication */
	int level;		/* level of error we should report */
	struct net_page *np;	/* Page our object is on */
	struct net_object *no;	/* Pointer to the object we're monitoring */
	int io_id;		/* ID for I/O attempts */
	struct ftp_setup *setup;		/* Setup data */
};

char *plugin_name() {
	return service;
}

char *plugin_label() {
	return label;
}

int plugin_type() {
	return TYPE_MONITOR;
}

int default_level() {
	return MONITOR_CRITICAL;
}

void reset(struct ftp_monitor *hm)
{
	if (hm->io_id > -1) {
		cancel_data(hm->io_id);
		hm->io_id = -1;
	}
	if (hm->s > -1) {
		close(hm->s);
		hm->s = -1;
	}
}

void stage5(void *data, int s, GdkInputCondition c)
{
	/* Called when we should be logged in */
	int res=0;
	struct ftp_monitor *hm = (struct ftp_monitor *)data;
	char buf[1024];
	hm->io_id = -1;
#if 0
	printf("Stage 5 and condition is %d!\n", c);
#endif
	read(hm->s, &buf, sizeof(buf));
	
	if (!strncmp(buf, "230", 3)) res++;
	
	if (res) {
		monitor_report(hm->no, hm->np, MONITOR_NOMINAL, service, "Nominal condition");
	} else {
		monitor_report(hm->no, hm->np, hm->level, service, "FTP server did not accept login");
	}
	reset(hm);
}
void stage4(void *data, int s, GdkInputCondition c)
{
	/* Called when password prompt should appear.  Make sure we have
	   a 331 response, then send password, and wait for the final OK */
	int res=0;
	struct ftp_monitor *hm = (struct ftp_monitor *)data;
	char buf[1024];
	char *passwd;
	hm->io_id = -1;
#if 0
	printf("Stage 4 and condition is %d!\n", c);
#endif
	if (hm->setup)
		passwd = hm->setup->passwd;
	else
		passwd = defpasswd;
	
	read(hm->s, &buf, sizeof(buf));
	
	if (!strncmp(buf, "331",3)) res++;
	
	snprintf(buf, sizeof(buf), "PASS %s\n", passwd);
	if (res && (write(s, buf, strlen(buf)) == strlen(buf))) {
		hm->io_id = wait_for_data(s, stage5, data);
		return;
	} else {
		monitor_report(hm->no, hm->np, hm->level, service, "FTP server did not prompt for password");
	}
	reset(hm);
}

void stage3(void *data, int s, GdkInputCondition c)
{
	/* Called when data is ready for reading.  Make sure we have
	   a 220 response, then send username, and wait for the password prompt */
	int res=0;
	struct ftp_monitor *hm = (struct ftp_monitor *)data;
	char buf[1024];
	char *username;
	hm->io_id = -1;
#if 0
	printf("Stage 3 and condition is %d!\n", c);
#endif
	if (hm->setup)
		username = hm->setup->username;
	else
		username = defusername;
	
	read(hm->s, &buf, sizeof(buf));
	
	if (!strncmp(buf, "220",3)) res++;
	
	snprintf(buf, sizeof(buf), "USER %s\n", username);
	if (res && (write(s, buf, strlen(buf)) == strlen(buf))) {
		if (hm->setup && hm->setup->login) {
			hm->io_id = wait_for_data(s, stage4, data);
			return;
		} else
			monitor_report(hm->no, hm->np, MONITOR_NOMINAL, service, "Nominal condition");
	} else {
		monitor_report(hm->no, hm->np, hm->level, service, "FTP server did prompt for username");
	}
	reset(hm);
}

void stage2(void *data, int s, GdkInputCondition c)
{
	/* Called when connect() has completed or failed, one or the other */
	struct ftp_monitor *hm = (struct ftp_monitor *)data;
	char writestr[256];
	
	int res;
	
	res = get_socket_error(s);
	
#if 0
	printf("Error is %s (%d)\n", strerror(res), res);
#endif
	
#if 0
	printf("Stage 2 and condition is %d!\n",c);
#endif
	
	/* If the socket is connected (i.e. available for write), then
	   we wait for data to read on it */
	
	hm->io_id = -1;
	if (!res) {
		hm->io_id = wait_for_data(s, stage3, data);
	} else {
		snprintf(writestr, sizeof(writestr), "Connection failed: %s", strerror(res));
		monitor_report(hm->no, hm->np, hm->level, service, writestr);
		reset(hm);
	}
}

void monitor(struct net_object *no, int level, void **data, void *setupdata)
{
	struct ftp_monitor *hm = (struct ftp_monitor *)(*data);
	struct ftp_setup *hs = (struct ftp_setup *)setupdata;
	unsigned short port=FTP_PORT;
	
	/* While the arguments to our monitoring function are straightforward
	   and the general idea (connect, write a request, read back and look
	   for our magic string) isn't that difficult, the function is somewhat
	   complicated by the requirement that we never make a blocking
	   I/O call.
	
	   This is the first step of the monitoring function.  First, we
	   allocate a monitor stucture to hold some important info like
	   the callback id's, our sockets, and the object and page. */
	
#if 0
	fprintf(stdout, "Monitoring FTP!\n");
#endif
	if (!*data) {
#if 0
		fprintf(stdout, "Making new monitor structure for %s\n", no->hostname);
#endif
		hm = g_new(struct ftp_monitor, 1);
		hm->s = -1;
		hm->io_id = -1;
		hm->no = no;
		hm->np = no->np;
		hm->setup = setupdata;
		*data = hm;
	}
	hm->level = level;
	if (hs)
		port = hs->port;

	/* Remember, we're not guaranteed that we have delivered
	   a reply by this time, so we need to close the socket
	   if it isn't already closed after first removing any
	   pending io callbacks */

	reset(hm);
		   
	hm->s = socket(AF_INET, SOCK_STREAM, IPPROTO_IP);
	
	/* Immediately report an error if socket creation failed */
	if (hm->s < 0) {
		monitor_report(no, no->np, hm->level, service, "Unable to create socket (local)");
		return;
	}
	
	/* monitor.c provides this as a convenience function.  We try to asynchronously
	   connect the socket to the given address.  When it is available for writing
	   or an  error has occured, stage2 will be called */
	
	hm->io_id = try_to_connect(hm->s, no->ip_addr, htons(port), hm, stage2);
}

void cleanup(void **data)
{
	/* Here, we clean up our ftp_monitor structure and 
	   remove any pending inputs we had, since we are no longer
	   being asked to work */
	   
	struct ftp_monitor *hm = (struct ftp_monitor *)(*data);
	if (hm->io_id > -1) {
		cancel_data(hm->io_id);
		hm->io_id = -1;
	}
	if (hm->s > -1) {
		close(hm->s);
	}

	g_free(hm);
	*data = NULL;
#if 0
	fprintf(stdout, "Cleaned up!\n");
#endif
}

static void close_setup_window(GtkWidget *w)
{
	struct ftp_setup *s = (struct ftp_setup *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	/* When they close the window, grab their info */
	strncpy(s->username, gtk_entry_get_text(GTK_ENTRY(s->usernamew)), sizeof(s->username));
	strncpy(s->passwd, gtk_entry_get_text(GTK_ENTRY(s->passwdw)), sizeof(s->passwd));
	s->port = atoi(gtk_entry_get_text(GTK_ENTRY(s->portw)));
	s->login = GTK_TOGGLE_BUTTON(s->loginw)->active;
	
	/* Destroy the window itself */
	gtk_widget_hide(s->window);
	gtk_widget_destroy(s->window);
	s->window = NULL;
}

void handle_click(GtkWidget *w)
{
	struct ftp_setup *s = (struct ftp_setup *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	int tmp = GTK_TOGGLE_BUTTON(w)->active;
	gtk_widget_set_sensitive(s->passwdw, tmp);
	gtk_widget_set_sensitive(s->usernamew, tmp);
}

void *setup(void *oldsetup, struct net_object *no)
{
	/* Allocate a ftp_setup structure, initialize some reasonable
	   defaults, display a window, and then return our structure.
	   
	   It's not important that the structure have final values in 
	   it when we return, since our callbacks will fill them in when
	   the user closes our setup window */
	   
	struct ftp_setup *s;
	
	GtkWidget *label;
	GtkWidget *bbox;
	GtkWidget *close;
	GtkWidget *vbox;
	
	char buf[256];
	
	if (oldsetup)
		s = oldsetup;
	else {
		s = g_new(struct ftp_setup, 1);
		memset(s, 0, sizeof(struct ftp_setup));
		s->port = FTP_PORT;
		s->login = 0;
		strncpy(s->username, defusername, sizeof(s->username));
		strncpy(s->passwd, defpasswd, sizeof(s->passwd));
	}
	
	if (s->window) {
		gtk_widget_show(s->window);
		return s;
	}
	
	/* Make our setup box */
	snprintf(buf, sizeof(buf), "FTP Monitor: %s", no->hostname);
	s->window = gtk_window_new(GTK_TOPLEVEL);
	gtk_widget_realize(s->window);
	gtk_window_set_title(GTK_WINDOW(s->window), buf);
	fix_icon(s->window->window);

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_widget_show(vbox);

	label = gtk_label_new(buf);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);
	
	label = gtk_label_new("Port Number");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	s->portw = gtk_entry_new();
	snprintf(buf, sizeof(buf), "%d", s->port);
	gtk_entry_set_text(GTK_ENTRY(s->portw), buf);
	gtk_widget_set_usize(s->portw, 50, 0);
	gtk_box_pack_start(GTK_BOX(vbox), s->portw, FALSE, FALSE, 5);
	gtk_widget_show(s->portw);

	s->loginw = gtk_check_button_new_with_label("Attempt Login");
	gtk_widget_show(s->loginw);
	gtk_box_pack_start(GTK_BOX(vbox), s->loginw, FALSE, FALSE, 5);

	label = gtk_label_new("Username:");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);
		
	s->usernamew = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(s->usernamew), s->username);
	gtk_box_pack_start(GTK_BOX(vbox), s->usernamew, FALSE, FALSE, 5);
	gtk_widget_show(s->usernamew);

	label = gtk_label_new("Password:");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	s->passwdw = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(s->passwdw), s->passwd);
	gtk_box_pack_start(GTK_BOX(vbox), s->passwdw, FALSE, FALSE, 5);
	gtk_widget_show(s->passwdw);

	gtk_object_set_user_data(GTK_OBJECT(s->loginw), s);
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(s->loginw), s->login);
	gtk_signal_connect(GTK_OBJECT(s->loginw), "clicked", GTK_SIGNAL_FUNC(handle_click), NULL);
	handle_click(s->loginw);

	bbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(bbox);
	
	close = gtk_button_new_with_label("  Close  ");
	gtk_object_set_user_data(GTK_OBJECT(close), s);
	gtk_signal_connect(GTK_OBJECT(close), "clicked", GTK_SIGNAL_FUNC(close_setup_window), NULL);
	gtk_object_set_user_data(GTK_OBJECT(s->window), s);
	gtk_signal_connect(GTK_OBJECT(s->window), "delete_event", GTK_SIGNAL_FUNC(close_setup_window), NULL);
	gtk_widget_show(close);
	
	gtk_box_pack_end(GTK_BOX(bbox), close, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(vbox), bbox, FALSE, FALSE, 5);	

	gtk_container_border_width(GTK_CONTAINER(s->window), 10);
	gtk_container_add(GTK_CONTAINER(s->window), vbox);
	gtk_widget_show(s->window);
	return s;
}

void setup_cleanup(void *data)
{
	struct ftp_setup *setup = (struct ftp_setup *)data;
	/* If we have a setup window open, destroy it, then free
	   our structure */
	if (setup->window)
		gtk_widget_destroy(setup->window);
#if 0
	printf("Cleaning up setup structure (%p)!\n", setup);
#endif
	g_free(setup);
}

char *setup2str(void *data)
{
	static char buf[256];
	struct ftp_setup *s = (struct ftp_setup *)data;
	
	/* When monitor stuff is saved, the saving code calls this
	   routing to convert the setup structure into a string that it
	   can include in the configuration file. */
	
	snprintf(buf, sizeof(buf), "%s!%s!%d!%d", s->username, s->passwd, s->port, s->login);
	return buf;
}

void *str2setup(char *buf, struct net_object *no)
{

	/* Hre, we read a setup string and return a setup pointer.  We use
	   "!" as delimiters because ^ is already used and ! are unlikely
	   to be found in the request */
	char *c;
	struct ftp_setup *s;

	s = g_new(struct ftp_setup, 1);
	s->port = FTP_PORT;
	strncpy(s->username, defusername, sizeof(s->username));
	strncpy(s->passwd, defpasswd, sizeof(s->passwd));
	s->window = NULL;

	c = strtok(buf, "!");
	if (c)
		strncpy(s->username, c, sizeof(s->username));
	c = strtok(NULL, "!");
	if (c)
		strncpy(s->passwd, c, sizeof(s->passwd));
	c = strtok(NULL, "!");
	if (c)
		s->port = atoi(c);
	c = strtok(NULL, "!");
	if (c)
		s->login = atoi(c);
	return s;
}

