/* keyb.c
 * 
 * This program is distributed under the GNU General Public License.
 * Copyright (C) 2000 Eugene Osintsev <gene@linuxave.net>
 */

#include <config.h>
#include <signal.h>
#include <curses.h>
#include <stdlib.h>
#include <string.h>
#include "font.h"
#include "scr.h"
#include "undo.h"
#include "help.h"
#include "defines.h"

extern struct CFE_FONT  font[2];
extern struct CFE_GLYPH glyph[2];

/* Signal handler */
static RETSIGTYPE sighandler(int sig);

/* Preparing for abnormal exit */
static void prepare_exit();

/* 0 -- left panel, 1 -- right one */
static int panel = 0;

/*------------------------------------------------------------------*/

void keyboard_loop()
{
    char goto_num[4] = "", info_str[80];
    int i, j, ch, len, num, addr, sig, height = font[0].height;
    int is_buffer = 0, is_diff, is_mod;
    unsigned char byte, buffer[MAX_FONT_HEIGHT];
#ifdef NCURSES_MOUSE_VERSION            
    MEVENT mouse_event;
#endif  /* NCURSES_MOUSE_VERSION */
    
    signal(SIGINT,   sighandler);
    signal(SIGHUP,   sighandler);
    signal(SIGTERM,  sighandler);
    signal(SIGWINCH, sighandler);

    while ((ch = getch()) != 'q') {
        if (!chk_win_size(0))
            continue;

        message("");
        switch (ch) {
        case C('p'):           /* Ctrl-P -- previous glyph on both panels */
            if (glyph[0].num != 0)
                glyph[0].num--;
            if (font[1].name && glyph[1].num != 0)
                glyph[1].num--;
            break;
        case C('n'):           /* Ctrl-N -- next glyph on both panels */
            if (glyph[0].num != (font[0].num_of_chars - 1))
                glyph[0].num++;
            if (font[1].name && glyph[1].num != (font[1].num_of_chars - 1))
                glyph[1].num++;
            break;
        case KEY_PPAGE:        /* PageUp -- previous glyph */
        case C('b'):
            if (glyph[panel].num != 0)
                glyph[panel].num--;
            break;
        case KEY_NPAGE:        /* PageDown -- next glyph */
        case C('f'):
            if (glyph[panel].num != (font[panel].num_of_chars - 1))
                glyph[panel].num++;
            break;
        case KEY_HOME:         /* Home -- first glyph */
        case C('a'):
            glyph[panel].num = 0;
            break;
        case KEY_END:          /* End -- last glyph */
        case C('e'):
            glyph[panel].num = font[panel].num_of_chars - 1;
            break;
        case KEY_UP:           /* UpArrow -- previous byte */
            if (glyph[panel].y != 0)
                glyph[panel].y--;
            break;
        case KEY_DOWN:         /* DownArrow -- next byte */
            if (glyph[panel].y != (height - 1))
                glyph[panel].y++;
            break;
        case KEY_LEFT:         /* LeftArrow -- left bit */
            if (glyph[panel].x != 0)
                glyph[panel].x--;
            break;
        case KEY_RIGHT:        /* RightArrow -- right bit */
            if (glyph[panel].x != 7)
                glyph[panel].x++;
            break;
            /* --- The end of navigation --- */

        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
            len = strlen(goto_num);
            if (len != 3) {
                goto_num[len] = ch;
                goto_num[len + 1] = '\0';
            }
            break;
        case 'g':              /* Go to glyph */
            if (strlen(goto_num) != 0) {
                num = atoi(goto_num);
                if (num > (font[panel].num_of_chars - 1))
                    alert("Glyph number is out of range");
                else
                    glyph[panel].num = num;
                goto_num[0] = '\0';
            }
            break;
        case 'm':              /* Go to next modified glyph */
            if (panel) {
                alert("Read-only panel");
                break;
            }
            is_mod = 0;
            for (num = glyph[0].num + 1; num < font[0].num_of_chars; num++) {
                if (glyph_modified(num)) {
                    glyph[0].num = num;
                    is_mod = 1;
                    break;
                }
            }
            if (!is_mod)
                message("No modified glyphs");
            break;
        case 'M':              /* Go to next unmodified glyph */
            if (panel) {
                alert("Read-only panel");
                break;
            }
            is_mod = 1;
            for (num = glyph[0].num + 1; num < font[0].num_of_chars; num++) {
                if (!glyph_modified(num)) {
                    glyph[0].num = num;
                    is_mod = 0;
                    break;
                }
            }
            if (is_mod)
                message("No unmodified glyphs");
            break;
        case 'i':       /* Show miscellaneous info */
            num = glyph[panel].num;
            addr = num * height + glyph[panel].y + font[panel].offset;
#ifdef HAVE_SNPRINTF
            snprintf(info_str, 80, "Address: 0x%04X (%d)    Char: 0x%X (%d)",
                     addr, addr, num, num);
#else
            sprintf(info_str, "Address: 0x%04X (%d)    Char: 0x%X (%d)",
                    addr, addr, num, num);
#endif  /* HAVE_SNPRINTF */
            message(info_str);
            break;
        case 'x':       /* Show Unicode numbers matched by current glyph */
            if (get_unicode_data(&font[panel], glyph[panel].num, info_str)) {
                message(info_str);
            } else {
                alert("No Unicode table");
            }
            break;
        case '\t':      /* Tab -- Change panel */
            if (panel)
                panel = 0;
            else if (font[1].name)
                panel = 1;
            break;
#ifdef NCURSES_MOUSE_VERSION            
        case KEY_MOUSE:
            if (getmouse(&mouse_event) == OK &&
                mouse_event.x >= 2 && mouse_event.x <= 16 &&
                mouse_event.x % 2 == 0 &&
                mouse_event.y >= 5 && mouse_event.y < (5 + height)) {

                panel = 0;
                glyph[0].x = mouse_event.x / 2 - 1;
                glyph[0].y = mouse_event.y - 5;
            } else {
                break;
            }
#endif  /* NCURSES_MOUSE_VERSION */
        case ' ':              /* Space -- Bit on/off */
            if (panel) {
                alert("Read-only panel");
                break;
            }
            put_undo(&glyph[0]);
            set_glyph_cur_row(get_glyph_cur_row() ^ (0200 >> glyph[0].x));
            break;
        case 'h':              /* Flip horizontally */
            if (panel) {
                alert("Read-only panel");
                break;
            }
            put_undo(&glyph[0]);
            for (i = 0; i < height / 2; i++) {
                byte = get_glyph_row(i);
                set_glyph_row(i, get_glyph_row(height - i - 1));
                set_glyph_row(height - i - 1, byte);
            }
            break;
        case 'v':              /* Flip vertically */
            if (panel) {
                alert("Read-only panel");
                break;
            }
            put_undo(&glyph[0]);
            for (i = 0; i < height; i++) {
                byte = 0;
                for (j = 0; j < 4; j++) {
                    byte |= (get_glyph_row(i) & (1 << j)) << (7 - 2 * j);
                    byte |= (get_glyph_row(i) & (0200 >> j)) >> (7 - 2 * j);
                }
                set_glyph_row(i, byte);
            }
            break;
        case 'c':              /* Copy glyph to buffer */
            is_buffer = 1;
            if (panel)
                memcpy(buffer, glyph[1].data, height);
            else
                memcpy(buffer, glyph[0].data, height);
            message("Glyph copied to buffer");
            break;
        case 'p':              /* Paste glyph from buffer */
            if (panel) {
                alert("Read-only panel");
                break;
            }
            if (is_buffer) {
                put_undo(&glyph[0]);
                memcpy(glyph[0].data, buffer, height);
            }
            break;
        case 's':              /* Save file from left panel */
            if (panel) {
                alert("Read-only panel");
                break;
            }
            if (file_modified()) {
                if (!save_font_file(&font[0])) {
                    prepare_exit();
                    perror(font[0].name);
                    exit(1);
                }
                clear_file_modified();
                message("File saved successfully");
            } else {
                alert("No changes to save");
            }
            break;
        case 'l':              /* Shift left */
            if (panel) {
                alert("Read-only panel");
                break;
            }
            put_undo(&glyph[0]);
            for (i = 0; i < height; i++)
                set_glyph_row(i, get_glyph_row(i) << 1);
            break;
        case 'r':              /* Shift right */
            if (panel) {
                alert("Read-only panel");
                break;
            }
            put_undo(&glyph[0]);
            for (i = 0; i < height; i++)
                set_glyph_row(i, get_glyph_row(i) >> 1);
            break;
        case 't':              /* Shift top */
            if (panel) {
                alert("Read-only panel");
                break;
            }
            put_undo(&glyph[0]);
            memmove(&glyph[0].data[0], &glyph[0].data[1], height - 1);
            set_glyph_row(height - 1, 0);
            break;
        case 'b':              /* Shift bottom */
            if (panel) {
                alert("Read-only panel");
                break;
            }
            put_undo(&glyph[0]);
            memmove(&glyph[0].data[1], &glyph[0].data[0], height - 1);
            set_glyph_row(0, 0);
            break;
        case 'u':              /* Undo last changes */
            if (panel) {
                alert("Read-only panel");
                break;
            }
            if (!get_undo(&glyph[0]))
                alert("Nothing to undo");
            break;
        case 'd':              /* Go to the next difference */
            if (!font[1].name) {
                alert("Only one file loaded");
                break;
            }
            if (font[0].num_of_chars != font[1].num_of_chars) {
                alert("Fonts have different number of glyphs");
                break;
            }

            /* Adjusting positions */
            if (panel == 0)
                glyph[1].num = glyph[0].num;
            else
                glyph[0].num = glyph[1].num;

            /* Comparing */
            is_diff = 0;
            for (num = glyph[0].num + 1; num < font[0].num_of_chars; num++) {
                for (i = 0; i < height; i++) {
                    if (font[0].data[num*height+i] !=
                        font[1].data[num*height+i]) {

                        glyph[0].num = glyph[1].num = num;
                        glyph[0].x = glyph[1].x = 0;
                        glyph[0].y = glyph[1].y = i;
                        is_diff = 1;
                        break;
                    }
                }
                if (is_diff)
                    break;
            }
            if (!is_diff)
                message("No differences");
            break;
        case KEY_F(1):
        case '?':              /* Help (this case must be above C('l') ) */
            if ((sig = show_help()) != 0) {
                sighandler(sig);
                continue;
            }
        case C('l'):           /* Redraw screen */
            draw_frame();
            break;
        }

        draw_glyph();
        draw_message();

        if (panel == 0)
            move(5 + glyph[0].y, 2 + glyph[0].x * 2);
        else
            move(5 + glyph[1].y, 40 + glyph[1].x * 2);

        refresh();
    }
    
    /* Exit */
    sighandler(SIGINT);
}

/*------------------------------------------------------------------*/

RETSIGTYPE sighandler(int sig)
{
    int ch;

    close_help(sig);

    switch (sig) {
    case SIGINT:
        if (file_modified()) {
            draw_frame();
            alert("Do you want to save file (y/n)?");
            draw_message();
            refresh();
            do {
                ch = getch();
            }
            while (ch != 'y' && ch != 'n');
            if (ch == 'y') {
                if (!save_font_file(&font[0])) {
                    prepare_exit();
                    perror(font[0].name);
                    exit(1);
                }
            }
        }
    case SIGTERM:
    case SIGHUP:
        prepare_exit();
        exit(1);
        break;
    case SIGWINCH:
        endwin();
        refresh();
        if (chk_win_size(1)) {
            draw_frame();
            draw_message();

            if (panel == 0)
                move(5 + glyph[0].y, 2 + glyph[0].x * 2);
            else
                move(5 + glyph[1].y, 40 + glyph[1].x * 2);
        }
        refresh();
        break;
    }
}

/*------------------------------------------------------------------*/

void prepare_exit()
{
    int i;
    
    for (i = 0; i <= 1 && font[i].name; i++)
        free(font[i].file);
    free_undo();
    clear();
    refresh();
    endwin();
}


/* EOF */
