/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/**
   \file cdw_button.c

   Implementation of "button" widget that can be embedded e.g.
   in dialog windows.
*/

#include <string.h>
#include <stdlib.h>

#include "cdw_button.h"
#include "cdw_debug.h"






/**
   \brief Create new CDW_BUTTON widget

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Remember to assign value to CDW_BUTTON->on_click

   \param parent - parent window, in which the button will be displayed
   \param begin_y - number of row of parent window, in which will be displayed upper border of button
   \param begin_x - number of column of parent window, in which will be displayed left border of button
   \param label - string that will be displayed as button label
   \param colors - color scheme of button

   \return pointer to new button widget on success
   \return NULL pointer if creating widget fails
*/
CDW_BUTTON *cdw_button_new(WINDOW *parent, int begin_y, int begin_x, const char *label, cdw_colors_t colors)
{
	cdw_assert (parent, "ERROR: parent window is null\n");

	CDW_BUTTON *button = (CDW_BUTTON *) malloc(sizeof(CDW_BUTTON));
	if (!button) {
		cdw_vdm ("ERROR: malloc()\n");
		return (CDW_BUTTON *) NULL;
	}

	button->strip = (WINDOW *) NULL;
	button->label = (char *) NULL;

	button->label = strdup(label);
	if (!(button->label)) {
		free(button);
		button = (CDW_BUTTON *) NULL;

		cdw_vdm ("ERROR: strdup()\n");
		return (CDW_BUTTON *) NULL;
	}

	button->begin_x = begin_x;
	button->begin_y = begin_y;
	button->colors = colors;
	button->on_click = (cdw_form_widget_function_t) NULL;
	button->parent = parent;

	int label_len = (int) strlen(button->label);
	/* "[ label ]" = 2 + len + 2 */
	button->strip = derwin(parent, 1, 2 + label_len + 2,
			       button->begin_y, button->begin_x);
	if (button->strip) {
		(void) wattrset(button->strip, COLOR_PAIR(button->colors));

		mvwprintw(button->strip, 0, 0, "[ %s ]", button->label);
		wrefresh(button->strip);

		return button;
	} else {
		cdw_button_delete(&button);
		cdw_vdm ("ERROR: can't create new button: derwin\n");

		return (CDW_BUTTON *) NULL;
	}
}





/**
   \brief Destroy CDW_BUTTON widget

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Deallocate all resources used by CDW_BUTTON widget. The function
   accepts NULL pointer. \p button is set to NULL on return.

   \param button - button to destroy
*/
void cdw_button_delete(CDW_BUTTON **button)
{
	cdw_assert (button, "ERROR: passed to the function a NULL pointer to the widget\n");

	if (!*button) {
		cdw_vdm ("WARNING: NULL button\n");
		return;
	}

	cdw_button_free(*button);
	*button = (CDW_BUTTON *) NULL;

	return;
}





/**
   \brief Free CDW_BUTTON widget

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Deallocate all resources used by CDW_BUTTON widget. The function
   accepts NULL pointer, but does not set its argument to NULL.

   \param button - button to free
*/
void cdw_button_free(CDW_BUTTON *button)
{
	if (!button) {
		cdw_vdm ("WARNING: NULL button\n");
		return;
	}

	if (button->strip) {
		delwin(button->strip);
		button->strip = (WINDOW *) NULL;
	}

	if (button->label) {
		free(button->label);
		button->label = (char *) NULL;
	}

	free(button);

	return;
}





/**
   \brief Highlight given button

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Redraw given button using its reversed 'colors' scheme. It will look as
   if the button has changed its state to 'has focus'.

   \param button - button to highlight
*/
void cdw_button_focus(CDW_BUTTON *button)
{
	cdw_assert (button, "ERROR: cannot focus NULL button\n");

	/* wattrset(button->strip, COLOR_PAIR(button->focus_color)); */
	(void) wattrset(button->strip, COLOR_PAIR(button->colors) | A_REVERSE);

	mvwprintw(button->strip, 0, 0, "[ %s ]", button->label);
	wrefresh(button->strip);

	return;
}





/**
   \brief Display given button as it is in its normal state

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Redraw given button using its normal color scheme. It will look
   as if the button has changed its state to 'has no focus'.

   \param button - button to unfocus
*/
void cdw_button_unfocus(CDW_BUTTON *button)
{
	cdw_assert (button, "ERROR: cannot unfocus NULL button\n");

	(void) wattrset(button->strip, COLOR_PAIR(button->colors));

	mvwprintw(button->strip, 0, 0, "[ %s ]", button->label);
	wrefresh(button->strip);

	return;
}
