/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/**
   \file cdw_dll.c

   \brief Bug-ridden implementation of doubly linked list
*/

#include <string.h>
#include <stdlib.h>
#include <stdbool.h>

#include "cdw_dll.h"
#include "cdw_debug.h"



/**
   \brief Create first item of list

   Call this function when user wants to add batch of data.
   This is not too elegant solution and probably will be put in a wrapper of
   some kind in future.

   \param head - pointer to beginning of list
   \param data - initial data to be put into list

   \return CDW_ERROR if some malloc error occurs
   \return CDW_NO if list is already initialized
   \return CDW_OK if success
*/
cdw_rv_t cdw_dll_init(cdw_dll_item_t **head, void *data)
{
	cdw_assert (*head == (cdw_dll_item_t *) NULL, "list already initialized\n");
	cdw_assert (data != (void *) NULL, "data should not be null\n");
	if (*head != (cdw_dll_item_t *) NULL) {
		return CDW_NO;
	}

	*head = (cdw_dll_item_t *) cdw_dll_new_item(data);

	if (*head == (cdw_dll_item_t *) NULL) {
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
 * \brief Returns last item of list
 *
 * Useful when you want to append new file at the end of list
 *
 * \return NULL if list is empty
 * \return pointer to last item on list if list is not empty
 */
cdw_dll_item_t *cdw_dll_last_item(cdw_dll_item_t *head)
{
	cdw_assert (head != (cdw_dll_item_t *) NULL, "trying to get last item from uninitialized list\n");

	cdw_dll_item_t *i;
	for (i = head; i->next != (cdw_dll_item_t *) NULL; i = i->next)
		;

	return i;
}






/**
 * \brief Allocate new list item, initialize it with given file
 *
 * This function creates its own copy of file.
 * Pointers to previous and next list item are initialized to
 * (cdw_dll_item_t *) NULL. cdw_file must have non-empty, non-NULL
 * fullpath field, otherwise the file is treated as invalid. Other
 * fields of cdw_file are not checked.
 *
 * \param cdw_file - file that you want to store in given list item
 *
 * \return Pointer to freshly allocated list item
 * \return NULL if malloc fails
 * \return NULL if 'fullpath' field of file is NULL or empty string
 */
cdw_dll_item_t *cdw_dll_new_item(void *data)
{
	if (data  == (void *) NULL) {
		return (cdw_dll_item_t *) NULL;
	}

	cdw_dll_item_t *i = (cdw_dll_item_t *) malloc(sizeof(cdw_dll_item_t));
	if (i == (cdw_dll_item_t *) NULL) {
		return (cdw_dll_item_t *) NULL;
	}

	i->data = data;

	i->next = (cdw_dll_item_t *) NULL;
	i->prev = (cdw_dll_item_t *) NULL;

	return i;
}





/**
 * \brief Append file at the end of list
 *
 * The function does not check if given file already exists on list.
 * In order to avoid duplicates on list you have to search for given
 * file on the list first.
 *
 * The function will initialize list if list is empty.
 *
 * \param cdw_file - file that you want to append to list
 *
 * \return CDW_GEN_ERROR if malloc fails
 * \return CDW_OK if success
 */
cdw_rv_t cdw_dll_append_non_unique(cdw_dll_item_t **head, void *data)
{
	cdw_assert (data != (void *) NULL, "data can't be null\n");

	if (cdw_dll_is_empty(*head)) {
		return cdw_dll_init(head, data);
	} else {
		cdw_dll_item_t *last = cdw_dll_last_item(*head);
		if (last == (cdw_dll_item_t *) NULL) {
			return CDW_ERROR;
		} else {
			cdw_dll_item_t *f = cdw_dll_new_item(data);
			if (f == (cdw_dll_item_t *) NULL) {
				return CDW_ERROR;
			} else {
				last->next = f;
				f->prev = last;
				f->next = (cdw_dll_item_t *) NULL;
				return CDW_OK;
			}
		}
	}
}





/**
 * \brief Append file at the end of list
 *
 * The function checks if given file already exists on list. File
 * that would be duplicate on the list is not appended and CDW_NO is returned.
 *
 * \param cdw_file - file that you want to store on list
 *
 * \return CDW_ERROR if malloc fails
 * \return CDW_NO if file is already on list
 * \return CDW_OK if success
 */
cdw_rv_t cdw_dll_append(cdw_dll_item_t **head, void *data, bool (*pred)(const void *, const void *))
{
	cdw_assert (data != (void *) NULL, "can't append null data\n");
	cdw_assert (pred != NULL, "predicate function is null\n");

	if (cdw_dll_is_member(*head, data, pred)) {
		return CDW_NO;
	} else {
		/* will init list if necessary */
		return cdw_dll_append_non_unique(head, data);
	}
}





/**
 * \brief Return pointer to i-th item on list
 *
 * \param i - index of item that you want to obtain (0-based)
 *
 * \return Pointer to i-th item of list
 * \return NULL if index is too large or list is empty
 */
cdw_dll_item_t *cdw_dll_ith_item(cdw_dll_item_t *head, size_t i)
{
	cdw_assert (head != (cdw_dll_item_t *) NULL, "trying to get element from uninitialized list\n");

	cdw_dll_item_t *item = (cdw_dll_item_t *) NULL;
	size_t j = 0;

	for (item = head, j = 0; item != (cdw_dll_item_t *) NULL && j <= i; item = item->next, j++) {
		if (j == i) {
			return item;
		}
	}

	return (cdw_dll_item_t *) NULL;
}





/**
 * \brief Remove i-th file from list
 *
 * \param i - index of file that should be removed (zero-based)
 *
 * \return CDW_ERROR if list is empty or other error occurred
 * \return CDW_OK if success
 */
cdw_rv_t cdw_dll_remove_ith_item(cdw_dll_item_t **head, size_t i)
{
	if (*head == (cdw_dll_item_t *) NULL) {
		return CDW_ERROR;
	}

	cdw_dll_item_t *f = cdw_dll_ith_item(*head, i);

	if (f == (cdw_dll_item_t *) NULL) {
		return CDW_ERROR;
	}

	/* works even if j is first or last node, NULLs are preserved */
	if (f->next != (cdw_dll_item_t *) NULL) {
		f->next->prev = f->prev;
	}
	if (f->prev != (cdw_dll_item_t *) NULL) {
		f->prev->next = f->next;
	}

	f->data = (void *) NULL;

	if (f == *head) { /* we have to create new list's head */
		*head = f->next;
	}

	free(f);
	f = (cdw_dll_item_t *) NULL;

	return CDW_OK;
}





/**
 * \brief Count files on list
 *
 * \return Number of files on list (0 if no files)
 */
size_t cdw_dll_length(cdw_dll_item_t *head)
{
	if (head == (cdw_dll_item_t *) NULL) {
		cdw_sdm ("called length with head == NULL\n");
	}
	cdw_dll_item_t *f;
	size_t i = 0;

	for (f = head; f; f = f->next) {
		i++;
	}

	return i;
}





/**
   \brief Remove all links of a list

   \date Function's top-level comment reviewed on 2012-02-16
   \date Function's body reviewed on 2012-02-16

   Function deallocates all links of a list, but does not
   deallocate payload of the list.

   \param head - beginning of a list

   \return CDW_OK
*/
cdw_rv_t cdw_dll_clean(cdw_dll_t *head)
{
	if (!head) {
		cdw_vdm ("WARNING: passed null list to the function\n");
		return CDW_OK;
	}

	for (cdw_dll_t *f = head; f; ) {
		cdw_dll_t *tmp = f->next;

		f->data = (void *) NULL;

		free(f);
		f = tmp;
	}

	return CDW_OK;
}





/**
   \brief Deallocate a list (links and payload)

   \date Function's top-level comment reviewed on 2012-02-16
   \date Function's body reviewed on 2012-02-16

   Function deallocates all links of a list, and all payload of the list.
   Deallocation of the payload is made using \p dealloc function.
   All pointers are set to NULL, including \p head.

   \param head - beginning of a list
   \param dealloc - deallocator function, freeing payload memory

   \return CDW_OK
*/
cdw_rv_t cdw_dll_delete(cdw_dll_t **head, cdw_deallocator_t dealloc)
{
	cdw_assert (head, "ERROR: passed NULL pointer to a list\n");

	if (!*head) {
		cdw_vdm ("WARNING: passed null list to the function\n");
		return CDW_OK;
	}

	for (cdw_dll_t *f = *head; f; ) {
		cdw_dll_t *tmp = f->next;

		dealloc(f->data);
		f->data = (void *) NULL;

		f->next = (cdw_dll_t *) NULL;
		f->prev = (cdw_dll_t *) NULL;

		free(f);
		f = tmp;
	}

	*head = (cdw_dll_t *) NULL;

	return CDW_OK;
}





/**
 * \brief Check if list is empty
 *
 * You could use cdw_dll_items() and check if its return value is
 * zero, but cdw_dll_is_empty() is faster when calling it for long,
 * non-empty lists.
 *
 * \return true if list is empty
 * \return false if list is not empty
 */
bool cdw_dll_is_empty(cdw_dll_item_t *head)
{
	if (head == (cdw_dll_item_t *) NULL) {
		return true;
	} else {
		return false;
	}
}





/**
 * \brief Check if given file is already on the list
 *
 * Search for given file on the list. Return true if
 * the file is on the list, return false if the file is not on
 * the list.
 *
 * The function also works for empty list: it returns false.
 *
 * This piece of code is used in only one place, but I put it in
 * function so that I can test it more easily. Additionally caller
 * code looks cleaner.
 *
 * \param cdw_file - file which you want to look for on the list
 *
 * \return true if given file is on the list
 * \return false if given file is not on the list (or the list is empty)
 */
bool cdw_dll_is_member(cdw_dll_item_t *head, void *data, bool (*pred)(const void *, const void *))
{
	cdw_assert (data != (void *) NULL, "data is null\n");
	cdw_assert (pred != NULL, "predicate function is null\n");

	if (head == (cdw_dll_item_t *) NULL) {
		return false;
	}

	cdw_dll_item_t *f;
	for (f = head; f != (cdw_dll_item_t *) NULL; f = f->next) {
		if (pred(data, f->data)) {
			return true;
		}
	}

	return false;
}





#ifdef CDW_UNIT_TEST_CODE


/* *********************** */
/* *** unit tests code *** */
/* *********************** */

/* I want to test if this dll implementation can correctly store and
   retrieve data of type a bit more complicated than simple int.
   I could use cdw_file_t, but in order to keep this unit tests module
   independent of other cdw modules, I'm creating new data type solely
   for purpose of testing dll module: cdw_dll_test_data_t. */

#define TEST_STRING_LEN 30+1
typedef struct {
	char string[TEST_STRING_LEN];
	char c;
	int i;
	bool boolean;
} cdw_dll_test_data_t;

#define N_TEST_ITEMS 7
cdw_dll_test_data_t test_data[N_TEST_ITEMS] = {
	{ "name_zero",  'a', 1,    true},
	{ "name_one",   'b', 22,   false },
	{ "name_two",   'c', 333,  true },
	{ "name_three", 'd', 4444, false },
	{ "name_four",  'e', 5,    true },
	{ "name_six",   'f', 66,   false }};


bool cdw_dll_test_data_equal(const void *data1, const void *data2);


static void test_cdw_dll_init(void);
static void test_cdw_dll_new_item(void);
static void test_cdw_dll_is_member(void);
static void test_cdw_dll_is_empty(void);
static void test_cdw_dll_append(void);
static void test_cdw_dll_ith_item(void);


void cdw_dll_run_tests(void)
{
	fprintf(stderr, "testing cdw_dll.c\n");

	test_cdw_dll_new_item();
	test_cdw_dll_init();
	test_cdw_dll_is_member();
	test_cdw_dll_is_empty();
	test_cdw_dll_append();
	test_cdw_dll_ith_item();

	fprintf(stderr, "done\n\n");

	return;
}




void test_cdw_dll_init(void)
{

	fprintf(stderr, "\ttesting cdw_dll_init()... ");

	cdw_dll_item_t *head = (cdw_dll_item_t *) NULL;
	cdw_rv_t crv = CDW_OK;

	cdw_dll_test_data_t *in = test_data;

	size_t len = 0;
	bool test;

	/* test length of empty list */
	len = cdw_dll_length(head);
	cdw_assert_test (len == 0, "ERROR: invalid length of empty list: %d\n", (int) len);

	/* test function checking empty list */
	test = cdw_dll_is_empty(head);
	cdw_assert_test (test == true, "ERROR: empty list recognized as non-empty");

	/* test initialization of list */
	crv = cdw_dll_init(&head, (void *) in);
	cdw_assert_test (crv == CDW_OK, "ERROR: failed to initialize\n");
	cdw_assert_test (head->data == in, "ERROR: head is invalid\n");

	/* test correctness of first element */
	cdw_dll_test_data_t *out = (cdw_dll_test_data_t *) head->data;
	cdw_assert_test (in == out, "ERROR: in and out are different\n");
	cdw_assert_test (cdw_dll_test_data_equal(in, out), "ERROR: in and out are different (2)\n");

	/* this test will trigger assertion in cdw_dll_init() in debug builds, and
	   will return CDW_NO when assertions are disabled */
	/*
	cdw_test_data_t *in2 = test_data + 1;
	crv = cdw_dll_init(&head, (void *) NULL);
	cdw_assert_test (crv == CDW_NO, "function tries to initialize initialized list\n");
	*/

	/* we won't test attempting to append second element to list by
	   init(), because the function isn't designed to do so */

	/* test that size of freshly initialized list is equal 1 */
	len = cdw_dll_length(head);
	cdw_assert_test (len == 1, "ERROR: invalid length of list with one element: %d\n", (int) len);

	/* test that list with one element is not empty */
	test = cdw_dll_is_empty(head);
	cdw_assert_test (test == false, "ERROR: empty list recognized as non-empty");

	crv = cdw_dll_clean(head);
	cdw_assert_test (crv == CDW_OK, "ERROR: list not cleaned properly\n");
	head = (cdw_dll_item_t *) NULL;

	fprintf(stderr, "OK\n");
}



void test_cdw_dll_new_item(void)
{
	fprintf(stderr, "\ttesting cdw_dll_new_item()... ");

	cdw_dll_item_t *item;

	/* test calling creator with null argument */
	item = cdw_dll_new_item((void *) NULL);
	cdw_assert_test (item == (cdw_dll_item_t *) NULL, "ERROR: function didn't recognize null argument\n");

	/* test creating new element from correct data/argument */
	cdw_dll_test_data_t *in = test_data;
	item = cdw_dll_new_item((void *) in);
	cdw_assert_test (item != (cdw_dll_item_t *) NULL, "ERROR: function didn't create non-null item\n");
	cdw_assert_test (item->next == (cdw_dll_item_t *) NULL, "ERROR: function didn't create proper ->next link\n");
	cdw_assert_test (item->prev == (cdw_dll_item_t *) NULL, "ERROR: function didn't create proper ->prev link\n");

	/* test that data was correctly stored in new element */
	cdw_dll_test_data_t *out = (cdw_dll_test_data_t *) item->data;
	cdw_assert_test (cdw_dll_test_data_equal(in, out), "ERROR: in and out data are not equal\n");

	/* test that single-element list has length equal to 1 */
	size_t len = cdw_dll_length(item);
	cdw_assert_test (len == 1, "ERROR: incorrect length of 1-elem list: %d\n", (int) len);

	/* test removing element from one-element list */
	cdw_rv_t crv = cdw_dll_remove_ith_item(&item, 0);
	cdw_assert_test (crv == CDW_OK, "ERROR: head of 1-elem list not removed properly\n");
	cdw_assert_test (item == (cdw_dll_item_t *) NULL, "ERROR: head of 1-elem list not null-ed properly\n");

	/* test that list with all elements removed has length equal to zero */
	len = cdw_dll_length(item);
	cdw_assert_test (len == 0, "ERROR: incorrect length of empty list: %d\n", (int) len);

	fprintf(stderr, "OK\n");

	return;

}





void test_cdw_dll_is_member(void)
{
	fprintf(stderr, "\ttesting cdw_dll_is_member() and cdw_dll_length()... ");

	size_t len = 0;
	bool test;

	cdw_dll_test_data_t *data1 = &(test_data[0]);
	cdw_dll_test_data_t *data2 = &(test_data[1]);
	cdw_dll_test_data_t *data3 = &(test_data[2]);
	cdw_dll_test_data_t *data4 = &(test_data[3]);
	cdw_dll_test_data_t *data5 = &(test_data[4]);
	cdw_dll_test_data_t *data6 = &(test_data[5]);
	cdw_dll_test_data_t *never_added_data7 = &(test_data[6]);

	/* 0-elem list */
	cdw_dll_item_t *item1 = (cdw_dll_item_t *) NULL;
	len = cdw_dll_length(item1);
	cdw_assert_test (len == 0, "ERROR: incorrect length of 0-elem list: %d\n", (int) len);

	test = cdw_dll_is_member(item1, (void *) data1, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: input data1 found on empty list\n");



	/* 1-elem list */
	item1 = cdw_dll_new_item((void *) data1);
	cdw_assert_test (item1 != (cdw_dll_item_t *) NULL, "ERROR: failed to create 1st item\n");
	cdw_dll_item_t *head = item1;

	len = cdw_dll_length(head);
	cdw_assert_test (len == 1, "ERROR: incorrect length of 1-elem list: %d\n", (int) len);

	test = cdw_dll_is_member(head, (void *) data1, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data0 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data2, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "data2 found on list before adding the data\n");


	/* 2-elem list */
	cdw_dll_item_t *item2 = cdw_dll_new_item((void *) data2);
	cdw_assert_test (item2 != (cdw_dll_item_t *) NULL, "ERROR: failed to create 2nd item\n");

	item1->next = item2;
	item2->prev = item1;

	len = cdw_dll_length(head);
	cdw_assert_test (len == 2, "ERROR: incorrect length of 2-elem list: %d\n", (int) len);

	test = cdw_dll_is_member(head, (void *) data1, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data0 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data2, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data1 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data3, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data2 found on list before adding the data\n");


	/* 3-elem list */
	cdw_dll_item_t *item3 = cdw_dll_new_item((void *) data3);
	cdw_assert_test (item3 != (cdw_dll_item_t *) NULL, "ERROR: failed to create 3rd item\n");

	item2->next = item3;
	item3->prev = item2;

	len = cdw_dll_length(head);
	cdw_assert_test (len == 3, "ERROR: incorrect length of 3-elem list: %d\n", (int) len);

	test = cdw_dll_is_member(head, (void *) data1, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data1 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data2, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data2 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data3, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data3 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data4, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data4 found on list before adding the data\n");

	test = cdw_dll_is_member(head, (void *) never_added_data7, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: never_added_data7 found on list\n");


	/* 4-elem list */
	cdw_dll_item_t *item4 = cdw_dll_new_item((void *) data4);
	cdw_assert_test (item4 != (cdw_dll_item_t *) NULL, "ERROR: failed to create 4th item\n");

	item3->next = item4;
	item4->prev = item3;

	len = cdw_dll_length(head);
	cdw_assert_test (len == 4, "ERROR: incorrect length of 4-elem list: %d\n", (int) len);

	test = cdw_dll_is_member(head, (void *) data1, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data1 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data2, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data2 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data3, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data3 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data4, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data4 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data5, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data5 found on list before adding the data\n");


	/* 5-elem list */
	cdw_dll_item_t *item5 = cdw_dll_new_item((void *) data5);
	cdw_assert_test (item5 != (cdw_dll_item_t *) NULL, "ERROR: failed to create 5th item\n");

	item4->next = item5;
	item5->prev = item4;

	len = cdw_dll_length(head);
	cdw_assert_test (len == 5, "ERROR: incorrect length of 5-elem list: %d\n", (int) len);

	test = cdw_dll_is_member(head, (void *) data1, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data1 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data2, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data2 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data3, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data3 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data4, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data4 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data5, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data5 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data6, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data6 found on list before adding the data\n");


	/* 6-elem list */
	cdw_dll_item_t *item6 = cdw_dll_new_item((void *) data6);
	cdw_assert_test (item6 != (cdw_dll_item_t *) NULL, "ERROR: failed to create 6th item\n");

	item5->next = item6;
	item6->prev = item5;

	len = cdw_dll_length(head);
	cdw_assert_test (len == 6, "ERROR: incorrect length of 6-elem list: %d\n", (int) len);

	test = cdw_dll_is_member(head, (void *) data1, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data1 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data2, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data2 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data3, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data3 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data4, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data4 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data5, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data5 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) data6, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data6 not found after adding it\n");

	test = cdw_dll_is_member(head, (void *) never_added_data7, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: never added data7 found on list\n");



	/* we have 6-elem list: removing element 3 */
	item3->prev->next = item3->next;
	item3->next->prev = item3->prev;

	free(item3);
	item3 = (cdw_dll_item_t *) NULL;

	len = cdw_dll_length(head);
	cdw_assert_test (len == 5, "ERROR: incorrect length of 5-elem list: %d\n", (int) len);

	test = cdw_dll_is_member(head, (void *) data1, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data1 not found\n");

	test = cdw_dll_is_member(head, (void *) data2, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data2 not found\n");

	test = cdw_dll_is_member(head, (void *) data3, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data3 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data4, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data4 not found\n");

	test = cdw_dll_is_member(head, (void *) data5, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data5 not found\n");

	test = cdw_dll_is_member(head, (void *) data6, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data6 not found\n");

	test = cdw_dll_is_member(head, (void *) never_added_data7, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: never added data7 found on list\n");


	/* we have 5-elem list: removed element 3, removing element 5 */
	item5->prev->next = item5->next;
	item5->next->prev = item5->prev;

	free(item5);
	item5 = (cdw_dll_item_t *) NULL;

	len = cdw_dll_length(head);
	cdw_assert_test (len == 4, "ERROR: incorrect length of 4-elem list: %d\n", (int) len);

	test = cdw_dll_is_member(head, (void *) data1, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data1 not found\n");

	test = cdw_dll_is_member(head, (void *) data2, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data2 not found\n");

	test = cdw_dll_is_member(head, (void *) data3, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data3 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data4, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data4 not found\n");

	test = cdw_dll_is_member(head, (void *) data5, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data5 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data6, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data6 not found\n");

	test = cdw_dll_is_member(head, (void *) never_added_data7, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: never added data7 found on list\n");


	/* we have 4-elem list: removed elements 3 and 5, removing element 1 */

	item1->next->prev = item1->prev;
	head = item1->next;

	free(item1);
	item1 = (cdw_dll_item_t *) NULL;

	len = cdw_dll_length(head);
	cdw_assert_test (len == 3, "ERROR: incorrect length of 3-elem list: %d\n", (int) len);

	test = cdw_dll_is_member(head, (void *) data1, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data1 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data2, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data2 not found\n");

	test = cdw_dll_is_member(head, (void *) data3, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data3 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data4, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data4 not found\n");

	test = cdw_dll_is_member(head, (void *) data5, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data5 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data6, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data6 not found\n");

	test = cdw_dll_is_member(head, (void *) never_added_data7, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: never added data7 found on list\n");

	/* we have 3-elem list: removed elements 3, 5 and 1, removing element 4 */

	item4->prev->next = item4->next;
	item4->next->prev = item4->prev;

	free(item4);
	item4 = (cdw_dll_item_t *) NULL;

	len = cdw_dll_length(head);
	cdw_assert_test (len == 2, "ERROR: incorrect length of 2-elem list: %d\n", (int) len);

	test = cdw_dll_is_member(head, (void *) data1, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data1 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data2, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data2 not found\n");

	test = cdw_dll_is_member(head, (void *) data3, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data3 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data4, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data4 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data5, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data5 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data6, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data6 not found\n");

	test = cdw_dll_is_member(head, (void *) never_added_data7, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: never added data7 found on list\n");


	/* we have 2-elem list: removed elements 3, 5, 1 and 4, removing element 6 */

	item6->prev->next = item6->next;
	//item6->next->prev = item6->prev; /* item 6 is last on list */

	free(item6);
	item6 = (cdw_dll_item_t *) NULL;

	len = cdw_dll_length(head);
	cdw_assert_test (len == 1, "ERROR: incorrect length of 1-elem list: %d\n", (int) len);

	test = cdw_dll_is_member(head, (void *) data1, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data1 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data2, cdw_dll_test_data_equal);
	cdw_assert_test (test == true, "ERROR: data2 not found\n");

	test = cdw_dll_is_member(head, (void *) data3, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data3 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data4, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data4 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data5, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data5 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data6, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data6 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) never_added_data7, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: never added data7 found on list\n");



	/* we have 1-elem list: removed elements 3, 5, 1, 4 and 6, removing element 2 */
	/* this is the last elem on list, any references to fields of
	   previous or next element are illegal
	item2->prev->next = item2->next;
	item2->next->prev = item2->prev;
	*/

	free(item2);
	item2 = (cdw_dll_item_t *) NULL;

	head = (cdw_dll_item_t *) NULL;

	len = cdw_dll_length(head);
	cdw_assert_test (len == 0, "ERROR: incorrect length of empty list: %d\n", (int) len);

	test = cdw_dll_is_member(head, (void *) data1, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data1 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data2, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data2 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data3, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data3 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data4, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data4 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data5, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data5 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) data6, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: data6 found after removing it\n");

	test = cdw_dll_is_member(head, (void *) never_added_data7, cdw_dll_test_data_equal);
	cdw_assert_test (test == false, "ERROR: never added data7 found on list\n");

	fprintf(stderr, "OK\n");

	return;
}



void test_cdw_dll_is_empty(void)
{
	fprintf(stderr, "\ttesting cdw_dll_is_empty()... ");

	cdw_dll_item_t *head = (cdw_dll_item_t *) NULL;

	bool test = cdw_dll_is_empty(head);
	cdw_assert_test (test == true, "ERROR: failed to recognize empty list\n");

	int var;
	head = (cdw_dll_item_t *) &var;

	test = cdw_dll_is_empty(head);
	cdw_assert_test (test == false, "ERROR: failed to recognize non-empty list\n");

	head = cdw_dll_new_item(&var);
	cdw_assert_test (test == false, "ERROR: failed to recognize non-empty list (2)\n");

	cdw_rv_t crv = cdw_dll_clean(head);
	cdw_assert_test (crv == CDW_OK, "ERROR: list not cleaned properly\n");
	head = (cdw_dll_item_t *) NULL;

	fprintf(stderr, "OK\n");

	return;
}




void test_cdw_dll_append(void)
{
	fprintf(stderr, "\ttesting cdw_dll_append()... ");

	cdw_dll_test_data_t *data1 = &(test_data[0]);
	cdw_dll_test_data_t *data2 = &(test_data[1]);
	cdw_dll_test_data_t *data3 = &(test_data[2]);
	cdw_dll_test_data_t *retrieved = (cdw_dll_test_data_t *) NULL;

	cdw_dll_item_t *head = (cdw_dll_item_t *) NULL;


	/* *** append first item *** */

	cdw_rv_t crv = cdw_dll_append(&head, (void *) data1, cdw_dll_test_data_equal);
	cdw_assert_test (crv == CDW_OK, "ERROR: data1 not appended correctly\n");

	/* appending data to empty list should not modify links */
	cdw_assert_test (head->next == (cdw_dll_item_t *) NULL, "invalid next (1)");
	cdw_assert_test (head->prev == (cdw_dll_item_t *) NULL, "invalid prev (1)");

	/* check data from head */
	cdw_assert_test (head->data != (void *) NULL, "ERROR: data of first item is null\n");
	retrieved = (cdw_dll_test_data_t *) head->data;
	cdw_assert_test (retrieved == data1, "ERROR: incorrect data pointer in first appended item\n");
	cdw_assert_test (cdw_dll_test_data_equal(retrieved, data1), "ERROR: incorrect data in first appended item\n");


	/* *** append second item *** */
	crv = cdw_dll_append(&head, (void *) data2, cdw_dll_test_data_equal);
	cdw_assert_test (crv == CDW_OK, "ERROR: data2 not appended correctly\n");

	/* first check links */
	cdw_assert_test (head->prev == (cdw_dll_item_t *) NULL, "ERROR: head's prev is not null\n");
	cdw_assert_test (head->next->prev == head, "ERROR: second element doesn't link to first\n");
	cdw_assert_test (head->next->next == (cdw_dll_item_t *) NULL, "ERROR: last element's next is not null\n");

	/* check data from head */
	cdw_assert_test (head->data != (void *) NULL, "ERROR: data of first item is null\n");
	retrieved = (cdw_dll_test_data_t *) head->data;
	cdw_assert_test (retrieved == data1, "ERROR: incorrect data pointer in first appended item\n");
	cdw_assert_test (cdw_dll_test_data_equal(retrieved, data1), "ERROR: incorrect data in first appended item\n");

	/* check data from second element of list */
	cdw_assert_test (head->next->data != (void *) NULL, "ERROR: data of second item is null\n");
	retrieved = (cdw_dll_test_data_t *) head->next->data;
	cdw_assert_test (retrieved == data2, "ERROR: incorrect data pointer in second appended item\n");
	cdw_assert_test (cdw_dll_test_data_equal(retrieved, data2), "ERROR: incorrect data in second appended item\n");



	/* *** append third item *** */
	crv = cdw_dll_append(&head, (void *) data3, cdw_dll_test_data_equal);
	cdw_assert_test (crv == CDW_OK, "ERROR: data3 not appended correctly\n");

	/* first check links */
	cdw_assert_test (head->prev == (cdw_dll_item_t *) NULL, "ERROR: head's prev is not null\n");

	cdw_assert_test (head->next->prev == head, "ERROR: second element doesn't link to first\n");
	cdw_assert_test (head->next->next != (cdw_dll_item_t *) NULL, "ERROR: second element's next is null\n");
	cdw_assert_test (head->next->next->prev == head->next, "ERROR: last element's prev is invalid\n");
	cdw_assert_test (head->next->next->next == (cdw_dll_item_t *) NULL, "ERROR: last elemeent's next is not null\n");

	/* check data from head */
	cdw_assert_test (head->data != (void *) NULL, "ERROR: data of first item is null\n");
	retrieved = (cdw_dll_test_data_t *) head->data;
	cdw_assert_test (retrieved == data1, "ERROR: incorrect data pointer in first appended item\n");
	cdw_assert_test (cdw_dll_test_data_equal(retrieved, data1), "ERROR: incorrect data in first appended item\n");

	/* check data from second element of list */
	cdw_assert_test (head->next->data != (void *) NULL, "ERROR: data of second item is null\n");
	retrieved = (cdw_dll_test_data_t *) head->next->data;
	cdw_assert_test (retrieved == data2, "ERROR: incorrect data pointer in second appended item\n");
	cdw_assert_test (cdw_dll_test_data_equal(retrieved, data2), "ERROR: incorrect data in second appended item\n");

	/* check data from third element of list */
	cdw_assert_test (head->next->next->data != (void *) NULL, "ERROR: data of third item is null\n");
	retrieved = (cdw_dll_test_data_t *) head->next->next->data;
	cdw_assert_test (retrieved == data3, "ERROR: incorrect data pointer in third appended item\n");
	cdw_assert_test (cdw_dll_test_data_equal(retrieved, data3), "ERROR: incorrect data in third appended item\n");


	/* *** attempt to append append data2 for a second time *** */
	size_t len1 = cdw_dll_length(head);
	crv = cdw_dll_append(&head, (void *) data2, cdw_dll_test_data_equal);
	cdw_assert_test (crv == CDW_NO, "ERROR: data2 was not recognized as being on list\n");
	size_t len2 = cdw_dll_length(head);
	cdw_assert_test (len1 == len2, "ERROR: length mismatch\n");
	cdw_assert_test (len2 == 3, "ERROR: incorrect length");

	/* check data in all three elements */
	retrieved = (cdw_dll_test_data_t *) head->data;
	cdw_assert_test (retrieved == data1, "ERROR: incorrect data pointer in first appended item\n");
	retrieved = (cdw_dll_test_data_t *) head->next->data;
	cdw_assert_test (retrieved == data2, "ERROR: incorrect data pointer in second appended item\n");
	retrieved = (cdw_dll_test_data_t *) head->next->next->data;
	cdw_assert_test (retrieved == data3, "ERROR: incorrect data pointer in third appended item\n");

	/* check links in last element */
	cdw_assert_test (head->next->next->next == (cdw_dll_item_t *) NULL, "ERROR: last element's next is incorrect\n");
	cdw_assert_test (head->next->next->prev->prev == head, "ERROR: incorrect chain\n");

	crv = cdw_dll_clean(head);
	cdw_assert_test (crv == CDW_OK, "ERROR: list not cleaned properly\n");
	head = (cdw_dll_item_t *) NULL;

	fprintf(stderr, "OK\n");
	return;
}





void test_cdw_dll_ith_item(void)
{
	fprintf(stderr, "\ttesting cdw_dll_ith_item() and cdw_dll_last_item()... ");

	cdw_dll_item_t *head = (cdw_dll_item_t *) NULL;

	for (int i = 0; i < N_TEST_ITEMS; i++) {
		cdw_rv_t crv = cdw_dll_append(&head, &(test_data[i]), cdw_dll_test_data_equal);
		cdw_assert_test (crv == CDW_OK, "ERROR: data%d not appended correctly\n", i + 1);

		cdw_dll_item_t *item = cdw_dll_last_item(head);
		cdw_dll_test_data_t *data = (cdw_dll_test_data_t *) item->data;
		cdw_assert_test (data == &(test_data[i]), "ERROR: failed to get correct last item\n");
		cdw_assert_test (cdw_dll_test_data_equal(data, (&test_data[i])), "ERROR: retrieved last data doesn't match original\n");
	}

	for (int i = 0; i < N_TEST_ITEMS; i++) {
		cdw_dll_item_t *item = cdw_dll_ith_item(head, (size_t) i);
		cdw_dll_test_data_t *data = (cdw_dll_test_data_t *) item->data;

		cdw_assert_test (data == &(test_data[i]), "ERROR: retrieved data%d pointer doesn't match original\n", i + 1);
		cdw_assert_test (cdw_dll_test_data_equal(data, (&test_data[i])), "ERROR: retrieved data%d doesn't match original\n", i + 1);
	}

	/* index larger than number of items on list */
	cdw_dll_item_t *item = cdw_dll_ith_item(head, N_TEST_ITEMS);
	cdw_assert_test (item == (cdw_dll_item_t *) NULL, "ERROR: function can't handle index larger than number of items\n");
	item = cdw_dll_ith_item(head, 100);
	cdw_assert_test (item == (cdw_dll_item_t *) NULL, "ERROR: function can't handle index larger than number of items (2)\n");

	cdw_rv_t crv = cdw_dll_clean(head);
	cdw_assert_test (crv == CDW_OK, "ERROR: list not cleaned properly\n");
	head = (cdw_dll_item_t *) NULL;

	fprintf(stderr, "OK\n");
	return;
}




bool cdw_dll_test_data_equal(const void *data1, const void *data2)
{
	const cdw_dll_test_data_t *d1 = (const cdw_dll_test_data_t *) data1;
	const cdw_dll_test_data_t *d2 = (const cdw_dll_test_data_t *) data2;

	if (strcmp(d1->string, d2->string)) {
		cdw_sdm ("INFO: string1 != string2: \"%s\" != \"%s\"\n", d1->string, d2->string);
		return false;
	}
	if (d1->c != d2->c) {
		cdw_sdm ("INFO: char1 != char2: \"%c\" != \"%c\"\n", d1->c, d2->c);
		return false;
	}
	if (d1->i != d2->i) {
		cdw_sdm ("INFO: int1 != int2: \"%d\" != \"%d\"\n", d1->i, d2->i);
		return false;
	}
	if (d1->boolean != d2->boolean) {
		cdw_sdm ("INFO: bool1 != bool2: \"%s\" != \"%s\"\n",
			 d1->boolean ? "true" : "false", d2->boolean ? "true" : "false");

		return false;
	}
	return true;
}


#endif /* CDW_UNIT_TEST_CODE */

