/* "shuffle.c" copyright 1998 daniel risacher  */


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/types.h>
#include <fcntl.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>
#include <sys/time.h>

#ifdef sun
#  include <sundev/srreg.h>
#elif defined linux
#  include <linux/cdrom.h>
#else
#  error Please fix includes for your system in shuffle.c
#endif

#include "config.h"
#include "commands.h"
#include "hardware.h"
#include "info.h"
#include "shuffle.h"
#include "main.h"

/* globals for this module */
static int   iDone      = FALSE;
static int   iNext      = FALSE;
static int   iBEVerbose = FALSE;
static char  *pszName   = NULL;

/************************************************************************/
/* Procedure:  huphdl
 * Purpose:    handler for SIGHUP
 * 
 * Inputs:     
 * Outputs:    TRUE to iNext flag -- do next track
 * Returns:    
 * Notes:  
 *   1.  Sending a HANGUP (SIGHUP,1) will cause cdshuffle to skip
 *       to the next track.  This may be used for debug or just if
 *       you don't like a song.  
 */
/************************************************************************/
void huphdl ()
{
    signal (SIGHUP, huphdl);
    iNext = TRUE;
    if (iBEVerbose)
	fprintf (stderr, "%s:  Hangup received....\n", pszName);
}

/************************************************************************/
/* Procedure:  inthdl
 * Purpose:    handler for SIGINT
 * 
 * Inputs:     
 * Outputs:    TRUE to iDone flag -- finished with processing
 * Returns:    
 * Notes:  
 *   1.  Sending an INTERRUPT (SIGINT,2) will cause cdshuffle to 
 *       stop playing, remove its PID file, and exit. 
 */
/************************************************************************/
void inthdl ()
{
    signal (SIGINT, inthdl);
    iDone = TRUE;
    if (iBEVerbose)
	fprintf (stderr, "%s:  Interrupt received....\n", pszName);

}

/************************************************************************/
/* Procedure:  wait_track
 * Purpose:    to wait for a track to finish
 * 
 * Inputs:     program name, CD file des., track number
 * Outputs:    none
 * Returns:    pointer to temp string
 * Notes:  
 *   1.  Polls the CD-ROM at a 1 second interval waiting for a 
 *       track to finish....
 */
/************************************************************************/
int wait_track(char* progname, int cdfile, int t)
{
  struct cdrom_subchnl subchnl;

#ifdef DEBUG
  fprintf (stderr, "%s:  waiting on track %d\n", 
	   progname, t);
#endif

  while (1) {
    
    subchnl.cdsc_format = CDROM_MSF;
    if ( ioctl(cdfile, CDROMSUBCHNL, &subchnl) == -1 ) {
	if (errno == EINTR)
	    return TRUE;

	fprintf(stderr, "%s: ioctl cdromsubchnl\n", progname);
	
#ifdef CDCTRL
	return (-1);
#else
	EXIT(1);
#endif

    }
    
    switch ( subchnl.cdsc_audiostatus ) {
    case CDROM_AUDIO_PAUSED:
    case CDROM_AUDIO_PLAY:
      sleep(1);
      if (iDone == TRUE) {
#ifdef DEBUG
	  fprintf (stderr, "%s:  done with audio r=0x%X", 
		   progname, subchnl.cdsc_audiostatus);
#endif
	  return TRUE;
      }

      if (iNext == TRUE) {
#ifdef DEBUG
	  fprintf (stderr, "%s:  next track processing r=0x%X\n",
		   progname, subchnl.cdsc_audiostatus);
#endif
	  iNext = FALSE;
	  return TRUE;
      }
      break;

    case CDROM_AUDIO_COMPLETED:  
#ifdef DEBUG
	fprintf (stderr, "%s:  Audio Completed r=0x%X", 
		 progname, subchnl.cdsc_audiostatus);
#endif
	return TRUE;

    default:
#ifdef DEBUG
	fprintf (stderr, "%s:  default r=0x%X\n", 
		 progname, subchnl.cdsc_audiostatus);
#endif
      return TRUE;
    }
  }
}

/************************************************************************/
/* Procedure:  do_shuffle
 * Purpose:    to play a CD-ROM in shuffle mode
 * 
 * Inputs:     program name, CD file descriptor, max tracks to play,
 *             count of tracks, verbose flag (T/F), , CD device,
 *             CR/CRLF flag
 * Outputs:    none
 * Returns:    pointer to temp string
 * Notes:  
 *   1.  Calls wait_track to wait for a track to finish playing....
 *   2.  This function should be modified for the following:
 *       A) Additon of a count of times to shuffle and max tracks to play.
 *       B) Change to call read_hw() as read_hw() includes additional 
 *          CD-ROM checks (e.g., if mounted).
 */
/************************************************************************/
void do_shuffle(char *progname, int cdfile, int iCDNum, int iTracks, 
		int iCount, int iVerbose, char *cddevice, int docr)
{
    /* struct cdrom_subchnl subchnl; */
  struct cdrom_tochdr tochdr;
  int tracks[100];
  int nent_tracks;
  int i, j, tmp, r;
  cdhw_t *hw;
  int iSaveTracks;

  pid_t my_pid;
  FILE  *fpPID;
  char  caBuf[40];

  iSaveTracks = iTracks;
  pszName = progname;
  iBEVerbose = iVerbose;

#ifdef DEBUG
  fprintf (stderr, "%s.do_shuffle:  file=%d, num=%d, trk=%d, cnt=%d, V=%d\n",
	   progname, cdfile, iCDNum, iTracks, iCount, iVerbose);
#endif

  hw = check_disc (progname, cdfile, DOLF, &i, cddevice);
  if (i == TRUE) {
        
      /* get number of tracks, if not available, error... */
      if ( ioctl(cdfile, CDROMREADTOCHDR, &tochdr) == -1 ) {
	  fprintf(stderr, "%s: ioctl cdromreadtochdr\n", progname);
#ifdef CDCTRL
	  return ;
#else
	  EXIT(1);
#endif
      }
      
      nent_tracks = tochdr.cdth_trk1;

      /* update the pid file "PID\nTIME\nFILE\n" */
      signal (SIGINT, inthdl);
      signal (SIGHUP, huphdl);
      my_pid = getpid();
      sprintf (caBuf, "%s%d", PID_FILE, iCDNum);
      fpPID = fopen (caBuf, "w");
      if (fpPID == NULL)
	  fprintf(stderr,"%s:  Error opening pid file %s\n", 
		  progname, caBuf);
      else {
	  if (iBEVerbose == TRUE)
	      fprintf (stderr, "%s:  Writing PID file %s\n", 
		       progname, caBuf);
	  sprintf (caBuf, "%d\n", my_pid);
	  fputs (caBuf, fpPID);
	  fclose (fpPID);
      }

      /* repeat play N time (if N = -1, do forever) */
      iDone = FALSE;
      while (iDone == FALSE) {

#ifdef DEBUG
	  fprintf(stderr,"do_shuf:  tracks=%d repeat=%d verbose=%d\n",
		  iTracks, iCount, iVerbose);
#endif

	  /* generate random list of tracks to play */
	  for (i=0; i<nent_tracks; i++)
	      tracks[i] = i+1;
	  
	  srand(time(NULL));
	  for (j=0; j<2; j++) {
	      for (i=0; i<nent_tracks; i++) {
		  r = (int) (nent_tracks * (rand() / (RAND_MAX+1.0)));

#ifdef DEBUG
		  printf ("do_shuf:  j=%d, i=%d, r=%d, tracks=%d\n", 
			  j, i, r, nent_tracks);
#endif
		  tmp = tracks[j];
		  tracks[j] = tracks[r];
		  tracks[r] = tmp;
	      }
	  }
#ifdef DEBUG
	  for (i=0; i<nent_tracks; i++) {
	      printf ("do_shuf:  I=%02d track=%02d\n", i, tracks[i]);
	  }
#endif
	  
	  /* play each track in random order */
	  for (i=0; i<nent_tracks; i++) {
#ifdef DEBUG
	      fprintf(stderr,"do_shuf:  playing %d\n", tracks[i]);
#endif
	      if (iVerbose) {
		  printf ("%s playing track %d", progname, tracks[i]);
		  DoCr(docr);
	      }
	      
	      /* this plays only this track */
	      do_play(progname, cdfile, tracks[i], tracks[i], FALSE);

	      /* if we are doing 1 track only, exit */
	      if ((iCount <= 1) && (iTracks == 1) &&
		  (iSaveTracks == 1))
		  break;

	      wait_track(progname, cdfile, tracks[i]);

	      /* if we have played the desired # of tracks, DONE */
	      iTracks--;
	      if (iTracks <= 0) {
		  iTracks = iSaveTracks;
		  break;
	      }

	      if (iDone == TRUE)
		  break;
	  }
	  
	  /* determine if we are to repeat the disc, iCount==0 is forever */
	  if (iCount > 0) {
	      iCount --;
	      if (iCount == 0)
		  break;
	  }
      }
  }
  free_hw_buf(hw);
  
  if ( ioctl(cdfile, CDROMSTOP) == -1 ) {
      fprintf(stderr, "%s: ioctl cdromstop\n", progname);
  }

  sprintf (caBuf, "%s%d", PID_FILE, iCDNum);
  if (iBEVerbose)
      fprintf (stderr, "%s:  removing lock file %s\n", 
	       progname, caBuf);
  unlink (caBuf);

}
