/* "info.c" copyright 1994 thomas insel 
 *          copyright 1995 sven oliver moll */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <string.h>

#ifdef sun
#  include <sundev/srreg.h>
#elif defined linux
#  include <linux/cdrom.h>
#else
#  error Please fix includes for your system in info.c
#endif

#include "config.h"
#include "database.h"
#include "hardware.h"
#include "info.h"
#include "main.h"

#ifdef CDCTRL
char caBuff[(BUFSIZ > (20 + 8*100)) ? BUFSIZ : (20 + 8*100)];
#endif


/************************************************************************/
/* Procedure:  tracks_line
 * Purpose:    to display a line of info on a CD-ROM track
 * 
 * Inputs:     hardware info
 * Outputs:    to temp string
 * Returns:    pointer to temp string
 * Notes:  
 *   1.
 */
/************************************************************************/
char *
tracks_line(cdhw_t *hw) 
{
    int i;

#ifdef CDCTRL
    char *tempstr = &caBuff[0];
#else
    char *tempstr = (char *)malloc(20 + 8*hw->tochdr.cdth_trk1);
#endif
  
    sprintf(tempstr, "tracks %u ", hw->tochdr.cdth_trk1);
    for (i=hw->tochdr.cdth_trk0; i<=hw->tochdr.cdth_trk1; i++)
	sprintf(tempstr + strlen(tempstr), "%u ",
		hw->tocentries[i-1].cdte_addr.msf.minute*4500 +
		hw->tocentries[i-1].cdte_addr.msf.second*75 +
		hw->tocentries[i-1].cdte_addr.msf.frame );
    sprintf(tempstr + strlen(tempstr), "%u", 
	    hw->tocentries[hw->tochdr.cdth_trk1].cdte_addr.msf.minute*60 +
	    hw->tocentries[hw->tochdr.cdth_trk1].cdte_addr.msf.second);
    
    return tempstr;
}

/************************************************************************/
/* Procedure:  check_disc 
 * Purpose:    to check the disc for mounted, data, and no disc
 * 
 * Inputs:     program name, CD file des., CD file des., pointer to flag,
 *             device of CDROM
 * Outputs:    debug to stdout, error to stdout
 * Returns:    pointer to HW structure
 * Notes:  
 *   1.
 */
/************************************************************************/
cdhw_t *check_disc (char *progname, int cdfile, int docr, int *piF,
		    char *cddevice)
{
    cdhw_t *hw;

    /* get hardware info */
    hw = read_hw(progname, cdfile, cddevice);

    /* check if NO CDROM */
    if ((hw->iStatus == CD_STATUS_NO_CD) ||
	(hw->iStatus == CD_STATUS_REMOVED)) {
	printf ("nodisc");
	DoCr (docr);
	*piF = FALSE;
    } else {

	/* check if mounted */
	if (hw->iStatus == CD_STATUS_MOUNTED) {
	    printf ("mounted");
	    DoCr (docr);
	    *piF = FALSE;
	} else {
	    
	    /* if data disc, print msg */
	    if (hw->iStatus == CD_STATUS_DATA_DISC) {
		printf ("datadisc %d track%c", hw->iData, (hw->iData>1)?'s':' ');
		DoCr (docr);
		*piF = FALSE;
		
	    } else {
		*piF = TRUE;
	    }
	}
    }
    return (hw);
}

/************************************************************************/
/* Procedure:  do_info
 * Purpose:    to print info on a CD-ROM
 * 
 * Inputs:     name of program, cd file descriptior, info level, CR flag,
 *             device of CDROM
 * Outputs:    to stdout/stderr
 * Returns:    n/a
 * Notes:  
 *   1.
 */
/************************************************************************/
void do_info(char *progname, int cdfile, int info, 
	     char *cddevice, int docr)
{
    cdhw_t *hw = NULL;
    int    iTemp;

#ifdef DEBUG1
    fprintf (stderr,"do_info:  skipped, progname=%s, cdfile=%d, info=%d, hw=0x%lx\n",
	     progname, cdfile, info, hw);
    return;
#endif
    hw = check_disc (progname, cdfile, docr, &iTemp, cddevice);
    if (iTemp == TRUE) {

/*     hw = read_hw(progname, cdfile, cd_device); */
/*     if ((hw->iStatus == CD_STATUS_NO_CD) || */
/* 	(hw->iStatus == CD_STATUS_REMOVED)) { */
/* 	printf ("nodisc"); */
/* 	DoCr (docr); */
/*     } else if (hw->iStatus == CD_STATUS_MOUNTED) { */
/* 	printf ("mounted"); */
/* 	DoCr (docr); */
/*     } else { */
	
	
#ifdef DEBUG
	fprintf (stderr,"do_info:  called, progname=%s, cdfile=%d, info=%d, hw=%lx\n",
		 progname, cdfile, info, (long)hw);
#endif
	
	/* audio status */
	if ((info == I_AUDIOSTATUS) || (info == I_DISP_ALL)) {
	    switch (hw->subchnl.cdsc_audiostatus) {
	    case CDROM_AUDIO_INVALID:
		printf("invalid ");
		break;
	    case CDROM_AUDIO_PLAY:
		printf("play ");
		break;
	    case CDROM_AUDIO_PAUSED:
		printf("paused ");
		break;
	    case CDROM_AUDIO_COMPLETED:
		printf("completed ");
		break;
	    case CDROM_AUDIO_ERROR:
		printf("error ");
		break;
	    case CDROM_AUDIO_NO_STATUS:
		printf("no-status ");
		break;
	    default:
		printf("unknown=%i ", hw->subchnl.cdsc_audiostatus);
		break;
	    }
	    DoCr (docr);
	} 
	
	/* track info */
	if ((info == I_TRACK) || (info == I_DISP_ALL)) {
	    printf("%u ",( (hw->subchnl.cdsc_audiostatus == CDROM_AUDIO_PLAY) ||
			   (hw->subchnl.cdsc_audiostatus == CDROM_AUDIO_PAUSED) )
		   ? hw->subchnl.cdsc_trk : 0);
	    DoCr (docr);
	}
	/* absolute time, i.e., since CD start */
	if ((info == I_DISP_ABS) || (info == I_DISP_ALL)) {
	    printf("%2u:%02u.%02u ",
		   hw->subchnl.cdsc_absaddr.msf.minute,
		   hw->subchnl.cdsc_absaddr.msf.second,
		   hw->subchnl.cdsc_absaddr.msf.frame);
	    DoCr (docr);
	}
	
	/* relative time, i.e., since track start */
	if ((info == I_DISP_REL) || (info == I_DISP_ALL)) {
	    printf("%2u:%02u.%02u ",
		   hw->subchnl.cdsc_reladdr.msf.minute,
		   hw->subchnl.cdsc_reladdr.msf.second,
		   hw->subchnl.cdsc_reladdr.msf.frame);
	    DoCr (docr);
	}
    }
#ifdef DEBUG
    fprintf (stderr,"do_info:  finished\n");
#endif
    /* free the hardware buffer.... */
    free_hw_buf(hw);
}

/************************************************************************/
/* Procedure:  do_dir
 * Purpose:    to print the directory of a CD-ROM
 * 
 * Inputs:     prog name, CD file des., format info, database, CR flag,
 *             device of CDROm
 * Outputs:    to stdout/stderr
 * Returns:    n/a
 * Notes:  
 *   1.
 */
/************************************************************************/
void do_dir(char *progname, int cdfile, int p_format, int usedb, 
	    char *cddevice, int docr)
{
    cdhw_t *hw;
    char *tracks_buffer;
    cd_t *cd;
    int  iTemp;
    
    int i;  /* generic index variable */

#ifdef DEBUG
    fprintf (stderr,"do_dir:  called, progname=%s, cdfile=%d, pfmt=%d, usedb=%d\n",
	     progname, cdfile, p_format, usedb);
#endif

    hw = check_disc (progname, cdfile, docr, &iTemp, cddevice);
    if (iTemp == TRUE) {

/*     hw = read_hw(progname, cdfile, cd_device); */
/*     if ((hw->iStatus == CD_STATUS_NO_CD) || */
/* 	(hw->iStatus == CD_STATUS_REMOVED)) { */
/* 	printf ("nodisc"); */
/* 	DoCr (docr); */
/*     } else if (hw->iStatus == CD_STATUS_MOUNTED) { */
/* 	printf ("mounted"); */
/* 	DoCr (docr); */
/*     } else { */

	tracks_buffer = tracks_line(hw);
	cd = read_db(tracks_buffer, usedb);
	
	/* print name & artist info */
	if (p_format == P_TEMPL) {
	    for (i=0; i<strlen(tracks_buffer); i++) /* print "tracks" line */
		putchar(tracks_buffer[i]);
	    DoCr(docr);
	    
	    if (cd->cdname[0])
		printf("cdname %s", cd->cdname);
	    else {
		printf("cdname ");
		DoCr(docr);
	    }
	    
	    if (cd->artist[0])
		printf("artist %s", cd->artist);
	    else {
		printf("artist ");
		DoCr(docr);
	    }
	    
	} else if (p_format != P_RAW) {
	    if (cd->artist[0])
		for (i=0; i<strlen(cd->artist)-1; i++) 
		    putchar(cd->artist[i]);
	    if (cd->cdname[0] && cd->artist[0])
		printf(" - ");
	    if (cd->cdname[0])
		for (i=0; i<strlen(cd->cdname)-1; i++) 
		    putchar(cd->cdname[i]);
	    if (!cd->artist[0] && !cd->cdname[0])
		printf("unknown cd");
	    if (p_format == P_LONG)
		printf(" - %2u:%02u in %u tracks",
		       hw->tocentries[hw->tochdr.cdth_trk1].cdte_addr.msf.minute,
		       hw->tocentries[hw->tochdr.cdth_trk1].cdte_addr.msf.second,
		       hw->tochdr.cdth_trk1);
	    DoCr(docr);
	}
	
	/* print tracks */
	switch(p_format) {
	case P_RAW:
	    printf("%2u-%2u",hw->tochdr.cdth_trk0, hw->tochdr.cdth_trk1);
	    DoCr(docr);
	    for (i = hw->tochdr.cdth_trk0; i<=hw->tochdr.cdth_trk1+1; i++) {
		printf("%2X %02u:%02u.%02u %c",
		       (i<hw->tochdr.cdth_trk1+1) ? ((i/10)*16+i%10) : 0xaa,
		       hw->tocentries[i-1].cdte_addr.msf.minute,
		       hw->tocentries[i-1].cdte_addr.msf.second,
		       hw->tocentries[i-1].cdte_addr.msf.frame,
		       (hw->tocentries[i-1].cdte_ctrl == 
			CDROM_DATA_TRACK) ? 'D' : 'A');
		DoCr(docr);
	    }
	    break;
	case P_LONG:
	    for (i = hw->tochdr.cdth_trk0; i<=hw->tochdr.cdth_trk1; i++) {
		int delfrms 
		    = ( hw->tocentries[i].cdte_addr.msf.minute
			-   hw->tocentries[i-1].cdte_addr.msf.minute ) * 4500
		    + ( hw->tocentries[i].cdte_addr.msf.second
			-   hw->tocentries[i-1].cdte_addr.msf.second ) * 75
		    + ( hw->tocentries[i].cdte_addr.msf.frame
			-   hw->tocentries[i-1].cdte_addr.msf.frame );
		printf(" %2u:%02u.%02u %2u ", 
		       delfrms/4500,(delfrms%4500)/75,delfrms%75, i);
		if (hw->tocentries[i-1].cdte_ctrl == CDROM_DATA_TRACK)
		    printf("[DATA] ");
		if (i == hw->subchnl.cdsc_trk) {
		    if (hw->subchnl.cdsc_audiostatus == CDROM_AUDIO_PLAY)
			printf("[PLAYING] ");
		    else if (hw->subchnl.cdsc_audiostatus == CDROM_AUDIO_PAUSED)
			printf("[PAUSED] ");
		}
		if (usedb && cd->track_names[i-1][0] != 0)
		    printf("%s",cd->track_names[i-1]);
		else {
		    DoCr(docr);
		}
	    }
	    break; /* P_LONG */
	    
	case P_QUICK:
	    switch(hw->subchnl.cdsc_audiostatus) {
	    case CDROM_AUDIO_PLAY: 
		printf(" - ");
		if (cd->track_names[hw->subchnl.cdsc_trk-1][0]) {
		    for (i=0; 
			 i<strlen(cd->track_names[hw->subchnl.cdsc_trk - 1])-1;
			 i++)
			putchar(cd->track_names[hw->subchnl.cdsc_trk-1][i]);
		    printf(" ");
		}
		printf("[%u]", hw->subchnl.cdsc_trk);
		DoCr(docr);
		break;
	    case CDROM_AUDIO_PAUSED:
		printf(" - paused on ");
		if (cd->track_names[hw->subchnl.cdsc_trk-1][0]) {
		    for (i=0; 
			 i<strlen(cd->track_names[hw->subchnl.cdsc_trk - 1])-1; i++)
			putchar(cd->track_names[hw->subchnl.cdsc_trk-1][i]);
		    printf(" ");
		}
		printf("[%u]", hw->subchnl.cdsc_trk);
		DoCr(docr);
		break;
	    default:
		DoCr(docr);
	    }
	    break; /* P_QUICK */
	    
	case P_TEMPL:
	    for (i = hw->tochdr.cdth_trk0; i<=hw->tochdr.cdth_trk1; i++)
		if (usedb && cd->track_names[i-1][0] != 0)
		    printf("track %s", cd->track_names[i-1]);
		else {
		    printf("track ");
		    DoCr(docr);
		}
	    break; /* P_TEMPL */
	    
	} /* switch */
    }
    free_hw_buf(hw);
}
