/*
cdcd - Command Driven CD player
Copyright (C)1998-99 Tony Arcieri

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <config.h>
#include <stdlib.h>
#include <time.h>
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>

#ifdef HAVE_LIBCDAUDIO
#include <libcdaudio.h>
#else
#include <cdaudio.h>
#endif

#ifdef HAVE_READLINE_H
#include <readline.h>
#define HAVE_READLINE 1
#endif

#ifdef HAVE_HISTORY_H
#include <history.h>
#define HAVE_HISTORY 1
#endif

#ifdef HAVE_READLINE_READLINE_H
#include <readline/readline.h>
#define HAVE_READLINE 1
#endif

#ifdef HAVE_READLINE_HISTORY_H
#include <readline/history.h>
#define HAVE_HISTORY 1
#endif

#define CDCD_CDDB_HOST "freedb.freedb.org"
#define CDCD_CDINDEX_HOST "www.cdindex.org"

#define MAX_ARGS 16

void get_input_text(char *buffer, char *prompt, int len);

/* Start track in playlist */
int play_track = 1;

/* End track in playlist */
int end_track = 0;

/* CDDB messages, a.k.a. have cdcd be in charge of CDDB operations */
int verbosity = 1;

/* CDDB timestamp */
int timestamp = 0;

/* Timestamped discid */
int timestamped_discid = 0;

/* CD-ROM device */
int cd_desc;

/* Device name */
char device[256];

struct cdcdrc {
   char device[256];
   int verbosity;
};

void cdcdrc_write(struct cdcdrc *conf);

void
signal_handler(int signal)
{
   switch(signal) {
    case SIGSEGV:
      puts("\nSIGSEGV caught: Segment violation");
      puts("\nPlease report what you were doing\nwhen this occoured to:\nbascule@inferno.tusculum.edu\n");
      abort();
    case SIGTERM:
      puts("\nSIGTERM caught, exiting.");
      exit(0);
   }
}

/* Change a whole string to lower case */
int
#if __STDC__
strnlower(char *dest, const char *src, int len)
#else
strnlower(dest, src, len)
   char *dest;
   const char *src;
   int len;
#endif
{
   int index;

   for(index = 0; index < len; index++) {
      dest[index] = tolower(src[index]);
      if(src[index] == '\0')
	return index;
   }
   
   return index;
}

/* Case insensative substring search with size limits*/
char
#if __STDC__
*strncasestr(const char *haystack, int haystacklen, const char *needle, int needlelen)
#else
*strncasestr(haystack, haystacklen, needle, needlelen)
   const char *haystack;
   int haystacklen;
   const char *needle;
   int needlelen;
#endif
{
   char haystacklower[256], needlelower[256];
   
   strnlower(haystacklower, haystack, haystacklen);
   strnlower(needlelower, needle, needlelen);
   
   return strstr(haystacklower, needlelower);
}

/* Case insensitive substring search */
char
#if __STDC__
*strcasestr(const char *haystack, const char *needle)
#else
*strcasestr(haystack, needle)
   const char *haystack;
   const char *needle;
#endif
{
   return strncasestr(haystack, strlen(haystack), needle, strlen(needle));
}

int
#if __STDC__
cdcdrc_data_mod(char *var, char *value, struct cdcdrc *conf)
#else
cdcdrc_data_mod(var, value, conf)
   char *var;
   char *value;
   struct cdcdrc *conf;
#endif
{
    if(value == NULL)
      value = "";
	
    if(strcasecmp(var, "DEVICE") == 0) {
      if(strlen(value) < 1)
	strncpy(conf->device, DEFAULT_DEVICE, 256);
      else
	strncpy(conf->device, value, 256);
    }
    if(strcasecmp(var, "VERBOSE") == 0) {
      if(strcasecmp(value, "ON") == 0)
	 conf->verbosity = 1;
      else
	 conf->verbosity = 0;
    }
	
    return 0;
}

void
#if __STDC__
cdcdrc_manual_configuration(struct cdcdrc *conf)
#else
cdcdrc_manual_configuration(conf)
   struct cdcdrc *conf;
#endif
{
   struct cddb_conf cddbconf;
   struct cddb_host proxy_host;
   struct cddb_serverlist list;
   struct cdcdrc writeconf;
   int useproxy = 0;
   char question_buffer[256], inbuffer[256], *proxy;
   
   conf->verbosity = 1;
   
   printf("Welcome to %s %s.\n\n", PACKAGE, VERSION);
   puts("You will now be asked a series of questions which will configure\ncdcd for your platform.  The default answers are given\nin brackets.\n");

   snprintf(question_buffer, 256, "What is the name of your CD-ROM device? [%s] ", DEFAULT_DEVICE);
   get_input_text(inbuffer, question_buffer, 256);
   
   if(strlen(inbuffer) > 0)
     strncpy(conf->device, inbuffer, 256);
   else
     strncpy(conf->device, DEFAULT_DEVICE, 256);
   
   get_input_text(inbuffer, "Are you connected to a network (y/n)? [y] ", 256);
   if(inbuffer[0] == 'n') {
      cddbconf.conf_access = CDDB_ACCESS_LOCAL;
      cddb_write_serverlist(cddbconf, list, proxy_host.host_server);
      return;
   } else
     cddbconf.conf_access = CDDB_ACCESS_REMOTE;

   puts("\nThere are two methods of retrieving CD track information, CDDB and CD Index.\nUsing the CD Index will be somewhat faster, but does not yet facilitate inexact\nmatching.  Both methods may be used through HTTP proxy servers.\n\nChoose a method:\n1) CDDB\n2) CD Index\n");
   get_input_text(inbuffer, "Which method? [1] ", 256);
   if(inbuffer[0] == '2') {
      list.list_len = 1;
      strncpy(list.list_host[0].host_server.server_name, "www.cdindex.org", 256);
      strncpy(list.list_host[0].host_addressing, "cgi-bin/cdi/get.pl", 256);
      list.list_host[0].host_server.server_port = 80;
      list.list_host[0].host_protocol = CDINDEX_MODE_HTTP;
   } else {
      list.list_len = 1;
      strncpy(list.list_host[0].host_server.server_name, "www.freedb.org", 256);
      strncpy(list.list_host[0].host_addressing, "cgi-bin/cddb.cgi", 256);
      list.list_host[0].host_server.server_port = 80;
      list.list_host[0].host_protocol = CDDB_MODE_HTTP;
   }
   
   if((proxy = getenv("http_proxy")) != NULL) {
      cddb_process_url(&proxy_host, proxy);
      printf("\nHTTP proxy server found: http://%s:%d/\n\n", proxy_host.host_server.server_name, proxy_host.host_server.server_port);
      get_input_text(inbuffer, "Use it? [y] ", 256);
      if(inbuffer[0] == 'n')
	useproxy = 1;
      else
	cddbconf.conf_proxy = 1;
   }
   
   if(proxy == NULL || useproxy == 1) {
      get_input_text(inbuffer, "Use a HTTP proxy server? [n] ", 256);
      if(inbuffer[0] == 'y') {
	 get_input_text(inbuffer, "Enter proxy server URL: ", 256);
	 cddb_process_url(&proxy_host, inbuffer);
	 cddbconf.conf_proxy = 1;
      }
   }
   
   putchar('\n');
   
   cddb_write_serverlist(cddbconf, list, proxy_host.host_server);
   memcpy(&writeconf, conf, sizeof(struct cdcdrc));
   cdcdrc_write(&writeconf);
   
   return;
}

int
#if __STDC__
cdcdrc_process_line(char *line, struct cdcdrc *conf)
#else
cdcdrc_process_line(line, conf)
   char *line;
   struct cdcdrc *conf;
#endif
{
   char *var, *value;
    
   if(strchr(line, '=') == NULL)
     return 0;
    
   line[strlen(line) - 1] = '\0';
	
   var = strtok(line, "=");
   if(var == NULL)
     return 0;
   value = strtok(NULL, "=");
   if(cdcdrc_data_mod(var, value, conf) < 0)
     return -1;
	
   return 0;
}

/* Read .cdcdrc */
int
#if __STDC__
cdcdrc_read(struct cdcdrc *conf)
#else
cdcdrc_read(conf)
   struct cdcdrc *conf;
#endif
{
   FILE *cdcdrcfile;
   int index;
   char inbuffer[256];
   char localconfpath[256];
   struct stat st;
  
   if(getenv("HOME") == NULL) {
      puts("$HOME is not set!");
      return -1;
   }
   
   snprintf(localconfpath, 256, "%s/.cdcdrc", getenv("HOME"));
   if(stat(localconfpath, &st) < 0) {
      cdcdrc_manual_configuration(conf);
      return 0;
   }
	
   cdcdrcfile = fopen(localconfpath, "r");
   fgets(inbuffer, 256, cdcdrcfile);
   if(strncmp(inbuffer, "# .cdcdrc revision 3", 20) != 0) {
      unlink(localconfpath);
      fclose(cdcdrcfile);
      cdcdrc_manual_configuration(conf);
      return 0;
   }
	
   while(!feof(cdcdrcfile)) {
      fgets(inbuffer, 256, cdcdrcfile);
      inbuffer[255] = '\0';
       
      for(index = 0; index < strlen(inbuffer); index++) 
	if(inbuffer[index] == '#') {
	   inbuffer[index] = '\0';
	   break;
	}
	   
      if(cdcdrc_process_line(inbuffer, conf) < 0)
	return -1;
   }
	
   fclose(cdcdrcfile);
	
   return 0;
}

/* Write .cdcdrc */
void
#if __STDC__
cdcdrc_write(struct cdcdrc *conf)
#else
cdcdrc_write(conf)
   struct cdcdrc *conf;
#endif
{
   FILE *cdcdrcfile;
   char outbuffer[256];
   char localconfpath[256];
	
   if(getenv("HOME") == NULL) {
      puts("$HOME is not set!");
      return;
   }
   
   snprintf(localconfpath, 256, "%s/.cdcdrc", getenv("HOME"));
   cdcdrcfile = fopen(localconfpath, "w");
	
   fputs("# .cdcdrc revision 3\n\n", cdcdrcfile);
   fprintf(cdcdrcfile, "VERBOSE=%s\n", conf->verbosity ? "ON" : "OFF");
   fprintf(cdcdrcfile, "DEVICE=%s\n", conf->device);
	
   fclose(cdcdrcfile);
}

/* Universal text input */
void
#if __STDC__
get_input_text(char *buffer, char *prompt, int len)
#else
get_input_text(buffer, prompt, len)
   char *buffer;
   char *prompt;
   int len;
#endif
{
#if HAVE_READLINE
   char *inputbuffer;
   inputbuffer = readline(prompt);
   if(inputbuffer == NULL) exit(0);
#endif
#if HAVE_HISTORY
   if(*inputbuffer) add_history(inputbuffer); 
#endif
   memset(buffer, '\0', len);
#if HAVE_READLINE
   strncpy(buffer, inputbuffer, len);
   free(inputbuffer);
#else
   printf(prompt);
   fgets(buffer, len, stdin);
   if(feof(stdin))
     exit(0);
   buffer[strlen(buffer) - 1] = '\0';
#endif
}

int
cddb_get_siteslist(struct cddb_serverlist *list, struct cddb_host host)
{
	int sock;
	struct cddb_hello hello;
	char http_string[512];
	
	strncpy(hello.hello_program, PACKAGE, 256);
	strncpy(hello.hello_version, VERSION, 256);
	
	switch(host.host_protocol) {
	  case CDDB_MODE_CDDBP:	
	    printf("Connecting to cddbp://%s:%d/\n", host.host_server.server_name, host.host_server.server_port);
	    if((sock = cddb_connect_server(host, NULL, hello)) < 0) {
	      perror("Couldn't connect to CDDB server!");
	      return -1;
	    }
	    puts("Connection established.\nDownloading server list...");
	    if(cddb_sites(sock, CDDB_MODE_CDDBP, list) < 0) {
	      fprintf(stderr, "CDDB error: %s\n", cddb_message);
	      return -1;
	    }
	    puts("Done.");
	    break;
	  case CDDB_MODE_HTTP:
	    printf("Connecting to http://%s:%d/%s\n", host.host_server.server_name, host.host_server.server_port, host.host_addressing);
	    if((sock = cddb_connect_server(host, NULL, hello, http_string, 512)) < 0) {
	      perror("Couldn't connect to CDDB server!");
	      return -1;
	    }
	    puts("Connection established.\nDownloading server list...");
	    if(cddb_sites(sock, CDDB_MODE_HTTP, list, http_string) < 0) {
	      fprintf(stderr, "CDDB error: %s\n", cddb_message);
	      return -1;
	    }
	    puts("Done.");
	}
   
   return 0;
}

int
#if __STDC__
cdcd_cd_stat(int cd_desc, struct disc_info *disc)
#else
cdcd_cd_stat(cd_desc, disc)
   int cd_desc;
   struct disc_info *disc;
#endif
{
   cd_stat(cd_desc, disc);
   if(!disc->disc_present) {
      cd_close(cd_desc);
      cd_stat(cd_desc, disc);
      if(!disc->disc_present) {
	 puts("No disc in drive");
	 return -1;
      }
   }
    
   return 0;
}
		  
   
/* CDDB processor */
void
#if __STDC__
cddb_lookup(int cd_desc, struct disc_data *data)
#else
cddb_lookup(cd_desc, data)
   int cd_desc;
   struct disc_data *data;
#endif
{
   int index, serverindex, selection, sock = -1;
   struct disc_info disc;
   struct disc_data outdata;
   struct cddb_conf conf;
   struct cddb_serverlist list;
   struct cddb_server *proxy;
   struct cddb_entry entry;
   struct cddb_hello hello;
   struct cddb_query query;
   char inbuffer[256], http_string[512], discid[CDINDEX_ID_SIZE];
   
   if(cdcd_cd_stat(cd_desc, &disc) < 0)
     return;
   
   if(!verbosity)
     cddb_read_disc_data(cd_desc, data);
   else {
      cddb_stat_disc_data(cd_desc, &entry);
      
      if(entry.entry_present) {
	 if(entry.entry_timestamp == timestamp && entry.entry_id == timestamped_discid)
	   return;
       
	 cddb_read_disc_data(cd_desc, data);
	 timestamp = entry.entry_timestamp;
	 timestamped_discid = entry.entry_id;
      } else {
	 proxy = (struct cddb_server *)malloc(sizeof(struct cddb_server));
	 cddb_read_serverlist(&conf, &list, proxy);
	 if(conf.conf_access == CDDB_ACCESS_LOCAL) {

	    free(proxy);
	    cddb_generate_unknown_entry(cd_desc, data);
	    return;
	 }
	 if(list.list_len < 1) {
	    puts("Couldn't perform CDDB/CD Index lookup: No servers available!");
	    return;
	 }
	 if(!conf.conf_proxy) {
	    free(proxy);
	    proxy = NULL;
	 } else 
	    printf("Using proxy http://%s:%d/\n", proxy->server_name, proxy->server_port);	
	      		
	 strncpy(hello.hello_program, PACKAGE, 256);
	 strncpy(hello.hello_version, VERSION, 256);
	 
	 serverindex = 0;
	      
         do {
	    switch(list.list_host[serverindex].host_protocol) {
	     case CDDB_MODE_CDDBP:
	       printf("Trying CDDB server cddbp://%s:%d/\n", list.list_host[serverindex].host_server.server_name, list.list_host[serverindex].host_server.server_port);
	       sock = cddb_connect_server(list.list_host[serverindex++], proxy, hello);
	       break;
	     case CDDB_MODE_HTTP:
	       printf("Trying CDDB server http://%s:%d/%s\n", list.list_host[serverindex].host_server.server_name, list.list_host[serverindex].host_server.server_port, list.list_host[serverindex].host_addressing);
	       sock = cddb_connect_server(list.list_host[serverindex++], proxy, hello, http_string, 512);
	       break;
	     case CDINDEX_MODE_HTTP:
	       printf("Trying CD Index server http://%s:%d/%s\n", list.list_host[serverindex].host_server.server_name, list.list_host[serverindex].host_server.server_port, list.list_host[serverindex].host_addressing);
	       sock = cdindex_connect_server(list.list_host[serverindex++], proxy, http_string, 512);
	       break;
	     default:
	       puts("Invalid protocol selected!");
	       return;
	    }
	    if(sock == -1) printf("Connection error: %s\n", cddb_message);
	 } while(serverindex < list.list_len && sock == -1);
	 
	 if(sock == -1) {
	    puts("Could not establish connection with any CDDB servers!");
	    if(conf.conf_proxy) free(proxy);
	    cddb_generate_unknown_entry(cd_desc, data);
	    return;
	 }
	 serverindex--;
         puts("Connection established.");
         
	 switch(list.list_host[serverindex].host_protocol) {
	  case CDDB_MODE_CDDBP:
	    printf("Retrieving information on %02lx.\n", cddb_discid(cd_desc));
            if(cddb_query(cd_desc, sock, CDDB_MODE_CDDBP, &query) < 0) { 
	       fprintf(stderr, "CDDB query error: %s", cddb_message);
	       if(conf.conf_proxy) free(proxy);
	       cddb_generate_unknown_entry(cd_desc, data);
	       return;
	    }
            break;
	  case CDDB_MODE_HTTP:
	    printf("Retrieving information on %02lx.\n", cddb_discid(cd_desc));
	    if(cddb_query(cd_desc, sock, CDDB_MODE_HTTP, &query, http_string) < 0) {
	       fprintf(stderr, "CDDB query error: %s", cddb_message);
	       if(conf.conf_proxy) free(proxy);
	       cddb_generate_unknown_entry(cd_desc, data);
	       return;
	    }
	    shutdown(sock, 2);
	    close(sock);
		 
	    if((sock = cddb_connect_server(list.list_host[serverindex], proxy, hello, http_string, 512)) < 0) {
	       perror("HTTP server reconnection error");
	       if(conf.conf_proxy) free(proxy);
	       cddb_generate_unknown_entry(cd_desc, data);
	       return;
	    }
	    break;
	  case CDINDEX_MODE_HTTP:
	    cdindex_discid(cd_desc, discid, CDINDEX_ID_SIZE);
	    printf("Retrieving information on %s.\n", discid);
	    if(cdindex_read(cd_desc, sock, data, http_string) < 0) {
	       printf("No match for %s.\n", discid);
	       if(conf.conf_proxy) free(proxy);
	       cddb_generate_unknown_entry(cd_desc, data);
	       return;
	    }
	    printf("Match for %s: %s / %s\nDownloading data...\n", discid, data->data_artist, data->data_title);
	    
	    memcpy(&outdata, data, sizeof(struct disc_data));
	    if(cddb_write_disc_data(cd_desc, outdata) < 0)
	      fprintf(stderr, "CDDB write error: %s", cddb_message);
	    return;
	 }
	 
	 if(conf.conf_proxy) free(proxy);
	 
	 if(list.list_host[serverindex].host_protocol == CDINDEX_MODE_HTTP);
	 
         switch(query.query_match) {
          case QUERY_EXACT:
	    if(strlen(query.query_list[0].list_artist) > 0)
	      printf("Match for %02lx: %s / %s\nDownloading data...\n", cddb_discid(cd_desc), query.query_list[0].list_artist, query.query_list[0].list_title);
	    else
	      printf("Match for %02lx: %s\nDownloading data...\n", cddb_discid(cd_desc), query.query_list[0].list_title);
	    entry.entry_genre = query.query_list[0].list_genre;
	    entry.entry_id = query.query_list[0].list_id;
	    switch(list.list_host[serverindex].host_protocol) {
	      case CDDB_MODE_CDDBP:
		if(cddb_read(cd_desc, sock, CDDB_MODE_CDDBP, entry, data) < 0) {
		   perror("CDDB read error");
		   cddb_generate_unknown_entry(cd_desc, data);
		   return;
		}	
		cddb_quit(sock);
		break;
	      case CDDB_MODE_HTTP:
		if(cddb_read(cd_desc, sock, CDDB_MODE_HTTP, entry, data, http_string) < 0) {
		   perror("CDDB read error");
		   cddb_generate_unknown_entry(cd_desc, data);
		   return;
		}
		    
		shutdown(sock, 2);
		close(sock);
		break;
	    }
	    break;
          case QUERY_INEXACT:
	    printf("Inexact match for %02lx.\n", cddb_discid(cd_desc));
	    puts("Please choose from the following inexact matches:");
	    for(index = 0; index < query.query_matches; index++)
	      if(strlen(query.query_list[index].list_artist) < 1)
		printf("%d: %s\n", index + 1, query.query_list[index].list_title);
	      else
	        printf("%d: %s / %s\n", index + 1, query.query_list[index].list_artist, query.query_list[index].list_title);
	    printf("%d: None of the above.\n", index + 1);
	    printf("> ");
	    fgets(inbuffer, 256, stdin);
	    selection = strtol(inbuffer, NULL, 10);
	    if(selection > 0 && selection <= query.query_matches) {
	       entry.entry_genre = query.query_list[selection - 1].list_genre;
	       entry.entry_id = query.query_list[selection - 1].list_id;
	       puts("Downloading data...");
	       switch(list.list_host[serverindex].host_protocol) {
	         case CDDB_MODE_CDDBP:
		   if(cddb_read(cd_desc, sock, CDDB_MODE_CDDBP, entry, data) < 0) {
		      perror("CDDB read error");
		      cddb_generate_unknown_entry(cd_desc, data);
		      return;
		   }
		   cddb_quit(sock);
		   break;
	         case CDDB_MODE_HTTP:
		   if(cddb_read(cd_desc, sock, CDDB_MODE_HTTP, entry, data, http_string) < 0) {
		      perror("CDDB read error");
		      cddb_generate_unknown_entry(cd_desc, data);
		      return;
		   }
		   shutdown(sock, 2);
		   close(sock);
		   break;
	       }
	       break;
	    }
          case QUERY_NOMATCH:
	    printf("No match for %02lx.\n", cddb_discid(cd_desc));
	    cddb_generate_unknown_entry(cd_desc, data);
         }
         close(sock);
	 
	 memcpy(&outdata, data, sizeof(struct disc_data));
         if(cddb_write_disc_data(cd_desc, outdata) < 0)
	   fprintf(stderr, "CDDB write error: %s\n", cddb_message);
      }
   }
   return; 
}

void
#if __STDC__
cddb_submit(int cd_desc, char *email_address)
#else
cddb_submit(cd_desc, email_address)
   int cd_desc;
   char *email_address;
#endif
{
   int index = 0;
   struct cddb_conf conf;
   struct cddb_serverlist list;
   struct disc_data data;
   struct cddb_server *proxy;
    
   proxy = (struct cddb_server *)malloc(sizeof(struct cddb_server));
   cddb_read_disc_data(cd_desc, &data);
   cddb_read_serverlist(&conf, &list, proxy);
   if(conf.conf_access == CDDB_ACCESS_LOCAL) {
      free(proxy);
      puts("Cannot complete CDDB submission in local mode");
      return;
   }
   if(!conf.conf_proxy) {
      free(proxy);
      proxy = NULL;
   }
	
   while(list.list_host[index].host_protocol != CDINDEX_MODE_HTTP && list.list_host[index].host_protocol != CDDB_MODE_HTTP && index < list.list_len)
      index++;
	
   if(index >= list.list_len) {
      puts("Could not locate any HTTP servers.  Try running 'sites refresh'.");
      if(conf.conf_proxy)
	free(proxy);
      return;
   }

   printf("Submitting to %s server http://%s:%d%s\n", (list.list_host[index].host_protocol == CDINDEX_MODE_HTTP) ? "CD Index" : "CDDB", list.list_host[index].host_server.server_name, list.list_host[index].host_server.server_port, (list.list_host[index].host_protocol == CDINDEX_MODE_HTTP) ? CDINDEX_SUBMIT_CGI : HTTP_SUBMIT_CGI);
   switch(list.list_host[index].host_protocol) {
    case CDINDEX_MODE_HTTP:
      cdindex_write_disc_data(cd_desc, data);
      if(cdindex_http_submit(cd_desc, list.list_host[index], proxy) < 0)
	fprintf(stderr, "Submission error: %s\n", cddb_message);
      else
	puts("Success!");
      
      break;
    case CDDB_MODE_HTTP:
      if(cddb_http_submit(cd_desc, list.list_host[index], proxy, email_address) < 0)
	fprintf(stderr, "Submission error: %s\n", cddb_message);
      else
	puts("Success!");
   }
}

#ifdef HAVE_READLINE
char
*command_matcher(char *text, int state)
{
   int list_index, len;
   char *name, *commands[30] = { "play", "stop", "open", "close", "pause", "resume", "ff", "rew", "sites", "next", "prev", "getvol", "setvol", "status", "rndplay", "slot", "list", "submit", "info", "tracks", "edit", "ext", "refresh", "device", "verbose", "quit", "access", "help", NULL};

   if(strlen(text) < 1)
     return NULL;
   
   if(!state) {
      list_index = 0;
      len = strlen(text);
   }
  
   while((name = commands[list_index]) != NULL) {
      list_index++;
      
      if(strncmp(name, text, len) == 0)
	return strdup(name);
   };
   
   return NULL;
}

char
*trackname_matcher(char *text, int state)
{
   int list_index, len, has_quote = 0;
   char name[256], *match_string;
   struct disc_info disc;
   struct disc_data data;
   
   if(strlen(text) <= 0)
     return NULL;
   
   cdcd_cd_stat(cd_desc, &disc);
   cddb_lookup(cd_desc, &data);
   
   if(!state) {
      list_index = 0;
      len = strlen(text);
   }
   
   if(text[0] == '\"') {
      match_string = text + 1;
      has_quote = 1;
      if(len > 0)
	len--;
   } else if(text[0] == '\'') {
      match_string = text + 1;
      has_quote = 2;
      if(len > 0)
	len--;
   } else if(text[0] == '`') {
      match_string = text + 1;
      has_quote = 3;
      if(len > 0)
	len--;
   } else {
      match_string = text;
      has_quote = 0;
   }
   
   while(list_index < disc.disc_total_tracks) {
      if(strncasecmp(data.data_track[list_index++].track_name, match_string, len) == 0) {
	 if(!has_quote)
	   snprintf(name, 256, "\"%s\"", data.data_track[list_index - 1].track_name);
	 else if(has_quote == 1)
	   snprintf(name, 256, "\"%s", data.data_track[list_index - 1].track_name);
	 else if(has_quote == 2)
	   snprintf(name, 256, "\'%s", data.data_track[list_index - 1].track_name);
	 else
	   snprintf(name, 256, "\'%s\'", data.data_track[list_index - 1].track_name);
	 
	 return strdup(name);
      }
   }
   
   return NULL;
}

char **cdcd_completion(char *text, int start, int end)
{
   char **matches = (char **)NULL;

   if(start > 0) {
      timestamp = 0;
      timestamped_discid = 0;
      
      matches = completion_matches(text, trackname_matcher);
      
      timestamp = 0;
      timestamped_discid = 0;
   }
   
   return matches;
}
#endif

/* One giant function to process the commands */
void
#if __STDC__
proc_command(char *command, int argc, char argv[MAX_ARGS][256])
#else
proc_command(command, argc, argv)
   char *command;
   int argc;
   char argv[MAX_ARGS][256];
#endif
{
   int index, track, arg, disp_track, disp_disc, cur_disc, location, destination;
   struct cdcdrc cdcdrc_data;
   struct disc_info disc;
   struct disc_data data;
   struct disc_timeval disc_time;
   struct disc_changer changer;
   struct disc_volume vol;
   struct cddb_entry entry;
   struct cddb_host host;
   struct cddb_conf conf;
   struct cddb_serverlist list;
   struct cddb_server proxy;
   char inbuffer[512], dispbuffer[4096], *inptr;
   
   if(strcmp(command, "help") == 0 || strcmp(command, "?") == 0) {
      if(argc == 0) {
	 puts("cdcd commands:");
	 puts(" play\tstop\topen\tclose\tpause\tresume\tff\trew\tsites");
	 puts(" next\tprev\tgetvol\tsetvol\tstatus\trndplay\tslot\tlist\tsubmit");
	 puts(" info\ttracks\tedit\text\trefresh\tdevice\tverbose\tquit\taccess");
	 puts("\nFor more specific help, type \'help <command>\'");
	 return;
      }
      if(strcmp(argv[0], "play") == 0) {
	 puts("Usage: play [starttrackname/track #] [endtrackname/track #] [min:sec]");
	 puts("\nBy default, play starts at the beginning of the CD.  You may specify tracks\nusing either the number of the track or a section of the track's name.  The\ndisc will play from where it starts to the end of the disc or a track you\nspecify.  You may also specify a position within the first track to start.");
      } else if(strcmp(argv[0], "stop") == 0) {
	 puts("Usage: stop");
	 puts("\nstop will stop the CD, if it is playing.");
      } else if(strcmp(argv[0], "open") == 0) {
	 puts("Usage: open (or eject)");
	 puts("\nopen will stop the CD, if it is playing, and open the CD tray.");
      } else if(strcmp(argv[0], "close") == 0) {
	 puts("Usage: close");
	 puts("\nclose will close the CD tray.");
      } else if(strcmp(argv[0], "pause") == 0) {
	 puts("Usage: pause");
	 puts("\npause will pause the CD, if it is playing.");
      } else if(strcmp(argv[0], "resume") == 0) {
	 puts("Usage: resume");
	 puts("\nresume will cause a paused CD to continue playing.");
      } else if(strcmp(argv[0], "ff") == 0) {
	 puts("Usage: ff [min:sec]");
	 puts("\nff will advance the current track 15 seconds, or the specified time.");
      } else if(strcmp(argv[0], "rew") == 0) {
	 puts("Usage: rew [min:sec]");
	 puts("\nrew will go back in the song 15 seconds, or the specified time.");
      } else if(strcmp(argv[0], "rndplay") == 0) {
	 puts("Usage: rndplay");
	 puts("\nrndplay selects a track at random from the disc and plays it.");
      } else if(strcmp(argv[0], "ext") == 0) {
	 puts("Usage: ext [trackname/track#, \"disc\"]");
	 puts("\next displays extended information for a track, if it available, or for the whole\ndisc, by passing the \"disc\" parameter");
      } else if(strcmp(argv[0], "refresh") == 0) {
	 puts("Usage: refresh");
	 puts("\nrefresh will update an entry in the local CDDB cache");
      } else if(strcmp(argv[0], "next") == 0) {
	 puts("Usage: next [min:sec]");
	 puts("\nnext advances to the next song and optionally at the specified position.");
      } else if(strcmp(argv[0], "prev") == 0) {
	 puts("Usage: prev [min:sec]");
	 puts("\nprev returns you to the previous song optionally at the specified position.");
      } else if(strcmp(argv[0], "info") == 0) {
	 puts("Usage: info");
	 puts("\ninfo will display statistics regarding the CD, such as name, artist, number of\ntracks, etc.  When you invoke info for the first time on a particular CD, it\nwill block as it attempts to retrive CD information from the CDDB.");
      } else if(strcmp(argv[0], "tracks") == 0) {
	 puts("Usage: tracks");
	 puts("\ntracks will display album, artist, and all track names.");
      } else if(strcmp(argv[0], "status") == 0) {
	 puts("Usage: status");
	 puts("\nstatus will display the bare-bones information about the status of the CD.\nFor more detailed information, use 'info'.");
      } else if(strcmp(argv[0], "edit") == 0) {
	 puts("Usage: edit [name/artist/genre/track #]");
	 puts("\nedit allows you to edit the disc name database.  After specifing what\nyou wish to edit, you will be allowed to edit the information.");
      } else if(strcmp(argv[0], "getvol") == 0) {
	 puts("Usage: getvol");
	 puts("\ngetvol displays the current volume");
      } else if(strcmp(argv[0], "setvol") == 0) {
	 puts("Usage: setvol left, [right] [rear left] [rear right]");
	 puts("\nsetvol sets the current volume.  Valid volumes: 0 - 255.");
      } else if(strcmp(argv[0], "seldisc") == 0) {
	 puts("Usage: slot disc #");
	 puts("\nslot specifies a CD-ROM changer slot to use.");
      } else if(strcmp(argv[0], "list") == 0) {
	 puts("Usage: list");
	 puts("\nlist displays the entire contents of your CD-ROM changer.");
      } else if(strcmp(argv[0], "device") == 0) {
	 puts("Usage: device [device name]");
	 puts("\ndevice allows you to change the default CD-ROM device (/dev/cdrom).  The new\ndevice name is stored in ~/.cdcdrc and will become the default used by cdcd.");
      } else if(strcmp(argv[0], "verbose") == 0) {
	 puts("Usage: verbose [on/off]");
	 puts("\nToggles whether commands display output when not absolutely nessecary.\nUse \"off\" for automated CDDB operation, or \"on\" for verbose.");
      } else if(strcmp(argv[0], "access") == 0) {
	 puts("Usage: access [local, remote, proxy] [url]");
	 puts("\nSpecifying 'remote' allows remote retrieval of CDDB entries.  Specify 'local'\nwhen you are offline.  If you require the use of a HTTP proxy, use the\n'proxy' option.");
      } else if(strcmp(argv[0], "sites") == 0) {
	 if(argc > 1) {
	    if(strcmp(argv[1], "refresh") == 0) {
	       puts("Usage: refresh [url]");
	       puts("\nUpdates the CDDB server list from the default CDDB server or the specified URL.");
	    } else if(strcmp(argv[1], "display") == 0) {
	       puts("Usage: display");
	       puts("\nDisplays the current CDDB server list.");
	    } else if(strcmp(argv[1], "add") == 0) {
	       puts("Usage: add [location] [url] [type]");
	       puts("\nAdds an entry to the CDDB server list.  'location' is a position on the list.\n'url' is the server URL.  'type' is the server type, either CDDB or CDI.");
	    } else if(strcmp(argv[1], "delete") == 0) {
	       puts("Usage: delete [location]");
	       puts("\nDeletes an entry from the CDDB server list.");
	    } else if(strcmp(argv[1], "move") == 0) {
	       puts("Usage: move [source] [destination]");
	       puts("\nMoves a CDDB server list entry from 'source' to 'destination'.");
	    }
		 
	    return;
	 }
	 puts("Usage: sites [refresh, display, add, delete, move] [...] [...]");
	 puts("\nEdits the CDDB server list. For more specific help type help sites followed by\none of the above parameters.");
      } else if(strcmp(argv[0], "submit") == 0) {
	 puts("Usage: submit [email address]");
	 puts("\nSubmits the CDDB entry for the current CD to the CDDB.  If you don't specify\nyour e-mail address, you will be prompted to enter it.");
      } else if(strcmp(argv[0], "quit") == 0) {
	 puts("Usage: quit");
	 puts("\nquit exits cdcd. (CD keeps playing, of course)");
      } else if(strcmp(argv[0], "<command>") == 0) {
	 puts("Umm, when I say \'help <command>\', I mean replace <command> with one of the\nabove commands.");
      } else if(strcmp(argv[0], "help") == 0) {
	 puts("Oh please, this isn't Windows.");
      } else
	printf("No help available on %s.\n", argv[0]);
   } else if(strcmp(command, "version") == 0) {
      cd_version(dispbuffer, 4096);
      printf("%s version %s, Copyright (C)1998-99 Tony Arcieri\n", PACKAGE, VERSION);
      puts("Distributed under the GNU General Public License. See file COPYING for details.");
      printf("Built with %s\n", dispbuffer);
   } else if(strcmp(command, "play") == 0) {
      end_track = 0;
      
      if(cdcd_cd_stat(cd_desc, &disc) < 0)
	return;
      
      arg = -1;
      for(index = 0; index < argc; index++)
	if(strchr(argv[index], ':') != NULL)
	  arg = index;
      
      if(arg != -1) {
	 if(argv[arg][0] == ':' || argv[arg][strlen(argv[arg]) - 1] == ':') {
	    puts("Please specify time in the format min:sec");
	    return;
	 }
	 disc_time.minutes = strtol(strtok(argv[arg], ":"), NULL, 10);
	 disc_time.seconds = strtol(strtok(NULL, ":"), NULL, 10);
	 disc_time.frames = 0;
      } else {
	 disc_time.minutes = 0;
	 disc_time.seconds = 0;
	 disc_time.frames = 0;
      }
      
      if(argc > 0 && arg != 0)
	play_track = strtol(argv[0], NULL, 10);
      else {
	if(disc.disc_mode == CDAUDIO_PLAYING)
	   play_track = disc.disc_current_track;
        else
	   play_track = 1;
      }
      
      if(argc > 1 && arg != 1)
	end_track = strtol(argv[1], NULL, 10);
      else
	end_track = disc.disc_total_tracks;
	
      if(play_track == 0) {
	 cddb_lookup(cd_desc, &data);
	 
	 for(track = 0; track < disc.disc_total_tracks; track++)
	   if(strncasestr(data.data_track[track].track_name, 256, argv[0], 512) != NULL) {
	      play_track = track + 1;
	      break;
	   }
	 
	 if(play_track == 0 && argv[0][0] != '0') {
	    puts("Unknown track title");
	    return;
	 } else if(argv[0][0] == '0') {
	    puts("Invalid track");
	    return;
	 }
      }
      
      if(end_track < play_track) {
	 cddb_lookup(cd_desc, &data);
	 
	 for(track = 0; track < disc.disc_total_tracks; track++)
	   if(strncasestr(data.data_track[track].track_name, 256, argv[1], 512) != NULL) {
	      end_track = track + 1;
	      break;
	   }
	 
	 if(end_track < play_track) end_track = disc.disc_total_tracks;
      }
      
      if(end_track > disc.disc_total_tracks) end_track = disc.disc_total_tracks;
      
      if((disc_time.minutes * 60 + disc_time.seconds) > (disc.disc_track[play_track - 1].track_length.minutes * 60 + disc.disc_track[play_track - 1].track_length.seconds)) {
	 play_track++;
	 if(play_track > disc.disc_total_tracks) {
	    cd_stop(cd_desc);
	    return;
	 }
	 
	 disc_time.minutes = 0;
	 disc_time.seconds = 0;
	 disc_time.frames = 0;
      }
      
      if(play_track < 1 || play_track > disc.disc_total_tracks) {
	 puts("Invalid track");
	 return;
      }
      
      if(disc.disc_track[play_track - 1].track_type == CDAUDIO_TRACK_DATA) {
         puts("Cannot play data track");
	 return;
      }	     
	   
      cd_play_track_pos(cd_desc, play_track, end_track, disc_time.minutes * 60 + disc_time.seconds);
   } else if(strcmp(command, "sites") == 0) {
      memset(inbuffer, '\0', 512);
      cddb_read_serverlist(&conf, &list, &proxy);
	   
      if(argc > 0)
	strncpy(inbuffer, argv[0], 512);
      else {
         puts("Editing server list:\n--------------------");
	 if(list.list_len == 0)
	   puts("Server list is empty");
	 for(index = 0; index < list.list_len; index++)
	   printf("%d: %s\t%s://%s:%d/%s\n", index + 1, (list.list_host[index].host_protocol == CDINDEX_MODE_HTTP) ? "CDI" : "CDDB", list.list_host[index].host_protocol ? "http" : "cddbp", list.list_host[index].host_server.server_name, list.list_host[index].host_server.server_port, list.list_host[index].host_addressing);
	 putchar('\n');
	 get_input_text(inbuffer, "Do what (refresh, display, add, delete, or move): ", 512);
	 if(strlen(inbuffer) < 1)
	   return;
	 inptr = strtok(inbuffer, " ");
	 argc = 0;
	 while(inptr != NULL) {
	    strncpy(argv[argc++], inptr, 256);
	    inptr = strtok(NULL, " ");
	 }		
	 strncpy(inbuffer, argv[0], 512);
      }
      if(strcmp(inbuffer, "refresh") == 0) {
	 if(argc > 1)
	   cddb_process_url(&host, argv[1]);
	 else {
	    host.host_protocol = CDDB_MODE_CDDBP;
	    strncpy(host.host_server.server_name, CDCD_CDDB_HOST, 256);
	    host.host_server.server_port = CDDBP_DEFAULT_PORT;
	 }
     
	 cddb_get_siteslist(&list, host);
	 if(list.list_len < 1) {
	   puts("Retrieved empty server list... not stored");
	   return;
	 }
	 conf.conf_access = CDDB_ACCESS_REMOTE;
	 cddb_write_serverlist(conf, list, proxy);
	 return;
      } else if(strcmp(inbuffer, "display") == 0) {
	 puts("Server list:\n------------");
	 if(list.list_len == 0)
	   puts("Empty.");
	 for(index = 0; index < list.list_len; index++)
	   printf("%d: %s\t%s://%s:%d/%s\n", index + 1, (list.list_host[index].host_protocol == CDINDEX_MODE_HTTP) ? "CDI" : "CDDB", list.list_host[index].host_protocol ? "http" : "cddbp", list.list_host[index].host_server.server_name, list.list_host[index].host_server.server_port, list.list_host[index].host_addressing); 
	      
	 return;
      } else if(strcmp(inbuffer, "add") == 0) {
	 if(argc > 1) 
	   strncpy(inbuffer, argv[1], 512);
	 else 	     
	   get_input_text(inbuffer, "Add to what location: ", 512);
	 if((location = strtol(inbuffer, NULL, 10)) == 0 || --location > list.list_len) {
	    puts("Invalid location");
	    return;
	 }
	 
	 if(argc > 2)
	   strncpy(inbuffer, argv[2], 512);
	 else
	   get_input_text(inbuffer, "Enter server URL: ", 512);			       
	 for(index = list.list_len; index > location; index--)
	    memcpy(&list.list_host[index], &list.list_host[index - 1], sizeof(struct cddb_host));
	 cddb_process_url(&list.list_host[location], inbuffer);
	 
	 if(argc > 3)
	   strncpy(inbuffer, argv[3], 512);
	 else if(list.list_host[location].host_protocol != CDDB_MODE_CDDBP)
	   get_input_text(inbuffer, "Enter server type: ", 512);
	 if(strcasecmp(inbuffer, "CDI") == 0)
	   list.list_host[location].host_protocol = CDINDEX_MODE_HTTP;
	 
	 list.list_len++;
	 cddb_write_serverlist(conf, list, proxy);
      } else if(strcmp(inbuffer, "delete") == 0 || strcmp(inbuffer, "del") == 0) {
	 if(argc > 1)
	   strncpy(inbuffer, argv[1], 512);
	 else
	   get_input_text(inbuffer, "Delete which server: ", 512);
	 if((location = strtol(inbuffer, NULL, 10)) == 0 || --location >= list.list_len) {
	    puts("Invalid server");
	    return;
	 }
	 for(index = location; index < list.list_len - 1; index++)
	    memcpy(&list.list_host[index], &list.list_host[index + 1], sizeof(struct cddb_host));
		
	 list.list_len--;
	 cddb_write_serverlist(conf, list, proxy);
      } else if(strcmp(inbuffer, "move") == 0) {
	 if(argc > 1)
	   strncpy(inbuffer, argv[1], 512);
	 else
	   get_input_text(inbuffer, "Move which server: ", 512);
	 if((location = strtol(inbuffer, NULL, 10)) == 0 || --location >= list.list_len) {
	    puts("Invalid server");
	    return;
	 }
	 if(argc > 2)
	   strncpy(inbuffer, argv[2], 512);
	 else
	   get_input_text(inbuffer, "Move to where: ", 512);
	 if((destination = strtol(inbuffer, NULL, 10)) == 0 || --destination >= list.list_len) {
	    puts("Invalid destination");
	    return;
	 }
	 memcpy(&host, &list.list_host[location], sizeof(struct cddb_host));
	 if(destination > location) {
	    for(index = location; index < destination; index++)
	       memcpy(&list.list_host[index], &list.list_host[index + 1], sizeof(struct cddb_host));
	 } else if(destination < location) {
	    for(index = location; index > destination; index--)
	       memcpy(&list.list_host[index], &list.list_host[index - 1], sizeof(struct cddb_host));
	 }
	 memcpy(&list.list_host[destination], &host, sizeof(struct cddb_host));
		     
	 cddb_write_serverlist(conf, list, proxy);
       
	cddb_process_url(&host, argv[0]);
      } else if(strlen(inbuffer) > 0)
	puts("Unknown sites command");
        
      return;
   } else if(strcmp(command, "access") == 0) {
      cddb_read_serverlist(&conf, &list, &proxy);
      if(argc > 0)
	strncpy(inbuffer, argv[0], 512);
      else {
	printf("CDDB access is currently set to %s.\n", conf.conf_access ? "REMOTE" : "LOCAL");
	if(conf.conf_proxy)
	  printf("HTTP proxy use is enabled: http://%s:%d/\n", proxy.server_name, proxy.server_port);
	else
	  puts("HTTP proxy use is disabled.");
	get_input_text(inbuffer, "Change access method (local, remote, proxy): ", 512);
      }
	   
      if(strcmp(inbuffer, "local") == 0) {
	 conf.conf_access = CDDB_ACCESS_LOCAL;
	 cddb_write_serverlist(conf, list, proxy);
      } else if(strcmp(inbuffer, "remote") == 0) {
	 conf.conf_access = CDDB_ACCESS_REMOTE;
	 cddb_write_serverlist(conf, list, proxy);
      } else if(strcmp(inbuffer, "proxy") == 0) {
	 if(argc > 1)
	   strncpy(inbuffer, argv[1], 512);
	 else
	   get_input_text(inbuffer, "Enter proxy URL ('off' to disable): ", 512);
	      
	 if(strcmp(inbuffer, "off") == 0) {
	    conf.conf_proxy = CDDB_PROXY_DISABLED;
	    cddb_write_serverlist(conf, list, proxy);
		 
	    return;
	 }
	      
	 cddb_process_url(&host, inbuffer);
	 conf.conf_proxy = CDDB_PROXY_ENABLED;
	 cddb_write_serverlist(conf, list, host.host_server);
      } else if(strlen(inbuffer) > 0)
	puts("What?");
	   
      return;
   } else if(strcmp(command, "ext") == 0) {
      if(cdcd_cd_stat(cd_desc, &disc) < 0)
	return;
      
      cddb_lookup(cd_desc, &data);

      if(argc > 0) {
	 if((disp_track = strtol(argv[0], NULL, 10)) == 0) {
	    for(track = 0; track < disc.disc_total_tracks; track++)
	      if(strncasestr(data.data_track[track].track_name, 256, argv[0], 512) != NULL) {
		 disp_track = track + 1;
		 break;
	      }
	 }
      } else
	disp_track = disc.disc_current_track;

      if(disp_track <= 0 || disp_track > disc.disc_total_tracks)
	disp_track = 1;
      
      if(argc > 0 && strcmp("disc", argv[0]) == 0) {
	 puts(data.data_title);
	 puts(data.data_extended);
      } else {
	 if(strncasecmp(data.data_track[disp_track - 1].track_extended, data.data_track[disp_track - 1].track_name, strlen(data.data_track[disp_track - 1].track_name)) != 0)
	   puts(data.data_track[disp_track - 1].track_name);
	 puts(data.data_track[disp_track - 1].track_extended);
      }
   } else if(strcmp(command, "stop") == 0) {
      cd_stop(cd_desc);
   } else if(strcmp(command, "open") == 0 || strcmp(command, "eject") == 0) {
      cd_stop(cd_desc);
      cd_eject(cd_desc);
   } else if(strcmp(command, "close") == 0) {
      cd_close(cd_desc);
   } else if(strcmp(command, "pause") == 0) {
      cd_pause(cd_desc);
   } else if(strcmp(command, "resume") == 0) {
      cd_resume(cd_desc);
   } else if(strcmp(command, "rew") == 0) {
      if(argc > 0) {
	 if(strchr(argv[0], ':') == NULL || argv[0][0] == ':' || argv[0][strlen(argv[0]) - 1] == ':') {
	    puts("Please specify the time in the form min:sec");
	    return;
	 }
	 disc_time.minutes = -strtol(strtok(argv[0], ":"), NULL, 10);
	 disc_time.seconds = -strtol(strtok(NULL, ":"), NULL, 10);
	 disc_time.frames = 0;
      } else {
	 disc_time.minutes = 0;
	 disc_time.seconds = -15;
	 disc_time.frames = 0;
      }
      
      printf("Rewinding %d:%02d.\n", -disc_time.minutes, -disc_time.seconds);
      cd_stat(cd_desc, &disc);      
      if(end_track < disc.disc_current_track || end_track > disc.disc_total_tracks)
	end_track = disc.disc_total_tracks;

      cd_track_advance(cd_desc, end_track, disc_time);
   } else if(strcmp(command, "ff") == 0) {
      if(argc > 0) {
	 if(strchr(argv[0], ':') == NULL || argv[0][0] == ':' || argv[0][strlen(argv[0]) - 1] == ':') {
	    puts("Please specify the time in the form min:sec");
	    return;
	 }
	 disc_time.minutes = strtol(strtok(argv[0], ":"), NULL, 10);
	 disc_time.seconds = strtol(strtok(NULL, ":"), NULL, 10);
	 disc_time.frames = 0;
      } else {
	 disc_time.minutes = 0;
	 disc_time.seconds = 15;
	 disc_time.frames = 0;
      }
      
      printf("Fast-forwarding %d:%02d.\n", disc_time.minutes, disc_time.seconds);
      cd_stat(cd_desc, &disc);      
      if(end_track < disc.disc_current_track || end_track > disc.disc_total_tracks)
	end_track = disc.disc_total_tracks;

      cd_track_advance(cd_desc, end_track, disc_time);
   } else if(strcmp(command, "rndplay") == 0) {
      cd_stat(cd_desc, &disc);

      play_track = (rand() % disc.disc_total_tracks) + 1;
      if(play_track > end_track) end_track = disc.disc_total_tracks;
      cd_play_track(cd_desc, play_track, end_track);
   } else if(strcmp(command, "next") == 0) {
      if(argc > 0) {
	if(strchr(argv[0], ':') == NULL || argv[0][0] == ':' || argv[0][strlen(argv[0]) - 1] == ':') {
	   puts("Please specify time in the format min:sec");
	   return;
	}
        disc_time.minutes = strtol(strtok(argv[0], ":"), NULL, 10);
	disc_time.seconds = strtol(strtok(NULL, ":"), NULL, 10);
	disc_time.frames = 0;
      } else {
	disc_time.minutes = 0;
        disc_time.seconds = 0;
	disc_time.frames = 0;
      }
      
      cd_stat(cd_desc, &disc);
      if(end_track < disc.disc_current_track || end_track > disc.disc_total_tracks)
	end_track = disc.disc_total_tracks;
	 
      if(disc.disc_current_track + 1 > end_track)
	cd_stop(cd_desc);
      else
	cd_play_track_pos(cd_desc, disc.disc_current_track + 1, end_track, disc_time.minutes * 60 + disc_time.seconds);
   } else if(strcmp(command, "prev") == 0) {
      if(argc > 0) {
	if(strchr(argv[0], ':') == NULL || argv[0][0] == ':' || argv[0][strlen(argv[0]) - 1] == ':') {
	   puts("Please specify time in the format min:sec");
	   return;
	}
	disc_time.minutes = strtol(strtok(argv[0], ":"), NULL, 10);
	disc_time.seconds = strtol(strtok(NULL, ":"), NULL, 10); 
	disc_time.frames = 0;
      } else {
	disc_time.minutes = 0;
	disc_time.seconds = 0;
	disc_time.frames = 0;
      }
	   
      cd_stat(cd_desc, &disc);
      if(end_track < disc.disc_current_track || end_track > disc.disc_total_tracks)
	end_track = disc.disc_total_tracks;
      
      if(disc.disc_current_track - 1 < 1)
	cd_play_track(cd_desc, 1, end_track);
      else
	cd_play_track_pos(cd_desc, disc.disc_current_track - 1, end_track, disc_time.minutes * 60 + disc_time.seconds);
   } else if(strcmp(command, "setvol") == 0) {
      if(argc > 0) {
	 vol.vol_front.left = strtol(argv[0], NULL, 10);
	 if(argc > 1)
	   vol.vol_front.right = strtol(argv[1], NULL, 10);
	 else
	   vol.vol_front.right = strtol(argv[0], NULL, 10);
	 if(argc > 2)
	   vol.vol_back.left = strtol(argv[2], NULL, 10);
	 else
	   vol.vol_back.left = strtol(argv[0], NULL, 10);
	 if(argc > 3)
	   vol.vol_back.right = strtol(argv[3], NULL, 10);
	 else
	   vol.vol_back.right = strtol(argv[0], NULL, 10);
	 if(cd_set_volume(cd_desc, vol) < 0)
	   puts("Invalid volume");
      } else
	puts("To what?");
   } else if(strcmp(command, "getvol") == 0) {
      cd_get_volume(cd_desc, &vol);
      printf("Left: %d\tRight:%d\n", vol.vol_front.left, vol.vol_front.right);
   } else if(strcmp(command, "status") == 0) {
      cd_stat(cd_desc, &disc);
      switch(disc.disc_mode) {
       case CDAUDIO_PLAYING:
	 printf("Playing %d %02d:%02d.%02d %02d:%02d.%02d\n", disc.disc_current_track, disc.disc_track_time.minutes, disc.disc_track_time.seconds, disc.disc_track_time.frames, disc.disc_time.minutes, disc.disc_time.seconds, disc.disc_time.frames);
	 break;
       case CDAUDIO_PAUSED:
	 printf("Paused %d %02d:%02d.%02d %02d:%02d.%02d\n", disc.disc_current_track, disc.disc_track_time.minutes, disc.disc_track_time.seconds, disc.disc_track_time.frames, disc.disc_time.minutes, disc.disc_time.seconds, disc.disc_time.frames);
	 break;
       case CDAUDIO_COMPLETED:
	 puts("Stopped");
	 break;
       case CDAUDIO_NOSTATUS:
	 puts("Stopped");
	 break;
      }
   } else if(strcmp(command, "info") == 0) {
      if(cdcd_cd_stat(cd_desc, &disc) < 0)
	return;
      
      cddb_lookup(cd_desc, &data);
      
      printf("Album name:   \t%s\n", data.data_title);
      if(strcmp(data.data_artist, "(various)") != 0 && strlen(data.data_artist) > 0) printf("Album artist: \t%s\n", data.data_artist);
      switch(disc.disc_mode) {
       case CDAUDIO_PLAYING:
         printf("Total tracks: \t%d\tDisc playing:\t%02d:%02d of %02d:%02d\n", disc.disc_total_tracks, disc.disc_time.minutes, disc.disc_time.seconds, disc.disc_length.minutes, disc.disc_length.seconds);
	 if(strlen(data.data_track[disc.disc_current_track - 1].track_artist) > 0)
	   printf("Playing:      \t%s / %s %02d %02d:%02d of %02d:%02d\n", data.data_track[disc.disc_current_track - 1].track_artist, data.data_track[disc.disc_current_track - 1].track_name, disc.disc_current_track, disc.disc_track_time.minutes, disc.disc_track_time.seconds, disc.disc_track[disc.disc_current_track - 1].track_length.minutes, disc.disc_track[disc.disc_current_track - 1].track_length.seconds);
	 else
	   printf("Playing:      \t%s %02d %02d:%02d of %02d:%02d\n", data.data_track[disc.disc_current_track - 1].track_name, disc.disc_current_track, disc.disc_track_time.minutes, disc.disc_track_time.seconds, disc.disc_track[disc.disc_current_track - 1].track_length.minutes, disc.disc_track[disc.disc_current_track - 1].track_length.seconds);
	 if(end_track != disc.disc_total_tracks && end_track != 0)
	   if(strlen(data.data_track[end_track - 1].track_artist) > 0)
	     printf("End track:    \t%s / %s %02d\n", data.data_track[end_track - 1].track_artist, data.data_track[end_track - 1].track_name, end_track);
	   else
	     printf("End track:    \t%s %02d\n", data.data_track[end_track - 1].track_name, end_track);
	 break;
       case CDAUDIO_PAUSED:
         printf("Total tracks: \t%d\tDisc paused:\t%02d:%02d of %02d:%02d\n", disc.disc_total_tracks, disc.disc_time.minutes, disc.disc_time.seconds, disc.disc_length.minutes, disc.disc_length.seconds);
	 printf("Paused:       \t%s %02d %02d:%02d of %02d:%02d\n", data.data_track[disc.disc_current_track - 1].track_name, disc.disc_current_track, disc.disc_track_time.minutes, disc.disc_track_time.seconds, disc.disc_track[disc.disc_current_track - 1].track_length.minutes, disc.disc_track[disc.disc_current_track - 1].track_length.seconds);
	 if(end_track != disc.disc_total_tracks && end_track != 0)
	   printf("End track:    \t%s %02d\n", data.data_track[end_track - 1].track_name, end_track);
	 break;
       case CDAUDIO_COMPLETED:
	 printf("Total tracks: \t%d\tDisc length:\t%02d:%02d\n", disc.disc_total_tracks, disc.disc_length.minutes, disc.disc_length.seconds);
	 puts("Stopped");
	 break;
       case CDAUDIO_NOSTATUS:
	 printf("Total tracks: \t%d\tDisc length:\t%02d:%02d\n", disc.disc_total_tracks, disc.disc_length.minutes, disc.disc_length.seconds);
	 puts("Stopped");
	 break;
      }
   } else if(strcmp(command, "tracks") == 0) {
      if(cdcd_cd_stat(cd_desc, &disc) < 0)
	return;
      
      cddb_lookup(cd_desc, &data);
      
      printf("Album name:   \t%s\n", data.data_title);
      if(strcmp(data.data_artist, "(various)") != 0 && strlen(data.data_artist) > 0) printf("Album artist: \t%s\n", data.data_artist);
      printf("Total tracks: \t%d\tDisc length:\t%02d:%02d\n\n", disc.disc_total_tracks, disc.disc_length.minutes, disc.disc_length.seconds);
          puts("Track  Length      Title\n-------------------------------------------------------------------------------");
      for(track = 0; track < disc.disc_total_tracks; track++)
	if(strlen(data.data_track[track].track_artist) > 0)
	  printf("%02d:    [%02d:%02d.%02d]  %s / %s %s\n", track + 1, disc.disc_track[track].track_length.minutes, disc.disc_track[track].track_length.seconds, disc.disc_track[track].track_length.frames, data.data_track[track].track_artist, data.data_track[track].track_name, (disc.disc_track[track].track_type == CDAUDIO_TRACK_DATA) ? "(data)" : "");
        else
	  printf("%02d:    [%02d:%02d.%02d]  %s %s\n", track + 1, disc.disc_track[track].track_length.minutes, disc.disc_track[track].track_length.seconds, disc.disc_track[track].track_length.frames, data.data_track[track].track_name, (disc.disc_track[track].track_type == CDAUDIO_TRACK_DATA) ? "(data)" : "");
   } else if(strcmp(command, "device") == 0) {
      strncpy(inbuffer, device, 256);
      if(argc > 0)
	strncpy(device, argv[0], 256);
      else {
	 printf("Current device: %s\n", device);
	 get_input_text(device, "Enter new device name: ", 256);
      }
      
      cd_finish(cd_desc);
      if((cd_desc = cd_init_device(device)) < 0) {
	 if(errno == EBUSY) {
	    printf("%s already appears to be mounted.  Sorry.\n", device);
	 } else
	   perror(device);
	 
	 strncpy(device, inbuffer, 256);
	 if((cd_desc = cd_init_device(device)) < 0) {
	    if(errno == EBUSY) {
	       printf("%s already appears to be mounted.  Sorry.\n", device);
	    } else
	      perror(device);
	 }
      } else {
	 cdcdrc_data.verbosity = verbosity;
	 strncpy(cdcdrc_data.device, device, 256);
	 cdcdrc_write(&cdcdrc_data);
      }
   } else if(strcmp(command, "edit") == 0) {
      cddb_lookup(cd_desc, &data);
      
      if(argc > 0)
	strncpy(inbuffer, argv[0], 512);
      else {
	 get_input_text(inbuffer, "Edit what (name, artist, genre, track#): ", 512);
      }
      if(strcmp(inbuffer, "name") == 0) {
	 if(argc > 1)
	   strncpy(data.data_title, argv[1], 256);
	 else {
	    printf("Current value: %s\n", data.data_title);
	    get_input_text(data.data_title, "Enter new value: ", 128);
	 }
      } else if(strcmp(inbuffer, "artist") == 0) {
	 if(argc > 1)
	   strncpy(data.data_artist, argv[1], 256);
	 else {
	    printf("Current value: %s\n", data.data_artist);
	    get_input_text(data.data_artist, "Enter new value: ", 128);
	 }
      } else if(strcmp(inbuffer, "genre") == 0) {
	 if(argc > 1)
	   strncpy(inbuffer, argv[1], 256);
	 else {
	    printf("Current value: %s\n", cddb_genre(data.data_genre));
	    get_input_text(inbuffer, "Enter new value: ", 512);
	 }
	 for(index = 1; index < 12; index++)
	   if(strncasecmp(inbuffer, cddb_genre(index), 4) == 0) {
	      cddb_erase_entry(data);
	      data.data_genre = index;
	      cddb_write_disc_data(cd_desc, data);
	      return;
	   }
	 puts("Sorry.  Valid genres are:");
	 for(index = 1; index < 12; index++)
	   puts(cddb_genre(index));
	 
	 return;
      } else if((track = strtol(inbuffer, NULL, 10)) != 0) {
	 if(argc > 1)
	   strncpy(data.data_track[track - 1].track_name, argv[1], 256);
	 else {
	    printf("Current value: %s\n", data.data_track[track - 1].track_name);
	    get_input_text(data.data_track[track - 1].track_name, "Enter new value: ", 128);
	 }
      } else if(strlen(inbuffer) > 0)
	puts("What?");
      
      cddb_write_disc_data(cd_desc, data);
   } else if(strcmp(command, "refresh") == 0) {
      if(cdcd_cd_stat(cd_desc, &disc) < 0)
	return;
      cddb_stat_disc_data(cd_desc, &entry);
      if(entry.entry_present) {
         cddb_lookup(cd_desc, &data);
         cddb_erase_entry(data);
      }
      cddb_lookup(cd_desc, &data);
   } else if(strcmp(command, "verbose") == 0) {
      if(argc > 0)
	strncpy(inbuffer, argv[0], 512);
      else {
	 if(verbosity)
	   puts("Verbosity is on");
	 else
	   puts("Verbosity is off");
	 get_input_text(inbuffer, "New verbosity setting (on/off): ", 256);
      }
      if(strcmp(inbuffer, "on") == 0)
	verbosity = 1;
      else if(strcmp(inbuffer, "off") == 0)
	verbosity = 0;
      cdcdrc_data.verbosity = verbosity;
      strncpy(cdcdrc_data.device, device, 256);
      cdcdrc_write(&cdcdrc_data);
   } else if(strcmp(command, "list") == 0) {
      cd_changer_stat(cd_desc, &changer);
      for(disp_disc = 0; disp_disc < changer.changer_slots; disp_disc++)
	if(changer.changer_disc[disp_disc].disc_present)
	  printf("Disc %d: \t%02d:%02d %s\n", disp_disc + 1, changer.changer_disc[disp_disc].disc_length.minutes, changer.changer_disc[disp_disc].disc_length.seconds, changer.changer_disc[disp_disc].disc_info);
   } else if(strcmp(command, "slot") == 0) {
      cur_disc = 0;
      
      if(argc > 0)
	cur_disc = strtol(argv[0], NULL, 10);
      
      if(cur_disc == 0) {
	 puts("Please specify which slot you wish to select");
	 return;
      }
      
      cd_changer_select_disc(cd_desc, cur_disc - 1);
   } else if(strcmp(command, "submit") == 0) {
      if(cdcd_cd_stat(cd_desc, &disc) < 0)
	return;
	   
      if(argc > 0)
	strncpy(inbuffer, argv[0], 256);
      else
        get_input_text(inbuffer, "Enter your e-mail address: ", 256);
	   
      cddb_submit(cd_desc, inbuffer);
   } else if(strcmp(command, "quit") == 0) {
      cd_finish(cd_desc);
      exit(0);
   } else if(strcmp(command, "debug") == 0) {
      if(cdcd_cd_stat(cd_desc, &disc) < 0)
	return;
      
      cddb_lookup(cd_desc, &data);

      puts("Debugging information:");
      printf("CDI id: \t%s\n", data.data_cdindex_id);
      printf("CDDB id:\t%08lx\n", data.data_id);
      printf("Disc genre:\t%s\n", cddb_genre(data.data_genre));
      printf("Disc mode:\t");
      switch(disc.disc_mode) {
       case CDAUDIO_PLAYING:
	 puts("Playing");
	 break;
       case CDAUDIO_PAUSED:
	 puts("Paused");
	 break;
       case CDAUDIO_COMPLETED:
	 puts("Stopped");
	 break;
       case CDAUDIO_NOSTATUS:
	 puts("Stopped");
	 break;
      }
      printf("Track time:\t%02d:%02d\n", disc.disc_track_time.minutes, disc.disc_track_time.seconds);
      printf("Disc time:\t%02d:%02d\n", disc.disc_time.minutes, disc.disc_time.seconds);
      printf("Disc length:\t%02d:%02d\n", disc.disc_length.minutes, disc.disc_length.seconds);
      printf("Disc track:\t%d\n", disc.disc_current_track);
      printf("Total tracks:\t%d\n", disc.disc_total_tracks);
      for(track = 0; track < disc.disc_total_tracks; track++) {
	 printf("Track %d:\n", track + 1);
	 printf(" Track LBA:\t%d\n", disc.disc_track[track].track_lba);
	 printf(" Track length:\t%02d:%02d\n", disc.disc_track[track].track_length.minutes, disc.disc_track[track].track_length.seconds);
      }
   } else if(strlen(command) != 0)
     fputs("Unknown command\n", stderr);
}

int
#if __STDC__
main(int argc, char **argv)
#else
main(argc, argv)
   int argc;
   char **argv;
#endif
{
   int command_argc, index, stringindex, instr;
   char command[256], buffer[256], string[256], stringbuffer[256];
   char *argptr, command_argv[MAX_ARGS][256];
   struct cdcdrc conf;
	
   signal(SIGSEGV, signal_handler);
   signal(SIGTERM, signal_handler);
   
   srand((unsigned int)time(NULL));

#ifdef HAVE_READLINE
   rl_readline_name = PACKAGE;
   rl_special_prefixes = "\"\'`";
   rl_completion_entry_function = (Function *)command_matcher;
   rl_attempted_completion_function = (CPPFunction *)cdcd_completion;
#endif
   
   cdcdrc_read(&conf);
   verbosity = conf.verbosity;
   strncpy(device, conf.device, 256);
   
   if((cd_desc = cd_init_device(device)) < 0) {
      if(errno == EBUSY) {
	 printf("%s already appears to be mounted.  Sorry.\n", device);
      } else
	perror(device);
   }
   
   if(argc > 1) {
      if(argc > 2) {
	 for(command_argc = 0; command_argc < argc - 2 && command_argc < MAX_ARGS; command_argc++)
	   strncpy(command_argv[command_argc], argv[command_argc + 2], 256);
      } else
	command_argc = 0;
	   
      proc_command(argv[1], command_argc, command_argv);
      
      exit(0);
      
   } else {
      proc_command("version", 0, command_argv);
      puts("Enter ? for help.");
      while(1) {
	 get_input_text(command, "cdcd> ", 256);
	 
	 /* The amazing superkludge string processor */
	 command_argc = 0;
	 instr = 0;
	 
	 for(index = 0; index < strlen(command); index++) {
	    if(command[index] == '\"')
	      instr = !instr;
	    if(command[index] == '\'' && instr == 0)
	      command[index] = '\"';
	 }
	 
	 instr = 0;
	 
	 if(strchr(command, ' ') != NULL) {
	    strtok(command, " ");
	    while((argptr = strtok(NULL, " ")) != NULL) {
	       if(strchr(argptr, '\"') != NULL) {
		  if(instr == 1) {
		     instr = 0;
		     stringindex = 0;
		     for(index = 0; index < strlen(argptr); index++) {
			if(argptr[index] != '\"')
			  buffer[stringindex++] = argptr[index];
			
			if(stringindex >= 255) break;
		     }
		     buffer[stringindex] = '\0';
		     
		     snprintf(command_argv[command_argc++], 256, "%s %s", string, buffer);
		  } else {
		     stringindex = 0;
		     for(index = 0; index < strlen(argptr); index++) {
			if(argptr[index] != '\"')
			  string[stringindex++] = argptr[index];
			else {
			   if(instr == 0)
			     instr = 1;
			   else
			     instr = 0;
			}
			
			if(stringindex >= 255) break;
		     }
		     string[stringindex] = '\0';
		     
		     if(instr == 0)
		       strncpy(command_argv[command_argc++], string, 256);
		  }
	       } else {
		  if(instr == 1) {
		     snprintf(stringbuffer, 256, "%s %s", string, argptr);
		     strncpy(string, stringbuffer, 256);
		  } else
		    strncpy(command_argv[command_argc++], argptr, 256);
	       }
	       if(command_argc >= MAX_ARGS)
		 break;
	    }
	 }
	 
	 if(instr == 1) {
	    instr = 0;
	    puts("Unterminated/malformed string");
	 } else
	   proc_command(command, command_argc, command_argv);
      }
   }
}
