/**
 * @file pcpar.h
 * Interface to accessing IBM PC like parallel ports
 * @author Marko Mkel (msmakela@nic.funet.fi)
 */

/*
 * Copyright  2001,2002 Marko Mkel
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef COMM_PC
# ifdef __linux__
#  include <unistd.h>
#  include <sys/io.h>
#  define outbyte(value, port) outb (value, port)
# endif /* __linux */
# ifdef __FreeBSD__
#  include <unistd.h>
#  include <machine/cpufunc.h>
#  define outbyte(value, port) outb (port, value)
# endif /* __FreeBSD__ */
# if defined __WIN32__ && defined __GNUC__
/* copied from Linux <sys/io.h> */
static __inline unsigned char
inb (unsigned short int port)
{
  unsigned char _v;
  __asm__ __volatile__ ("inb %w1,%0":"=a" (_v):"Nd" (port));
  return _v;
}
static __inline void
outbyte (unsigned char value, unsigned short int port)
{
  __asm__ __volatile__ ("outb %b0,%w1": :"a" (value), "Nd" (port));
}
# endif /* __WIN32__ */
# ifdef __BCC__
#  define outbyte(value,port) oportb(port, value)
#  define inb(port) inportb(port)
# endif /* __BCC__ */
#endif /* COMM_PC */

/** the base address for the printer port, and the address of the data port */
extern unsigned baseaddr;
/** the address of the status port */
extern unsigned stataddr;
/** the address of the control port */
extern unsigned ctrladdr;

/** Open a parallel port
 * @param dev	the "device name" of the port
 * @return	the base address of the port (-1=ppdev), or 0 on failure
 */
unsigned
open_port (const char* dev);

#ifdef USE_PPDEV
/** Close the communication channel to the parallel port */
void
close_port (void);

/** write data bits to the parallel port
 * @param data	the data bits to be written
 * @return	zero on success, nonzero on failure
 */
int
write_data (char data);

/** write control bits to the parallel port
 * @param ctrl	the control bits to be written
 * @return	zero on success, nonzero on failure
 */
int
write_ctrl (char ctrl);

/** read the status bits of the parallel port
 * @return	the read status bits
 */
char
read_stat (void);

/** read the control bits of the parallel port
 * @return	the read control bits
 */
char
read_ctrl (void);

/** output a data byte
 * @param data	the data to be output
 */
# define out_data(data) write_data (data)
/** output a control byte
 * @param ctrl	the control byte to be output
 */
# define out_ctrl(ctrl) write_ctrl (ctrl)
/** read the status bits */
# define in_stat() read_stat ()
/** read the control bits */
# define in_ctrl() read_ctrl ()
#else /* USE_PPDEV */
/** output a data byte
 * @param data	the data to be output
 */
# define out_data(data) outbyte (data, baseaddr)
/** output a control byte
 * @param ctrl	the control byte to be output
 */
# define out_ctrl(ctrl) outbyte (ctrl, ctrladdr)
/** read the status bits */
# define in_stat() inb (stataddr)
/** read the control bits */
# define in_ctrl() inb (ctrladdr)
/** terminate the connection */
# define close_port()
#endif /* USE_PPDEV */

/* some definitions for inline assembler */
# ifdef __BCC__
#  define I #
#  define buf_ 4[bp]
#  define len_ 6[bp]
# endif
