/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2005 The Caudium Group
 * Copyright  1999 Bill Welliver <hww3@riverweb@com>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: vhs_sqllogger.pike,v 1.1.4.2 2005/05/06 14:39:46 kiwi Exp $
 */

#include <module.h>
#include <caudium.h>
inherit "module";
inherit "caudiumlib";

constant module_type  = MODULE_LOGGER;
constant module_name  = "VHS - SQL logger";
constant module_doc   = "This logger uses a SQL database server which has pike "
                        "support to log all access for a virtual server.<br />"
			"This module has been designed to work with mod_log_sql2 from"
			"apache distribution.";
constant module_unique= 1;
constant thread_safe  = 1;
constant cvs_version  = "$Id: vhs_sqllogger.pike,v 1.1.4.2 2005/05/06 14:39:46 kiwi Exp $";

//
//! module: SQL logger
//!  This logger uses a SQL database server which has pike support to log
//!  all access for a virtual server.<br />Please see
//!  <tt>examples/sqllogger/</tt> files for database formats examples.
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_LOGGER
//! cvs_version: $Id: vhs_sqllogger.pike,v 1.1.4.2 2005/05/06 14:39:46 kiwi Exp $
//

object db;		// The db stack
int num;
string logtable; 

#if constant(thread_create)
static inherit Thread.Mutex;
#define THREAD_SAFE
#define LOCK() do { object key; catch(key=lock())
#define UNLOCK() key=0; } while(0)
#else
#undef THREAD_SAFE
#define LOCK() do {
#define UNLOCK() } while(0)
#endif


class db_handler {
#ifdef THREAD_SAFE
  static inherit Thread.Mutex;
#endif
  
  array (object) dbs = ({});
  string db_url;
  int num_dbs;  

  void create(string _dburl) {
    db_url = _dburl;
    num_dbs=num;
    mixed err;
    for(int i = 0; i < num; i++) {
      err=catch( dbs += ({ Sql.Sql(db_url) }));
      if(err) perror("Error creating db object:\n" + describe_backtrace(err)+"\n");
    }
  }
  
  void|object handle(void|object d) {
    LOCK();
    int count;
    dbs -= ({0});
    if(objectp(d)) {
      if(search(dbs, d) == -1) {
	if(sizeof(dbs)>(2*num_dbs)) {
	  werror("Dropping db because of inventory...\n");
	  destruct(d);
	} else {
	  dbs += ({d});
	}
      }
    } else {
      if(!sizeof(dbs)) {
	d = Sql.Sql(db_url);
      } else {
	d = dbs[0];
	dbs -= ({d});
      }
    }
    UNLOCK();
    return d;
  }
}


void create() {
  defvar("dburl", "mysql://user:pass@host/database", "Database URL", 
	 TYPE_STRING,
	 "This is the conncetion to the database in a SQL-URL.\n");
  defvar("logtable", "access_log", "Log table", 
	 TYPE_STRING,
	 "This is the table into which all client names will be put.\n");
  defvar("dbcount", 3, "Number of Connections", 
	 TYPE_INT,
	 "Number of connections to make.\n");
  defvar("failtime", 5,"Warning Timeout",
	 TYPE_INT, 
	 "Time between reconnect attempts if SQL server is down, in minutes.\n");
}

void start() {
  num=QUERY(dbcount);
  db=db_handler( QUERY(dburl) );   
  logtable=QUERY(logtable);
}


void stop() {
  destruct(db); 		// We're done, so close the connections.
}


void log(object id, mapping file)  {
  string log_query, referer, host;
  array auth;
  object sql_conn=db->handle();


  referer = (arrayp(id->referer))?(array)id->referer*" ":(string)id->referer;

  if (!sizeof(referer)) referer = "-";
  
  host = (string)id->misc->host; 
  if(search(host,":")) {
    host = (host / ":")[0];
  }

  int|mapping user=id->get_user();

  log_query=sprintf("INSERT INTO %s (agent,bytes_sent,referer,remote_host,remote_user,"
                    "request_duration,request_method,request_protocol,"
                    "request_uri,request_args,status,time_stamp,virtual_host) VALUES("
                    "'%s', %s, '%s', '%s', '%s', '0', '%s', '%s', '%s', %s, %d,%d,'%s')",
		    logtable,
		    (string)(id->useragent||"-"),
		    (string)file->len,
		    referer,
		    (string)id->remoteaddr,
		    (user?user->username:"-"),
		    (string)id->method,
		    (string)id->prot,
		    (string)id->not_query,
                    id->query?sprintf("'?%s'",(string)id->query):"NULL",
		    Logging.unsigned_to_bin(file->error||200),
		    time(),
		    host,
		    );

//  perror("SQL %s\n",log_query);
  
  if(catch(sql_conn->query(log_query)))
    perror("VHS - logSQL: Error running query.\n");		
  
  db->handle(sql_conn);
  return; 
}


/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: dburl
//! This is the conncetion to the database in a SQL-URL.
//!
//!  type: TYPE_STRING
//!  name: Database URL
//
//! defvar: logtable
//! This is the table into which all client names will be put.
//!
//!  type: TYPE_STRING
//!  name: Log table
//
//! defvar: dbcount
//! Number of connections to make.
//!
//!  type: TYPE_INT
//!  name: Number of Connections
//
//! defvar: failtime
//! Time between reconnect attempts if SQL server is down, in minutes.
//!
//!  type: TYPE_INT
//!  name: Warning Timeout
//

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

