(***********************************************************************)
(*                                                                     *)
(*                           Objective Caml                            *)
(*                                                                     *)
(*            Franois Pessaux, projet Cristal, INRIA Rocquencourt     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*            Jun Furuse, projet Cristal, INRIA Rocquencourt           *)
(*                                                                     *)
(*  Copyright 1999,2000,2001,2002,2001,2002                            *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)
(* Indexed 16 bit depth image format *)

type elt = int

type t = { 
    width: int;
    height: int;
    colormap: Color.map;
    mutable transparent: elt;
    mutable infos: Info.info list;
    data: Bitmap.t }

(* creation functions *)

val create_with : int -> int -> Color.map -> elt -> string -> t
    (* [create_with width height colormap transparent infos data] creates
       an index16 image of a size [width] x [height], a colormap [colormap],
       a transparent color index [transparent] (-1 means no transparent
       color, and a string representation 
       of the bitmap [data] *)
val create : int -> int -> t
    (* Creates a non-initialized image with null colormap and null infos. *)
val make : int -> int -> elt -> t
    (* Creates an image with null colormap and null infos filled with elt. *)

(* methods *)

val get : t -> int -> int -> elt
val set : t -> int -> int -> elt -> unit
    (* [get image x y] and [set image x y v] reads/writes the pixel 
       information at (x,y) of [image]. If (x,y) is out of the image,
       they raise Out_of_image exception. *)

val unsafe_get : t -> int -> int -> elt
val unsafe_set : t -> int -> int -> elt -> unit
    (* [unsafe_get] and [unsafe_set] are the same functions as [get] 
       and [set], but they lack the image region checks. So it is fast.
       But you have to use them with being sure that the specified point 
       is in the image. Otherwise the result is unknown, and sometimes 
       a runtime error occurs. *)

val get_rgb : t -> int -> int -> Color.rgb
val unsafe_get_rgb : t -> int -> int -> Color.rgb
    (* [get_rgb] and [unsafe_get_rgb] are as same as [get] and 
       [unsafe_get], but return the color of the point instead of 
       the index. *)

val set_colormap : t -> Color.map -> unit

val destroy : t -> unit
    (* You need to call [destroy t] to free the swap files maybe created
       for [t]. *)

val sub : t -> int -> int -> int -> int -> t
(* [sub dst x y width height] returns sub-bitmap of [dst],
   at (x,y)-(x+width-1,y+height-1). *)

val blit : t -> int -> int -> t -> int -> int -> int -> int -> unit
(* [blit src sx sy dst dx dy width height] copies the rectangle
   region of [src] at (sx,sy)-(sx+width-1,sy+height-1) to [dst], at
   (dx,dy)-(dx+width-1,dy+height-1) *)

val to_rgb24 : t -> Rgb24.t
val to_rgba32 : t -> Rgba32.t
