/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2001 The Caudium Group
 * Copyright  1994-2001 Roxen Internet Software
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_layout_fs.pike,v 1.14.2.6 2001/06/19 17:46:11 oliv3 Exp $
 */

// #define DEBUG 1

/*
  oliv3 TODO: load from a tarball

  <Grendel> why the TAR - because right now for each template screen you waste two inodes
*/

#include <module.h>
#include <camas/screens.h>

inherit "module";
inherit "caudiumlib";

//
//! module: CAMAS: Filesystem Layout
//!  Filesystem Layout for CAMAS.<br /><br />
//!  This module will fetch the layout files, one file per CAMAS screen.<br />
//!  e.g.: if the filename is '<b>layout.rxml</b>', the path '<b>/path/to/template</b>',
//!  <br /> and the screen '<b>compose</b>', <br />CAMAS will look for
//!  '<b>/path/to/template/compose/layout.rxml</b>'.<br /> If the layout is not found,
//!  the default will be used ('<b>/path/to/template/layout.rxml</b>').<br />
//!  If there is no default file, the layout from <i>camas_layout_default</i> module
//!  will be used.<br />The name of the layout is used for the dialog boxes and
//!  the preferences file.
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_PROVIDER
//! cvs_version: $Id: camas_layout_fs.pike,v 1.14.2.6 2001/06/19 17:46:11 oliv3 Exp $
//

constant cvs_version = "$Id: camas_layout_fs.pike,v 1.14.2.6 2001/06/19 17:46:11 oliv3 Exp $";
constant module_type = MODULE_PROVIDER;
constant module_name = "CAMAS: Filesystem Layout";
constant module_doc  = "Filesystem Layout for CAMAS.<br /><br />"
"This module will fetch the layout files, one file per CAMAS screen.<br />"
"e.g.: if the filename is '<b>layout.rxml</b>', the path '<b>/path/to/template</b>',<br />"
"and the screen '<b>compose</b>',<br />CAMAS will look for '<b>/path/to/template/compose/layout.rxml</b>'.<br />"
"If the layout is not found, the default will be used ('<b>/path/to/template/layout.rxml</b>').<br />"
"If there is no default file, the layout from the <i>camas_layout_default</i> module will be used.<br />"
"The name of the layout is used for the dialog boxes and the preferences file.";


constant module_unique = 0;
constant thread_safe = 1;		// I think this module should be :)

#if constant(thread_create)
object global_lock = Thread.Mutex();
#endif

mapping (string: mapping) screens = ([ ]);
int files_loaded = 0;

void create () {
  defvar ("name", "NONE", "Name", TYPE_STRING,
	  "The name of the template.");

  defvar ("location", "", "Template path", TYPE_DIR, "The path to search for template files.");

  defvar ("template", "layout.rxml", "Template filename", TYPE_STRING,
	  "Filename to search for template files.");

  defvar ("frameset", 0, "Uses frames", TYPE_FLAG,
	  "If set to 'yes', you will be able to create a frameset-based layout. The frameset "
	  "<b>MUST</b> (for now) be called 'frameset.rxml' and be at the top of the template path "
	  "(where the default layout file is)");
}

string check_variable (string var, mixed value) {
  switch (var) {
  case "name": 
    if (!sizeof (value) || ((string)value == "NONE")) {
      return "You have to set a name.";
    }; break;

  case "location": 
    if (sizeof (value)) {
      array stat = file_stat (value);
      if (!stat || stat[1] != -2)
	return "Must be a directory.";
    }; 
    files_loaded = 0; break;
  }
  return 0;
}

void start (int num, object conf) {
  module_dependencies (conf, ({ "camas_layout_manager" }));

  if (QUERY (name) != "NONE")
    conf->get_provider ("camas_layout_manager")->add_layout (this_object ());

  load_layout_files ();
}

void stop () {
  if (QUERY (name) != "NONE")
    my_configuration ()->get_provider ("camas_layout_manager")->remove_layout (this_object ());
}

string status () {
  string ret = "CAMAS Filesystem Layout: " + QUERY (name) + "<br /><br />";
  if (sizeof (screens))
    {
      ret += "Provides these screens:<ul>";
      foreach (sort (indices (screens)), string s)
	{
	  ret += "<li>" + s;
	  if (screens[s]->default_screen)
	    ret += " <font size=-1><i>(loaded from the default file)</i></font>";
	  ret += "</li>";
	}
      ret += "</ul>";
    }
  return ret;
}

string name () { 
  return QUERY (name);
}

string query_provides () { 
  return "camas_layout";
}

string read (string file) {
  string result = "";
  object o = Stdio.File ();

  if (o->open (file, "r"))
    result = o->read ();
  o->close ();
  return result;
}

string layout_file (void|string screen) { //void: default_screen
  string file = QUERY (location) + "/";
  if (screen == "frameset")
    file += "frameset.rxml";
  else {
    file += (screen) ? screen : "";
    file += "/" + QUERY (template);
  }
  return file;
}


int newer_screen (string screen) {
  string file = layout_file (screen);
  array fstat = file_stat (file);
  int newer = (fstat) ? (fstat[3] > screens[screen]->modified) : 0;
#ifdef DEBUG
  if (newer)
    write ("have to reload screen: " + screen + "\n");
#endif
  return newer;
}

void load_screen (string screen) {
#if constant(thread_create)
  object lock = global_lock->lock();
#endif
  string file = layout_file (screen);

#ifdef DEBUG
  write ("load_screen (" + screen + ") ...");
#endif
  array fstat = file_stat (file);
  int flastmod = fstat ? fstat[3] : -1;
  if (fstat && ((flastmod > screens[screen]->modified) || (screens[screen]->default_screen))) {
    // found the layout file
#ifdef DEBUG
    write ("new[er] file.\n");
#endif
    screens[screen]->data = read (file);
    screens[screen]->default_screen = 0;
    screens[screen]->modified = flastmod;
  } else {
    // file is missing, try to load a default layout
    file = layout_file ();
    fstat = file_stat (file);
    if (fstat) {
      // found default layout
#ifdef DEBUG
      write ("using default.\n");
#endif
      flastmod = fstat ? fstat[3] : -1;
      screens[screen]->data = read (file);
      screens[screen]->default_screen = 1;
      screens[screen]->modified = flastmod;
    }
    else {
      // oops. not even a default layout
#ifdef DEBUG
      write ("nothing found.\n");
#endif
      screens[screen]->data = 0;
      screens[screen]->default_screen = 1;
      screens[screen]->modified = -1;
    }
  }
#if constant(thread_create)
  destruct(lock);
#endif
}

void load_layout_files () {
  foreach (indices (screennames), int s) {
    string sn = screennames[s][0];
    if (search (indices (screens), sn) == -1) { // new screen
      screens += ([ sn: ([ "data": "", "modified": -1, "default_screen": 0 ]) ]);
      load_screen (sn);
    }
  }

  if (QUERY (frameset)) {
    screens += ([ "frameset": ([ "data": "", "modified": -1, "default_screen": 0 ]) ]);
    load_screen ("frameset");
  }

  files_loaded = 1;
}

string get_layout (int _screen) {
  string screen = screennames[_screen][0];
  int changed = newer_screen (screen);

  if (changed)
    load_screen (screen);

  return screens[screen]->data;
}

int has_frameset () {
  //write("FS::has_frameset: " + QUERY (frameset) + "\n");
  return QUERY (frameset);
}

string get_frameset () {
  /* oliv3 FIXME quick & dirty hack, clean up !!! */
  //write("FS::get_frameset: " + screens["frameset"]->data + "\n");
  return (screens["frameset"] && has_frameset ()) ? screens["frameset"]->data : "";
}

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: name
//! The name of the template.
//!  type: TYPE_STRING
//!  name: Name
//
//! defvar: location
//! The path to search for template files.
//!  type: TYPE_DIR
//!  name: Template path
//
//! defvar: template
//! Filename to search for template files.
//!  type: TYPE_STRING
//!  name: Template filename
//
//! defvar: frameset
//! If set to 'yes', you will be able to create a frameset-based layout. The frameset <b>MUST</b> (for now) be called 'frameset.rxml' and be at the top of the template path (where the default layout file is)
//!  type: TYPE_FLAG
//!  name: Uses frames
//
