/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2001 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_imho.pike,v 1.18.2.9.2.5 2001/10/17 10:34:31 oliv3 Exp $
 */
//
//! module: CAMAS: IMHO Tags
//!  This module gives backward IMHO compatibility for &lt;imho_..&gt;
//!  tags and containers.<br />
//!  So you can use old IMHO tags on old layouts...<br />
//!  <b>This module is automatically selected if you select "CAMAS: Main 
//!  module".</b>
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_PARSER|MODULE_PROVIDER
//! cvs_version: $Id: camas_imho.pike,v 1.18.2.9.2.5 2001/10/17 10:34:31 oliv3 Exp $
//
#include <module.h>
inherit "module";
inherit "caudiumlib";
#include <camas/globals.h>

constant cvs_version   = "$Id: camas_imho.pike,v 1.18.2.9.2.5 2001/10/17 10:34:31 oliv3 Exp $";
constant module_type   = MODULE_PARSER|MODULE_PROVIDER;
constant module_name   = "CAMAS: IMHO Tags";
constant module_doc    = "This module gives backward IMHO compatibility for "
                         "&lt;imho_..&gt; tag and containers.<br />"
		         "So you can use old IMHO tags on old layouts..."
                         "<br /><b>This module is automatically selected if you "
                         "select \"CAMAS: Main module\".</b>";
constant module_unique = 1;
constant thread_safe = 1;

// Definition to give us an easy way to make the code and understand it :)
#define CAMAS_MODULE  id->conf->get_provider("camas_main")

// Language supports
#define MSG(m) sessobj->lang_module->msg (sessobj, m)
#define MSGA(m, a) sessobj->lang_module->msg (sessobj, m, a)

// Faster debugging
//#define CAMAS_DEBUG
#ifdef CAMAS_DEBUG
# define DEBUG(X)	if(QUERY(debug)) werror("CAMAS IMHO: "+X+"\n");
#else
# define DEBUG(X)
#endif

void create()
{
#ifdef CAMAS_DEBUG
  defvar("debug",0,"Debug",TYPE_FLAG,"Debug the call / errors into Caudium "
         "error log ?");
#endif
  // Was in main module
  defvar("banner", "CAMAS", "Banner text", TYPE_STRING,
	 "Banner text. This text is inserted on top of every page if you are "
	 "using the default layout. Can be inserted using "
	 "&lt;imho_banner&nbsp;/&gt; in a layout file.");
}

void start(int cnt, object conf)
{
 module_dependencies(conf, 
	({ "tablify",	// <imho_servers>
	 }));
}

// What we provides
string query_provides ()
{
  return "camas_imho";
}

// Tags now

//
//! tag: imho_servers
//!  Gives a list of CAMAS Imap servers and domains served in a
//!  table. (need the tablify support).
//
string tag_imho_servers(string tag_name, mapping arg, object id, object file)
{
 object camas_module = CAMAS_MODULE;
 
 if(objectp(camas_module))
 {
   string tab = "<tablify nicer>";
   DEBUG(tag_name);
  
   tab += "Domain\tServer\n"; 
   foreach (camas_module->imap_servers(), array arr) {
     tab+= arr[0] + "\t" + arr[1] + "\n";
   }
   tab += "</tablify>";
   return tab;

 }
 return "";
}

//
//! tag: imho_banner
//!  Return the contents of the CAMAS's defvar "banner".
//
string tag_imho_banner(string tag_name, mapping arg, object id, object file)
{
 DEBUG(tag_name);
 return QUERY(banner);
}

//
//! tag: imho_title
//!  Return the symbolic name of the current screen eg mailindex, compose, etc..
//
string tag_imho_title(string tag_name, mapping arg, object id, object file)
{
  if (id->misc->imho && id->misc->imho->sessobj) {
    DEBUG(tag_name);
    mapping sessobj = id->misc->imho->sessobj;
    if (screennames[sessobj->status])
      return (MSG(screennames[sessobj->status][1]));
    else
      return("");
  }
}

//
//! tag: imho_name
//!  Resturn the current username on the current session
//! attribute: first
//!  Will give the first part of the name before a blank space
//! attribute: second
//!  Will give the second part of the name after the blank space...
//
string tag_imho_name(string tag_name, mapping arg, object id, object file)
{
  if (id->misc->imho && id->misc->imho->sessobj)
    {
      DEBUG(tag_name);
      array username = id->misc->imho->sessobj->name/" ";
      
      if (arg->first)
	{
	  return username[0];
	}
      else
	{
	  if (arg->second) {
	    if (sizeof(username) > 1)
	      return username[1];
	    else
	      return "";
	  }
	  return id->misc->imho->sessobj->name;
	}
    }
  else
    return "";

}

//
//! tag: imho_frame
//!  Wrapper to HTML &lt;frame ... &gt; tag for CAMAS framed layout. It use 
//!  internal data to set the src= arg with correct CAMAS arguments.
//! attribute: name
//!  Internal CAMAS' symbolic frame name used to generate HTML src parameter
//!  for standaart &lt;frame src=..&gt;
//
string tag_imho_frame(string tag_name, mapping arg, object id, object file)
{
  object camas_module = CAMAS_MODULE;
  if (id->misc->imho && id->misc->imho->sessobj && objectp(camas_module)) {
    DEBUG(tag_name);
    string src = id->misc->imho->nexturl+arg->name + "-" + CAMAS.Tools.rand_string (8);
    string ret = "<frame src=\"" + src + "\" ";
    foreach ((indices (arg) - ({ "/" })), string a)
      ret += " " + a + "=" + html_encode_tag_value (arg[a]) + "";
    ret += ">";
    return ret;
  }
  return "";
}

//
//! tag: imho_main
//!  Main CAMAS tag for (allmost) all functions.
//! see also: imho_mailindex
//!  Show the index of all mails for one mailbox
//! see also: imho_compose
//!  Compose a mail
//! see also: imho_readmail
//!  Read a mail
//! see also: imho_folderlist
//!  List the folders
//! see also: imho_mailfilter
//!  Access to mail filters
//! see also: imho_files
//!  Access to files
//! see also: imho_addressbook
//!  Access to the address book
//! see also: imho_setup
//!  Setup the CAMAS preferences
//! attribute: thcolor
//!  To be documented
//! attribute: thbgcolor
//!  To be documented
//! attribute: tdbgcolor
//!  To be documented
//! attribute: checkcolor
//!  To be documented
//! attribute: checksize
//!  Size of the check box
//! attribute: arrowsize
//!  Size of the arrow
//! attribute: check
//!  Draw a checkbox (to be verified)
//! attribute: arrowup
//!  Draw a up arrow
//! attribute: arrowdown
//!  Draw a down arrow
//! attribute: folder
//!  To be documented
//! 
// FIXME: see how we can move this from camas.pike....
string tag_imho_main(string tag_name, mapping arg, object id, object file)
{
  object camas_module = CAMAS_MODULE;
  if (id->misc->imho && id->misc->imho->sessobj && objectp(camas_module)) {
    string thcolor=arg->thcolor || "#ffffff";
    string thbgcolor=arg->thbgcolor || "#000070";
    string tdbgcolor=arg->tdbgcolor || "#fcfce0";
    string checkcolor=arg->checkcolor || "#000000";
    int checksize=(int)arg->checksize || 15;
    int arrowsize=(int)arg->arrowsize || 15;
    string check=arg->check||"";
    string arrowup=arg->arrowup||"";
    string arrowdown=arg->arrowdown||"";
    string folder=arg->folder||"";
    camas_module->setup_state(id,arg);

    int screen = id->misc->imho->sessobj->status;
    if (screen != DIALOGBOX) {
      switch(tag_name) {
       case "imho_mailindex":
	screen = MAILINDEX;
	break;
       case "imho_compose":
	screen = COMPOSE;
	break;
       case "imho_readmail":
	screen = READMAIL;
	break;
       case "imho_folderlist":
	screen = FOLDERLIST;
	break;
       case "imho_mailfilter":
        screen = MAILFILTER;
        break;
       case "imho_files":
	screen = FILES;
	break;
       case "imho_addressbook":
	screen = ADDRESSBOOK;
	break;
       case "imho_setup":
	screen = SETUP;
	break;
      }
        }

    string main = camas_module->create_main (id, screen);

    main = replace (main,
		    ({ "&imho_folder;", "&imho_thcolor;", "&imho_thbgcolor;", 
		       "&imho_tdbgcolor;", "&imhocheckfg;", "&imhochecksrc;",
		       "&imhoarrowupsrc;", "&imhoarrowdownsrc;", "&imhochecksize;",
		       "&imhoarrowsize;" }),
		    ({ (sizeof (folder) ? folder : "internal-gopher-menu"), thcolor, thbgcolor,
		       tdbgcolor, checkcolor, check,
		       arrowup, arrowdown, (string)(checksize * 4), (string)(arrowsize * 4) }) );
    return main;
  }
  return "";
}

//
//! tag: imho_about
//!  Show version of CAMAS and eventualy a link to camas website.
//! attribute: nolink
//!  Do not add a link to camas website
//
string tag_imho_about(string tag_name, mapping arg, object id, object file)
{
  object camas_module = CAMAS_MODULE;
  if (objectp(camas_module)) 
  {
   DEBUG(tag_name);
   return (arg->nolink ? "" : "<a target=\"_top\" href=\"" 
                              + camas_module->server_url (id)  
                              + camas_module->QUERY(location) + "0/about\">" )
          + "CAMAS v" + camas_module->camas_version 
          + (arg->nolink ? "" : "</a>");
  }
  return "";
}

//
//! tag: imho_buttons
//!  Returns the standart CAMAS buttons (eg compose mail, mail index,
//!  folderlist, files, address book, filter book, setup and logout).
//! attribute: bg
//!  Backgroup color of the table (to be verified).
//! attribute: fg
//!  Color of the text
//! attribute: magic_fg
//!  Color of the used when gtext is used.
//! attribute: scale
//!  Scale of the gtext image generated (see gtext tag).
//! attribute: xsize
//!  To be documented
//! attribute: breakline
//!  Add a &lt;br /&gt; after each buttons...
//
string tag_imho_buttons(string tag_name, mapping arg, object id, object file)
{
  object camas_module = CAMAS_MODULE;

  if(id->misc->imho && id->misc->imho->sessobj && objectp(camas_module)) {
    DEBUG(tag_name);
    mapping sessobj = id->misc->imho->sessobj;
    string bg=arg->bg || "#c0c0c0";
    string fg=arg->fg || "#000000";
    string magic_fg=arg->magic_fg || "#303030";
    string scale=arg->scale || "0.4";
    string ret = "", folder="";
    camas_module->setup_state(id,arg);

    if (id->misc->imho && id->misc->imho->sessobj) {
      folder= id->misc->imho->sessobj->mailbox[MB_HIERARCHY_IDX][-1];
      folder = sessobj->imapclient->translate_frommbox(sessobj, folder);
    }
    else
      folder = MSG(M_INBOX);

    string args="";
    if (arg->xsize)
      args += " xsize="+arg->xsize+" ";

    ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->imho->nexttarget+"?actioncompose=1\" target=\""+id->misc->imho->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+MSG(M_NEWMAIL)+"</imho_text>"+(arg->breakline?"<br />":"");
    ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->imho->nexttarget+"?actionindex=1\" target=\""+id->misc->imho->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+(camas_module->feature(FEAT_MAILBOXES)?(arg->noupdate?MSG(M_CURRMAILBOX):  replace(MSGA(M_MAILBOX,({"\0"})),"\0",folder)  ):MSG(M_INBOX))+"</imho_text>"+(arg->breakline?"<br />":"");
    if(camas_module->feature(FEAT_MAILBOXES))
      ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->imho->nexttarget+"?actionfolderlist=1\" target=\""+id->misc->imho->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+MSG(M_MAILBOXES)+"</imho_text>"+(arg->breakline?"<br />":"");
    if(camas_module->feature(FEAT_ATTACHMENTS) && sizeof(camas_module->QUERY(uploaddir)))
      ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->imho->nexttarget+"?actionfiles=1\" target=\""+id->misc->imho->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+MSG(M_FILES)+"</imho_text>"+(arg->breakline?"<br />":"");
    if(camas_module->feature(FEAT_ADDRESSBOOK))
      ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->imho->nexttarget+"?actionaddressbook=1\" target=\""+id->misc->imho->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+MSG(M_ADDRESSBOOKTITLE)+"</imho_text>"+(arg->breakline?"<br />":"");
    if(camas_module->feature(FEAT_MAILFILTER))
      ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->imho->nexttarget+"?actionfilterbook=1\" target=\""+id->misc->imho->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+MSG(M_FILTERBOOKTITLE)+"</imho_text>"+(arg->breakline?"<br />":"");
    if(camas_module->feature(FEAT_USEREDITSETUP))
      ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->imho->nexttarget+"?actionsetup=1\" target=\""+id->misc->imho->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+MSG(M_PREFS)+"</imho_text>"+(arg->breakline?"<br />":"");
    ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->imho->nexttarget+"?actionsetup=1\" target=\""+id->misc->imho->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+MSG(M_PREFS)+"</imho_text>"+(arg->breakline?"<br />":"");

    if(camas_module->QUERY(logout))
    {
      string logout_url = id->misc->imho->nexturl + "_top" + "-" + CAMAS.Tools.rand_string (8) + "?actionlogout=1";
      ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\"";
      ret += ((camas_module->QUERY(jslogout) && id->supports->javascript) ?
	    ("javascript:CAMAS_Confirm=0;top.location='" + logout_url + "';") : logout_url);
      ret += "\" target=\"_top\" magic magic_fg=" + magic_fg + " bevel=4 fg=" + fg;
      ret += " xspacing=10 yspacing=5>" + MSG (M_LOGOUT) + "</imho_text>";
      ret += (arg->breakline ? "<br />" : "");
    }
    return ret;
  }
  return "";
}

//
//! tag: imho_string
//!  Returns the string from the current language translation
//! attribute: no
//!  Number of the message (see camas/language.h file)
//
string tag_imho_string(string tag_name, mapping arg, object id, object file)
{
  if (id->misc->imho) {
    DEBUG(tag_name);
    mapping sessobj = id->misc->imho->sessobj;
    if (sessobj && arg["no"])
	return(MSG((int) arg["no"]));
  }
  return "";
}

//
//! tag: imho_submit
//!  To be documented (err... what's this tag for ?)
//! attribute: type
//!  To be documented
//! attribute: submit
//!  To be documented
//! attribute: class
//!  To be documented
//! attribute: button
//!  To be documented
//! attribute: value
//!  To be documented
//
string tag_imho_submit(string tag_name, mapping arg, object id, object file)
{
  if (id->misc->imho) {
    DEBUG(tag_name);
    mapping args = ([ "type": "submit", "class": "button", "value": arg->value ]);
    // Allow all arguments. Lets e.g. javascript through.
    foreach ((indices (arg) - ({ "/" })), string a) {
      if (!(args[a]))
	args[a] = arg[a];
    }
    string tag = make_tag ("input", args);
    return tag[0..sizeof (tag)-2] + " />";
  }
  return "";
}

//
//! tag: imho_mailbox
//!  Returns the name of the current mailbox
//
string tag_imho_mailbox(string tag_name, mapping arg, object id, object file)
{
  if (id->misc->imho && id->misc->imho->sessobj) {
    DEBUG(tag_name);
    mapping sessobj = id->misc->imho->sessobj;
    string mbox = sessobj->mailbox[MB_HIERARCHY_IDX][-1];
    mbox = sessobj->imapclient->translate_frommbox(sessobj, mbox);
     
    return(html_encode_string(mbox));
  }
  return "";
}

//
//! tag: imho_activemailboxbuttons
//!  Insert a list of user defined inboxes
//! attribute: bg
//!  Background color of the cells
//! attribute: fg
//!  Foreground color of the cells
//! attribute: magic_fg
//!  Magic color when gtext is used
//! attribute: scale
//!  Gtext scale
//! attribute: xsize
//!  To be documented
//! attribute: noupdate
//!  This set of button will not be updated for every action, so it don't
//!  display e.g. the name of the current mailbox.
string tag_imho_activemailboxbuttons(string tag_name, mapping arg, object id, object file)
{
  object camas_module = CAMAS_MODULE;
  if(id->misc->imho && id->misc->imho->sessobj && objectp(camas_module)) {
    DEBUG(tag_name);
    mapping sessobj=id->misc->imho->sessobj;
    string bg=arg->bg || "#c0c0c0";
    string fg=arg->fg || "#803000";
    string magic_fg=arg->magic_fg || "#402000";
    string scale=arg->scale || "0.3";
    string ret = "", folder;
    array mbox_parts;
    camas_module->setup_state(id,arg);

    string args="";
    if (arg->xsize)
      args += " xsize="+arg->xsize+" ";

    string name = sessobj->imapclient->translate_frommbox(sessobj, "INBOX");
    ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->imho->nexttarget+"?actionindex=1&mbox=INBOX\" target=\""+id->misc->imho->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+name+(arg->noupdate?"":(" "+sessobj->cache["INBOX"]->messages))+"</imho_text>"+(arg->breakline?"<br />":"");
    return ret;
  }
  return "";
}

//
//! tag: imho_url
//!  Return the url that is used to reach internal objects like mailboxes...
//! attribute: href
//!  To be documented
//! attribute: mbox
//!  Give the link to reach mailbox named by this attribute
//! attribute: target
//!  Send it to framed named by this target.
string tag_imho_url(string tag_name, mapping arg, object id, object file)
{
  object camas_module = CAMAS_MODULE;
  if(id->misc->imho && id->misc->imho->sessobj && objectp(camas_module)) {
    DEBUG(tag_name);
    string ret = "", action="", mbox="";
    string tag = "";
    mapping sessobj = id->misc->imho->sessobj;
    string target = "";
    camas_module->setup_state(id,arg);
    int mboxno = -1; 
    foreach(indices(arg), string a) {
      switch(a) {
        case "href":
          action = arg[a];
          break;
        case "mbox":
          mboxno = sessobj->imapclient->find_mailbox(sessobj, arg[a]);
          if (mboxno == -1)
    	    mboxno = 0;
          break;
        case "target":
          break;
        case "/": // avoid: <imho_url href=\"checkactivemailboxes\" /> ==> / is not an attribute !
          break;
        default:
          ret += "&"+a+"="+arg[a];
          break;
      }
    }
    ret=id->misc->imho->nexttarget+"?action"+action+"=1"+ret;
    if (mboxno >= 0)
      ret+="&mbox="+http_encode_url(sessobj->mailboxes[mboxno][MB_FOLDERNAME_IDX]);
    return(ret);
  }
  return "";

}

//
//! tag: imho_countnewmails
//!  Return the number of new/unread mails in the current inbox or 
//!  &lt;FALSE&gt; if the is NO new mail at all in the current mailbox
//!  If attribute is used then It will send a the correct string for
//!  each states.
//! attribute: nomessages
//!  The string given will be sent when there is no new message in the 
//!  current mailbox
//! attribute: onemessage
//!  The string given will be sent when there is ONE new message in the
//!  current mailbox
//! attribute: message
//!  The string given will be sent when there is more than one new
//!  message in the current mailbox.
//! attribute: prefix
//!  Is the string to insert before the number of messages
//! attribute: prehave
//!  Is the string to insert before the number of message (backward 
//!  backward compatibility with IMHO 0.97 templates).
//
string tag_imho_countnewmails(string tag_name, mapping arg, object id, object file)
{
  if (id->misc->imho && id->misc->imho->sessobj) {
    DEBUG(tag_name);
    
    mapping sessobj=id->misc->imho->sessobj;

    mapping email;
    int cpt=0;
    int nbmessages=sizeof(sessobj->mails);
    if (nbmessages) {
      foreach(sessobj->mails, email)
      {
	if (sizeof(email->imap->FLAGS) == 0) cpt++;
	else
	  if (search(email->imap->FLAGS,"\\Seen") == -1) cpt++;
      }
      nbmessages=cpt;
    }

    if (arg->nomessages && arg->onemessage && arg->messages && (arg->prefix||arg->prehave))
      {
	if (nbmessages==0)
	  return arg->nomessages;
	
	// So there is at least one message here ;-)
	if (nbmessages==1)
	  return sprintf("%s %d %s",(arg->prefix||arg->prehave), nbmessages, arg->onemessage);
	
	// So there is more than one message ;-)
	return sprintf("%s %d %s",(arg->prefix||arg->prehave), nbmessages, arg->messages);
      }
    else
      {
	return sprintf("%d%s", nbmessages,(nbmessages==0)?"<FALSE>":"<TRUE>");
      }
  }
  else
    return "";
}

//
//! tag: imho_address
//!  Returns the email address from preferences or the domain name from
//!  email address saved/entered in preferences
//! attribute: domain
//!  Give only the domain name
//
string tag_imho_address(string tag_name, mapping arg, object id, object file)
{
  if (id->misc->imho && id->misc->imho->sessobj) {
    DEBUG(tag_name);
    if (arg->domain)
      return sprintf("%s",((id->misc->imho->sessobj->defaultaddress)/"@")[1]);
    else
      return sprintf("%s",id->misc->imho->sessobj->defaultaddress);
  }
  else
    return "";
}

//
//! tag: imho_screen
//!  Will give the name of the current screen. Usefull for a help system
//!  or "unknown" when the screen name is not known...
//
string tag_imho_screen(string tag_name, mapping arg, object id, object file)
{
  if (id->misc->imho && id->misc->imho->sessobj)
  {
    DEBUG(tag_name);
    string screen="unknown";
    mapping sessobj=id->misc->imho->sessobj;

    if (screennames[sessobj->status])
      screen=screennames[sessobj->status][0];
    return screen;
  }
  else
    return "";
}

//
//! tag: imho_dumpid
//!  Will return the mapping ID in human readable form. Usefull for debug
//
string tag_imho_dumpid(string tag_name, mapping arg, object id, object file)
{
  DEBUG(tag_name);
  return (sprintf("<pre>Id:%O\n</pre>",mkmapping(indices(id),values(id))));
}

//
//! tag: imho_totalsize
//!  Return the totalsize of all folders in kB
//
string tag_imho_totalsize(string tag_name, mapping arg, object id, object file)
{
  object camas_module = CAMAS_MODULE;

  if(objectp(camas_module));
  {
    DEBUG(tag_name);
    return (string)camas_module->get_totalsize (id);
  }
  return "";
}

//
//! tag: imho_use
//!  Return the current usage in percent of the diskquota the user has
//
string tag_imho_use (string tag_name, mapping arg, object id, object file) {
  object camas_module = CAMAS_MODULE;
  if (id->misc->imho && id->misc->imho->sessobj && (int)id->misc->imho->sessobj->quota && objectp(camas_module)) {
    DEBUG(tag_name);
    float size = (float)camas_module->get_totalsize (id);
    return (string) ((int)(size / (float)id->misc->imho->sessobj->quota * 100.0));
  }
  return "";
}


//
//! tag: imho_doctype
//!  Return the doctype used
//
string tag_imho_doctype(string tag_name, mapping arg, object id, object file)
{
 DEBUG(tag_name);
 return "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" \"http://www.w3.org/TR/html4/loose.dtd\">\n";
}

//
//! tag: imho_logout
//!  This destroy and leave the camas session.<br />
//!  Use this tag in logout screen page in your templates.<br />
//!  This tags also set the name of the screen to "logout".
//
string tag_imho_logout (string tag_name, mapping args, object id, object file) {
  object camas_module = CAMAS_MODULE;

  if (id->misc->imho && id->misc->imho->sessobj && objectp(camas_module)) {
    DEBUG(tag_name);
    object sess123 = id->conf->get_provider ("123sessions");
    mapping sessobj = id->misc->session_variables;
    id->misc->imho->screen="logout";
    id->misc->imho->title=MSG(M_LOGGEDOUT);
#if constant(thread_create)
    object lock = camas_module->global_lock->lock();
#endif
    destruct(sessobj->imapclient);
#if constant(thread_create)
    destruct(lock);
#endif
    if (id->misc->session_id) 
      sess123->delete_session (id, id->misc->session_id, 1);
    }
 return "";
} 

//
//! tag: imho_quota
//!  Return the diskquota the user can use.
//
string tag_imho_quota (string tag_name, mapping arg, object id, object file)
{
  object camas_module = CAMAS_MODULE;
  DEBUG(tag_name);
  if ( objectp(camas_module))
   return (id->misc->imho && id->misc->imho->sessobj) ? (string)id->misc->imho->sessobj->quota : (string) camas_module->QUERY (quota);
  return "";
}

//
//! tag: imho_jslogout
//!  Little javascript devil to autoclose the CAMAS session when
//!  use close his brower :))
//
string tag_imho_jslogout (string tag_name, mapping arg, object id, object file) {
  object camas_module = CAMAS_MODULE;
  
  if (objectp(camas_module)) 
  DEBUG(tag_name);
  {
    string res = "";
    mapping sessobj = id->misc->imho->sessobj;
    if (id->supports->javascript && camas_module->QUERY (jslogout)) {
      res = "<script language=\"JavaScript\" type=\"text/javascript\"><!--\n";
      res += "var CAMAS_Width;\nvar CAMAS_Height;\nvar CAMAS_Confirm=1;\n\n";
      res += "function popup () {\n";
      res += "  if (CAMAS_Confirm) {\n";
      res += "    if (confirm ('" + MSG (M_ENDSESSION) + "')) {\n";

      string logout_url = "http://" + id->host 
        + (sizeof (id->prestate) ? ("/(" + indices (id->prestate) * "," + ")") : "")
        + camas_module->QUERY (location) + "_top?actionlogout=1";
      if (QUERY (debug))
        write("js_logout: onunload () => '" + logout_url + "');\n");
      res += "      window.open ('" + logout_url + "');\n";

      res += "    }\n";
      res += "    else {\n";
      res += "      day = new Date ();\n";
      res += "      id = day.getTime ();\n";
      res += "      window.open (this.location, 'CAMAS' + id, 'toolbar=1,scrollbars=1,location=1,";
      res += "status=1,menubar=1,resizable=1,personalbar=1,width=CAMAS_Width,height=CAMAS_Height');\n";
      res += "    }\n";
      res += "  }\n";
      res += " } \n\n";
    

      res += "function set_size () {\n";
      if (id->supports->msie) {
        res += "  CAMAS_Width  = document.body.offsetWidth;\n";
        res += "  CAMAS_Height = document.body.offsetHeight;\n";
      }
      else {
        res += "  CAMAS_Width  = window.outerWidth;\n";
        res += "  CAMAS_Height = window.outerHeight;\n";
      }
      res += " }\n\n";

      res += "window.onload = set_size;\n";
      res += "window.onunload = popup;\n\n";
    
      res += "//-->\n</script>\n";
    }
    return res;
  }
  return "";
}

//
//! tag: jsnotify
//!  RXML code to handle notify window
//
string tag_imho_jsnotify (string tag_name, mapping arg, object id, object file) {
  DEBUG(tag_name);
  string res = "";

  // Why use the RXML parser since we have there the whole CAMAS variables
  // to handle this kind of replacements ?
  // --Xavier
  if (id->misc->_xml_parser) {
    res += "<cset preparse=\"\" scope=\"var\" variable=\"url\"><imho_url href=\"checkactivemailboxes\" /></cset>\n";
    res += "<script language=\"JavaScript\" type=\"text/javascript\">\n";
    res += "  window.setTimeout('window.location=\"&var.url;\";',60000);\n";
    res += "</script>\n";
  }
  else {
    res += "<set variable=\"url\" eval=\"<imho_url href=\"checkactivemailboxes\" />\">\n";
    res += "<formoutput quote=\"$\">\n";
    res += "<script language=\"JavaScript\" type=\"text/javascript\">\n";
    res += "  window.setTimeout('window.location=\"$url:quote=none$\";',60000);\n";
    res += "</script>\n";
    res += "</formoutput>\n";
  }

  return res;
}


// Tags that this module provides
mapping query_tag_callers()
{
  return([ 
    "imho_servers":tag_imho_servers,
    "imho_banner":tag_imho_banner,
    "imho_title":tag_imho_title,
    "imho_name":tag_imho_name,
    "imho_frame":tag_imho_frame,
    "imho_about":tag_imho_about, 
    "imho_main":tag_imho_main,
    "imho_buttons":tag_imho_buttons,
    "imho_mailindex":tag_imho_main,
    "imho_compose":tag_imho_main,
    "imho_readmail":tag_imho_main,
    "imho_folderlist":tag_imho_main,
    "imho_files":tag_imho_main,
    "imho_addressbook":tag_imho_main,
    "imho_mailfilter":tag_imho_main,
    "imho_setup":tag_imho_main,
    "imho_string":tag_imho_string,
    "imho_submit":tag_imho_submit,
    "imho_mailbox":tag_imho_mailbox,
    "imho_activemailboxbuttons":tag_imho_activemailboxbuttons,
    "imho_url":tag_imho_url,
    "imho_countnewmails":tag_imho_countnewmails,
    "imho_address":tag_imho_address,
    "imho_screen":tag_imho_screen,
    "imho_dumpid":tag_imho_dumpid,
    "imho_totalsize":tag_imho_totalsize,
    "imho_use":tag_imho_use,
    "imho_doctype":tag_imho_doctype,
    "imho_emailaddress":tag_imho_address,
    "imho_href":tag_imho_url,
    "imho_username":tag_imho_name,
    "imho_logout": tag_imho_logout,
    "imho_quota": tag_imho_quota,
    "imho_jslogout": tag_imho_jslogout,
    "imho_jsnotify": tag_imho_jsnotify,
  ]);
}

// Containers now

//
//! container: imho_text
//!  Container used intead of &lt;gtext&gt; and make normal text
//!  even we don't use gtext. Font face and foreground color will
//!  be preserved even if nogtext
//! attribute: href
//!  like a href=
//! attribute: fg
//!  foreground color
// todo: suppress the code remaining in camas.pike 'cause of imho_kludge...
//
string cont_imho_text(string tag_name, mapping arg, string content, object id)
{
  DEBUG(tag_name);
  string ret = "";
  if (id->misc->imho && id->misc->imho->nogtext) {
    if (arg->href) {
      ret += "<a";
      foreach(indices(arg), string a)
	ret += " "+a+"=\""+arg[a]+"\"";
      ret += ">";
    }
    if (arg->fg || id->misc->imho->face || id->misc->imho->size) {
      ret += "<font ";
      if (arg->fg)
	ret += " color="+arg->fg;
      if (id->misc->imho->face)
	ret += " face=\""+id->misc->imho->face+"\"";
      // **XB**
      if (id->misc->imho->size)
        ret += " size=\"" + id->misc->imho->size + "\"";
      ret += ">";
      if (arg->href)
	ret += "[&nbsp;"+replace(content," ","&nbsp;")+"&nbsp;]";
      else
	ret += content;
      ret += "</font>";
    }
    else {
      if (arg->href)
	ret += "[&nbsp;"+replace(content," ","&nbsp;")+"&nbsp;]";
      else
	ret += content;
    }
    if (arg->href)
      ret += "</a> ";
  } else {
    ret += "<gtext";
    foreach(indices(arg), string a) {
      if (a == "nfont" && id->misc->imho->face)
	ret += " "+a+"=\""+id->misc->imho->face+"\"";
      else
	ret += " "+a+"="+html_encode_tag_value(arg[a])+"";
    }
    ret += ">"+content+"</gtext>";
  }
  return ret;
}

//
//! container: imho_gtext
//!  Container used for &lt;a&gt; with links to CAMAS actions
//! see also: imho_a
//! attribute: bg
//!  The background color
//! attribute: fg
//!  The foreground color
//! attribute: breakline
//!  Insert a &lt;br&gt; after each buttons
//! attribute: magic_fg
//!  The color of the highlighted button
//! attribute: noupdate
//!  This set of button will not be updated for every action, so don't
//!  don't display e.g. the name of the current mailbox.
//! attribute: scale
//!  The gtext scale.
//! attribute: xsize
//!  Specify the xsize of all buttons (usefull if placed in a column)
//
string cont_imho_href(string tag_name, mapping arg, string content, object id)
{
  object camas_module = CAMAS_MODULE;
  if (id->misc->imho && id->misc->imho->sessobj && objectp(camas_module)) {
    DEBUG(tag_name); 
    string ret = "", action="", mbox="", redirect_url="";
    string tag = "";
    sscanf(tag_name,"imho_%s", tag);
    ret += "<"+tag;
    mapping sessobj = id->misc->imho->sessobj;
    string target = "";
    camas_module->setup_state(id,arg);
    int mboxno = -1; 
    foreach(indices(arg), string a) {
      switch(a) {
       case "href":
        action = arg[a];
        break;
       case "redirect_url":
        redirect_url = arg[a];
        break;
       case "mbox":
        mboxno = sessobj->imapclient->find_mailbox(sessobj, arg[a]);
        if (mboxno == -1)
	  mboxno = 0;
        break;
       default:
        ret+=" "+a+"="+html_encode_tag_value(arg[a])+"";
      }
    }
    ret+=" href=\""+id->misc->imho->nexttarget+"?action"+action+"=1";
    if (sizeof(redirect_url) > 1) ret += "&redirect_url="+redirect_url;
    if (mboxno >= 0)
      ret+="&mbox="+http_encode_url(sessobj->mailboxes[mboxno][MB_FOLDERNAME_IDX]);
    ret+="\">"+content+"</"+tag+">";
    return(ret);
  }
  return "";

}

//
//! container: imho_buttonoutput
//!  To be documented
//! attribute: buttons
//!  To be documented
//! attribute: noupdate
//!  To be documented
//! bugs:
//!  Fix this doc...
//
string cont_imho_buttonoutput(string tag_name, mapping arg, string content, object id)
{
  object camas_module = CAMAS_MODULE;

  if (id->misc->imho && id->misc->imho->sessobj && objectp(camas_module)) {
    DEBUG(tag_name);
    mapping sessobj = id->misc->imho->sessobj;
    array vars = ({ });
    string folder = "";
    camas_module->setup_state (id, arg);

    if (id->misc->imho && id->misc->imho->sessobj) {
      folder = id->misc->imho->sessobj->mailbox[MB_HIERARCHY_IDX][-1];
      folder = sessobj->imapclient->translate_frommbox(sessobj, folder);
    }
    else
      folder = MSG(M_INBOX);

    //oliv3 FIXME next line: (args->button || "main") == "inboxes" ???
    if (lower_case (arg->buttons || "main") == "inboxes") {
      string name = sessobj->imapclient->translate_frommbox (sessobj, "INBOX");
      vars += ({
	([ 
	  "button-url": id->misc->imho->nexttarget + "?actionindex=1&mbox=INBOX",
	  "button-target": id->misc->imho->target,
	  "button-text": name + (arg->noupdate ? "" : (" " + sessobj->cache["INBOX"]->messages))
	]) });
    }
    else {
      vars += ({ 
	([ 
	  "button-name":"compose",
	  "button-url":id->misc->imho->nexttarget+"?actioncompose=1", 
	  "button-target":id->misc->imho->target,
	  "button-text":MSG(M_NEWMAIL)
	]),
	
	([
	  "button-name":"mailindex",
	  "button-url":id->misc->imho->nexttarget+"?actionindex=1", 
	  "button-target":id->misc->imho->target, 
	  "button-text":(camas_module->feature (FEAT_MAILBOXES)?(arg->noupdate?MSG(M_CURRMAILBOX): replace(MSGA(M_MAILBOX,({"\0"})),"\0",folder)  ):"INBOX")
	]) 
      });

      if(camas_module->feature(FEAT_MAILBOXES))
	vars += ({
	  ([ 
	    "button-name":"mailboxes",
	    "button-url":id->misc->imho->nexttarget+"?actionfolderlist=1", 
	    "button-target":id->misc->imho->target,
	    "button-text":MSG(M_MAILBOXES)
	  ])
	});
      
      
      if(camas_module->feature(FEAT_ATTACHMENTS) && sizeof(camas_module->QUERY(uploaddir)))
	vars += ({
	  ([ 
	    "button-name":"files",
	    "button-url":id->misc->imho->nexttarget+"?actionfiles=1", 
	    "button-target":id->misc->imho->target,
	    "button-text":MSG(M_FILES)
	  ])
	});
      
      if(camas_module->feature(FEAT_ADDRESSBOOK))
	vars += ({
	  ([ 
	    "button-name":"addressbook",
	    "button-url":id->misc->imho->nexttarget+"?actionaddressbook=1", 
	    "button-target":id->misc->imho->target,
	    "button-text":MSG(M_ADDRESSBOOKTITLE)
	  ])
	});
      
      if(camas_module->feature(FEAT_MAILFILTER))
        vars += ({
	  ([
	    "button-name":"mailfilters",
	    "button-url":id->misc->imho->nexttarget+"?actionfilterbook=1",
	    "button-target":id->misc->imho->target,
	    "button-text":MSG(M_FILTERBOOKTITLE)
	  ])
	});

      if(camas_module->feature(FEAT_USEREDITSETUP))
	vars += ({
	  ([ 
	    "button-name":"preferences",
	    "button-url":id->misc->imho->nexttarget+"?actionsetup=1", 
	    "button-target":id->misc->imho->target,
	    "button-text":MSG(M_PREFS)
	  ])
	});
        string logout_url = id->misc->imho->nexturl + "_top" + "-" 
                            + CAMAS.Tools.rand_string (8) + "?actionlogout=1";
        vars += ({ 
  	([ 
  	  "button-name":"logout",
  	  "button-url":
	  ((camas_module->QUERY (jslogout) && id->supports->javascript) ?
	   ("javascript:CAMAS_Confirm=0;top.location='" + logout_url + "';") : logout_url),
	  "button-target":"_top",
	  "button-text":MSG(M_LOGOUT)
	])
        });
    }

    return do_output_tag (arg, vars, content, id);
  }
  return "";
}

//
//! container: imho_jump
//!  Custom CAMAS logout button
//! attribute: redirect
//!  The url to be redirected to when quitting CAMAS
//! example: rxml
//!  To add an exit button to Caudium website 
//!   {imho_jump redirect="http://caudium.net"}Exit to Caudium Website{/imho_jump}
//
string cont_imho_jump(string tag_name, mapping arg, string content, object id)
{
  DEBUG(tag_name);
  object camas_module = CAMAS_MODULE;
  if(objectp(camas_module)) 
  {
    if (!arg->redirect)
      return "";
    else
    {
      string redir = "<a target=\"_top\" href=\"" ;
      redir += camas_module->server_url (id) ;
      redir += camas_module->QUERY (location);
      redir += "0/jump/" + http_encode_url (arg->redirect) + "\">";
      redir += content + "</a>";
      return redir;
    }
  }
  return "";
}


// Containers that this module provides
mapping query_container_callers() { 
  return ([
    "imho_text": cont_imho_text,
    "imho_a":cont_imho_href,
    "imho_gtext":cont_imho_href,
    "imho_buttonoutput":cont_imho_buttonoutput,
    "imho_jump":cont_imho_jump
  ]); 
}

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! Debug the call / errors into Caudium error log ?
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: banner
//! Banner text. This text is inserted on top of every page if you are using the default layout. Can be inserted using &lt;imho_banner&#xa0;/&gt; in a layout file.
//!  type: TYPE_STRING
//!  name: Banner text
//
