/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BSCommander is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BSCommander is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *-------------------------------------------------------------------
 * Project: Beesoft Commander
 * Author : Piotr Pszczolkowski
 * Contact: piotr@beesoft.org
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "RightView.h"
#include "Settings.h"
#include "Config.h"
#include "Workspace.h"
#include <qlayout.h>
#include <qtabbar.h>
#include <qapplication.h>

/*------- local constants:
-------------------------------------------------------------------*/
const QString RightView::ArgPrefix       = "-rp=";
const QString RightView::ConfigGroupName = "RigthView";

//*******************************************************************
// RoghtPanel
//*******************************************************************
RightView::RightView( QWidget* const in_parent, Workspace* const in_workspace )
: ViewGeneric( in_parent, in_workspace )
{
	connect( d_tab_bar, SIGNAL( selected( int )      ), this, SLOT( slot_set_tab( int )         ));
	connect( this     , SIGNAL( disconnect_signals() ), d_workspace, SLOT( disconnect_signals() ));

	QString call_path = QString::null;
	for( int i = 0; i < qApp->argc(); ++i ) {
		QString arg = qApp->argv()[i];
		if( arg.startsWith( ArgPrefix ) ) {
			call_path = arg.mid( ArgPrefix.length() );
			break;
		}
	}

	if( Shared::is_dir_ok( call_path ) ) {
		add_new_tab( call_path, "" );
	}
	else {
		QString dname;
		QString fname;
		Settings::read( ConfigGroupName, ViewWindow::CONFIG_KEY_PATH, dname );
		Settings::read( ConfigGroupName, ViewWindow::CONFIG_KEY_FILE, fname );
		add_new_tab( dname, fname );
	}
}
// end of RightView

//*******************************************************************
// ~RightView
//*******************************************************************
RightView::~RightView()
{
	ViewWindow* const view = current_view();
	if( view ) {
		Settings::save( ConfigGroupName, ViewWindow::CONFIG_KEY_PATH , view->get_dir() );
		Settings::save( ConfigGroupName, ViewWindow::CONFIG_KEY_FILE , view->get_current_fname() );
		Settings::save( ConfigGroupName, ViewWindow::CONFIG_KEY_COMBO, view->get_combox_items() );
	}
}
// end of ~RightView

//*******************************************************************
// set_dir_combox                                            PRIVATE
//-------------------------------------------------------------------
// Odczytanie z pliku konfiguracyjnego nazw katalogow, ktore
// uzytkownik wprowadzil 'z reki'.
//*******************************************************************
void RightView::set_dir_combox( ViewWindow* const in_view )
{
	QStringList items;
	Settings::read( ConfigGroupName, ViewWindow::CONFIG_KEY_COMBO, items );
	in_view->set_combox_items( items );
}
// end of set_dir_combox

//*******************************************************************
// reconnect                                                  PUBLIC
//-------------------------------------------------------------------
// Funkcja wywolywana jest w 'workspace' po wylaczeniu wszystkich
// sygnalow zwiazanych z 'workspace'.
// Tutaj nawiazywane sa wszystkie polaczenia ponownie, z tym ze
// dla aktualnego widoku(!).
//*******************************************************************
void RightView::reconnect()
{
	connect( d_workspace, SIGNAL( tab_new_right()    ), this, SLOT( slot_new_tab()       ));
	connect( d_workspace, SIGNAL( tab_remove_right() ), this, SLOT( slot_remove_tab()    ));
	connect( d_workspace, SIGNAL( looks_refresh()    ), this, SLOT( slot_looks_refresh() ));
	connect( d_workspace, SIGNAL( update_lang()      ), this, SLOT( slot_update_lang()   ));
	
	ViewWindow* const view = current_view();
	if( view ) {
		connect( view       , SIGNAL( open_new_tab( const QString& )), this, SLOT( slot_new_tab( const QString& )));
		connect( view       , SIGNAL( tab_update( const QString& )  ), this, SLOT( slot_tab_update( const QString& )));
		connect( d_workspace, SIGNAL( ftp_start_right() ), view       , SLOT  ( slot_ftp_start() ));
		connect( d_workspace, SIGNAL( ftp_end()         ), view       , SLOT  ( slot_ftp_end()   ));
		connect( d_workspace, SIGNAL( filter( bool )    ), view       , SLOT  ( filter( bool )   ));
		connect( d_workspace, SIGNAL( reload()          ), view       , SLOT  ( cd_refresh()     ));
		connect( d_workspace, SIGNAL( select_right()    ), view       , SLOT  ( slot_select()    ));
		connect( d_workspace, SIGNAL( unselect_right()  ), view       , SLOT  ( slot_unselect()  ));
		connect( d_workspace, SIGNAL( revers_right()    ), view       , SLOT  ( slot_revers()    ));
		connect( view       , SIGNAL( ftp_rejected()    ), d_workspace, SLOT  ( slot_ftp_rejected()   ));
		// Obsluga klawiszy funkcyjnych.
		connect( d_workspace, SIGNAL( F2ToRight() ), view, SLOT( F2() ));
		connect( d_workspace, SIGNAL( F3ToRight() ), view, SLOT( F3() ));
		connect( d_workspace, SIGNAL( F4ToRight() ), view, SLOT( F4() ));
		connect( d_workspace, SIGNAL( F6ToRight() ), view, SLOT( F6() ));
		connect( d_workspace, SIGNAL( F7ToRight() ), view, SLOT( F7() ));
		connect( d_workspace, SIGNAL( F8ToRight() ), view, SLOT( F8() ));
		// sygnaly zwrotne z context-menu
		connect( view       , SIGNAL( F5()               ), d_workspace, SLOT( slotF5()              ));
		connect( view       , SIGNAL( F9()               ), d_workspace, SLOT( slotF9()              ));
		connect( d_workspace, SIGNAL( update_selection() ), view       , SIGNAL( selection_changed() ));
		connect( d_workspace, SIGNAL( tree_right()       ), view       , SLOT( tree()                ));
		connect( d_workspace, SIGNAL( md5_create_right() ), view       , SLOT( md5_create()          ));
		connect( d_workspace, SIGNAL( md5_check_right()  ), view       , SLOT( md5_check()           ));
	}
}
// end of create_connections

//*******************************************************************
// slot_new_tab                                         PRIVATE slot
//*******************************************************************
void RightView::slot_new_tab( const QString& in_dir )
{
	if( in_dir.isEmpty() ) {
		ViewWindow* const view = current_view();
		add_new_tab( view->get_dir(), view->get_current_fname() );
	}
	else {
		add_new_tab( in_dir );
	}
}
// end of slot_new_tab

//*******************************************************************
// add_new_tab                                               PRIVATE
//*******************************************************************
void RightView::add_new_tab( const QString& in_dir, const QString& in_fname )
{
	int id = -1;
	ViewWindow* const view = new_tab( in_dir, id );
	if( view ) {
		set_dir_combox( view );
		view->disp_this_location( in_dir, in_fname );
		d_tab_bar->setCurrentTab( id );
	}
}
// end of add_new_tab

//*******************************************************************
// slot_remove_tab                                      PRIVATE slot
//*******************************************************************
void RightView::slot_remove_tab()
{
	int index = remove_tab();
	
	if( index != -1 ) {
		// wybor nastepnej wyswietlanej zakladki
		// regula: nastepna lub ostatnia
		if( index >= d_tab_bar->count() ) {
			index = d_tab_bar->count() - 1;
		}
		QTab* const tab = d_tab_bar->tabAt( index );
		d_tab_bar->setCurrentTab( tab );
		slot_set_tab( d_tab_bar->currentTab() );	// !!!
	}
}
// end of slot_remove_tab

//*******************************************************************
// slot_set_tab                                         PRIVATE slot
//-------------------------------------------------------------------
// Zmiana zakladki - funckja wywolywana przez QTabBar.
//*******************************************************************
void RightView::slot_set_tab( const int in_id )
{
	set_tab( in_id );

	ViewWindow* const view = current_view();
	if( view ) {
		d_workspace->set_right_view_window( view );
		emit disconnect_signals();
		view->set_focus();
	}
}
// end of slot_set_tab
