/******************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BSCommander is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BSCommander is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "FTPCopyTo.h"
#include <qmessagebox.h>
#include <qpushbutton.h>

//*******************************************************************
// FTPCopyTo                                           CONSTRUCTOR
//*******************************************************************
FTPCopyTo::FTPCopyTo(   QWidget* const in_parent,
                        const FtpServerInfo& in_fsi,
                        const ViewTable::SelectedItems& in_items,
                        const QString& in_src_dir )
: CopyDialog    ( in_parent, TRUE )
, d_fsi         ( in_fsi )
, d_path        ( "" )
, d_dir_name    ( "" )
, d_cmd         ( CMD_NONE )
, d_pwd_cmd_id  ( -1 )
, d_running     ( FALSE )
, d_break       ( FALSE )
{
    d_dir.cd( in_src_dir );

    ViewTable::SelectedItems::const_iterator it = in_items.begin();
    while( it != in_items.end() ) {
        if( (*it)->is_dir() ) d_dirs_on_dir.push ( (*it)->name() );
        else                  d_files_on_dir.push( (*it)->name() );
        ++it;
    }

    connect( d_left_btn, SIGNAL( clicked() ), this, SLOT( run() ));
    connect( d_right_btn, SIGNAL( clicked() ), this, SLOT( cancel() ));
    
    connect( &d_ftp, SIGNAL( dataTransferProgress( int, int)), this, SLOT( transfer_progress( int, int )));
    connect( &d_ftp, SIGNAL( done( bool )), this, SLOT( done( bool )));
    connect( &d_ftp, SIGNAL( commandFinished( int, bool )), this, SLOT( raw_cmd_finish( int, bool )));
    connect( &d_ftp, SIGNAL( rawCommandReply( int, const QString& )), this, SLOT( raw_cmd_reply( int, const QString& )));
}
// end of FTPCopyTo

//*******************************************************************
// ~FTPCopyTo                                           DESTRUCTOR
//*******************************************************************
FTPCopyTo::~FTPCopyTo()
{
    d_ftp.abort();
}
// end of ~FTPCopyTo


//###################################################################
//#                                                                 #
//#                      P R I V A T E                              #
//#                                                                 #
//###################################################################


//*******************************************************************
// doit                                                      PRIVATE
//*******************************************************************
void FTPCopyTo::doit()
{
    Shared::idle();

    if( d_break ) {
        close();
    }
    else {
        // Najpierw kopiujemy konsekwetnie wszystkie pliki
        // az do oproznienia stosu.
        if( !d_files_on_dir.empty() ) {
            put();
        }
        // Jesli wszystkie pliki aktualnego katalogu sa juz przekopiowane
        // zabieramy sie do rekursywnego przechodzenia przez katalogogi.
        else if( !d_dirs_on_dir.empty() ) {
            // Jezeli aktualny katalog znajduje na szczycie stosu
            // oznacza to, ze przekopiowalismy cala jego zawartosc.
            if( d_dir_name == d_dirs_on_dir.top() ) {
                cd_up();
            }
            else {
                // Sa jeszcze do przekopiowania podkatalogi.
                const QString fname = d_dirs_on_dir.top();
                if( fname.isEmpty() || !Shared::is_regular_file( fname )) {
                    d_dirs_on_dir.pop();
                    doit();
                }
                else {
                    d_dirs.push( d_dir_name );
                    d_dir_name = fname;
                    cd( d_dir_name );
                }
            }
        }
        // Wszystko usuniete.
        else {
            close();
        }
    }
}
// end of doit

//*******************************************************************
// read_directory                                            PRIVATE
//*******************************************************************
void FTPCopyTo::read_directory()
{
    const QFileInfoList* const fil = d_dir.entryInfoList( Shared::get_file_filter(), QDir::Unsorted  );
    if( fil ) {
        QFileInfo* fi = 0;
        QString fname = "";
        QFileInfoListIterator it( *fil );
        while( it.current() ) {
            fi = it.current();
            fname = fi->fileName();
            if( fi->isDir() ) d_dirs_on_dir.push( fname );
            else d_files_on_dir.push( fname );
            ++it;
        }
    }
}
// end of read_directory

//*******************************************************************
// put                                                       PRIVATE
//-------------------------------------------------------------------
// Inicjacja kopiowania pliku z dysku na serwer.
//*******************************************************************
void FTPCopyTo::put()
{
    Shared::idle();

    if( d_break ) {
        close();
    }
    else {
        const QString fname = d_files_on_dir.top();
        d_files_on_dir.pop();

        QString lfs_fpath = d_dir.absPath();
        if( !lfs_fpath.endsWith( "/" ) ) lfs_fpath += "/";
        lfs_fpath += fname;
        
        QString ftp_fpath = d_path;
        if( !ftp_fpath.endsWith( "/" ) ) ftp_fpath += "/";
        ftp_fpath += fname;

        d_file.setName( lfs_fpath );

        if( d_file.open( IO_ReadOnly ) ) {
            display_info(  lfs_fpath, ftp_fpath );
            d_cmd = CMD_PUT;
            d_ftp.put( &d_file, ftp_fpath );
        }
        // W przeciwnym przypadku wyswietlamy informacje o bledzie
        // i kontynuujemy kopiowanie pozostalych plikow.
        // UWAGA: informacje o tym pliku zostala juz zdjeta ze stosu.
        else {
            read_open_error( lfs_fpath );
            doit();
        }
    }
}
// end of get

//*******************************************************************
// cd_up                                                     PRIVATE
//*******************************************************************
void FTPCopyTo::cd_up()
{
    d_dir.cdUp();
    d_cmd = CMD_CDUP;
    d_ftp.cd( ".." );
}
// end of cd_up

//*******************************************************************
// cd                                                        PRIVATE
//*******************************************************************
void FTPCopyTo::cd( const QString& in_dir_name )
{
    bool is_cd_ok = FALSE;
    
    // wchodzimy do katalogu na dysku lokalnym
    d_dir.mkdir( in_dir_name, FALSE );
    if( d_dir.cd( in_dir_name, FALSE ) ) {
        if( Shared::is_dir_ok( d_dir.absPath() )) {
            read_directory();
            is_cd_ok = TRUE;
        }
    }
    // jesli nie bylo problemow probujemy utworzyc
    // taki sam katalog na serwerze
    if( is_cd_ok ) {
        d_cmd = CMD_MKDIR;
        d_ftp.mkdir( in_dir_name );
//        check_dir( in_dir_name );
    }
    else {
        doit();
    }
}
// end of cd

//*******************************************************************
// pwd                                                       PRIVATE
//*******************************************************************
void FTPCopyTo::pwd()
{
    static const QString cmd("PWD");
    
    d_cmd = CMD_NONE;
    d_pwd_cmd_id = d_ftp.rawCommand( cmd );
}
// end of ftp_pwd

//*******************************************************************
// check_dir                                                 PRIVATE
//*******************************************************************
void FTPCopyTo::check_dir( const QString& in_dir_name )
{
/*
    static const QString ls( "LIST " );
    
    QString ftp_fpath = d_path;
    if( !ftp_fpath.endsWith( "/" ) ) ftp_fpath += "/";
    ftp_fpath += in_dir_name;
    
    
    const QString cmd = ls + ftp_fpath + "\r\n";
    qWarning( "FTPCopyTo::check_dir() - " + cmd );
    
    d_cmd = CMD_NONE;
    d_check_dir_id = d_ftp.rawCommand( cmd );
*/
}
// end of check_dir

//*******************************************************************
// close                                                     PRIVATE
//*******************************************************************
void FTPCopyTo::close()
{
    d_cmd = CMD_CLOSE;
    d_ftp.close();
}
// end of close


//###################################################################
//#                                                                 #
//#                          S L O T S                              #
//#                                                                 #
//###################################################################


//*******************************************************************
// run                                                  PRIVATE slot
//-------------------------------------------------------------------
// Funkcja jest uruchamiana po nacisnieciu lewego przycisku
// w dialogu 'FTPCopyDialog'.
//*******************************************************************
void FTPCopyTo::run()
{
    d_cmd = CMD_LOGIN;
    d_ftp.connectToHost( d_fsi.addr(), d_fsi.port() );
    d_ftp.login( d_fsi.user(), d_fsi.pass() );
}
// end of run

//*******************************************************************
// cancel                                               PRIVATE slot
//*******************************************************************
void FTPCopyTo::cancel()
{
    if( d_running ) {
        d_break = TRUE;
    }
    else {
        accept();
        emit done();
    }
}
// end of run

//*******************************************************************
// transfer_progress                                    PRIVATE slot
//*******************************************************************
void FTPCopyTo::transfer_progress( const int in_done, const int in_total )
{
    set_progress( in_done, in_total );
}
// end of transfer_progress

//*******************************************************************
// done                                                 PRIVATE slot
//*******************************************************************
void FTPCopyTo::done( bool in_error )
{
    if( in_error ) {
        if( CMD_MKDIR == d_cmd ) {
            d_cmd = CMD_CD;
            d_ftp.cd( d_dir_name );
        }
        else {
            if( d_cmd != CMD_CLOSE ) {
                QMessageBox::critical( this, tr( d_caption ), d_ftp.errorString() );
                close();
            }
            else {
                reject();
                emit done();
            }
        }
    }
	else {
        switch( d_cmd ) {
            case CMD_LOGIN:
                d_running = TRUE;
                runned();
                d_cmd = CMD_CD;
                d_ftp.cd( d_fsi.dir()  );
                break;
            case CMD_MKDIR:
                d_cmd = CMD_CD;
                d_ftp.cd( d_dir_name );
                break;
            case CMD_CD:
                pwd();
                break;
            case CMD_CDUP:
                d_dirs_on_dir.pop();
                if( d_dirs.empty() ) {
                    d_dir_name = "";
                }
                else {
                    d_dir_name = d_dirs.top();
                    d_dirs.pop();
                }
                d_cmd = CMD_NONE;
                doit();
                break;
            case CMD_PUT:
                d_file.close();
                d_cmd = CMD_NONE;
                doit();
                break;
            case CMD_CLOSE:
                d_cmd = CMD_NONE;
                accept();
                emit done();
        }
    }
}
// end of done

//*******************************************************************
// raw_cmd_finish                                       PRIVATE slot
//*******************************************************************
void FTPCopyTo::raw_cmd_finish( int in_id, bool in_error )
{
    if( in_error ) {
        QMessageBox::critical( this, tr( d_caption ), d_ftp.errorString() );
        close();
    }
}
// end of raw_list_finish

//*******************************************************************
// raw_cmd_reply                                        PRIVATE slot
//-------------------------------------------------------------------
// Jako ospowiedz otrzymujemy tekst np. "/test/a" is current directory.
// Jak widac nazwa aktualnego katalogu jest zawarta w cudzyslowiu.
//*******************************************************************
void FTPCopyTo::raw_cmd_reply( int, const QString& in_detail )
{
    if( d_pwd_cmd_id != -1 ) {
        d_path = "";
        
        const QChar DELIMITER = '\"';    // cudzyslow
        const int idx_start = in_detail.find( DELIMITER );
        if( idx_start != -1 ) {
            const int idx_end = in_detail.find( DELIMITER, idx_start + 1 );
            if( idx_end != -1 ) {
                d_path = in_detail.mid( idx_start + 1, idx_end - idx_start - 1 );
			}
		}

		d_pwd_cmd_id = -1;
        d_cmd = CMD_NONE;
        doit();
	}
}
// end of ftp_cmd_reply
