/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BSCommander is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BSCommander is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "CmpDirs.h"
#include "Shared.h"
#include "CmpFiles.h"
#include "Config.h"
#include <qlayout.h>
#include <qgroupbox.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qlistview.h>
#include <qdir.h>
#include <qmessagebox.h>
#include <vector>
#include <algorithm>
using namespace std;

/*------- local constants:
-------------------------------------------------------------------*/
const QString CmpDirs::Caption         = QT_TR_NOOP( "Directories comparing" );
const QString CmpDirs::InfoCaption     = QT_TR_NOOP( "Infomation about the file" );
const QString CmpDirs::InfoMsg         = QT_TR_NOOP( "Path: %1\nSize: %2 bytes\nLast modyfication: %3\nOwner: %4\nGroup: %5\nExecutable: %6" );
const QString CmpDirs::GboxDirsCaption = QT_TR_NOOP( "Compared directories" );
const QString CmpDirs::GboxTblCaption  = QT_TR_NOOP( "Result" );
const QString CmpDirs::LeftDirLabel    = QT_TR_NOOP( "Left panel directory" );
const QString CmpDirs::RightDirLabel   = QT_TR_NOOP( "Right panel directory" );
const QString CmpDirs::FileNotPresents = QT_TR_NOOP( "not presents" );
const QString CmpDirs::Equal           = "=";
const QString CmpDirs::NotEqual        = "!=";
const QString CmpDirs::Error           = "x";
const QString CmpDirs::Dir             = "<dir>";
const QString CmpDirs::ViewHdrCaption[ ColsNumber ] = {
	QT_TR_NOOP( "File from left panel" ),
	QT_TR_NOOP( "Result" ),
	QT_TR_NOOP( "File from right panel" )
};

//*******************************************************************
// CmpDirs                                               CONSTRUCTOR
//*******************************************************************
CmpDirs::CmpDirs( QWidget* const in_parent, const QString& in_src_dir, const QString& in_dst_dir )
: QDialog       ( in_parent )
, d_src_dir     ( in_src_dir )
, d_dst_dir     ( in_dst_dir )
, d_run_btn     ( new QPushButton( tr(Shared::RunBtnLabel), this ))
, d_close_btn   ( new QPushButton( tr(Shared::CloseBtnLabel), this ))
, d_src_dir_path( 0 )
, d_dst_dir_path( 0 )
, d_view        ( 0 )
{
	setCaption( tr(Caption) );
	setFont( Config::instance()->lfs_default_font() );
	
	QHBoxLayout* const main_layout = new QHBoxLayout( this );
	main_layout->setMargin( Shared::LayoutMargin  );
	main_layout->setSpacing( Shared::LayoutSpacing );
	
	QVBoxLayout* const wrk_layout = new QVBoxLayout;
	{
		QGroupBox* const box = new QGroupBox( 2, Qt::Horizontal, tr(GboxDirsCaption), this );
		//.............................................
		new QLabel( tr(LeftDirLabel), box );
		d_src_dir_path = new InfoField( box );
		//.............................................
		new QLabel( tr(RightDirLabel), box );
		d_dst_dir_path = new InfoField( box );
		//.............................................
		wrk_layout->addWidget( box );
	}
	{
		QGroupBox* const box = new QGroupBox( 1, Qt::Vertical, tr(GboxTblCaption), this );
		d_view = new QListView( box );
		d_view->setPaletteBackgroundColor( Config::instance()->lfs_default_bkg_color() );
		d_view->setShowSortIndicator( TRUE );
		d_view->setSelectionMode( QListView::Extended );
		d_view->setAllColumnsShowFocus( TRUE );
		for( int i = 0; i < ColsNumber; ++i ) {
			d_view->addColumn( tr(ViewHdrCaption[i]) );
		}
		d_view->setColumnAlignment ( 0, Qt::AlignLeft );
		d_view->setColumnAlignment ( 1, Qt::AlignHCenter );
		d_view->setColumnAlignment ( 2, Qt::AlignLeft );
		wrk_layout->addWidget( box );
	}
	main_layout->addLayout( wrk_layout );

	QVBoxLayout* const btn_layout = new QVBoxLayout;
	btn_layout->addStretch( Shared::OverStretch );
	btn_layout->addWidget( d_run_btn );
	btn_layout->addWidget( d_close_btn );
	main_layout->addLayout( btn_layout );
	
	connect( d_run_btn  , SIGNAL( clicked()  ),
				this       , SLOT  ( slot_run() ));
	connect( d_close_btn, SIGNAL( clicked()  ),
				this       , SLOT( slot_close() ));
	connect( d_view     , SIGNAL( rightButtonClicked( QListViewItem*, const QPoint&, int )),
				this       , SLOT  ( info( QListViewItem*, const QPoint& ) ));
}
// end of CmpDirs

//*******************************************************************
// polish                                          PRIVATE inherited
//*******************************************************************
void CmpDirs::polish()
{
	Shared::polish( this, 40, 60 );
}
// end of polish

//*******************************************************************
// showEvent                                       PRIVATE inherited
//*******************************************************************
void CmpDirs::showEvent( QShowEvent* e )
{
	QDialog::showEvent( e );
	
	const QFontMetrics fm( font() );
	QString dir;

	Shared::clip_path( fm, d_src_dir_path->width(), dir = d_src_dir );
	d_src_dir_path->setText( dir );
	Shared::clip_path( fm, d_dst_dir_path->width(), dir = d_dst_dir );
	d_dst_dir_path->setText( dir );
}
// end of showEvent

//*******************************************************************
// slot_run                                             PRIVATE slot
//*******************************************************************
void CmpDirs::slot_run()
{
	d_run_btn->hide();
	compare( d_src_dir, d_dst_dir );
}
// end of slot_run

//*******************************************************************
// slot_close                                           PRIVATE slot
//*******************************************************************
void CmpDirs::slot_close()
{
	accept();
}
// end of slot_close

//*******************************************************************
// info                                                 PRIVATE slot
//*******************************************************************
void CmpDirs::info( QListViewItem* in_item, const QPoint& in_point )
{
	const QHeader* const header = d_view->header();
	const int h0_xl = header->mapToGlobal( header->sectionRect( 0 ).topLeft() ).x();
	const int h0_xr = header->mapToGlobal( header->sectionRect( 0 ).bottomRight() ).x();
	const int h2_xl = header->mapToGlobal( header->sectionRect( 2 ).topLeft() ).x();
	const int h2_xr = header->mapToGlobal( header->sectionRect( 2 ).bottomRight() ).x();
	const int x = in_point.x();

	const ViewItem* const item = dynamic_cast<ViewItem*>( in_item );
	if( item ) {
		QString path = QString::null;
		//.............................................
		     if(( x > h0_xl ) && ( x < h0_xr )) path = item->lft_path();
		else if(( x > h2_xl ) && ( x < h2_xr )) path = item->rgt_path();
		//.............................................
		if( path ) {
			const QFileInfo fi( path );
			const QString msg = tr(InfoMsg).arg( path )
												.arg( Shared::num2str( fi.size() ))
												.arg( fi.lastModified().toString( Shared::DateTimeMuster ))
												.arg( fi.owner() )
												.arg( fi.group() )
												.arg( fi.isExecutable() ? tr(Shared::Yes) : tr(Shared::No) ) ;
			QMessageBox::information( this, tr(InfoCaption), msg, QMessageBox::Ok );
		}
		//.......................................................
	}
}
// end of info

//*******************************************************************
// compare                                                   PRIVATE
//*******************************************************************
bool CmpDirs::compare( const QString& lpath, const QString& rpath )
{
	bool retval = FALSE;
	
	QDir ldir( lpath, QString::null, QDir::Unsorted );
	QDir rdir( rpath, QString::null, QDir::Unsorted );
	
	if( ldir.exists() && ldir.isReadable() ) {
		if( rdir.exists() && rdir.isReadable() ) {
			const QFileInfoList* const lft_items = ldir.entryInfoList( QDir::All, QDir::Unsorted  );
			const QFileInfoList* const rgt_items = rdir.entryInfoList( QDir::All, QDir::Unsorted  );
			if( lft_items && rgt_items ) {
				//...........................................
				files( lft_items, d_lft_files );
				files( rgt_items, d_rgt_files );
				compare_files();
				d_lft_files.clear();
				d_rgt_files.clear();
				//...........................................
				DataMap lft_dirs;
				DataMap rgt_dirs;
				dirs( lft_items, lft_dirs );
				dirs( rgt_items, rgt_dirs );
				
				DataMap dirs( lft_dirs );
				dirs.insert( rgt_dirs.begin(), rgt_dirs.end() );
				
				if( FALSE == dirs.empty() ) {
					DataMap::const_iterator lft_it;
					DataMap::const_iterator rgt_it;
					DataMap::const_iterator it = dirs.begin();
					while( it != dirs.end() ) {
						const QString name = it->first;
						//.............................................
						lft_it = lft_dirs.find( name );
						if( lft_it != lft_dirs.end() ) {
							d_lft_name = name;
							d_lft_path = lft_it->second->absFilePath();
						}
						else {
							d_lft_name = tr(FileNotPresents);
							d_lft_path = QString::null;
						}
						//.............................................
						rgt_it = rgt_dirs.find( name );
						if( rgt_it != rgt_dirs.end() ) {
							d_rgt_name = name;
							d_rgt_path = rgt_it->second->absFilePath();
						}
						else {
							d_rgt_name = tr(FileNotPresents);
							d_rgt_path = QString::null;
						}
						//.............................................
						new ViewItem( d_view, d_lft_name, Dir, d_rgt_name, d_lft_path, d_rgt_path );
						//.............................................
						if( d_lft_path && d_rgt_path ) {
							compare( d_lft_path, d_rgt_path );
						}
						++it;
					}
				}
			}
		}
	}
	
	return retval;
}
// end of compare

//*******************************************************************
// files                                                     PRIVATE
//*******************************************************************
void CmpDirs::files( const QFileInfoList* const in_list, DataMap& in_v )
{
	QFileInfoListIterator it( *in_list );
	while( it.current() ) {
		if( (*it)->isFile() ) {
			in_v.insert(  make_pair( (*it)->fileName(), *it ) );
		}
		++it;
	}
}
// end of files

//*******************************************************************
// compare_files                                             PRIVATE
//*******************************************************************
void CmpDirs::compare_files()
{
	DataMap total( d_lft_files );
	total.insert( d_rgt_files.begin(), d_rgt_files.end() );

	DataMap::const_iterator lft_it;
	DataMap::const_iterator rgt_it;
	DataMap::const_iterator it = total.begin();
	
	while( it != total.end() ) {
		const QString name = it->first;
		//.............................................
		lft_it = d_lft_files.find( name );
		if( lft_it != d_lft_files.end() ) {
			d_lft_name = name;
			d_lft_path = lft_it->second->absFilePath();
		}
		else {
			d_lft_name = tr(FileNotPresents);
			d_lft_path = QString::null;
		}
		//.............................................
		rgt_it = d_rgt_files.find( name );
		if( rgt_it != d_rgt_files.end() ) {
			d_rgt_name = name;
			d_rgt_path = rgt_it->second->absFilePath();
		}
		else {
			d_rgt_name = tr(FileNotPresents);
			d_rgt_path = QString::null;
		}
		//.............................................
		ViewItem* const item = new ViewItem( d_view, d_lft_name, "", d_rgt_name, d_lft_path, d_rgt_path );
		//.............................................
		if( d_lft_path && d_rgt_path ) {
			QString cmp_result;
			CmpFiles cmp( this, d_lft_path, d_rgt_path );
			const int retval = cmp.compare();
			switch( retval ) {
				case CmpFiles::NotEqual:
					cmp_result = NotEqual;
					break;
				case CmpFiles::Equal:
					cmp_result = Equal;
					break;
				case CmpFiles::Break:
					reject();
					break;
				default:
					cmp_result = Error;
			}
			item->setText( 1, cmp_result );
		}
		++it;
	}
}
// end of compare_files

//*******************************************************************
// dirs                                                      PRIVATE
//*******************************************************************
void CmpDirs::dirs( const QFileInfoList* const in_list, DataMap& in_v )
{
	QFileInfoListIterator it( *in_list );
	while( it.current() ) {
		if( (*it)->isDir() ) {
			const QString name = (*it)->fileName();
			if( Shared::is_regular_file( name ) ) {
				in_v.insert(  make_pair( name, *it ) );
			}
		}
		++it;
	}
}
// end of dirs
