/***************************************************************************
 *            readcd.c
 *
 *  dim jan 22 18:06:10 2006
 *  Copyright  2006  Rouquier Philippe
 *  brasero-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <stdlib.h>

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <nautilus-burn-drive.h>

#include "burn-basics.h"
#include "burn-common.h"
#include "burn-readcd.h"
#include "burn-imager.h"
#include "burn-process.h"
#include "burn-job.h"
#include "burn-caps.h"
#include "brasero-ncb.h"

static void brasero_readcd_class_init (BraseroReadcdClass *klass);
static void brasero_readcd_init (BraseroReadcd *sp);
static void brasero_readcd_finalize (GObject *object);
static void brasero_readcd_iface_init_image (BraseroImagerIFace *iface);

static BraseroBurnResult
brasero_readcd_read_stderr (BraseroProcess *process, const char *line);
static BraseroBurnResult
brasero_readcd_set_argv (BraseroProcess *process,
			 GPtrArray *argv,
			 gboolean has_master,
			 GError **error);
static BraseroBurnResult
brasero_readcd_post (BraseroProcess *process, BraseroBurnResult retval);

static BraseroBurnResult
brasero_readcd_set_source (BraseroJob *job,
			   const BraseroTrackSource *source,
			   GError **error);
static BraseroBurnResult
brasero_readcd_set_output (BraseroImager *imager,
			   const char *output,
			   gboolean overwrite,
			   gboolean clean,
			   GError **error);
static BraseroBurnResult
brasero_readcd_set_output_type (BraseroImager *imager,
				BraseroTrackSourceType type,
				GError **error);
static BraseroBurnResult
brasero_readcd_get_track (BraseroImager *imager,
			  BraseroTrackSource **track,
			  GError **error);
static BraseroBurnResult
brasero_readcd_get_size (BraseroImager *imager,
			 gint64 *size,
			 gboolean sectors,
			 GError **error);

static BraseroBurnResult
brasero_readcd_get_rate (BraseroJob *job,
			 gint64 *rate);
static BraseroBurnResult
brasero_readcd_get_written (BraseroJob *job,
			    gint64 *written);

static BraseroBurnResult
brasero_readcd_get_track_type (BraseroImager *imager,
			       BraseroTrackSourceType *track_type);

typedef enum {
	BRASERO_READCD_ACTION_IMAGING,
	BRASERO_READCD_ACTION_GETTING_SIZE,
	BRASERO_READCD_ACTION_NONE
} BraseroReadcdAction;

struct BraseroReadcdPrivate {
	BraseroReadcdAction action;
	GTimer *timer;

	BraseroBurnCaps *caps;

	BraseroTrackSourceType track_type;
	BraseroTrackSource *source;

	gint64 sectors_num;
	gint64 current_sector;
	gint64 current_rate;
	int start;

	char *toc;
	char *output;

	int overwrite:1;
	int clean:1;

	int track_ready:1;
	int is_DVD:1;
};

static GObjectClass *parent_class = NULL;

GType
brasero_readcd_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BraseroReadcdClass),
			NULL,
			NULL,
			(GClassInitFunc)brasero_readcd_class_init,
			NULL,
			NULL,
			sizeof (BraseroReadcd),
			0,
			(GInstanceInitFunc)brasero_readcd_init,
		};

		static const GInterfaceInfo imager_info =
		{
			(GInterfaceInitFunc) brasero_readcd_iface_init_image,
			NULL,
			NULL
		};

		type = g_type_register_static (BRASERO_TYPE_PROCESS,
					       "BraseroReadcd",
					       &our_info,
					       0);

		g_type_add_interface_static (type,
					     BRASERO_TYPE_IMAGER,
					     &imager_info);
	}

	return type;
}

static void
brasero_readcd_class_init (BraseroReadcdClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS(klass);
	BraseroJobClass *job_class = BRASERO_JOB_CLASS (klass);
	BraseroProcessClass *process_class = BRASERO_PROCESS_CLASS (klass);

	parent_class = g_type_class_peek_parent(klass);
	object_class->finalize = brasero_readcd_finalize;

	job_class->get_rate = brasero_readcd_get_rate;
	job_class->set_source = brasero_readcd_set_source;
	job_class->get_written = brasero_readcd_get_written;

	process_class->stderr_func = brasero_readcd_read_stderr;
	process_class->set_argv = brasero_readcd_set_argv;
	process_class->post = brasero_readcd_post;
}

static void
brasero_readcd_iface_init_image (BraseroImagerIFace *iface)
{
	iface->set_output_type = brasero_readcd_set_output_type;
	iface->set_output = brasero_readcd_set_output;

	iface->get_track_type = brasero_readcd_get_track_type;
	iface->get_track = brasero_readcd_get_track;
	iface->get_size = brasero_readcd_get_size;
}

static void
brasero_readcd_init (BraseroReadcd *obj)
{
	obj->priv = g_new0 (BraseroReadcdPrivate, 1);
	obj->priv->caps = brasero_burn_caps_get_default ();
	obj->priv->track_type = BRASERO_TRACK_SOURCE_DEFAULT;
}

static void
brasero_readcd_finalize (GObject *object)
{
	BraseroReadcd *cobj;
	cobj = BRASERO_READCD(object);

	g_object_unref (cobj->priv->caps);
	cobj->priv->caps = NULL;

	if (cobj->priv->timer) {
		g_timer_destroy (cobj->priv->timer);
		cobj->priv->timer = NULL;
	}

	if (cobj->priv->toc) {
		if (cobj->priv->clean)
			g_remove (cobj->priv->toc);

		g_free (cobj->priv->toc);
		cobj->priv->toc= NULL;
	}

	if (cobj->priv->output) {
		if (cobj->priv->clean)
			g_remove (cobj->priv->output);

		g_free (cobj->priv->output);
		cobj->priv->output= NULL;
	}

	if (cobj->priv->source) {
		brasero_track_source_free (cobj->priv->source);
		cobj->priv->source = NULL;
	}

	g_free(cobj->priv);
	G_OBJECT_CLASS(parent_class)->finalize(object);
}

static BraseroBurnResult
brasero_readcd_get_track (BraseroImager *imager,
			  BraseroTrackSource **track,
			  GError **error)
{
	BraseroReadcd *readcd;
	BraseroTrackSource *retval;
	BraseroTrackSourceType target;

	readcd = BRASERO_READCD (imager);

	if (!readcd->priv->source)
		return BRASERO_BURN_NOT_READY;

	if (readcd->priv->track_type == BRASERO_TRACK_SOURCE_UNKNOWN)
		return BRASERO_BURN_NOT_READY;

	if (!readcd->priv->track_ready) {
		NautilusBurnDrive *drive;
		BraseroBurnResult result;
		NautilusBurnMediaType media;

		drive = readcd->priv->source->contents.drive.disc;
		media = nautilus_burn_drive_get_media_type (drive);
		if (media > NAUTILUS_BURN_MEDIA_TYPE_CDRW)
			readcd->priv->is_DVD = TRUE;
		else
			readcd->priv->is_DVD = FALSE;

		readcd->priv->current_sector = 0;
		readcd->priv->action = BRASERO_READCD_ACTION_IMAGING;
		result = brasero_job_run (BRASERO_JOB (readcd), error);
		readcd->priv->action = BRASERO_READCD_ACTION_NONE;

		if (result != BRASERO_BURN_OK)
			return result;

		readcd->priv->current_sector = readcd->priv->sectors_num;
		readcd->priv->track_ready = 1;
	}

	/* see if we are ready */
	retval = g_new0 (BraseroTrackSource, 1);

	/* the disc is the same as when we called set_argv so is the target */
	target = readcd->priv->track_type;
	if (target == BRASERO_TRACK_SOURCE_DEFAULT)
		target = brasero_burn_caps_get_imager_default_target (readcd->priv->caps,
								      readcd->priv->source);
	else
		target = readcd->priv->track_type;

	retval->type = target;
	if (target == BRASERO_TRACK_SOURCE_ISO
	||  target == BRASERO_TRACK_SOURCE_ISO_JOLIET) {
		retval->contents.iso.image = g_strdup_printf ("file://%s", readcd->priv->output);
	}
	else if (target == BRASERO_TRACK_SOURCE_RAW) {
		retval->contents.raw.toc = g_strdup_printf ("file://%s", readcd->priv->toc);
		retval->contents.raw.image = g_strdup_printf ("file://%s", readcd->priv->output);
	}

	*track = retval;

	return BRASERO_BURN_OK;
}

static BraseroBurnResult
brasero_readcd_get_size (BraseroImager *imager,
			 gint64 *size,
			 gboolean sectors,
			 GError **error)
{
	BraseroReadcd *readcd;
	BraseroTrackSourceType target;
	BraseroBurnResult result = BRASERO_BURN_OK;

	readcd = BRASERO_READCD (imager);

	if (!readcd->priv->source)
		return BRASERO_BURN_NOT_READY;

	if (readcd->priv->sectors_num == 0) {
		if (brasero_job_is_running (BRASERO_JOB (imager)))
			return BRASERO_BURN_RUNNING;

		readcd->priv->action = BRASERO_READCD_ACTION_GETTING_SIZE;
		result = brasero_job_run (BRASERO_JOB (readcd), error);
		readcd->priv->action = BRASERO_READCD_ACTION_NONE;

		if (result != BRASERO_BURN_OK)
			return result;
	}

	target = readcd->priv->track_type;
	if (target == BRASERO_TRACK_SOURCE_DEFAULT)
		target = brasero_burn_caps_get_imager_default_target (readcd->priv->caps,
								      readcd->priv->source);

	if (sectors)
		*size = readcd->priv->sectors_num;
	else if (target == BRASERO_TRACK_SOURCE_ISO
	      ||  target == BRASERO_TRACK_SOURCE_ISO_JOLIET)
		*size = readcd->priv->sectors_num * 2048;
	else	/* when used with -clone option */
		*size = readcd->priv->sectors_num * 2448;

	return result;
}

static BraseroBurnResult
brasero_readcd_get_rate (BraseroJob *job,
			 gint64 *rate)
{
	BraseroReadcd *readcd;

	readcd = BRASERO_READCD (job);

	if (!readcd->priv->current_rate)
		return BRASERO_BURN_NOT_READY;

	if (rate)
		*rate = readcd->priv->current_rate;

	return BRASERO_BURN_OK;
}

static BraseroBurnResult
brasero_readcd_get_written (BraseroJob *job,
			    gint64 *written)
{
	BraseroReadcd *readcd;

	readcd = BRASERO_READCD (job);

	if (written) {
		BraseroTrackSourceType target;

		target = readcd->priv->track_type;
		if (target == BRASERO_TRACK_SOURCE_DEFAULT)
			target = brasero_burn_caps_get_imager_default_target (readcd->priv->caps,
									      readcd->priv->source);

		if (target == BRASERO_TRACK_SOURCE_ISO
		||  target == BRASERO_TRACK_SOURCE_ISO_JOLIET)
			*written = readcd->priv->current_sector * 2048;
		else	/* when used with -clone option */
			*written = readcd->priv->current_sector * 2448;
	}

	return BRASERO_BURN_OK;
}

static BraseroBurnResult
brasero_readcd_get_track_type (BraseroImager *imager,
			       BraseroTrackSourceType *track_type)
{
	BraseroReadcd *readcd;

	readcd = BRASERO_READCD (imager);

	*track_type = readcd->priv->track_type;

	return BRASERO_BURN_OK;
}

static BraseroBurnResult
brasero_readcd_set_source (BraseroJob *job,
			   const BraseroTrackSource *source,
			   GError **error)
{
	BraseroReadcd *readcd;

	readcd = BRASERO_READCD (job);

	if (readcd->priv->source) {
		brasero_track_source_free (readcd->priv->source);
		readcd->priv->source = NULL;
	}

	if (readcd->priv->clean && readcd->priv->track_ready) {
		if (readcd->priv->output)
			g_remove (readcd->priv->output);

		if (readcd->priv->toc)
			g_remove (readcd->priv->toc);
	}

	readcd->priv->sectors_num = 0;
	readcd->priv->track_ready = 0;

	if (source->type != BRASERO_TRACK_SOURCE_DISC)
		return BRASERO_BURN_NOT_SUPPORTED;

	readcd->priv->source = brasero_track_source_copy (source);
	return BRASERO_BURN_OK;	
}

static BraseroBurnResult
brasero_readcd_set_output_type (BraseroImager *imager,
				BraseroTrackSourceType type,
				GError **error)
{
	BraseroReadcd *readcd;

	readcd = BRASERO_READCD (imager);

	if (readcd->priv->track_type == type)
		return BRASERO_BURN_OK;

	if (readcd->priv->clean && readcd->priv->track_ready) {
		if (readcd->priv->output)
			g_remove (readcd->priv->output);

		if (readcd->priv->toc)
			g_remove (readcd->priv->toc);
	}

	readcd->priv->sectors_num = 0;
	readcd->priv->track_ready = 0;

	/* NOTE: it's not good to call burn_caps here to get the default
	 * target for the output since the disc might change before we 
	 * call get track */
	readcd->priv->track_type = BRASERO_TRACK_SOURCE_UNKNOWN;
	if (type != BRASERO_TRACK_SOURCE_DEFAULT
	&&  type != BRASERO_TRACK_SOURCE_RAW
	&&  type != BRASERO_TRACK_SOURCE_ISO
	&&  type != BRASERO_TRACK_SOURCE_ISO_JOLIET)
		return BRASERO_BURN_NOT_SUPPORTED;

	readcd->priv->track_type = type;

	return BRASERO_BURN_OK;
}

static BraseroBurnResult
brasero_readcd_set_output (BraseroImager *imager,
			   const char *output,
			   gboolean overwrite,
			   gboolean clean,
			   GError **error)
{
	BraseroReadcd *readcd;

	readcd = BRASERO_READCD (imager);

	if (readcd->priv->output) {
		if (readcd->priv->clean && readcd->priv->track_ready)
			g_remove (readcd->priv->output);

		g_free (readcd->priv->output);
		readcd->priv->output = NULL;
	}

	if (readcd->priv->toc) {
		if (readcd->priv->clean && readcd->priv->track_ready)
			g_remove (readcd->priv->toc);

		g_free (readcd->priv->toc);
		readcd->priv->toc = NULL;
	}
	readcd->priv->track_ready = 0;

	if (output)
		readcd->priv->output = g_strdup (output);

	readcd->priv->overwrite = overwrite;
	readcd->priv->clean = clean;

	return BRASERO_BURN_OK;
}

static BraseroBurnResult
brasero_readcd_read_stderr (BraseroProcess *process, const char *line)
{
	BraseroReadcd *readcd;
	char *pos;

	readcd = BRASERO_READCD (process);
	if ((pos = strstr (line, "addr:"))) {
		int sector;
		double fraction;
		long remaining = -1;

		pos += strlen ("addr:");
		sector = strtoll (pos, NULL, 10);

		/* readcd reports very often when copying a DVD */
		if (readcd->priv->is_DVD
		&&  readcd->priv->current_sector + 256 > sector)
			return BRASERO_BURN_OK;

		readcd->priv->current_sector = sector;
		fraction = (double) sector / readcd->priv->sectors_num;

		if (!readcd->priv->timer && sector > 10) {
			/* let the speed regulate before keeping track */
			readcd->priv->timer = g_timer_new ();
			readcd->priv->start = sector;
		}

		if (readcd->priv->sectors_num && readcd->priv->timer) {
			gdouble elapsed;
			double rate;

			elapsed = g_timer_elapsed (readcd->priv->timer, NULL);
			rate = (sector - readcd->priv->start) / elapsed;

			if (rate > 0.0) {
				/* here the ave_rate = sectors / second
				 * 1 sec = 75 sectors
				 * speed 1 => 75 / second for a CD */
				readcd->priv->current_rate = rate * 2352;
				remaining = (long) ((gdouble) readcd->priv->sectors_num /
							        rate - (gdouble) elapsed);
			}
		}

		if (remaining >= 0)
			brasero_job_progress_changed (BRASERO_JOB (readcd),
						      fraction,
						      remaining);
		else
			brasero_job_progress_changed (BRASERO_JOB (readcd),
						      fraction,
						      -1);
	}
	else if ((pos = strstr (line, "Capacity:"))) {
		if (readcd->priv->action == BRASERO_READCD_ACTION_GETTING_SIZE) {
			pos += strlen ("Capacity:");
			readcd->priv->sectors_num = strtoll (pos, NULL, 10);
			brasero_job_finished (BRASERO_JOB (readcd));
		}
		else {
			readcd->priv->timer = g_timer_new ();
			brasero_job_action_changed (BRASERO_JOB (readcd),
						    BRASERO_BURN_ACTION_DRIVE_COPY,
						    FALSE);
		}
	}
	else if (strstr (line, "Device not ready.")) {
		brasero_job_error (BRASERO_JOB (readcd),
				   g_error_new (BRASERO_BURN_ERROR,
						BRASERO_BURN_ERROR_BUSY_DRIVE,
						_("the drive is not ready")));
	}
	else if (strstr (line, "Device or resource busy")) {
		brasero_job_error (BRASERO_JOB (readcd),
				   g_error_new (BRASERO_BURN_ERROR,
						BRASERO_BURN_ERROR_BUSY_DRIVE,
						_("you don't seem to have the required permissions to access the drive")));
	}
	else if (strstr (line, "Cannot open SCSI driver.")) {
		brasero_job_error (BRASERO_JOB (readcd),
				   g_error_new (BRASERO_BURN_ERROR,
						BRASERO_BURN_ERROR_BUSY_DRIVE,
						_("you don't seem to have the required permissions to access the drive")));		
	}
	else if (strstr (line, "Cannot send SCSI cmd via ioctl")) {
		brasero_job_error (BRASERO_JOB (readcd),
				   g_error_new (BRASERO_BURN_ERROR,
						BRASERO_BURN_ERROR_SCSI_IOCTL,
						_("you don't seem to have the required permissions to access the drive")));
	}
	else if (strstr (line, "Time total:")) {
		brasero_job_finished (BRASERO_JOB (process));
	}

	return BRASERO_BURN_OK;
}

static BraseroBurnResult
brasero_readcd_set_argv (BraseroProcess *process,
			 GPtrArray *argv,
			 gboolean has_master,
			 GError **error)
{
	BraseroBurnResult result = FALSE;
	BraseroTrackSourceType target;
	NautilusBurnMediaType type;
	NautilusBurnDrive *drive;
	BraseroReadcd *readcd;
	char *outfile_arg;
	char *dev_str;

	readcd = BRASERO_READCD (process);
	brasero_job_set_run_slave (BRASERO_JOB (readcd), FALSE);

	if (!readcd->priv->source)
		return BRASERO_BURN_NOT_READY;

	g_ptr_array_add (argv, g_strdup ("readcd"));

	drive = readcd->priv->source->contents.drive.disc;
	if (NCB_DRIVE_GET_DEVICE (drive))
		dev_str = g_strdup_printf ("dev=%s", NCB_DRIVE_GET_DEVICE (drive));
	else
		return BRASERO_BURN_ERR;

	g_ptr_array_add (argv, dev_str);

	if (readcd->priv->action == BRASERO_READCD_ACTION_GETTING_SIZE) {
		g_ptr_array_add (argv, g_strdup ("-sectors=0-0"));
		brasero_job_action_changed (BRASERO_JOB (readcd),
					    BRASERO_BURN_ACTION_GETTING_SIZE,
					    FALSE);
		return BRASERO_BURN_OK;
	}

	g_ptr_array_add (argv, g_strdup ("-nocorr"));

	type = nautilus_burn_drive_get_media_type (drive);
	target = readcd->priv->track_type;
	if (target == BRASERO_TRACK_SOURCE_DEFAULT)
		target = brasero_burn_caps_get_imager_default_target (readcd->priv->caps,
								      readcd->priv->source);

	if (type > NAUTILUS_BURN_MEDIA_TYPE_CDRW 
	&&  target != BRASERO_TRACK_SOURCE_ISO
	&&  target != BRASERO_TRACK_SOURCE_ISO_JOLIET) {
		g_set_error (error,
			     BRASERO_BURN_ERROR,
			     BRASERO_BURN_ERROR_GENERAL,
			     _("raw images cannot be created with DVDs"));
		return BRASERO_BURN_ERR;
	}

	if (target == BRASERO_TRACK_SOURCE_RAW) {
		/* NOTE: with this option the sector size is 2448 
		 * because it is raw96 (2352+96) otherwise it is 2048  */
		g_ptr_array_add (argv, g_strdup ("-clone"));
	}
	else /* This is for *.iso images to ease the process */
		g_ptr_array_add (argv, g_strdup ("-noerror"));

	if (!has_master) {
		if (target == BRASERO_TRACK_SOURCE_ISO
		||  target == BRASERO_TRACK_SOURCE_ISO_JOLIET) {
			result = brasero_burn_common_check_output (&readcd->priv->output,
								   readcd->priv->overwrite,
								   NULL,
								   error);
		}
		else if (target == BRASERO_TRACK_SOURCE_RAW) {	
			result = brasero_burn_common_check_output (&readcd->priv->output,
								   readcd->priv->overwrite,
								   &readcd->priv->toc,
								   error);
		}

		if (result != BRASERO_BURN_OK)
			return result;

		outfile_arg = g_strdup_printf ("-f=%s", readcd->priv->output);
		g_ptr_array_add (argv, outfile_arg);
	}
	else if (target == BRASERO_TRACK_SOURCE_ISO
	      ||  target == BRASERO_TRACK_SOURCE_ISO_JOLIET) {
		outfile_arg = g_strdup ("-f=-");
		g_ptr_array_add (argv, outfile_arg);
	}
	else 	/* unfortunately raw images can't be piped out */
		return BRASERO_BURN_NOT_SUPPORTED;

	return BRASERO_BURN_OK;
}

static BraseroBurnResult
brasero_readcd_post (BraseroProcess *process, BraseroBurnResult retval)
{
	BraseroReadcd *readcd;

	readcd = BRASERO_READCD (process);

	if (readcd->priv->timer) {
		g_timer_destroy (readcd->priv->timer);
		readcd->priv->timer = NULL;
	}

	readcd->priv->current_rate = 0;
	readcd->priv->start = 0;

	return BRASERO_BURN_OK;
}

BraseroReadcd *
brasero_read_cd_new (void)
{
	BraseroReadcd *obj;

	obj = BRASERO_READCD (g_object_new (BRASERO_TYPE_READCD, NULL));

	return obj;
}
