#ifndef _SYMBOL_H_
#define _SYMBOL_H_

#include "event.h"

/**
 * Instances of this class are symbols (or text-elements). They do not need a (graphical) presentation of their own. They can be displayed
 * graphically within the presentation of the parts, they belong to.
 *
 * Symbols have an ID number, a displacement, in some cases an additional parameter and in some cases a text.
 *
 * These symbols are not associated with a certain note. So far the following symbol IDs exist:
 * <pre>
 *
 * AS_LABEL1
 * AS_LABEL2
 * AS_LABEL3
 * AS_CASE1
 * AS_CASE2
 * AS_PEDAL1
 * AS_PEDAL2
 * AS_NOTEDOT
 * AS_NOTE3
 * AS_NOTE6
 * AS_NOTEXX
 * AS_8VA
 * AS_15VA
 * AS_TRILLX
 * AS_TRILLX2
 * AS_DECRESCENDO
 * AS_CRESCENDO
 * AS_BRACKETUP
 * AS_BRACKETDOWN
 * AS_ARPEGGIO
 * AS_TEXT
 * DYN_PPP
 * DYN_PP
 * DYN_P
 * DYN_MP
 * DYN_MF
 * DYN_F
 * DYN_FF
 * DYN_FFF
 * DYN_SFZ
 * DYN_SF
 * DYN_SFF
 * DYN_FP
 * </pre>
 **/
class Symbol : public Event
{
 private:
  char * _text;
  int _displacement;
  int _symbol;

 public:
  /**
   * Default constructor
  **/
  Symbol();

  /**
   * Constructor defining a symbol sym at a position p
  **/
  Symbol(Position p, int sym);

  /**
   * Constructor defining a symbol sym at a position p with a displacement displ
  **/
  Symbol(Position p, int sym, int displ);

  /**
   * Constructor defining a symbol sym at a position p with a displacement displ and a parameter par
  **/
  Symbol(Position p, int sym, int displ, int par);

  /**
   * Constructor defining a text tx at a position p
  **/
  Symbol(Position p, char * tx);

  /**
   * Constructor defining a text tx at a position p with a displacement displ
  **/
  Symbol(Position p, char * tx, int displ);

  /**
   * Copy Constructor
   **/
  Symbol(const Symbol&);

  /**
   * Returns true if this symbol contains an additional parameter
   **/
  bool isParameterized() const { return (duration()!=0); }

  /**
   * Returns true if this symbol contains text
   **/
  bool isText() const { return (_text!=0); }

  /**
   * Returns the symbol ID
   **/
  int symbol() const { return _symbol; }

  /**
   * Returns the displacement
   **/
  int displacement() const { return _displacement; }

  /**
   * Returns the text
   **/
  const char * text() const { return _text; }

  /**
   * Returns the additional parameter
   **/
  int parameter() const { return duration(); }

  /**
   * Sets the symbol ID
   **/
  void setSymbol(int);

  /**
   * Sets the displacement
   **/
  void setDisplacement(int);

  /**
   * Sets the text
   **/
  void setText(char*);

  /**
   * Sets the additional parameter
   **/
  void setParameter(int);

  /**
   * Implementation of the print method
  **/
  virtual ostream & print(int,ostream&) const;

  /**
   * Implementation of the flush method
  **/
  virtual void flush(const char*) const;

  /**
   * Implementation of the copy method
  **/
  virtual Element * copy() const;

  /**
   * Implementation of the load method
  **/
  static Element * load(char*,ifstream*&,Element*);
};


#endif
