#ifndef _PART_H_
#define _PART_H_

#include "compound.h"
#include "position.h"

class Reference;
class Event;
class Table;
class PrPart;
class Track;
class Symbol;

/** A Part is thought of as a container for events (like Notes, MasterEvents, etc. An Event is
 * an Atom which has a position and a length). These events have to appear ordered in their
 * position property. More than one event at the same position are admitted.
 *
 * Each Part owns a Pointer to a current event. This pointer can move back and forth within
 * the list of events like the head of a turing machine. Loops are achived via these operations.
 *
 * A Part also contains a memory in terms of a table. With certain methods, references to events
 * can be put into memory and taken out again. This is important for the play-operation
 */
class Part : public Compound
{
  friend class Player;

 private:
  Position _pos;
  int _key;
  int _clef;
  int _meter0;
  int _meter1;
  int _program;

  Track * _track;
  Event * _cur[2];
  Table * _mem;
  Reference * _memref;

  // player functions:

  bool rewindPlayer();
  bool windPlayer(Position p);
  bool currentPlayer(Type);
  Event * currentPlayer();
  Part * currentPlayerInc();
  bool currentPlayerStartsAt(long p);
  bool currentPlayerStartsBehind(long p);
  void rememberCurrentPlayer();

 protected:
  PrPart * _pr;

 public:
  
  /**
   ** The Part's Constructors
   ** -----------------------
   **/

  /** Constructs an empty part with an empty memory */
  Part();
  
  /** Constructs a part from another part. Caution: Memory is not duplicated! */
  Part(Part*);

  /** Constructs a part and remembers the track it belongs to (important for presentations) */
  Part(Track*);

  /** Destructor */
  ~Part();
  
  /**
   ** Manipulationg the Part's Properties
   ** -----------------------------------
   **/

  /** returns the start position of the part */
  Position start();

  /** returns the key */
  int key();

  /** returns the clef */
  int clef();

  /** returns the numerator of the measure */
  int meter0();

  /** returns the denominator of the measure */
  int meter1();

  /** returns the midi program */
  int program();

  /** returns the track the part belongs to, or 0 if this informaition does not exist! */
  Track * track();

  /** sets the start position of the part */
  void setStart(Position);

  /** sets the key */
  void setKey(int key);

  /** sets the clef */
  void setClef(int clef);

  /** sets the numerator and denominator of the measure */
  void setMeter(int numerator,int denominator);

  /** sets the numerator of the measure */
  void setMeter0(int numerator);

  /** sets the denominator of the measure */
  void setMeter1(int denominator);

  /** sets the midi program */
  void setProgram(int program);

  /** sets the track */
  void setTrack(Track *);

  /**
   ** Each Part has an internal pointer to an event, called the current
   ** event. It can be moved around the eventlist, like the head of a
   ** turing machine.
   ** The following methods deal with this internal pointer
   ** -----------------------------------------------------------------
   **/


  /** sets the pointer to the beginning
   *  @see #rewind(), #wind(), #current(), #currentStartsBefore(), #currentStartsBehind(), #currentStartsAt(), #currentInc()
   */
  void rewind(int num=0);

  /** sets the pointer to the event at
   *  position p (or later)
   *  returns true in case of success
   *  @see #rewind(), #wind(), #current(), #currentStartsBefore(), #currentStartsBehind(), #currentStartsAt(), #currentInc()
   */
  bool wind(Position p, int num=0);

  /** checks type of the event pointed to by the pointer
   *  @see #rewind(), #wind(), #current(), #currentStartsBefore(), #currentStartsBehind(), #currentStartsAt(), #currentInc()
   */
  bool current(Type, int num=0);

  /** returns true if the pointer starts before position p
   *  @see #rewind(), #wind(), #current(), #currentStartsBefore(), #currentStartsBehind(), #currentStartsAt(), #currentInc()
   */
  bool currentStartsBefore(long p, int num=0);

  /** returns true if the pointer starts behind position p
   *  @see #rewind(), #wind(), #current(), #currentStartsBefore(), #currentStartsBehind(), #currentStartsAt(), #currentInc()
   */
  bool currentStartsBehind(long p, int num=0);

  /** returns true if the pointer starts at position p
   *  @see #rewind(), #wind(), #current(), #currentStartsBefore(), #currentStartsBehind(), #currentStartsAt(), #currentInc()
   */
  bool currentStartsAt(long p, int num=0);

  /** returns true if the pointer ends behind position p
   */
  bool currentEndsBehind(long p, int num=0);

  /** returns the absolute position of the current event (including the part's offset!)
   */
  long currentPos(int num=0);

  /** returns the pointer (only use if you have to!!!)
   *  @see #rewind(), #wind(), #current(), #currentStartsBefore(), #currentStartsBehind(), #currentStartsAt(), #currentInc()
   */
  Event * current(int num=0);

  /** increments the pointer. Returns false if incrementation cannot be done
   *  @see #rewind(), #wind(), #current(), #currentStartsBefore(), #currentStartsBehind(), #currentStartsAt(), #currentInc()
   */
  bool currentInc(int num=0);

  /** flushes properties of current pointer to std out.
   *  @see #rewind(), #wind(), #current(), #currentStartsBefore(), #currentStartsBehind(), #currentStartsAt(), #currentInc()
   */
  void flushCurrent();


  /**
   ** The following funtions are concerned with the memory management
   ** which is achieved by a Table of References.
   ** ---------------------------------------------------------------
   **/

  /** adds the current event to the internal memory
   *  @see #startMem(), #incMem(), #moreMem(), #cutMem(), #clearMem(), #memEndsAt(), #mem(), #rememberCurrent()
   */
  void rememberCurrent(int num=0);

  /** sets the internal memory pointer (not to be with the internal event pointer s.a.!)
   *  to the beginning of the memory table
   *  @see #startMem(), #incMem(), #moreMem(), #cutMem(), #clearMem(), #memEndsAt(), #mem(), #rememberCurrent()
   */
  void startMem();

  /** returns true if the memory pointer exists and false if it is zero,
   *  which is the case at the end of a loop
   *  @see #startMem(), #incMem(), #moreMem(), #cutMem(), #clearMem(), #memEndsAt(), #mem(), #rememberCurrent()
   */
  bool moreMem();

  /** increments the memory pointer
   *  @see #startMem(), #incMem(), #moreMem(), #cutMem(), #clearMem(), #memEndsAt(), #mem(), #rememberCurrent()
   */
  void incMem();

  /** removes the memory pointers reference from the memory table
   *  @see #startMem(), #incMem(), #moreMem(), #cutMem(), #clearMem(), #memEndsAt(), #mem(), #rememberCurrent()
   */
  void cutMem();

  /** clears the memory table
   *  @see #startMem(), #incMem(), #moreMem(), #cutMem(), #clearMem(), #memEndsAt(), #mem(), #rememberCurrent()
   */
  void clearMem();

  /** returns true if the event pointed to by the memory pointer
   *  ends at the given position p and false otherwise
   *  @see #startMem(), #incMem(), #moreMem(), #cutMem(), #clearMem(), #memEndsAt(), #mem(), #rememberCurrent()
   */
  bool memEndsAt(long p);

  /** returns the event pointed to by the memory pointer.
   *  (Don't use if you can avoid it)
   *  @see #startMem(), #incMem(), #moreMem(), #cutMem(), #clearMem(), #memEndsAt(), #mem(), #rememberCurrent()
   */
  Event * mem();



  /**
   ** Concerning the Presentation
   ** ---------------------------
   */

  /** This funktion returns a pointer to its presentation
   */
  PrPart * presentation();

  void hide();

  void show();

  /**
   ** Other Functionality
   ** -------------------
   */

  /** The parts have to be ordered with increasing position. This method ensures consistency.
   */
  
  /** returns a list of references containing events between left and right, where notes also
   * have their pitch to be between lowPitch and highPitch
   */
  Reference * makeRefs(int lowPitch, int highPitch, long left, long right);
  void reorder();
  
  void partCopy();

  /** glues two parts. Returns the 'lost' part which should not be deleted if the operation is to be undone
   */
  Part * partGlue();

  /** splits the part at position p and returns the event after p (or null)
   */
  Event * partSplit(Position p);

  /** this add method differs from the compounds add method by inserting the new element not at the end of the part, but
   * at the correct position
   */
  virtual void add(Element *);

  Symbol * setSymbol(Position p, int sym, int off, int par=-1);

  /**
   ** Virtual operations, each element has to implement
   ** -------------------------------------------------
   **/

  virtual ostream & print(int,ostream&);
  virtual void flush(char*);
  virtual Element * copy();

  static Element * load(char*,ifstream*&,Element*);

};

#endif
