#ifndef _PRSCOREPAINTER_H_
#define _PRSCOREPAINTER_H_

class ScoreBar;
class ScoreGroup;
class ScoreChord;
class ScoreBreak;
class Table;
class PrScoreEditor;
class Note;
class Lyrics;

extern const char * gmNames[];
extern int sharpTab[];
extern int flatTab[];
extern int signShift[];
extern int scrSigns[8];
extern int allSigns[15][7];
extern int invPitch[];
extern int sign[];
extern int yClef[];
extern int enhF[5][12];
extern int enhS[5][12];

#define TOP_DOWN_TRANSITION 16
#define STEM_LENGTH 6

enum ScoreObjectType { BREAK, SINGLE, MULTIPLE };

/* For performance reasons, the structures ChordGeometry and GroupGeometry are not defined as part of ScoreChord and ScoreGroup. For the same
reason we have a maximal number of elements per group. */

#define MAX_ELEMENTS_PER_GROUP 24

struct ChordGeometry {
  int  enh;
  int  step;
  int  sgn;
  int  len;
  int  display_len;      // tuplets need a different length value to be displayed!
  int  ypos;
  int  ypos_old;
  bool neighbour_offset; // if two notes are only one pitch-step apart, one has to be moved in position
  bool middle_stem;      // if two notes of the chord are only one pitch-step apart, this is true
  int  num_of_notes;     // number of notes inside the chord
  int  yAverage;         // average y-position
  int  yBottom;
  int  yTop;
  int  xPos;             // The xPos is the left most position of the note body.
  int  stemPos;          // The stemPos is the position of the stem: With an up-stem, this is the right most position of the note body
  int  stemDir;
  int  nwidth;
  int  yscale;
  Lyrics * lyrics;
};

struct GroupGeometry {
  int  xPosLeft[MAX_ELEMENTS_PER_GROUP];
  int  xPosRight[MAX_ELEMENTS_PER_GROUP];
  int  yPos[MAX_ELEMENTS_PER_GROUP]; // beam position
  int  yBottom[MAX_ELEMENTS_PER_GROUP];
  int  yTop[MAX_ELEMENTS_PER_GROUP];
  int  flags[MAX_ELEMENTS_PER_GROUP];
  int  yBottomS;
  int  yTopS;
  int  elements;
  int  dir;
  int  tuplet;
  int  xTupletLeft;   // for tuplets' text position ("3", "5", etc)
  int  xTupletRight;  // for tuplets' text position ("3", "5", etc)
  int  yTupletLeft;   // for tuplets' text position ("3", "5", etc)
  int  yTupletRight;  // for tuplets' text position ("3", "5", etc)
};


/** The ScorePainter is instantiated only once (as asingleton)!
  * Thus it should not contain any editor-specific members. It is
  * thought only to provide methods to paint the ScoreObjects
  * (bars, groups, chords and breaks).
  * The few private members are always(!!!) to used temporarily.
  * The PrScoreEditor is an abstract class!
  */

class PrScorePainter
{
 private:
  char   txt[12];

  void setUpY(int start, double slope);
  void setDownY(int start, double slope);

 protected:

  struct ChordGeometry * c_geometry;
  struct GroupGeometry * g_geometry;
  double unitsPerTick;
  int    key;
  int    clef;

  /** this method returns the midi program name, for the specified program
    */
  const char * programName(int prg) const ;

  /** this method resets the key-array. It should usually be called at the beginning
    * of a bar presentation
    */
  void resetSigns(int key);

  /** use this method to initialize the chord's geometry and specify the width of the note body as a parameter
    */
  void initChordGeometry(int nwidth, int yscale);

  /** use this method to initialize the group's geometry.
    */
  void initGroupGeometry(ScoreGroup * group, int xoff, int nwidth);


  /** this method helps to set the geometry. Call it with each note within a chord.
   * It requires the geometry to be initialized by the initGeometry method, as well
   * as key and clef to be set correctly.
   */
  void use(Note * note, ScoreChord * chord);

  void use(ScoreObjectType t, ChordGeometry * cg, int xpos);

  /** call this method after all notes for a chord are processed by the use method
    * to conclude the geometry calculation. The direction of the stem is
    * determined as well as the position of the stem, for which the parameters xoff
    * (offset in x direction) and nwidth (the width of a note, which should be
    * equal to the shift in x-direction) are needed.
    */
  void makeChordGeometry(int xoff);

  void makeGroupGeometry(bool horizontalBeams);

  /** this method returns the number of dots that occurr for a given length. If no
    * length is specified, the geometry's length is used.
    */
  int dot(int len=0) const;

  /** this method returns the number of flags of a chord. (1/8 has one flag, 1/16
    * has 2 flags, 1/32 has 3 flags etc.)
    * If no length is specified, the geometry's length will be used.
    */
  int flags(int len = -1) const;

  /** this method returns the ordered length number (0=whole, 1=half, 2=quater, 3=eigth, etc)
    * If no length is specified, the geometry's length will be used.
    */
  int lengthOrd(int len = -1) const;

  /** returns the integer parameter as a char*
    */
  const char * string(int num);

  /** returns the geometry's lyrics
    */
  const char * lyrics() const;

  int stemLength() const;

  bool hasStem() const;

  bool hasFlags() const;

 public:
  PrScorePainter();
  virtual ~PrScorePainter();

  int indent(ScoreBar*);

  /** Overload this to paint a bar at the specified position */
  virtual void paintBar(ScoreBar*, PrScoreEditor*, Table*, int, int, int)=0;

  /** Overload this to paint a bar at the specified position */
  virtual void paintGroup(ScoreGroup*, PrScoreEditor*, Table*, int, int)=0;

  /** Overload this to paint a bar at the specified position */
  virtual void paintChord(ScoreChord*, PrScoreEditor*, Table*, int, int)=0;

  /** Overload this to paint a bar at the specified position */
  virtual void paintBreak(ScoreBreak*, PrScoreEditor*, int, int)=0;

  /** Overload this to return the number of units with which the notes have to be indented due to clef and key inside the specified bar */
  virtual int indentSystem(ScoreBar*)=0;
};

#endif
