# -*- coding: utf-8 -*-

# Bluemindo: A really simple but powerful audio player in Python/PyGTK.
# Copyright (C) 2007-2009  Erwan Briand

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3 of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from gobject import timeout_add
from gtk.glade import XML as glade_XML
from gtk.gdk import POINTER_MOTION_MASK, pixbuf_new_from_file, INTERP_BILINEAR
from gtk import STOCK_INFO, EventBox, Image, status_icon_new_from_file
from os.path import join
from os import getcwd

try:
    from egg.trayicon import TrayIcon
    trayicon_type = 'egg'
except ImportError:
    trayicon_type = 'gtk'

from media.gstreamer import GStreamer
from common.functions import Functions
functions = Functions()

class Trayicon:
    def __init__(self, extensionsloader):
        self.extensions = extensionsloader
        self.module = {'name': 'Trayicon',
                       'logo': STOCK_INFO,
                       'configurable': False}

    def start_module(self):
        """This function starts the module."""

        def start_bluemindo(glade):
            self.window = glade.get_widget('window1')
            default = join(functions.datadir, 'image', 'logo_head_small.png')

            if trayicon_type == 'egg':
                self.tray = TrayIcon('Bluemindo')
                ebox = EventBox()
                ebox.set_visible_window(False)
                ebox.set_events(POINTER_MOTION_MASK)
                ebox.connect('button-press-event', self.on_clicked)
                ebox.connect('enter-notify-event', self.on_tray_enter)
                ebox.connect('leave-notify-event', self.on_tray_leave)

                self.tray_icon = Image()
                pixbuf = pixbuf_new_from_file(default)
                pixbuf = pixbuf.scale_simple(20, 20, INTERP_BILINEAR)
                self.tray_icon.set_from_pixbuf(pixbuf)

                ebox.add(self.tray_icon)
                self.tray.add(ebox)
                self.tray.show_all()

                self.is_the_cursor_on_tray = False
            else:
                self.tray = status_icon_new_from_file(default)
                self.tray.connect('activate', self.on_gtk_clicked)
                self.tray.connect('popup-menu', self.trayicon_menu)

            self.extensions.load_event('OnTrayIconAnswer', self.tray)

        # Connect to Bluemindo's signals
        self.extensions.connect('OnBluemindoStarted', start_bluemindo)
        self.extensions.connect('OnBluemindoWindowClosed', self.hideshow)
        self.extensions.connect('OnStopPressed', self.handler_stop)
        self.extensions.connect('OnPlayPressed', self.handler_play)
        self.extensions.connect('OnPlayNewSong', self.handler_play)

    def hideshow(self):
        """Hide or show Bluemindo's main window."""
        if self.window.get_properties('visible')[0]:
            functions.close_bluemindo(self.window, False)
            self.window.hide()
        elif not self.window.get_properties('visible')[0]:
            self.window.show()
            functions.open_bluemindo(self.window)

    def on_gtk_clicked(self, widget):
        self.hideshow()

    def trayicon_menu(self, status_icon=None, button=0, activate_time=0):
        def tray_window(widget):
            """Hide or show the window."""
            self.hideshow()

        def tray_quit(widget):
            """Quit Bluemindo. :'("""
            self.extensions.load_event('OnBluemindoQuitted')
            functions.close_bluemindo(self.window)

        def tray_play(widget):
            """Load the Play button."""
            self.extensions.load_event('OnPlayPressed')

        def tray_next(widget):
            """Load the next button."""
            self.extensions.load_event('OnNextPressed')

        def tray_previous(widget):
            """Load the Previous button."""
            self.extensions.load_event('OnPreviousPressed')

        # Right button: context menu
        wdg = glade_XML(join(getcwd(), 'modules', 'trayicon',
                 'traycontext.glade'), 'menu1', domain='bluemindo')
        wdg.get_widget('menu1').popup(None, None, None, button, activate_time)

        wdg.get_widget('tray-bluemindo').connect('activate', tray_window)
        wdg.get_widget('tray-quit').connect('activate', tray_quit)
        wdg.get_widget('menu-play').connect('activate', tray_play)
        wdg.get_widget('menu-previous').connect('activate', tray_previous)
        wdg.get_widget('menu-next').connect('activate', tray_next)

    def on_clicked(self, widget, event):
        if event.button == 1:
            # Left button: hide/show
            self.hideshow()
        elif event.button == 3:
            # Functions for the right click context menu
            self.trayicon_menu()

    def on_tray_enter(self, widget, event):
        """The tray is focused by the mouse: show current song if any."""
        def show_notif():
            if self.is_the_cursor_on_tray:
                self.extensions.load_event('OnNotificationRequest')

        self.is_the_cursor_on_tray = True
        timeout_add(2000, show_notif)

    def on_tray_leave(self, widget, event):
        self.is_the_cursor_on_tray = False

    def handler_stop(self):
        """Updates the status icon when we stop our music."""
        self._set_icon('stop')

    def handler_play(self, song=None):
        """Updates the status icon when we play or pause our music."""
        gst = GStreamer()
        cur = gst.getstatus()

        if cur == 'PAUSED':
            self._set_icon('pause')
        else:
            self._set_icon('play')

    def _set_icon(self, what):
        if trayicon_type == 'gtk':
            return

        wtf = join(functions.datadir, 'image', 'logo_head_small_%s.png' % what)

        pixbuf = pixbuf_new_from_file(wtf)
        pixbuf = pixbuf.scale_simple(20, 20, INTERP_BILINEAR)
        self.tray_icon.set_from_pixbuf(pixbuf)