/* Bluefish HTML Editor
 * coloursel2.c - Colour Picker
 *
 * Copyright (C) 2000 Olivier Sessink
 *
 * this is a rewrite from scratch, the original colour picker
 * was Copyright (C) 1998 Neil Millar (neil@millar.u-net.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. 
 *
 */
#include "default_include.h"

#include <string.h>  	/* strncpy */
#include <stdlib.h>  /* strtol() */

#include "coloursel.h"  /* myself */
#include "gtk_easy.h"	/* window_destroy() */
#include "bluefish.h" 	/* main_v */
#include "document.h" 	/* insert_dbl_text */
#include "pixmaps.h" 	/* new_pixmap() */

typedef struct {
	GtkWidget *win;
	GtkWidget *hexentry;
	GtkWidget *websafe;
	GtkWidget *csel;
	gint is_modal;
	gint startpos;
	gint endpos;
	gint hex_changed_id;
	gint csel_changed_id;
} Tcolsel;

static gint string_is_color(gchar *color) {
	gint i;

	if (!color) {
		DEBUG_MSG("string_is_color, pointer NULL\n");
		return 0;
	}
	if (strlen(color) != 7) {
		DEBUG_MSG("string_is_color, strlen(%s) != 7\n", color);
		return 0;
	}
	if (color[0] != '#') {
		DEBUG_MSG("string_is_color, 0 in %s is not #\n", color);
		return 0;
	}
	for (i = 1; i <7 ; i++) {
		if ((color[i] > 102) 
				|| (color[i] < 48) 
				|| ((color[i] > 57) && (color[i] < 65))
				|| ((color[i] > 70) && (color[i] < 97))) {
			DEBUG_MSG("string_is_color, %d in %s is not from a color, it is %d\n", i, color, color[i]);
			return 0;
		}
	}
	DEBUG_MSG("string_is_color, %s is color\n", color);
	return 1;

}

static gchar *gdouble_arr_to_hex(gdouble *color, gint websafe)
{
	gchar *tmpstr;
	unsigned int red_int;
	unsigned int green_int;
	unsigned int blue_int;
	gdouble red;
	gdouble green;
	gdouble blue;
	
	red = color[0];
	green = color[1];
	blue = color[2];

	if (websafe) {
		red_int = 0x33*((unsigned int) (red * 255 / 0x33));
		green_int = 0x33*((unsigned int) (green * 255/0x33));
		blue_int = 0x33*((unsigned int) (blue * 255/0x33));
	} else {
		red_int = (unsigned int) (red * 255);
		green_int = (unsigned int) (green * 255);
		blue_int = (unsigned int) (blue * 255);
	}
	tmpstr = g_malloc(8*sizeof(char));
	g_snprintf (tmpstr, 8,"#%.2X%.2X%.2X", red_int, green_int, blue_int);
	return tmpstr;
}

static gdouble *hex_to_gdouble_arr(gchar *color)
{
	static gdouble tmpcol[4];
	gchar tmpstr[8];
	long tmpl;

	if (!string_is_color(color)) {
		DEBUG_MSG("hex_to_gdouble_arr, color is not a valid #...... color\n");
		return NULL;
	}
	strncpy(tmpstr, &color[1], 2);
	tmpl = strtol(tmpstr, NULL, 16);
	tmpcol[0] = (gdouble) tmpl;
	
	strncpy(tmpstr, &color[3], 2);
	tmpl = strtol(tmpstr, NULL, 16);
	tmpcol[1] = (gdouble) tmpl;
	
	strncpy(tmpstr, &color[5], 2);
	tmpl = strtol(tmpstr, NULL, 16);
	tmpcol[2] = (gdouble) tmpl;

	DEBUG_MSG("hex_to_gdouble_arr, R=%d, G=%d, B=%d\n", color[0], color[1], color[2]);

	tmpcol[3] = 0;
	return tmpcol;
}
 
static void colsel_destroy_lcb(GtkWidget *widget, GdkEvent *event, Tcolsel *csd) {
	if (csd->is_modal) {
		gtk_main_quit();
	} else {
		gchar *tmpstr;
		
		tmpstr = gtk_editable_get_chars(GTK_EDITABLE(csd->hexentry), 0, -1);
		if (strlen(tmpstr) == 7) {
			if (csd->startpos || csd->endpos) {
				replace_text(tmpstr, csd->startpos, csd->endpos);
			} else {
				insert_dbl_text(tmpstr, NULL);
			}
		}
		g_free(tmpstr);
		
		window_destroy(csd->win);
		g_free(csd);
	}
}

static void colsel_ok_clicked_lcb(GtkWidget *widget, Tcolsel *csd) { 
	colsel_destroy_lcb(NULL, NULL, csd);
}

static void colsel_cancel_clicked_lcb(GtkWidget *widget, Tcolsel *csd) {
	gtk_entry_set_text(GTK_ENTRY(csd->hexentry), "");
	colsel_destroy_lcb(NULL, NULL, csd);
}
/* declaration needed to connect/disconnect callback */
static void colsel_color_changed(GtkWidget *widget, Tcolsel *csd);

static void hexentry_color_changed(GtkWidget *widget, Tcolsel *csd) {
	gdouble *color;
	gchar *tmpstr;
	tmpstr=gtk_editable_get_chars(GTK_EDITABLE(csd->hexentry), 0, -1);
	if (strlen(tmpstr) == 7){
		color = hex_to_gdouble_arr(tmpstr);
		if (color) {
			gtk_signal_disconnect(GTK_OBJECT(csd->csel), csd->csel_changed_id);
			gtk_color_selection_set_color(GTK_COLOR_SELECTION(csd->csel), color);
			csd->csel_changed_id = gtk_signal_connect(GTK_OBJECT(csd->csel), "color-changed", colsel_color_changed, csd);
		} else {
			tmpstr=NULL;
		}
	}
	g_free(tmpstr);
}


static void colsel_color_changed(GtkWidget *widget, Tcolsel *csd) {
	gdouble color[4];
	gchar *tmpstr;

	gtk_color_selection_get_color(GTK_COLOR_SELECTION(csd->csel), color);
	tmpstr = gdouble_arr_to_hex(color, gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(csd->websafe)));

	gtk_signal_disconnect(GTK_OBJECT(csd->hexentry), csd->hex_changed_id);
	gtk_entry_set_text(GTK_ENTRY(csd->hexentry), tmpstr);
	csd->hex_changed_id = gtk_signal_connect(GTK_OBJECT(csd->hexentry), "changed", hexentry_color_changed, csd);
	g_free(tmpstr);
}

static Tcolsel *colsel_dialog(gchar *setcolor, gint modal, gint startpos, gint endpos) {

	Tcolsel *csd;
	GtkWidget *vbox, *hbox, *but;
	gdouble *color;
	gchar *this_color=setcolor;

	csd = g_malloc(sizeof(Tcolsel));
	csd->is_modal = modal;
	csd->startpos = startpos;
	csd->endpos = endpos;
	DEBUG_MSG("colsel_dialog, malloced at %p\n", csd);
	csd->win = window_full(_("Bluefish: Select color"), GTK_WIN_POS_MOUSE, GTK_WINDOW_DIALOG
			, 5, colsel_destroy_lcb, csd);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(csd->win), vbox);
	csd->csel = gtk_color_selection_new();
	if (this_color) {
		color = hex_to_gdouble_arr(this_color);
		if (color) {
			gtk_color_selection_set_color(GTK_COLOR_SELECTION(csd->csel), color);
		} else {
			this_color=NULL;
		}
	}
	csd->csel_changed_id = gtk_signal_connect(GTK_OBJECT(csd->csel), "color-changed", colsel_color_changed, csd);
	gtk_box_pack_start(GTK_BOX(vbox), csd->csel, TRUE, TRUE, 0);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	
	csd->hexentry = boxed_entry_with_text(this_color, 7, hbox);
	csd->hex_changed_id = gtk_signal_connect(GTK_OBJECT(csd->hexentry), "changed", hexentry_color_changed, csd);
	csd->websafe = boxed_checkbut_with_value(_("websafe"), 0, hbox);

	hbox = gtk_hbutton_box_new();
	gtk_hbutton_box_set_layout_default(GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 1);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

	but = bf_stock_ok_button(colsel_ok_clicked_lcb, csd);
	gtk_window_set_default(GTK_WINDOW(csd->win), but);
	gtk_box_pack_start(GTK_BOX(hbox), but, TRUE, TRUE, 0);
	but = bf_stock_cancel_button(colsel_cancel_clicked_lcb, csd);
	gtk_box_pack_start(GTK_BOX(hbox), but, TRUE, TRUE, 0);
	gtk_widget_show_all(csd->win);
	DEBUG_MSG("colsel_dialog, finished\n");
	return csd;
}

void sel_colour_cb(GtkWidget *widget, gpointer data) {
	Tcolsel *csd;
	gchar *tmpstr=NULL;
	gint startpos=0;
	gint endpos=0;

	if (GTK_EDITABLE(main_v->current_document->textbox)->has_selection) {
		DEBUG_MSG("sel_colour_cb, selection found\n");
		startpos = GTK_EDITABLE(main_v->current_document->textbox)->selection_start_pos;
		endpos = GTK_EDITABLE(main_v->current_document->textbox)->selection_end_pos;
		if (startpos > endpos) {
			gint tmpint;
			tmpint = startpos;
			startpos = endpos;		
			endpos = tmpint;
		}
		if ((endpos - startpos) == 7) {
			tmpstr = gtk_editable_get_chars(GTK_EDITABLE(main_v->current_document->textbox) ,startpos, endpos);
			if (!string_is_color(tmpstr)) {
				startpos = endpos = 0;
			}
			DEBUG_MSG("sel_colour_cb, tmpstr = %s (startpos=%d, endpos=%d)\n", tmpstr, startpos, endpos);
		} else {
			DEBUG_MSG("sel_colour_cb, startpos=%d, endpos=%d\n", startpos, endpos);
			startpos = endpos = 0;
		}
	} else {
		DEBUG_MSG("sel_colour_cb, NO selection found\n");
	}

	csd = colsel_dialog(tmpstr, 0, startpos, endpos);
	if (tmpstr) {
		g_free(tmpstr);
	}
}
 
gchar *return_color(gchar *start_value) {

	Tcolsel *csd;
	gchar *return_text;

	csd = colsel_dialog(start_value, 1, 0, 0);
	DEBUG_MSG("return color, started\n");
	gtk_grab_add(csd->win);
	gtk_main();

	return_text = gtk_editable_get_chars(GTK_EDITABLE(csd->hexentry), 0, -1);
	window_destroy(csd->win);
	g_free(csd);
	DEBUG_MSG("return_color, the new gtk_main stopped, return_text=%s, %p\n", return_text, return_text);
	return return_text;

}

static void color_but_clicked(GtkWidget * widget, GtkWidget * entry)
{
	gchar *tmpstring, *tmpstr2;

	DEBUG_MSG("color_but_clicked, before return_color\n");
	tmpstr2 = gtk_editable_get_chars(GTK_EDITABLE(entry),0 ,-1);
	tmpstring = return_color(tmpstr2);
	DEBUG_MSG("color_but_clicked, return_color=%s\n", tmpstring);
	gtk_entry_set_text(GTK_ENTRY(entry), tmpstring);
	g_free(tmpstring);
	g_free(tmpstr2);

}

GtkWidget *color_but_new(GtkWidget * which_entry, GtkWidget * win)
{

	GtkWidget *pixmap, *color_but;

	color_but = gtk_button_new();
	pixmap = new_pixmap(176, win->window, NULL);
	gtk_widget_show(pixmap);
	gtk_container_add(GTK_CONTAINER(color_but), pixmap);
	gtk_signal_connect(GTK_OBJECT(color_but), "clicked", color_but_clicked, which_entry);
	gtk_widget_show(color_but);
	return color_but;
}
