/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "NonCopyable.h"
#include "cache/AbstractItem.h"
#include "cache/Item.h"
#include "cache/RequestResolution.h"
#include "cache/ResponseResolution.h"
#include "cache/AbstractResponseReader.h"
#include "cache/AbstractResponseWriter.h"
#include "cache/AbstractResponseHandler.h"
#include "cache/AbstractFileStorage.h"
#include "cache/FileStorage.h"
#include "TempDir.h"
#include "RequestCacheControl.h"
#include "HttpRequestMetadata.h"
#include "HttpResponseMetadata.h"
#include "HttpRequestLine.h"
#include "HttpStatusLine.h"
#include "HttpHeadersCollection.h"
#include "HttpHeader.h"
#include "HttpVersion.h"
#include "BString.h"
#include "SplittableBuffer.h"
#include "Date.h"
#include "URI.h"
#include "IntrusivePtr.h"
#include "AbstractCommand.h"
#include "types.h"
#include <ace/config-lite.h>
#include <ace/OS_NS_unistd.h>
#include <boost/test/auto_unit_test.hpp>
#include <memory>
#include <sstream>
#include <algorithm>
#include <string>
#include <string.h>
#include <stddef.h>

using namespace HttpCache;

namespace
{

class Fixture
{
	DECLARE_NON_COPYABLE(Fixture)
public:
	static int const DEFAULT_EXPIRATION;
	
	Fixture();
	
	~Fixture();
	
	void testInterestedInResponse();
	
	void testWillingToCache();
	
	void testCacheThenRead(int const* expiration = &DEFAULT_EXPIRATION);
	
	void testCacheThenReloadThenRead();
	
	void testGarbageCollection();
	
	void testCacheThenSuccessfulValidation();
	
	void testCacheThenUnsuccessfulValidation();
	
	void testExpireThenUpdateThenRead();
private:
	void reload(uint32_t version);
	
	void cacheObject(
		std::string const& body, int const* expiration = &DEFAULT_EXPIRATION);
	
	bool checkCachedResponse(
		AbstractResponseReader& reader, std::string const& body);
	
	bool checkCachedResponse(std::string const& body);
	
	static HttpRequestLine getRequestLine();
	
	static std::auto_ptr<HttpRequestMetadata> getRequestMetadata();
	
	static HttpResponseMetadata getCacheableResponseMetadata(
		size_t body_size, int const* expiration = &DEFAULT_EXPIRATION);
	
	static HttpResponseMetadata getNotModifiedResponseMetadata();
	
	TempDir m_cacheDir;
	IntrusivePtr<AbstractItem> m_ptrItem;
};


int const Fixture::DEFAULT_EXPIRATION = 100000;

Fixture::Fixture()
{
	reload(0);
}

Fixture::~Fixture()
{
}

void
Fixture::testInterestedInResponse()
{
	ConstRequestPtr const request(getRequestMetadata().release());
	RequestResolution const req_res(m_ptrItem->handleRequest(request));
	BOOST_CHECK(req_res.handler.get());
	BOOST_CHECK(!req_res.reader.get());
}

void
Fixture::testWillingToCache()
{
	ConstRequestPtr const request(getRequestMetadata().release());
	RequestResolution req_res(m_ptrItem->handleRequest(request));
	BOOST_REQUIRE(req_res.handler.get());
	
	HttpResponseMetadata const response_metadata(getCacheableResponseMetadata(100000));
	ResponseResolution resp_res(req_res.handler->handleResponse(response_metadata));
	BOOST_CHECK(resp_res.writer.get());
	BOOST_CHECK(!resp_res.reader.get());
}

void
Fixture::testCacheThenRead(int const* expiration)
{
	std::string body(100000, 'a');
	cacheObject(body, expiration);
	BOOST_CHECK(checkCachedResponse(body));
}

void
Fixture::testCacheThenReloadThenRead()
{
	std::string body(100000, 'a');
	cacheObject(body);
	
	reload(1);
	
	BOOST_CHECK(checkCachedResponse(body));
}

void
Fixture::testGarbageCollection()
{
	std::string body(100000, 'a');
	cacheObject(body);
	
	BOOST_REQUIRE(checkCachedResponse(body));
	BOOST_CHECK(!m_ptrItem->prepareForGC(AbstractItem::GC_ONLY_UNCACHED));
	BOOST_CHECK(m_ptrItem->prepareForGC());
}

void
Fixture::testCacheThenSuccessfulValidation()
{
	std::string body(100000, 'a');
	cacheObject(body);
	
	RequestPtr const request(getRequestMetadata().release());
	request->headers().setHeader(BString("Cache-Control"), BString("max-age=0"));
	RequestResolution req_res(m_ptrItem->handleRequest(request));
	BOOST_REQUIRE(req_res.handler.get());
	BOOST_CHECK(!req_res.reader.get());
	
	ConstRequestPtr const outgoing_request(req_res.handler->getOutgoingRequest());
	BOOST_CHECK(outgoing_request->headers().hasHeader(BString("If-None-Match")));
	
	ResponseResolution resp_res(
		req_res.handler->handleResponse(getNotModifiedResponseMetadata())
	);
	BOOST_REQUIRE(resp_res.reader.get());
	BOOST_CHECK(!resp_res.writer.get());
	
	BOOST_CHECK(checkCachedResponse(*resp_res.reader, body));
}

void
Fixture::testCacheThenUnsuccessfulValidation()
{
	std::string body(100000, 'a');
	cacheObject(body);
	
	RequestPtr const request(getRequestMetadata().release());
	request->headers().setHeader(BString("Cache-Control"), BString("max-age=0"));
	RequestResolution req_res(m_ptrItem->handleRequest(request));
	BOOST_REQUIRE(req_res.handler.get());
	BOOST_CHECK(!req_res.reader.get());
	
	ConstRequestPtr const outgoing_request(req_res.handler->getOutgoingRequest());
	BOOST_CHECK(outgoing_request->headers().hasHeader(BString("If-None-Match")));
	
	ResponseResolution resp_res(
		req_res.handler->handleResponse(getCacheableResponseMetadata(100000))
	);
	BOOST_CHECK(resp_res.writer.get());
	BOOST_CHECK(!resp_res.reader.get());
}

void
Fixture::testExpireThenUpdateThenRead()
{
	std::string body(100000, 'a');
	int const expiration = 2;
	cacheObject(body, &expiration);
	
	ACE_OS::sleep(expiration);
	
	RequestPtr const request(getRequestMetadata().release());
	RequestResolution req_res(m_ptrItem->handleRequest(request));
	BOOST_REQUIRE(req_res.handler.get());
	BOOST_CHECK(!req_res.reader.get());
	
	ConstRequestPtr const outgoing_request(req_res.handler->getOutgoingRequest());
	BOOST_CHECK(outgoing_request->headers().hasHeader(BString("If-None-Match")));
	
	ResponseResolution resp_res(
		req_res.handler->handleResponse(getNotModifiedResponseMetadata())
	);
	BOOST_REQUIRE(resp_res.reader.get());
	BOOST_CHECK(!resp_res.writer.get());
	
	// Because our "304 Not Modified" response has the Date header, the response
	// will be considered fresh for another 2 seconds.
	
	req_res = m_ptrItem->handleRequest(request);
	BOOST_REQUIRE(req_res.reader.get());
	BOOST_CHECK(!req_res.handler.get());
	
	BOOST_CHECK(checkCachedResponse(*req_res.reader, body));
}

void
Fixture::reload(uint32_t const version)
{
	FileId const file_id(FileId::Key(), version);
	m_ptrItem.reset(0); // It's not good to have two identical Item's at the same time.
	IntrusivePtr<AbstractFileStorage> file_storage(
		FileStorage::create(m_cacheDir.get(), 150000, IntrusivePtr<AbstractCommand>())
	);
	bool file_ok = false;
	m_ptrItem = Item::create(file_storage, file_id, file_ok);
}

void
Fixture::cacheObject(std::string const& body, int const* expiration)
{
	ConstRequestPtr const request(getRequestMetadata().release());
	RequestResolution req_res(m_ptrItem->handleRequest(request));
	BOOST_REQUIRE(req_res.handler.get());
	
	HttpResponseMetadata const response_metadata(
		getCacheableResponseMetadata(body.size(), expiration)
	);
	ResponseResolution resp_res(req_res.handler->handleResponse(response_metadata));
	BOOST_REQUIRE(resp_res.writer.get());
	
	char const* pos = body.c_str();
	char const* const end = pos + body.size();
	while (pos != end) {
		size_t const chunk = std::min<size_t>(end - pos, 8000);
		ssize_t const written = resp_res.writer->writeBodyData(pos, chunk);
		BOOST_REQUIRE(written > 0);
		pos += written;
	}
	resp_res.writer->commit();
}

bool
Fixture::checkCachedResponse(AbstractResponseReader& reader, std::string const& body)
{
	std::auto_ptr<HttpResponseMetadata> metadata(reader.retrieveMetadata());
	if (!metadata.get()) {
		return false;
	}
	if (!metadata->headers().hasHeader(BString("X-Special-Header"))) {
		return false;
	}
	
	char const* pos = body.c_str();
	char const* const end = pos + body.size();
	char buf[4096];
	while (pos != end) {
		ssize_t const chunk = reader.readBodyData(buf, sizeof(buf));
		if (chunk == -1) {
			break;
		}
		if (chunk > end - pos) {
			return false;
		}
		if (memcmp(pos, buf, chunk)) {
			return false;
		}
		pos += chunk;
	}
	
	return pos == end && reader.readBodyData(buf, sizeof(buf)) == 0;
}

bool
Fixture::checkCachedResponse(std::string const& body)
{
	ConstRequestPtr const request(getRequestMetadata().release());
	RequestResolution req_res(m_ptrItem->handleRequest(request));
	if (!req_res.reader.get()) {
		return false;
	}
	
	return checkCachedResponse(*req_res.reader, body);
}

HttpRequestLine
Fixture::getRequestLine()
{
	return HttpRequestLine(
		BString("GET"), URI(BString("http://localhost/")),
		HttpVersion::HTTP_1_1
	);
}

std::auto_ptr<HttpRequestMetadata>
Fixture::getRequestMetadata()
{
	std::auto_ptr<HttpRequestMetadata> metadata(
		new HttpRequestMetadata(getRequestLine())
	);
	metadata->setBodyStatus(HttpRequestMetadata::BODY_FORBIDDEN);
	metadata->headers().setHeader(
		BString("Host"), metadata->requestLine().getURI().getHost()
	);
	return metadata;
}

HttpResponseMetadata
Fixture::getCacheableResponseMetadata(
	size_t const body_size, int const* const expiration)
{
	HttpStatusLine const sline(HttpVersion::HTTP_1_1, 200);
	HttpResponseMetadata metadata(sline);
	metadata.setBodyStatus(HttpResponseMetadata::BODY_SIZE_KNOWN);
	metadata.setBodySize(body_size);
	metadata.headers().setHeader(BString("Date"), Date::formatCurrentTime());
	metadata.headers().setHeader(BString("ETag"), BString("\"123\""));
	if (expiration) {
		std::ostringstream strm;
		strm << "max-age=" << *expiration;
		metadata.headers().setHeader(
			BString("Cache-Control"), BString(strm.str())
		);
	}
	metadata.headers().setHeader(BString("X-Special-Header"), BString("1"));
	return metadata;
}

HttpResponseMetadata
Fixture::getNotModifiedResponseMetadata()
{
	HttpStatusLine const sline(HttpVersion::HTTP_1_1, 304);
	HttpResponseMetadata metadata(sline);
	metadata.setBodyStatus(HttpResponseMetadata::BODY_FORBIDDEN);
	metadata.headers().setHeader(BString("Date"), Date::formatCurrentTime());
	metadata.headers().setHeader(BString("ETag"), BString("\"123\""));
	return metadata;
}

} // anonymous namespace


BOOST_AUTO_UNIT_TEST(test_interested_in_response)
{
	Fixture fixture;
	fixture.testInterestedInResponse();
}

BOOST_AUTO_UNIT_TEST(test_willing_to_cache)
{
	Fixture fixture;
	fixture.testWillingToCache();
}

BOOST_AUTO_UNIT_TEST(test_cache_then_read)
{
	Fixture fixture;
	fixture.testCacheThenRead();
}

BOOST_AUTO_UNIT_TEST(test_cache_then_read_heuristic_expiration)
{
	Fixture fixture;
	fixture.testCacheThenRead(0);
}

BOOST_AUTO_UNIT_TEST(test_cache_then_reload_then_read)
{
	Fixture fixture;
	fixture.testCacheThenReloadThenRead();
}

BOOST_AUTO_UNIT_TEST(test_garbage_collection)
{
	Fixture fixture;
	fixture.testGarbageCollection();
}

BOOST_AUTO_UNIT_TEST(test_cache_then_successful_validation)
{
	Fixture fixture;
	fixture.testCacheThenSuccessfulValidation();
}

BOOST_AUTO_UNIT_TEST(test_cache_then_unsuccessful_validation)
{
	Fixture fixture;
	fixture.testCacheThenUnsuccessfulValidation();
}

BOOST_AUTO_UNIT_TEST(test_expire_then_update_then_read)
{
	Fixture fixture;
	fixture.testExpireThenUpdateThenRead();
}
