/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "BasicConfigDialog.h"
#include "Application.h"
#include "OperationLog.h"
#include "GlobalState.h"
#include "Conf.h"
#include "ConfigFile.h"
#include "StringUtils.h"
#include "SymbolicInetAddr.h"
#include "types.h"
#include <ace/config-lite.h>
#include <boost/lexical_cast.hpp>
#include <wx/frame.h>
#include <wx/icon.h>
#include <wx/panel.h>
#include <wx/string.h>
#include <wx/button.h>
#include <wx/radiobox.h>
#include <wx/arrstr.h>
#include <wx/textctrl.h>
#include <wx/checkbox.h>
#include <wx/stattext.h>
#include <wx/sizer.h>
#include <wx/msgdlg.h>
#include <list>
#include <sstream>
#include <assert.h>

using namespace std;

namespace wxGUI
{

class BasicConfigDialog::Impl : public wxFrame
{
public:
	Impl();

	virtual ~Impl();

	static void show();
private:
	void onWindowClose(wxCloseEvent& evt);
	
	void loadConfig();
	
	void setCurrentListenPort(wxString const& port);
	
	void onApply(wxCommandEvent& evt);

	void onCancel(wxCommandEvent& evt);
	
	void showError(wxString const& msg);
	
	void showWarning(wxString const& msg);
	
	static Impl* m_spInstance;
	wxString m_curListenPort;
	wxTextCtrl* m_pListenPort;
	wxCheckBox* m_pTrayAnimation;
	wxRadioBox* m_pPageCleanup;

	DECLARE_EVENT_TABLE()
};

BEGIN_EVENT_TABLE(BasicConfigDialog::Impl, wxFrame)
	EVT_CLOSE(onWindowClose)
	EVT_BUTTON(wxID_OK, onApply)
	EVT_BUTTON(wxID_CANCEL, onCancel)
END_EVENT_TABLE()


void
BasicConfigDialog::show()
{
	Impl::show();
}


BasicConfigDialog::Impl* BasicConfigDialog::Impl::m_spInstance = 0;

BasicConfigDialog::Impl::Impl()
:	wxFrame(wxGetApp().GetTopWindow(), -1, _T("Basic Configuration")),
	m_pListenPort(0),
	m_pTrayAnimation(0),
	m_pPageCleanup(0)
{
	SetExtraStyle(wxWS_EX_BLOCK_EVENTS);
	SetIcon(wxIcon(_T("AppIcon"), wxBITMAP_TYPE_ICO_RESOURCE, 16, 16));

	wxBoxSizer *top_sizer = new wxBoxSizer(wxVERTICAL);
	SetSizer(top_sizer);
	
	wxPanel *panel = new wxPanel(this, -1);
	top_sizer->Add(panel, 1, wxEXPAND, 0);
	wxBoxSizer *panel_sizer = new wxBoxSizer(wxVERTICAL);
	panel->SetSizer(panel_sizer);

	wxBoxSizer *port_sizer = new wxBoxSizer(wxHORIZONTAL);
	panel_sizer->Add(port_sizer, 0, wxALIGN_CENTRE|wxALL, 5);
	port_sizer->Add(new wxStaticText(panel, -1, _T("Listen port")), 0, wxALIGN_CENTRE|wxALL, 5);
	m_pListenPort = new wxTextCtrl(panel, -1, wxEmptyString, wxDefaultPosition, wxSize(45, -1));
	port_sizer->Add(m_pListenPort, 0, wxALIGN_CENTRE|wxLEFT, 5);
	
	wxArrayString cleanup_options;
	cleanup_options.Add(_T("Substitute ads with clickable replacements."));
	cleanup_options.Add(_T("Remove ads completely, where it's safe."));
	cleanup_options.Add(_T("Always remove ads completely."));

	m_pPageCleanup = new wxRadioBox(
		panel, -1, _T("Ads"), wxDefaultPosition, wxDefaultSize,
		cleanup_options, 0, wxRA_SPECIFY_ROWS
	);
	panel_sizer->Add(m_pPageCleanup, 0, wxALIGN_CENTER|wxLEFT|wxRIGHT, 5);
	
	panel_sizer->AddSpacer(5);

	m_pTrayAnimation = new wxCheckBox(panel, -1, _T("Tray icon animation"));
	panel_sizer->Add(m_pTrayAnimation, 0, wxLEFT|wxRIGHT, 5);
	
	wxBoxSizer *bbox_sizer = new wxBoxSizer(wxHORIZONTAL);
	panel_sizer->Add(bbox_sizer, 0, wxALIGN_CENTRE|wxALL, 5);
	bbox_sizer->Add(new wxButton(panel, wxID_OK, _T("OK")), 0, wxALIGN_CENTRE|wxALL, 2);
	wxButton* cancel_btn = new wxButton(panel, wxID_CANCEL, _T("Cancel"));
	bbox_sizer->Add(cancel_btn, 0, wxALIGN_CENTRE|wxALL, 5);
	cancel_btn->SetDefault();
	
	panel_sizer->SetSizeHints(panel);
	top_sizer->SetSizeHints(this);
	
	loadConfig();
}

BasicConfigDialog::Impl::~Impl()
{
}

void
BasicConfigDialog::Impl::show()
{
	if (!m_spInstance) {
		m_spInstance = new Impl;
		m_spInstance->Show();
		m_spInstance->m_pListenPort->SetSelection(0, 0);
		// Why the hell it gets selected?
		// And why unselecting it only works after Show()?
	} else {
		m_spInstance->Show();
		m_spInstance->Raise();
	}
}

void
BasicConfigDialog::Impl::onWindowClose(wxCloseEvent& evt)
{
	if (m_spInstance == this) {
		m_spInstance = 0;
	}
	Destroy();
}

void
BasicConfigDialog::Impl::loadConfig()
{
	Config config(GlobalState::ReadAccessor()->config());
	list<SymbolicInetAddr> listen_addrs(config.getListenAddresses());
	if (listen_addrs.size() > 1) {
		setCurrentListenPort(wxEmptyString),
		showWarning(
			_T("Your current config has multiple listen addresses.\n")
			_T("You should be using the Advanced Configuration dialog.")
		);
	} else if (listen_addrs.size() == 1) {
		wxString val;
		val << listen_addrs.front().getPort();
		setCurrentListenPort(val);
	} else {
		setCurrentListenPort(wxEmptyString);
	}
	
	switch (config.getPageCleanupLevel()) {
	case Config::CLEANUP_OFF:
		m_pPageCleanup->SetSelection(0);
		break;
	case Config::CLEANUP_SAFE:
		m_pPageCleanup->SetSelection(1);
		break;
	case Config::CLEANUP_MAXIMUM:
		m_pPageCleanup->SetSelection(2);
		break;
	}

	m_pTrayAnimation->SetValue(config.isTrayAnimationEnabled());
}

void
BasicConfigDialog::Impl::setCurrentListenPort(wxString const& port)
{
	m_curListenPort = port;
	m_pListenPort->SetValue(port);
}

void
BasicConfigDialog::Impl::onApply(wxCommandEvent&)
{
	bool const listen_port_changed = (m_pListenPort->GetValue() != m_curListenPort);
	uint16_t listen_port = 0;
	if (listen_port_changed) {
		try {
			string port = StringUtils::trimStr(m_pListenPort->GetValue().mb_str());
			listen_port = boost::lexical_cast<uint16_t>(port);
		} catch (boost::bad_lexical_cast&) {}
		if (listen_port == 0) {
			showError(_T("Listen port must be an integer in range of [1, 65535]"));
			m_pListenPort->SetFocus();
			return;
		}
	}
	
	Config const old_config(GlobalState::ReadAccessor()->config());
	Config new_config(old_config);
	if (listen_port_changed) {
		new_config.setListenAddresses(
			list<SymbolicInetAddr>(1, SymbolicInetAddr("127.0.0.1", listen_port))
		);
	}
	
	switch (m_pPageCleanup->GetSelection()) {
	case 0:
		new_config.setPageCleanupLevel(Config::CLEANUP_OFF);
		break;
	case 1:
		new_config.setPageCleanupLevel(Config::CLEANUP_SAFE);
		break;
	case 2:
		new_config.setPageCleanupLevel(Config::CLEANUP_MAXIMUM);
		break;
	default:
		assert(0);
	}

	new_config.setTrayAnimationEnabled(m_pTrayAnimation->GetValue());
	
	Application& app = *Application::instance();
	
	if (!app.configFile().applyAndSave(new_config)) {
		app.showLogDialog();
		return;
	}

	Close();
}

void
BasicConfigDialog::Impl::onCancel(wxCommandEvent&)
{
	Close();
}

void
BasicConfigDialog::Impl::showError(wxString const& msg)
{
	wxMessageDialog* dialog = new wxMessageDialog(
		this, msg, _T("Error"), wxOK|wxICON_ERROR
	);
	dialog->ShowModal();
}

void
BasicConfigDialog::Impl::showWarning(wxString const& msg)
{
	wxMessageDialog* dialog = new wxMessageDialog(
		this, msg, _T("Warning"), wxOK|wxICON_EXCLAMATION
	);
	dialog->ShowModal();
}

} // namespace wxGUI
