/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef FORWARDING_H_
#define FORWARDING_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "NonCopyable.h"
#include "IntrusivePtr.h"
#include "RefCountable.h"
#include "RefCounter.h"
#include "ProxyDescriptor.h"
#include <string>
#include <memory>
#include <iosfwd>
#include <list>
#include <vector>
#include <map>
#include <stddef.h>

class URI;
class ConfErrorHandler;

namespace Forwarding
{

/**
 * \brief A minimal wrapper around std::string.
 *
 * This class mostly serves as a reminder for client code that it's dealing
 * with utf-8 data.
 */
class Utf8String
{
public:
	Utf8String() {}
	
	explicit Utf8String(std::string const& raw) : m_storage(raw) {}
	
	// Member-wise copying is OK.
	
	std::string& raw() { return m_storage; }
	
	std::string const& raw() const { return m_storage; }
	
	void swap(Utf8String& other) { m_storage.swap(other.m_storage); }
private:
	std::string m_storage;
};

inline void swap(Utf8String& o1, Utf8String& o2)
{
	o1.swap(o2);
}

inline bool operator==(Utf8String const& lhs, Utf8String const& rhs)
{
	return lhs.raw() == rhs.raw();
}

inline bool operator!=(Utf8String const& lhs, Utf8String const& rhs)
{
	return lhs.raw() != rhs.raw();
}

inline bool operator<(Utf8String const& lhs, Utf8String const& rhs)
{
	return lhs.raw() < rhs.raw();
}


class Matcher : public RefCountable<RefCounter<ACE_MT_SYNCH> >
{
public:
	virtual ~Matcher() {}
	
	virtual bool matches(URI const& uri) const = 0;
};

typedef IntrusivePtr<Matcher const> MatcherConstPtr;


/**
 * \brief A base class for children of \<bypass\> tag.
 */
class MatchPattern : public RefCountable<RefCounter<ACE_NULL_SYNCH> >
{
public:
	virtual ~MatchPattern() {}
	
	/**
	 * \brief Get a matcher object.
	 * \return Returns a smart pointer to a matcher object.
	 *         In case of a syntax error in the pattern, a null
	 *         smart pointer is returned.
	 */
	virtual MatcherConstPtr getMatcher() const = 0;

	virtual void toStream(std::ostream& strm, int indent) const = 0;
};


class BypassVisitor;


/**
 * \brief Another base class for children of \<bypass\> tag.
 */
class BypassPattern : public MatchPattern
{
public:
	/**
	 * \brief A part of the "Visitor" design pattern.
	 */
	virtual void accept(BypassVisitor& visitor) const = 0;
};


typedef IntrusivePtr<BypassPattern const> BypassPatternConstPtr;


/**
 * \brief Represents the \<bypass\> tag in forwarding.xml.
 */
class BypassList : public std::list<BypassPatternConstPtr>
{
public:
	// Member-wise copying is OK.
	
	/**
	 * \brief A part of the "Visitor" design pattern.
	 */
	void accept(BypassVisitor& visitor) const;

	void toStream(std::ostream& strm, int indent) const;
};

inline void swap(BypassList& o1, BypassList& o2)
{
	o1.swap(o2);
}


/**
 * \brief Represents the \<domain\> tag in forwarding.xml.
 */
class DomainMatchPattern : public MatchPattern
{
public:
	DomainMatchPattern(Utf8String const& pattern) : m_pattern(pattern) {}
	
	virtual MatcherConstPtr getMatcher() const;
	
	virtual void toStream(std::ostream& strm, int indent) const;
private:
	Utf8String m_pattern;
};


/**
 * \brief Represents the \<subnet\> tag in forwarding.xml.
 */
class SubnetMatchPattern : public MatchPattern
{
public:
	SubnetMatchPattern(Utf8String const& pattern) : m_pattern(pattern) {}
	
	virtual MatcherConstPtr getMatcher() const;
	
	virtual void toStream(std::ostream& strm, int indent) const;
private:
	Utf8String m_pattern;
};


/**
 * \brief Represents the \<host-mask\> tag in forwarding.xml.
 */
class HostMaskMatchPattern : public BypassPattern
{
public:
	HostMaskMatchPattern(Utf8String const& pattern) : m_pattern(pattern) {}
	
	Utf8String const& getPattern() const { return m_pattern; }

	virtual MatcherConstPtr getMatcher() const;

	virtual void accept(BypassVisitor& visitor) const;
	
	virtual void toStream(std::ostream& strm, int indent) const;
private:
	Utf8String m_pattern;
};


/**
 * \brief Represents the \<simple-hostnames\> tag in forwarding.xml.
 *
 * This pattern matches hostnames without dots.
 * It corresponds to "Exclude simple hostnames" option in OSX
 * and "Bypass proxy server for local addresses" in Windows.
 */
class SimpleHostnamesMatchPattern : public BypassPattern
{
public:
	SimpleHostnamesMatchPattern() {}
	
	virtual MatcherConstPtr getMatcher() const;
	
	virtual void accept(BypassVisitor& visitor) const;

	virtual void toStream(std::ostream& strm, int indent) const;
};


/**
 * \brief Used to traverse BypassList elements.
 */
class BypassVisitor
{
public:
	virtual ~BypassVisitor() {}

	virtual void visit(HostMaskMatchPattern const& p) = 0;

	virtual void visit(SimpleHostnamesMatchPattern const& p) = 0;
};


/**
 * \brief Represents the \<proxy-chain\> tag in forwarding.xml.
 */
class ProxyDescriptorList : public std::list<ProxyDescriptor>
{
public:
	// Member-wise copying is OK.
	
	void toStream(std::ostream& strm, int indent) const;
private:
	void itemToStream(
		ProxyDescriptor const& item,
		std::ostream& strm, int indent) const;
};

inline void swap(ProxyDescriptorList& o1, ProxyDescriptorList& o2)
{
	o1.swap(o2);
}


/**
 * \brief Represents the \<option\> tag in forwarding.xml.
 */
class Option
{
public:
	Option(Utf8String const& name);
	
	~Option();
	
	// Member-wise copying is OK.
	
	Utf8String const& getName() const { return m_name; }
	
	void setName(Utf8String const& name) { m_name = name; }
	
	bool isSelected() const { return m_isSelected; }
	
	void setSelected(bool val = true) { m_isSelected = val; }
	
	BypassList& bypassList() { return m_bypassList; }
	
	BypassList const& bypassList() const { return m_bypassList; }
	
	ProxyDescriptorList& proxyChain() { return m_proxyChain; }
	
	ProxyDescriptorList const& proxyChain() const { return m_proxyChain; }
	
	void swap(Option& other);
	
	void toStream(std::ostream& strm, int indent) const;
private:
	Utf8String m_name;
	bool m_isSelected;
	BypassList m_bypassList;
	ProxyDescriptorList m_proxyChain;
};

inline void swap(Option& o1, Option& o2)
{
	o1.swap(o2);
}


/**
 * \brief Represents a collection of \<option\> tags in forwarding.xml.
 */
class OptionList : public std::list<Option>
{
public:
	// Member-wise copying is OK.
	
	void toStream(std::ostream& strm, int indent) const;
};

inline void swap(OptionList& o1, OptionList& o2)
{
	o1.swap(o2);
}


/**
 * \brief Represents forwarding.xml config file.
 */
class Config
{
public:
	Config();
	
	~Config();
	
	// Member-wise copying is OK.
	
	OptionList& options() { return m_options; }
	
	OptionList const& options() const { return m_options; }
	
	void swap(Config& other);
	
	/**
	 * \brief Writes an XML file to a stream.
	 */
	void toStream(std::ostream& strm) const;
	
	/**
	 * \brief Loads an XML file into *this.
	 *
	 * In case of a critical error, false is returned and *this
	 * is left unmodified.  In other cases (even if some non-critical
	 * errors were detected), true is returned and *this is modified.
	 * If false if returned, the problem is guaranteed to be reported
	 * through \a eh.
	 */
	bool fromStream(std::istream& strm, ConfErrorHandler& eh);
private:
	OptionList m_options;
};

inline void swap(Config& o1, Config& o2)
{
	o1.swap(o2);
}


/**
 * \brief Represents a forwarding option based on the current OSX network
 *        profile (aka Location).
 *
 * If the current profile name begins with "(BF) ", the next-hop proxy
 * is taken from its non-(BF) counterpart, otherwise it's taken from
 * the current profile itself.  The name always corresponds to the current
 * network profile, be it a (BF) profile or not.
 */
class SystemOption
{
public:
	SystemOption(Utf8String const& name);
	
	~SystemOption();
	
	// Member-wise copying is OK.
	
	Utf8String const& getName() const { return m_name; }
	
	void setName(Utf8String const& name) { m_name = name; }
	
	std::string const& getErrorMessage() const { return m_errorMsg; }
	
	void setErrorMessage(std::string const& msg) { m_errorMsg = msg; }
	
	bool hasError() const { return !m_errorMsg.empty(); }
	
	BypassList& bypassList() { return m_bypassList; }
	
	BypassList const& bypassList() const { return m_bypassList; }
	
	ProxyDescriptorList& proxyChain() { return m_proxyChain; }
	
	ProxyDescriptorList const& proxyChain() const { return m_proxyChain; }
private:
	Utf8String m_name;
	std::string m_errorMsg;
	BypassList m_bypassList;
	ProxyDescriptorList m_proxyChain;
};


/**
 * \brief A ref-countable vector of ProxyDescriptor objects.
 */
class ProxyChain :
	public RefCountable<RefCounter<ACE_MT_SYNCH> >,
	public std::vector<ProxyDescriptor>
{
};

typedef IntrusivePtr<ProxyChain const> ProxyChainConstPtr;


class ResolverException : public std::exception
{
public:
	ResolverException(std::string const& msg) : m_msg(msg) {}
	
	virtual ~ResolverException() throw() {}
	
	virtual char const* what() const throw() { return m_msg.c_str(); }
private:
	std::string m_msg;
};


/**
 * \brief Decides which proxy chain (if any) to use for a given URL.
 */
class Resolver
{
	DECLARE_NON_COPYABLE(Resolver)
public:
	Resolver();
	
	Resolver(Config const& config);
	
	~Resolver();
	
	/**
	 * \brief Selects an option specified by its name.
	 * \throw ResolverException
	 *        In case the specified option is not found,
	 *        ResolverException is thrown.
	 */
	void selectOption(Utf8String const& option_name);
	
	/**
	 * \brief Selects an option based on the current OSX network profile
	 *        (aka Location).
	 * \note System option will be overriden with a non-system option
	 *       having the same name (in case such an option exists).
	 */
	void selectSystemOption(SystemOption const& option);
	
	/**
	 * \brief Decides which proxy chain (if any) to use for a given URL.
	 * \return Returns a smart pointer to ProxyChain. The pointer is
	 *         guaranteed to be non null, but the chain may be empty.
	 * \throw ResolverException
	 *        In case of a configuration problem, ResolverException
	 *        is thrown.  An example would be:
	 *        "Unsupported feature: proxy auto configuration" that comes
	 *        from SystemOption.
	 */
	ProxyChainConstPtr resolve(URI const& url) const;
	
	void swap(Resolver& other);
private:
	typedef std::vector<MatcherConstPtr> BypassSet;
	
	struct Opt
	{
		BypassSet bypassSet;
		ProxyChainConstPtr proxyChain;
	};
	
	struct SysOpt : public Opt
	{
		std::string errorMessage;
	};
	
	static BypassSet createBypassSet(BypassList const& list);
	
	static ProxyChainConstPtr createProxyChain(
		ProxyDescriptorList const& proxies);
	
	static bool matches(URI const& url, BypassSet const& bypass_set);
	
	ProxyChainConstPtr const m_ptrEmptyProxyChain;
	std::map<Utf8String, Opt> m_options; // by name
	std::auto_ptr<SysOpt> m_ptrSystemOption;
	Opt* m_pSelectedOption;
};

inline void swap(Resolver& o1, Resolver& o2)
{
	o1.swap(o2);
}


} // namespace Forwarding

#endif
