/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef HTTPCACHE_ABSTRACTFILEIO_H_
#define HTTPCACHE_ABSTRACTFILEIO_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <ace/config-lite.h>
#include <ace/Basic_Types.h> // for ssize_t, off_t
#include <stddef.h>

namespace HttpCache
{

class AbstractFileIO
{
public:
	virtual ~AbstractFileIO() {}
	
	/**
	 * \brief Read up to \p len bytes.
	 * \return The number of bytes actually read, 0 on EOF, or -1 on error.
	 */
	virtual ssize_t read(void* buf, size_t len) = 0;
	
	/**
	 * \brief Read exactly \p len bytes.
	 * \return The number of bytes actually read.
	 * \note If the returned value is less than \p len, it means it was
	 *       impossible to read more. It's possible to tell if the reason
	 *       was an error or an EOF by providing the \p last_result parameter.
	 */
	virtual size_t read_n(void* buf, size_t len, ssize_t* last_result = 0) = 0;
	
	/**
	 * \brief Write up to \p len bytes.
	 * \return The number of bytes actually written, or -1 on error.
	 */
	virtual ssize_t write(void const* buf, size_t len) = 0;
	
	/**
	 * \brief Write exactly \p len bytes.
	 * \return The number of bytes actually written.
	 * \note If the returned value is less than \p len, it means it was
	 *       impossible to write more.
	 */
	virtual size_t write_n(void const* buf, size_t len) = 0;
	
	/**
	 * \brief Seek and return the resulting absolute position.
	 *
	 * \param offset Positive or negative offset relative to seek origin.
	 * \param whence One of SEEK_SET, SET_CUR or SEEK_END.  Just like with fseek().
	 */
	virtual off_t seek(off_t offset, int whence) = 0;
	
	virtual off_t tell() = 0;
	
	virtual int truncate(off_t len) = 0;
};

} // namespace HttpCache

#endif
