/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CachingServer.h"
#include "NonCopyable.h"
#include "AbstractRequestHandler.h"
#include "AbstractResponseHandler.h"
#include "ImmediateResponse.h"
#include "ErrorDescriptor.h"
#include "HttpRequestMetadata.h"
#include "HttpResponseMetadata.h"
#include "HttpStatusLine.h"
#include "RequestPtr.h"
#include "RequestTag.h"
#include "DataChunk.h"
#include "SplittableBuffer.h"
#include "ErrorDescriptor.h"
#include "ErrorFactory.h"
#include "ErrorCodes.h"
#include "cache/ObjectStorage.h"
#include "cache/RequestResolution.h"
#include "cache/ResponseResolution.h"
#include "cache/AbstractResponseReader.h"
#include "cache/AbstractResponseWriter.h"
#include "cache/AbstractResponseHandler.h"
#include <assert.h>

class CachingServer::CachedResponse : public ImmediateResponse
{
	DECLARE_NON_COPYABLE(CachedResponse)
public:
	CachedResponse(std::auto_ptr<HttpCache::AbstractResponseReader> reader);
	
	virtual ~CachedResponse();
	
	virtual void output(
		AbstractResponseHandler& handler,
		ConstRequestPtr const& request,
		RequestStatus& request_status,
		RequestTag const& request_tag);
private:
	virtual void eventGotProvisionalResponse(
		std::auto_ptr<HttpResponseMetadata> metadata);
		
	virtual void eventGotMetadata(
		std::auto_ptr<HttpResponseMetadata> metadata, bool is_persistent);
	
	std::auto_ptr<HttpCache::AbstractResponseReader> m_ptrReader;
	std::auto_ptr<HttpResponseMetadata> m_ptrResponseMetadata;
};


class CachingServer::CachingFilter : public AbstractResponseHandler
{
	DECLARE_NON_COPYABLE(CachingFilter)
public:
	CachingFilter(
		std::auto_ptr<HttpCache::AbstractResponseHandler> cache_handler,
		IntrusivePtr<AbstractResponseHandler> const& next_hop_handler,
		ConstRequestPtr const& request, RequestTag const& request_tag);
	
	virtual ~CachingFilter();
	
	virtual void processProvisionalResponse(
		RequestStatus& status,
		std::auto_ptr<HttpResponseMetadata> metadata);
	
	virtual void processResponseMetadata(
		RequestStatus& status,
		std::auto_ptr<HttpResponseMetadata> metadata);
	
	// postcondition: data is empty
	virtual void processBodyData(
		RequestStatus& status,
		SplittableBuffer& data, bool eof);
	
	virtual void processError(
		RequestStatus& status,
		std::auto_ptr<ErrorDescriptor> edesc);
private:
	std::auto_ptr<HttpCache::AbstractResponseHandler> m_ptrCacheHandler;
	IntrusivePtr<AbstractResponseHandler> m_ptrNextHopHandler;
	std::auto_ptr<HttpCache::AbstractResponseWriter> m_ptrWriter;
	ConstRequestPtr m_ptrRequest;
	RequestTag m_requestTag;
	bool m_usingCachedResponse;
	bool m_cachedResponseServedOK;
};


CachingServer::CachingServer(AbstractServer& delegate, bool dont_create)
:	m_rDelegate(delegate),
	m_dontCreate(dont_create)
{
}

CachingServer::~CachingServer()
{
}

IntrusivePtr<AbstractRequestHandler>
CachingServer::submitRequest(
	ConstRequestPtr const& request,
	RequestTag const& request_tag,
	IntrusivePtr<AbstractResponseHandler> const& handler)
{
	HttpCache::RequestResolution resolution(
		HttpCache::ObjectStorage::instance()->handleRequest(
			request, m_dontCreate
		)
	);
	if (resolution.reader.get()) {
		request_tag->flags().set(RequestTag::RESPONSE_CACHED);
		return m_rDelegate.submitRequest(
			request, request_tag, handler,
			std::auto_ptr<ImmediateResponse>(
				new CachedResponse(resolution.reader)
			)
		);
	} else if (resolution.handler.get()) {
		ConstRequestPtr const outgoing_request(
			resolution.handler->getOutgoingRequest()
		);
		return m_rDelegate.submitRequest(
			outgoing_request, request_tag,
			IntrusivePtr<AbstractResponseHandler>(
				new CachingFilter(
					resolution.handler, handler,
					request, request_tag
				)
			)
		);
	} else {
		return m_rDelegate.submitRequest(
			request, request_tag, handler
		);
	}
}

IntrusivePtr<AbstractRequestHandler>
CachingServer::submitRequest(
	ConstRequestPtr const& request_metadata,
	RequestTag const& request_tag,
	IntrusivePtr<AbstractResponseHandler> const& handler,
	std::auto_ptr<ImmediateResponse> response)
{
	return m_rDelegate.submitRequest(
		request_metadata, request_tag, handler, response
	);
}

bool
CachingServer::isIdle() const
{
	return m_rDelegate.isIdle();
}

bool
CachingServer::isOverloaded() const
{
	return m_rDelegate.isOverloaded();
}


/*====================== CachingServer::CachedResponse ===================*/

CachingServer::CachedResponse::CachedResponse(
	std::auto_ptr<HttpCache::AbstractResponseReader> reader)
:	m_ptrReader(reader)
{
}

CachingServer::CachedResponse::~CachedResponse()
{
}

void
CachingServer::CachedResponse::output(
	AbstractResponseHandler& handler,
	ConstRequestPtr const& request,
	RequestStatus& request_status,
	RequestTag const& request_tag)
{
	handler.processResponseMetadata(request_status, m_ptrReader->retrieveMetadata());
	
	BString chunk;
	char* wptr = 0;
	char* eptr = 0;
	SplittableBuffer body_data;
	
	while (!request_status.isCancelled()) {
		if (wptr == eptr) {
			std::auto_ptr<DataChunk> data_chunk(
				DataChunk::create(READ_BUF_SIZE)
			);
			wptr = data_chunk->getDataAddr();
			eptr = wptr + data_chunk->getDataSize();
			BString(data_chunk).swap(chunk);
		}
		ssize_t const res = m_ptrReader->readBodyData(wptr, eptr - wptr);
		if (res > 0) {
			body_data.append(BString(chunk, wptr, wptr + res));
			wptr += res;
		} else if (res < 0) {
			std::string const problem("error reading cached response");
			request_status.cancel(
				ErrorFactory::errGenericError(
					ErrorCodes::ERROR_READING_CACHED_RESPONSE,
					problem, request->requestLine().getURI(),
					problem, HttpStatusLine::SC_INTERNAL_SERVER_ERROR
				)
			);
			return;
		}
		bool const eof = (res == 0);
		handler.processBodyData(request_status, body_data, eof);
		if (eof) {
			break;
		}
	}
}

void
CachingServer::CachedResponse::eventGotProvisionalResponse(
	std::auto_ptr<HttpResponseMetadata>)
{
}

void
CachingServer::CachedResponse::eventGotMetadata(
	std::auto_ptr<HttpResponseMetadata> metadata, bool)
{
	m_ptrResponseMetadata = metadata;
}

/*====================== CachingServer::CachingFilter ====================*/

CachingServer::CachingFilter::CachingFilter(
	std::auto_ptr<HttpCache::AbstractResponseHandler> cache_handler,
	IntrusivePtr<AbstractResponseHandler> const& next_hop_handler,
	ConstRequestPtr const& request, RequestTag const& request_tag)
:	m_ptrCacheHandler(cache_handler),
	m_ptrNextHopHandler(next_hop_handler),
	m_ptrRequest(request),
	m_requestTag(request_tag),
	m_usingCachedResponse(false),
	m_cachedResponseServedOK(false)
{
}

CachingServer::CachingFilter::~CachingFilter()
{
}

void
CachingServer::CachingFilter::processProvisionalResponse(
	RequestStatus& status,
	std::auto_ptr<HttpResponseMetadata> metadata)
{
	m_ptrNextHopHandler->processProvisionalResponse(status, metadata);
}

void
CachingServer::CachingFilter::processResponseMetadata(
	RequestStatus& status,
	std::auto_ptr<HttpResponseMetadata> metadata)
{
	HttpCache::ResponseResolution resolution(
		m_ptrCacheHandler->handleResponse(*metadata)
	);
	if (resolution.reader.get()) {
		m_usingCachedResponse = true;
		m_requestTag->flags().set(RequestTag::RESPONSE_CACHED);
		CachedResponse cached_response(resolution.reader);
		cached_response.output(
			*m_ptrNextHopHandler, m_ptrRequest,
			status, m_requestTag
		);
		m_cachedResponseServedOK = !status.isCancelled();
	} else {
		m_ptrNextHopHandler->processResponseMetadata(status, metadata);
	}
	m_ptrWriter = resolution.writer;
}

void
CachingServer::CachingFilter::processBodyData(
	RequestStatus& status,
	SplittableBuffer& data, bool eof)
{
	if (m_ptrWriter.get()) {
		BString buf(data.toBString());
		while (!buf.empty()) {
			ssize_t const res = m_ptrWriter->writeBodyData(
				buf.begin(), buf.size()
			);
			if (res > 0) {
				buf.trimFront(res);
			} else {
				m_ptrWriter.reset();
				break;
			}
		}
		if (eof && m_ptrWriter.get()) {
			m_ptrWriter->commit();
			m_ptrWriter.reset();
		}
	}
	if (!m_usingCachedResponse) {
		m_ptrNextHopHandler->processBodyData(status, data, eof);
	}
}

void
CachingServer::CachingFilter::processError(
	RequestStatus& status,
	std::auto_ptr<ErrorDescriptor> edesc)
{
	m_ptrWriter.reset();
	if (m_usingCachedResponse && m_cachedResponseServedOK) {
		// This error is about the original response that we don't care about.
		return;
	}
	m_ptrNextHopHandler->processError(status, edesc);
}
