/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef REFPTR_H_
#define REFPTR_H_

#include <CoreFoundation/CoreFoundation.h>

enum Retain { NORETAIN = 0, RETAIN = 1  };

template<typename T>
class RefPtr
{
public:
	RefPtr() : m_pObj(0) {}
	
	RefPtr(RefPtr const& other) : m_pObj(other.m_pObj) { retain(); }
	
	template<typename OT>
	RefPtr(RefPtr<OT> const& other) : m_pObj(other) { retain(); }
	
	template<typename OT>
	RefPtr(OT obj, Retain r) : m_pObj(obj) { if (r) retain(); }
	
	RefPtr& operator=(RefPtr const& other) {
		RefPtr(other).swap(*this);
		return *this;
	}
	
	template<typename OT>
	RefPtr& operator=(RefPtr<OT> const& other) {
		RefPtr(other).swap(*this);
		return *this;
	}
	
	T get() const { return m_pObj; }
	
	template<typename OT>
	RefPtr<OT> const& cast() const {
		return reinterpret_cast<RefPtr<OT> const&>(*this);
	}
	
	operator T() const { return m_pObj; }
	
	void swap(RefPtr& other) {
		T temp = m_pObj;
		m_pObj = other.m_pObj;
		other.m_pObj = temp;
	}
	
	void retain() {
		if (m_pObj) {
			m_pObj = (T)CFRetain(m_pObj);
		}
	}
	
	void release() {
		if (m_pObj) {
			CFRelease(m_pObj);
		}
	}
private:
	/*
	Prevent compiling this type of code:
	RefPtr<CFTypeRef> createSomething();
	RefPtr<CFTypeRef> ptr(createSomething(), NORETAIN);
	*/
	template<typename OT>
	RefPtr(RefPtr<OT> const&, Retain);
	
	T m_pObj;
};

typedef RefPtr<CFTypeRef> TypeRef;
typedef RefPtr<CFStringRef> StringRef;
typedef RefPtr<CFMutableStringRef> MutableStringRef;
typedef RefPtr<CFDictionaryRef> DictRef;
typedef RefPtr<CFMutableDictionaryRef> MutableDictRef;
typedef RefPtr<CFArrayRef> ArrayRef;
typedef RefPtr<CFMutableArrayRef> MutableArrayRef;
typedef RefPtr<CFNumberRef> NumberRef;
typedef RefPtr<CFPropertyListRef> PropListRef;

#endif
